/* $Id: mach64glx.c,v 1.25 2000/03/25 06:52:32 gareth Exp $ */

/*
 * GLX Hardware Device Driver for ATI Rage Pro
 * Copyright (C) 1999 Gareth Hughes
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on MGA driver: mgaglx.c 1.6
 *
 *    Gareth Hughes <gareth@precisioninsight.com>
 */

#include <stdlib.h>

#include "context.h"
#include "depth.h"
#include "macros.h"
#include "texstate.h"
#include "triangle.h"
#include "vb.h"
#include "types.h"

#include "xsmesaP.h"
#include "glx_log.h"
#include "mesaglx/context.h"
#include "mesaglx/matrix.h"
#include "mesaglx/types.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "servermd.h" /* PixmapBytePad */
#include "scrnintstr.h"
#include "regionstr.h"
#include "windowstr.h"
#undef GC

#include "mach64glx.h"

#include "glx_symbols.h"

/* our collected globals */
mach64Glx_t		mach64glx;
mach64BufferPtr		mach64DB = NULL;
mach64ContextPtr	mach64Ctx = NULL;


/*
 * ForceSoftwareBuffers
 * Free any hardware buffers and reallocate only in
 * virtual memory, so hardware acceleration can't be used.
 */
static void ForceSoftwareBuffers( mach64BufferPtr buf ) {
	if ( buf->backBufferBlock ) {
		/* free the hw buffer */
		mmFreeMem( buf->backBufferBlock );
		buf->backBufferBlock = NULL;
	}

	/* allocate a main memory buffer */
	hwMsg( 1, "moving back buffer to system memory to disable hw accel.\n" );
	buf->backBuffer = malloc( buf->pitch * buf->height * mach64glx.bytesPerPixel );
	if ( !buf->backBuffer ) {
		FatalError( "Malloc for back buffer failed" );
	}

	if ( buf->depthBufferBlock ) {
		/* free the hw buffer */
		mmFreeMem( buf->depthBufferBlock );
		buf->depthBufferBlock = NULL;
	}

	/* allocate a main memory buffer */
	hwMsg( 1, "moving depth buffer to system memory to disable hw accel.\n" );
	buf->depthBuffer = malloc( buf->pitch * buf->height * 2 );
	if ( !buf->depthBuffer ) {
		FatalError( "Malloc for depth buffer failed" );
	}
}


void mach64GLXCreateDepthBuffer( GLcontext* ctx ) {
	/* we create the depth buffer when creating the image,
	 * so there is nothing to do here.
	 */
}

/*
 * mach64GLXDestroyImage
 */
void mach64GLXDestroyImage( GLXImage* image ) {
	mach64BufferPtr	buf;

	hwMsg( 1, "mach64GLXDestroyImage( %p )\n", image->devPriv );

	buf = (mach64BufferPtr)image->devPriv;

	/* free memory either in card space or VM */
	if ( buf->backBufferBlock ) {
		mmFreeMem( buf->backBufferBlock );
	} else {
		free( buf->backBuffer );
	}

	if ( buf->depthBufferBlock ) {
		mmFreeMem( buf->depthBufferBlock );
	} else {
		free( buf->depthBuffer );
	}

	buf->magic = 0;
	free( buf );

	if ( hwGetLogLevel() >= 1 )  {
		mmDumpMemInfo( cardHeap );
	}

	xfree( image );
}

/*
 * mach64GLXCreateImage
 *
 * This will always succeed, but if the buffer couldn't be allocated in
 * card memory, HW accel won't be usable.
 *
 * The mach64 doesn't have any flexibility in visual selection.  The
 * back buffer must be the same pixel format as the front buffer and
 * the depth buffer can only be 16 bits.
 */
GLXImage *mach64GLXCreateImage( WindowPtr pwindow, GLvisual *visual,
				int width, int height, GLXImage *old_image )
{
	mach64BufferPtr		buf;
	GLXImage		*image;
	int			depth;

	hwMsg( 1, "mach64GLXCreateImage( %i, %i )\n", width, height );

	/* free the old image */
	if ( old_image && old_image->devPriv ) {
		mach64GLXDestroyImage( old_image );
	}

	depth = visual->RedBits + visual->GreenBits + visual->BlueBits;

	buf = (mach64BufferPtr)calloc(1,sizeof(mach64Buffer));
	if ( !buf ) {
		FatalError( "Malloc for buf failed\n" );
	}

	buf->magic = mach64BufferMagic;
	buf->pwindow = pwindow;
	buf->visual = *visual;
	buf->width = width;
	buf->height = height;
	buf->pitch = ( width + 63 ) & ~63;

	buf->visual.DepthBits = 16;


	/* throw out all texture to make sure we have as much
	 * room as possible.
	 */
	mach64FlushAllTextures();

	/* allocate the back buffer */
	buf->backBufferBlock = mmAllocMem( cardHeap, buf->pitch * buf->height *
					   mach64glx.bytesPerPixel, 7, 0 );

	/* allocate the depth buffer */
 	buf->depthBufferBlock = mmAllocMem( cardHeap, buf->pitch * buf->height * 2, 7, 0 );


	/* fall back to software rendering if buffers didn't fit on card.
	 * it is possible to accelerate some operations with a back
	 * buffer in hardware and a depth buffer in software, but it
	 * will often be slower due to no caching on the framebuffer,
	 * so just force both of them to software
	 */
	if ( buf->backBufferBlock == 0 || buf->depthBufferBlock == 0 ) {
		ForceSoftwareBuffers( buf );
	} else {
		buf->backBuffer = (unsigned char *)
			mach64glx.linearBase + buf->backBufferBlock->ofs;
		buf->depthBuffer = (unsigned char *)
			mach64glx.linearBase + buf->depthBufferBlock->ofs;
		if ( hwGetLogLevel() >= 1 )  {
			mmDumpMemInfo( cardHeap );
		}
	}

	// the image structure is just for GLX's use, the hardware renderer
	// doesn't use it for much of anything

	image = (GLXImage *) xalloc( sizeof(GLXImage) );
	if ( !image ) {
		FatalError( "Malloc for back ximage failed" );
	}
	image->pwin = pwindow;
	// X's drawing routines won't take arbitrary width / pitch combinations,
	// so set width equal to pitch
	image->width = buf->pitch;
	image->height = height;
	image->bits_per_pixel = depth;
	image->data = buf->backBuffer;
	image->bytes_per_line = buf->pitch * mach64glx.bytesPerPixel;
	image->devPriv = buf;

	return image;
}


/*
 * DoMakeCurrent
 * Changing either the context or the buffer will force this.
 */
static void DoMakeCurrent( XSMesaContext c, XSMesaBuffer b )
{
	mach64ContextPtr	ctx;
	mach64BufferPtr		buf;

	hwMsg( 10, "DoMakeCurrent( %p, %p )\n", c, b );

	mach64Ctx = NULL;
	mach64DB = NULL;

	if ( !c || !b ) {
		return;
	}

	ctx = c->hw_ctx;
	if ( !ctx || ctx->magic != mach64ContextMagic ) {
		FatalError( "DoMakeCurrent: ctx->magic != mach64ContextMagic" );
	}

	/* if we don't have a backimage, create one now.  This happens
	 * because of the way software GLX doesn't allocate a
	 * buffer when front buffer rendering, but we need one
	 * anyway
	 */
	if ( !b->backimage ) {
		GLvisual	visual;

		memset( &visual, 0, sizeof( visual ) );
		visual.RedBits = 5;
		visual.GreenBits = 6;
		visual.BlueBits = 5;
		visual.DepthBits = 16;

		b->backimage = mach64GLXCreateImage( (WindowPtr)b->frontbuffer,
						     &visual, b->width, b->height, NULL );
	}

	buf = (mach64BufferPtr) b->backimage->devPriv;
	if ( buf->magic != mach64BufferMagic ) {
		FatalError( "DoMakeCurrent: != mach64BufferMagic" );
	}

	if ( !ctx || !buf ) {
		return;
	}

	/* for development, we can turn off hardware accel on a per-context basis
	 * the most global way to prevent any hardware acceleration is to force the
	 * back buffer into main memory instead of card memory.
	 */
	if ( c->no_accel && ( buf->backBufferBlock || buf->depthBufferBlock ) ) {
		ForceSoftwareBuffers( buf );
	}

	/* valid conditions to render with */
	buf->xsBuf = b;	/* save the xsbuffer so we can get the window for front swaps */
	ctx->DB = buf;
	mach64Ctx = ctx;
	mach64DB = buf;
}


/*
 * mach64GLXMakeCurrent
 * Note that GLX calls this every batch of commands,
 * so it will show up several times a frame even if
 * the client application isn't issuing makeCurrent calls.
 *
 * This is called from several places in glxcmds.c
 */
GLboolean mach64GLXMakeCurrent( XSMesaContext c )
{
	if ( c == XSMesa ) {
		return GL_TRUE;
	}

	/* mesa 3.1 will sometimes leave unflushed vertexes */
	if ( XSMesa ) {
		glFlush();
	}

	hwMsg( 10, "mach64GLXMakeCurrent( %p )\n", c );
	XSMesa = c;

	if ( !c ) {
		mach64Ctx = NULL;
		mach64DB = NULL;
		gl_make_current( NULL, NULL );
		return GL_TRUE;
	}

	DoMakeCurrent( c, c->xsm_buffer );
	gl_make_current( c->gl_ctx,
			 ( c->xsm_buffer ) ? c->xsm_buffer->gl_buffer : NULL );

	if ( mach64Ctx ) {
		mach64_setup_DD_pointers( mach64Ctx->gl_ctx );
	}

	return GL_TRUE;
}

/*
 * Bind buffer b to context c and make c the current rendering context.
 * This is only called by glxcmds.c:GLMakeCurrent()
 */
GLboolean mach64GLXBindBuffer( XSMesaContext c, XSMesaBuffer b )
{
	hwMsg( 10, "mach64GLXBindBuffer( %p, %p )\n", c, b );

#if 0
	if ( !c ) {
		return GL_TRUE;
	}

	c->xsm_buffer = b;
	XSMesa = NULL;

	return mach64GLXMakeCurrent( c );
#else
	DoMakeCurrent( c, b );

	return XSMesaBindBuffer( c, b );
#endif
}



/*
 * Create a new XSMesaContext.
 * Input:  v - XSMesaVisual
 *         share_list - another XSMesaContext with which to share display
 *                      lists or NULL if no sharing is wanted.
 * Return:  an XSMesaContext or NULL if error.
 */
XSMesaContext mach64GLXCreateContext( XSMesaVisual v,
				      XSMesaContext share_list )
{
	XSMesaContext	c;

	hwMsg( 0, "### Creating new xsmesa context for Rage Pro...\n" );

	c = (XSMesaContext) calloc( 1, sizeof(struct xsmesa_context) );
	if ( !c ) {
		return NULL;
	}

	c->gl_ctx = gl_create_context( v->gl_visual,
				       ( share_list ) ? share_list->gl_ctx : NULL,
				       (void *) c,
				       GL_TRUE /* direct rendering */ );
	if ( !c->gl_ctx ) {
		free( c );
		return NULL;
	}

	c->xsm_visual = v;
	c->xsm_buffer = NULL;			/* set later by XSMesaMakeCurrent */
	c->pixelformat = v->dithered_pf;	/* Dithering is enabled by default */
	c->hw_ctx = (mach64ContextPtr) calloc( 1, sizeof(mach64Context) );
	if ( !c->hw_ctx ) {
		hwError( "Cannot create hardware specific context.\n" );
		return NULL;
	}

	((mach64Context *)c->hw_ctx)->magic = mach64ContextMagic;
	((mach64Context *)c->hw_ctx)->gl_ctx = c->gl_ctx;

	/* clear extensions we don't support */
	mach64DDExtensionsInit( c->gl_ctx );

	c->gl_ctx->Driver.UpdateState = mach64_setup_DD_pointers;

	/* Ask mesa to clip fog coordinates for us. */
	c->gl_ctx->TriangleCaps |= DD_CLIP_FOG_COORD;

	/* modify pipeline for fog / cva / etc */
	if ( c->gl_ctx->NrPipelineStages ) {
		c->gl_ctx->NrPipelineStages =
			mach64DDRegisterPipelineStages( c->gl_ctx->PipelineStage,
							c->gl_ctx->PipelineStage,
							c->gl_ctx->NrPipelineStages );
	}

	hwMsg( 1, "mach64GLXCreateContext succeeded: %p\n", c );

	return c;
}



void mach64GLXDestroyContext( XSMesaContext c )
{
	hwMsg( 1, "mach64GLXDestroyContext( %p )\n", c );

	/* make sure all drawing is completed */
	mach64DmaFinish();

	if ( c->hw_ctx == mach64Ctx ) {
		mach64Ctx = NULL;
	}
	if ( XSMesa == c ) {
		XSMesa = 0;
	}

	((mach64Context *)c->hw_ctx)->magic = 0;
	free( c->hw_ctx );

	XSMesaDestroyContext( c );
}


/*
 * Local Variables:
 * mode: c
 * tab-width: 8
 * c-basic-offset: 8
 * End:
 */
