/* $Id: mach64clear.c,v 1.16 2000/03/25 06:52:32 gareth Exp $ */

/*
 * GLX Hardware Device Driver for ATI Rage Pro
 * Copyright (C) 1999 John Carmack, Gareth Hughes
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *    John Carmack <johnc@idsoftware.com>
 *    Gareth Hughes <gareth@precisioninsight.com>
 */

#include "mach64glx.h"
#include "glx_symbols.h"
#include "hw_buffer.h"

/*
 * mach64Clear
 * perform hardware accelerated clearing of the color and/or depth
 * buffer.  Software may still clear stencil buffers.
 * If all==GL_TRUE, clear whole buffer, else just clear region defined
 * by x,y,width,height
 */
GLbitfield mach64Clear( GLcontext *ctx, GLbitfield mask, GLboolean all,
			GLint x, GLint y, GLint width, GLint height )
{
	DMALOCALS;
	int		screenFormat;

	hwMsg( 10, "mach64Clear( %i, %i, %i, %i, %i )\n", mask, x, y, width, height );

	/* if we are software rendering, we must clear the buffer ourself.
	 * mesa doesn't handle it in the fallback.
	 */
	if ( !mach64DB->backBufferBlock ) {
		return FallbackBufferClear( ctx, mask, all, x, y, width, height );
	}

	if (all == GL_TRUE) {
		x = 0;
		y = 0;
		width = mach64DB->width;
		height = mach64DB->height;
	}

	if ( y + height > mach64DB->height ) {
		height = mach64DB->height - y;
	}
	if ( x + width > mach64DB->width ) {
		width = mach64DB->width - x;
	}
	if ( x < 0 ) {
		width += x;
		x = 0;
	}
	if ( y < 0 ) {
		height += y;
		y = 0;
	}
	if ( x >= mach64DB->width || y >= mach64DB->height || width < 1 || height < 1 ) {
		return 0;
	}

	/* bump the performance counter */
	mach64glx.c_clears++;

	/* flip top to bottom */
	y = mach64DB->height - y - height;

	DMAGETPTR( 100 );

	/* color buffer */
	if ( mask & GL_COLOR_BUFFER_BIT ) {
		int		r, g, b,a;
		int		color;

		r = 255 * ctx->Color.ClearColor[0];
		g = 255 * ctx->Color.ClearColor[1];
		b = 255 * ctx->Color.ClearColor[2];
		a = 255 * ctx->Color.ClearColor[3];

		color = mach64PackColor( r, g, b, a );

		switch( mach64glx.depth ) {
		case 15: screenFormat = 3; break;
		case 16: screenFormat = 4; break;
		case 32: screenFormat = 6; break;
		}

		/* hardware accelerated clear */
		DMAOUTREG( MACH64_Z_CNTL, 0 );
		DMAOUTREG( MACH64_SCALE_3D_CNTL, 0 );

		DMAOUTREG( MACH64_SC_LEFT_RIGHT, 0 | ( ( mach64DB->width - 1 ) << 16 ) );
		DMAOUTREG( MACH64_SC_TOP_BOTTOM, 0 | ( ( mach64DB->height - 1 ) << 16 )  );

		DMAOUTREG( MACH64_CLR_CMP_CNTL, 0 );			/* disable color compare */
		DMAOUTREG( MACH64_GUI_TRAJ_CNTL, DST_X_LEFT_TO_RIGHT | DST_Y_TOP_TO_BOTTOM );

		DMAOUTREG( MACH64_DP_PIX_WIDTH,
			   ( screenFormat << 0 )
			   | ( screenFormat << 4 )
			   | ( screenFormat << 8 )
			   | ( screenFormat << 16 )
			   | ( screenFormat << 28 )
			   );

		DMAOUTREG( MACH64_DP_FRGD_CLR, color );
		DMAOUTREG( MACH64_DP_WRITE_MASK, 0xffffffff );		/* write to all */
		DMAOUTREG( MACH64_DP_MIX, BKGD_MIX_D | FRGD_MIX_S );	/* bkground leave alone */
		DMAOUTREG( MACH64_DP_SRC, BKGD_SRC_FRGD_CLR | FRGD_SRC_FRGD_CLR | MONO_SRC_ONE );

		DMAOUTREG( MACH64_DST_OFF_PITCH,
			   ( (mach64DB->pitch/8) << 22 ) | (mach64DB->backBufferBlock->ofs>>3) ) ;
		DMAOUTREG( MACH64_DST_X_Y, (y << 16) | x );
		DMAOUTREG( MACH64_DST_WIDTH_HEIGHT, (height << 16) | width );

		mask &= ~GL_COLOR_BUFFER_BIT;
	}

	/* depth buffer */
	if ( mask & GL_DEPTH_BUFFER_BIT ) {
		if ( 1 /* !ctx->Depth.Mask FIXME: check this! */ ) {
			hwUI32	zval;

			zval = (hwUI32) (ctx->Depth.Clear * DEPTH_SCALE);

			if ( mach64DB->depthBufferBlock ) {
				/* hardware accelerated clear */
				DMAOUTREG( MACH64_Z_CNTL, 0 );
				DMAOUTREG( MACH64_SCALE_3D_CNTL, 0 );

				DMAOUTREG( MACH64_SC_LEFT_RIGHT, 0 | ( ( mach64DB->width - 1 ) << 16 ) );
				DMAOUTREG( MACH64_SC_TOP_BOTTOM, 0 | ( ( mach64DB->height - 1 ) << 16 )  );

				DMAOUTREG( MACH64_CLR_CMP_CNTL, 0 );		/* disable color compare */
				DMAOUTREG( MACH64_GUI_TRAJ_CNTL, DST_X_LEFT_TO_RIGHT | DST_Y_TOP_TO_BOTTOM );

				/* depth buffer is always 16 bit */
				DMAOUTREG( MACH64_DP_PIX_WIDTH,
					   ( 3 << 0 )
					   | ( 3 << 4 )
					   | ( 3 << 8 )
					   | ( 3 << 16 )
					   | ( 3 << 28 )
					   );

				DMAOUTREG( MACH64_DP_FRGD_CLR, zval );
				DMAOUTREG( MACH64_DP_WRITE_MASK, 0xffffffff );		/* write to all */
				DMAOUTREG( MACH64_DP_MIX, BKGD_MIX_D | FRGD_MIX_S );	/* bkground leave alone */
				DMAOUTREG( MACH64_DP_SRC, BKGD_SRC_FRGD_CLR | FRGD_SRC_FRGD_CLR | MONO_SRC_ONE );

				DMAOUTREG( MACH64_DST_OFF_PITCH,
					   ( (mach64DB->pitch/8) << 22 ) | (mach64DB->depthBufferBlock->ofs>>3) ) ;
				DMAOUTREG( MACH64_DST_X_Y, (y << 16) | x );
				DMAOUTREG( MACH64_DST_WIDTH_HEIGHT, (height << 16) | width );
			}
		}
		mask &= ~GL_DEPTH_BUFFER_BIT;
	}

	DMAADVANCE();

	return mask;
}


/*
 * Local Variables:
 * mode: c
 * tab-width: 8
 * c-basic-offset: 8
 * End:
 */
