/*
  GLX Hardware Device Driver for Intel i810
  Copyright (C) 1999 Keith Whitwell <keithw@precisioninsight.com>
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
 original by Jeff Hartmann <slicer@ionet.net>
  6/16/99: rewrite by John Carmack <johnc@idsoftware.com>
  Oct 99: port to i180 by Keith Whitwell <keithw@precisioninsight.com>
*/

#ifndef I810DMA_H
#define I810DMA_H

#include "i810common.h"
#include "i810lib.h"
#include "i810.h"
#include "mm.h"

/* the main initialization of the entire i810 hardware driver */
GLboolean i810InitGLX( void );

/* a flush command will guarantee that all data added to the dma buffer
is on its way to the card, and will eventually complete with no more
intervention.
*/
void i810DmaFlush(void);

/* the overflow function is called when a block can't be allocated
in the current dma buffer.  It flushes the current buffer and
records some information */
void i810DmaOverflow(int newDwords);

/* a finish command will guarantee that all dma commands have actually
been consumed by the card.  Note that there may still be a couple primitives
that have not yet been executed out of the internal FIFO, so this does not
guarantee that the drawing engine is idle. */
void i810DmaFinish( void );

/* a i810WaitDrawingEngine command will guarantee that the framebuffer
is safe to read or write for software rendering */
int i810WaitDrawingEngine( void );





typedef struct {
   I810MemRange mem;
   char *virtual_start;
   int head;
   int space;
   int additional_space;
   int texture_age;
} i810BatchBuffer;


#define I810_NO_FLUSH 0
#define I810_USE_BATCH 1


#if I810_USE_BATCH

#define BEGIN_BATCH(n)							\
   unsigned int outbatch;						\
   volatile char *virt;							\
   if (I810_DEBUG & DEBUG_VERBOSE_RING)					\
      fprintf(stderr,							\
      "BEGIN_BATCH(%d) in %s\n"						\
      "(spc left %d/%d, head %x vstart %x start %x)\n",			\
      n, __FUNCTION__, i810glx.dma_buffer->space,			\
      i810glx.dma_buffer->mem.Size - i810glx.dma_buffer->head,		\
      i810glx.dma_buffer->head,						\
      i810glx.dma_buffer->virtual_start,				\
      i810glx.dma_buffer->mem.Start );					\
   if (i810glx.dma_buffer->space < n*4)					\
       i810DmaOverflow(n);						\
   outbatch = i810glx.dma_buffer->head;					\
   virt = i810glx.dma_buffer->virtual_start;			


#define OUT_BATCH(val) {					\
   *(volatile unsigned int *)(virt + outbatch) = val;		\
   if (I810_DEBUG & DEBUG_VERBOSE_RING)				\
      fprintf(stderr, "OUT_BATCH %x: %x\n", outbatch/4, val);	\
   outbatch += 4;						\
}
   

#define ADVANCE_BATCH() {						\
   if (I810_DEBUG & DEBUG_VERBOSE_RING)					\
      fprintf(stderr, "ADVANCE_BATCH(%f) in %s\n",			\
                  (outbatch - i810glx.dma_buffer->head) / 4.0,		\
                  __FUNCTION__);					\
   i810glx.dma_buffer->space -= outbatch - i810glx.dma_buffer->head;	\
   i810glx.dma_buffer->head = outbatch;					\
}

#define FINISH_PRIM()

#else

/* Need some work to be able to do primitives this way.
 */
#define BEGIN_BATCH(n) BEGIN_LP_RING(n)
#define ADVANCE_BATCH() ADVANCE_LP_RING()
#define OUT_BATCH(val) OUT_RING(val)
#define FINISH_PRIM() OUTREG(GLXSYM(I810LpRing).base_reg + RING_TAIL, GLXSYM(I810LpRing).tail)

#endif


/* These are only required by i810direct.c:
 */
extern GLuint i810ActiveDmaBuffer;
extern int (*i810DoDmaFlush)( int, int * );
extern i810BatchBuffer *dmaBuffers[2];

extern void i810DmaResetBuffer( void );

extern void i810DmaExecute( GLuint *code, int dwords );

extern void I810Sync( void ) ;
extern void I810WaitLpRing( int n );


#endif
