#include <X11/Xlib.h>
#include <X11/Xlibint.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>

#undef Bool
#include "dixstruct.h"
#include "glxcommon.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "servermd.h" /* PixmapBytePad */
#include "scrnintstr.h"
#include "regionstr.h"
#include "windowstr.h"
#undef GC

#include "X.h"
#include "input.h"
#include "scrnintstr.h"

#include "xf86.h"
#include "xf86Priv.h"
#include "xf86_OSlib.h"

#include <errno.h>

#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_SYS_MMAN_H
# include <sys/mman.h>
#endif

#include <dlfcn.h>
#include "glxvisual.h"

#include "glx_clients.h"
#include "glx_init.h"

#define GLX_SYM_NO_DEFINES
#include "glx_symbols.h"
#include "x11_symbols.h"

#define MAX_DIRECT_RESOURCE 20

typedef struct {
   GCPtr ptr;
   GC gc;
   XGCValues xgcv;
   unsigned int mask;
} gc_map;


static GC find_client_gc_from_ptr( void *ptr );


static ScreenRec directScreen;
static gc_map gc_list[MAX_DIRECT_RESOURCE];
static int nr_active_gcs = 0;
static ExtensionEntry directExtension;
static ScreenInfo screenInfo;
static PaddingInfo PixmapWidthPaddingInfo[33];

static Display *clientDisplay;
static int clientScreenNum;

static struct {
   XPixmapFormatValues *(*XListPixmapFormats)(Display*, int*);
   XVisualInfo *(*XGetVisualInfo)(Display*, long, XVisualInfo*, int*);
   Status (*XAllocColor)(Display*, Colormap, XColor*);
   int (*XQueryColors)(Display*, Colormap, XColor*, int); 
   int (*XChangeGC)(Display*, GC, unsigned long, XGCValues*);
   int (*XFreeGC)(Display*, GC);
   GC (*XCreateGC)(Display*, Drawable, unsigned long, XGCValues*); 
   Status (*XGetGeometry)(Display*, Drawable, Window*, int*,
			  int*, unsigned int*, unsigned int*, 
			  unsigned int*, unsigned int*);
   XErrorHandler (*XSetErrorHandler) (XErrorHandler);
   Status (*XGetWindowAttributes)(Display*, Window, XWindowAttributes*);
   int (*XDrawPoints)(Display*, Drawable, GC, XPoint*, int, int);
} x11sym;

#define X11SYM(x) (*(x11sym.x))

static int any_error;
static void *my_dlsym(void *handle, const char *name)
{
   void *tmp = dlsym(handle, name);
   char *error;
   if ((error = dlerror()) != 0) {
      fputs(error, stderr);
      any_error = 1;
      return 0;
   }
   return tmp;
}

int glxHookX11ClientSymbols( void )
{
   void *handle = dlopen (0, RTLD_LAZY);
   
   if (!handle) {
      fputs (dlerror(), stderr);
      fprintf(stderr, 
	      "Cannot self-dlopen client - Direct rendering disabled\n");
      return 0;
   }   

   x11sym.XListPixmapFormats = my_dlsym(handle, "XListPixmapFormats");
   x11sym.XGetVisualInfo = my_dlsym(handle, "XGetVisualInfo");
   x11sym.XAllocColor = my_dlsym(handle, "XAllocColor");
   x11sym.XQueryColors = my_dlsym(handle, "XQueryColors");
   x11sym.XChangeGC = my_dlsym(handle, "XChangeGC");
   x11sym.XFreeGC = my_dlsym(handle, "XFreeGC");
   x11sym.XCreateGC = my_dlsym(handle, "XCreateGC");
   x11sym.XGetGeometry = my_dlsym(handle, "XGetGeometry");
   x11sym.XSetErrorHandler = my_dlsym(handle, "XSetErrorHandler"); 
   x11sym.XGetWindowAttributes = my_dlsym(handle, "XGetWindowAttributes");
   x11sym.XDrawPoints = my_dlsym(handle, "XDrawPoints");

   glxsym.WriteToClient = my_dlsym(handle, "WriteToClient");
   glxsym.FlushAllOutput = my_dlsym(handle, "FlushAllOutput");

   if (!any_error) 
      fprintf(stderr, "Sucessfully loaded X11 client symbols\n");
   else
      fprintf(stderr, "X11 client symbol(s) not found\n");

   dlclose(handle);
   return !any_error;
}


static GC find_client_gc_from_ptr( void *ptr )
{
   int i;

   for (i = 0 ; i < nr_active_gcs ; i++)
      if (gc_list[i].ptr == ptr) {
	 if (gc_list[i].mask) {
	    X11SYM(XChangeGC)( clientDisplay,
			       gc_list[i].gc,
			       gc_list[i].mask,
			       &gc_list[i].xgcv );
	    
	    gc_list[i].mask = 0;
	 }
	 return gc_list[i].gc;
      }

   return 0;
}

static gc_map *find_gc_map_from_ptr( void *ptr )
{
   int i;

   for (i = 0 ; i < nr_active_gcs ; i++)
      if (gc_list[i].ptr == ptr) 
	 return &gc_list[i];

   return 0;
}



typedef struct {
   DrawablePtr ptr;
   Drawable drawable;
} drawable_map;

typedef struct {
   PixmapPtr ptr;
   Pixmap pixmap;
} pixmap_map;

typedef struct {
   WindowPtr ptr;
   Window window;
} window_map;

static drawable_map drawable_list[MAX_DIRECT_RESOURCE];
static int nr_active_drawables = 0;

static pixmap_map pixmap_list[MAX_DIRECT_RESOURCE];
static int nr_active_pixmaps = 0;

static window_map window_list[MAX_DIRECT_RESOURCE];
static int nr_active_windows = 0;


static Drawable find_client_drawable_from_ptr( void *ptr )
{
   int i;

   for (i = 0 ; i < nr_active_drawables ; i++)
      if (drawable_list[i].ptr == ptr) 
	 return drawable_list[i].drawable;

   fprintf(stderr, "couldn't find drawable matching ptr %p\n", ptr);
   return 0;
}

static DrawablePtr build_pixmap_rec( int id )
{
   PixmapPtr pPix;
   Window dummy;
   int x, y;
   unsigned int width_return, height_return;
   unsigned int border_width_return;
   unsigned int depth_return;

   if (nr_active_drawables == MAX_DIRECT_RESOURCE) 
      return 0;

   pPix = (PixmapPtr) calloc(1, sizeof(PixmapRec));

   pPix->drawable.type = DRAWABLE_PIXMAP;
   pPix->drawable.pScreen = &directScreen;
   pPix->drawable.id = id;

   if (!X11SYM(XGetGeometry)( clientDisplay, 
			      id,
			      &dummy,
			      &x, &y,
			      &width_return, &height_return,
			      &border_width_return,
			      &depth_return ))
   {
      free(pPix);
      fprintf(stderr, "couldn't get geometry for pixmap id %d\n", id);
      return 0;
   }

   pPix->drawable.x = x;
   pPix->drawable.y = y;
   pPix->drawable.width = width_return;
   pPix->drawable.height = height_return;
   pPix->drawable.depth = depth_return;
   pPix->drawable.bitsPerPixel = depth_return; /* hmm */

   pixmap_list[nr_active_pixmaps].pixmap = id;
   pixmap_list[nr_active_pixmaps].ptr = pPix;
   nr_active_pixmaps++;

   drawable_list[nr_active_drawables].drawable = id;
   drawable_list[nr_active_drawables].ptr = (DrawablePtr) pPix;
   nr_active_drawables++;

   return (DrawablePtr) pPix;
}

static DrawablePtr build_window_rec( int id, XWindowAttributes *xwa )
{
   WindowPtr pWin = calloc(1, sizeof(WindowRec));

   if (nr_active_drawables == MAX_DIRECT_RESOURCE) 
      return 0;

   if (xwa->class == InputOnly) {
      pWin->drawable.type = UNDRAWABLE_WINDOW;
   } else {
      pWin->drawable.type = DRAWABLE_WINDOW;
   }

   pWin->drawable.pScreen = &directScreen;
   pWin->drawable.id = id;
   pWin->drawable.x = xwa->x;
   pWin->drawable.y = xwa->y;
   pWin->drawable.width = xwa->width;
   pWin->drawable.height = xwa->height;
   pWin->drawable.depth = xwa->depth;
   pWin->drawable.bitsPerPixel = xwa->depth;

   pWin->optional = (WindowOptPtr) calloc(sizeof(WindowOptRec), 1);
   pWin->optional->visual = xwa->visual->visualid;
   pWin->optional->colormap = xwa->colormap;
   pWin->parent = 0;

   window_list[nr_active_windows].window = id;
   window_list[nr_active_windows].ptr = pWin;
   nr_active_windows++;

   drawable_list[nr_active_drawables].drawable = id;
   drawable_list[nr_active_drawables].ptr = (DrawablePtr) pWin;
   nr_active_drawables++;

   return (DrawablePtr) pWin;
}

static int x_error;

static int tmp_handler( Display *dpy, XErrorEvent *ev )
{
   x_error = 1;
   return 0;
}


static DrawablePtr find_server_drawable_from_id( int id )
{
   int i;
   int (*handler)(Display *, XErrorEvent *);
   XWindowAttributes xwa;
   DrawablePtr pDraw;
   
   for (i = 0 ; i < nr_active_drawables ; i++)
      if (drawable_list[i].drawable == id)
	 return drawable_list[i].ptr;


   /* For the purposes of the driver, just need to establish
    * whether this an InputOnly window or not.
    */
   x_error = 0;
   handler = X11SYM(XSetErrorHandler)(tmp_handler);
   if (!X11SYM(XGetWindowAttributes)( clientDisplay, id, &xwa )) {
      x_error = 1;
   }
   X11SYM(XSetErrorHandler)(handler);
   
   if (x_error) {
      pDraw = build_pixmap_rec( id );
   } else {
      pDraw = build_window_rec( id, &xwa );
   }

   return pDraw;
}


static PixmapPtr find_server_pixmap_from_id( int id )
{
   int i;

   for (i = 0 ; i < nr_active_pixmaps ; i++)
      if (pixmap_list[i].pixmap == id)
	 return pixmap_list[i].ptr;

   fprintf(stderr, "couldn't find pixmap ptr for id %d\n", id);

   return 0;
}


/* Always build the optional struct, so this should never be called.
 */
static WindowPtr FindWindowWithOptional(WindowPtr w)
{
   fprintf(stderr, "FindWindowWithOptional called\n");
   abort();
   return 0;
}



#define MAX_DIRECT_RESOURCE_TYPE 5

typedef struct {
   void *ptr;
   int id;
} resource_map;

static resource_map res_maps[MAX_DIRECT_RESOURCE_TYPE][MAX_DIRECT_RESOURCE];
static int res_nr[MAX_DIRECT_RESOURCE_TYPE];
static int max_resource_type;


static pointer LookupIDByType(XID id, RESTYPE rtype)
{
   /* For pre-existing resourse types, check the local database for an
    * existing entry.  If none exists, query the server.  The hole in
    * this scheme is that we never learn of updates, and the X
    * protocol doesn't give us all the information anyway.  However we
    * get enough to get by.  Buffer swapping has to be done by the
    * server though.  
    */
   switch (rtype) {
   case RT_COLORMAP:
      return (ColormapPtr) id;	/* hello sailor */
   case RT_PIXMAP:
      return find_server_pixmap_from_id( id );
   case RT_FONT:
      return 0;			/* not implemented [need new proto request] */
   case RT_WINDOW:
   case RT_GC:
   case RT_CURSOR:
   case RT_CMAPENTRY:
   case RT_OTHERCLIENT:
   case RT_PASSIVEGRAB:
   case RT_LASTPREDEF:
   case RT_NONE:
      return 0;			/* not used in glx.so */
   default:
      /* It's one of ours:
       */
      rtype -= RT_LASTPREDEF + 1;
      if (rtype >= 0 && rtype < max_resource_type) {
	 resource_map *array = res_maps[rtype];
	 int i;
	 for (i = 0 ; i < res_nr[rtype] ; i++)
	    if (array[i].id == id) 
	       return array[i].ptr;
      }

      return 0;
   }
}

static pointer LookupIDByClass(XID id, RESTYPE classes)
{
   switch (classes) {
   case RC_DRAWABLE:
      return find_server_drawable_from_id( id );
   default:
      return 0;
   }
}

static pointer SecurityLookupIDByClass(ClientPtr client, XID id, 
				       RESTYPE rtype, Mask bleagh)
{
   return(LookupIDByClass(id, rtype));
}

static pointer SecurityLookupIDByType(ClientPtr client, XID id, 
				      RESTYPE rtype, Mask bleagh)
{
   return(LookupIDByType(id, rtype));
}

static RESTYPE CreateNewResourceType(DeleteType deleteFunc)
{
   if (max_resource_type == MAX_DIRECT_RESOURCE_TYPE)
      return 0;

   return RT_LASTPREDEF + ++max_resource_type;
}

static Bool AddResource(XID id, RESTYPE rtype, pointer value)
{
   rtype -= RT_LASTPREDEF + 1;
   if (rtype >= 0 && rtype < max_resource_type) {
      resource_map *array = res_maps[rtype];
      int idx;
      
      if (res_nr[rtype] == MAX_DIRECT_RESOURCE) {
	 fprintf(stderr, "resource array %ld full\n", rtype);
	 return 0;
      }

      idx = res_nr[rtype]++;
      array[idx].id = id;
      array[idx].ptr = value;
      return 1;
   }

   return 0;
}

static void FreeResource(XID id, RESTYPE rtype)
{
   rtype -= RT_LASTPREDEF + 1;
   if (rtype >= 0 && rtype < max_resource_type) {
      resource_map *array = res_maps[rtype];
      int i;
      
      for (i = 0 ; i < res_nr[rtype] ; i++)
	 if (array[i].id == id) {
	    res_nr[rtype]--;
	    array[i] = array[res_nr[rtype]];
	    return;
	 }
   }
}

static int LocalClient(ClientPtr client)
{
   return 1;
}

static ExtensionEntry *AddExtension(char* name,
				    int NumEvents,
				    int NumErrors,
				    int (*MainProc)(ClientPtr),
				    int (*SwappedMainProc)(ClientPtr),
				    void (*CloseDownProc)(ExtensionEntry *),
				    unsigned short (*MinorOpProc)(ClientPtr))
{
   directExtension.MinorOpcode = MinorOpProc;
   directExtension.CloseDown = CloseDownProc;
   directExtension.eventBase = 0;
   directExtension.eventLast = NumEvents;
   directExtension.errorBase = 0;
   directExtension.errorLast = NumErrors;
   directExtension.name = name;
   return &directExtension;
}

static unsigned short StandardMinorOpcode(ClientPtr client)
{
   return 1;			
}



static void fake_PolyPoint(DrawablePtr pDrawable,
			   GCPtr pGC, int mode, int npt, DDXPointPtr pptInit)
{
   GC gc = find_client_gc_from_ptr( pGC );
   Drawable d = find_client_drawable_from_ptr( pDrawable );

   if (gc && d) {
      XPoint *points = (XPoint *)pptInit;
      X11SYM(XDrawPoints)( clientDisplay, d, gc, points, npt, mode );
   }
}

static void nop ( void )
{
}



static struct _GCOps fake_gcops = {
   (void *)nop, (void *)nop, (void *)nop, (void *)nop, (void *)nop,
   fake_PolyPoint, 
   (void *)nop, (void *)nop, (void *)nop, (void *)nop,
   (void *)nop, (void *)nop, (void *)nop, (void *)nop, (void *)nop, 
   (void *)nop, (void *)nop, (void *)nop, (void *)nop, (void *)nop,
#ifdef NEED_LINEHELPER
   0,
#endif
   { 0 }
};


static int DoChangeGC(GCPtr pGC, BITS32 mask, XID* pval, int fPointer)
{
   gc_map *gc = find_gc_map_from_ptr( pGC );

   if (!gc) return 0;

   switch (mask) {
   case GCFunction:
      gc->xgcv.function = * pval;
      break;
   case GCForeground:
      gc->xgcv.foreground = * pval;
      break;
   case GCPlaneMask:
      gc->xgcv.plane_mask = * pval;
      break;
   default:
      /* Since I'm not even pretending that this is complete, I'll
       * leave it at that.
       */
      fprintf(stderr, "non-implemented fake gc mask %x\n", (int) mask);
      return 1;
   }

   gc->mask |= mask;

   return 1;
}



static void ValidateGC(DrawablePtr pDraw, GCPtr pGC)
{
}

static GCPtr GetScratchGC(unsigned depth, ScreenPtr pScreen)
{
   Display *dpy = clientDisplay;
   Drawable d = RootWindow(dpy, clientScreenNum);
   XGCValues gcv;
   GCPtr pGC;
   
   if (pScreen->rootDepth != depth)
      return 0;

   if (nr_active_gcs == MAX_DIRECT_RESOURCE) 
      return 0;

   gc_list[nr_active_gcs].gc = X11SYM(XCreateGC)( dpy, d, 0, &gcv );

   if (!gc_list[nr_active_gcs].gc) 
      return 0;

   pGC = (GCPtr) calloc( sizeof(*pGC), 1 );
   pGC->ops = &fake_gcops;
   pGC->depth = depth;

   gc_list[nr_active_gcs].ptr = pGC;
   gc_list[nr_active_gcs].mask = 0;
   nr_active_gcs++;
   return pGC;
}


static void FreeScratchGC(GCPtr pGC)
{
   int i;
   
   for (i = 0 ; i < nr_active_gcs ; i++)
      if (gc_list[i].ptr == pGC) {
	 X11SYM(XFreeGC)( clientDisplay, gc_list[i].gc );
	 free(pGC);
	 nr_active_gcs--;
	 if (nr_active_gcs > i) 
	    gc_list[i] = gc_list[nr_active_gcs];
      }	       
}

static int AllocColor(ColormapPtr pmap,
		      unsigned short* pred,
		      unsigned short* pgreen,
		      unsigned short* pblue,
		      Pixel* pPix,
		      int client)
{
   Status rv = -1;
   Colormap cmap = 0;
   XColor col = { *pPix, *pred, *pgreen, *pblue, ~0, 0 };
   Display *dpy = clientDisplay;
   int scr = clientScreenNum;

   if (!cmap) cmap = DefaultColormap(dpy, scr);

   rv = X11SYM(XAllocColor)(dpy, cmap, &col);

   *pPix = col.pixel;
   *pred = col.red;
   *pgreen = col.green;
   *pblue = col.blue;

   return rv;
}

static int QueryColors(ColormapPtr pmap,
		       int count,
		       Pixel* ppixIn,
		       xrgb* prgbList)
{
   Colormap cmap = 0;
   XColor *colors = (XColor *) malloc(count * sizeof(XColor));
   int rv, i;

   if (!cmap) cmap = DefaultColormap(clientDisplay, clientScreenNum);

   for (i = 0 ; i < count ; i++) 
      colors[i].pixel = ppixIn[i];

   rv = X11SYM(XQueryColors)( clientDisplay, cmap, colors, count );
		 
   for (i = 0 ; i < count ; i++) {
      prgbList[i].red = colors[i].red;
      prgbList[i].green = colors[i].green;
      prgbList[i].blue = colors[i].blue;
   }

   free(colors);
   return rv;
}



static void ErrorF(char* f, ...)
{
   va_list ap;
   va_start(ap, f);
   (void) vfprintf(stderr, f, ap);
   va_end(ap);
}

static void FatalError(char* f, ...)
{
   va_list ap;
   va_start(ap, f);
   (void) vfprintf(stderr, f, ap);
   va_end(ap);
   exit(1);
}

static unsigned long *Xalloc(unsigned long amount)
{
   return (unsigned long *)malloc(amount);
}

static void Xfree(void * ptr)
{
   free(ptr);
}

static XID FakeClientID(int client)
{
   static int i;
   return ++i;
}


static void InstallLocalXFree86Symbols( void )
{
   glxsym.AddExtension = &AddExtension;
   glxsym.AddResource = &AddResource;
   glxsym.CreateNewResourceType = &CreateNewResourceType;
   glxsym.FreeResource = &FreeResource;
   glxsym.LookupIDByClass = &LookupIDByClass;
   glxsym.LookupIDByType = &LookupIDByType;
   glxsym.SecurityLookupIDByClass = &SecurityLookupIDByClass;
   glxsym.SecurityLookupIDByType = &SecurityLookupIDByType;
   glxsym.AllocColor = &AllocColor;
   glxsym.QueryColors = &QueryColors;
   glxsym.ErrorF = &ErrorF;
   glxsym.FatalError = &FatalError;
   glxsym.FreeScratchGC = &FreeScratchGC;
   glxsym.GetScratchGC = &GetScratchGC;
   glxsym.DoChangeGC = &DoChangeGC;
   glxsym.ValidateGC = &ValidateGC;
/*     glxsym.GlxExtensionInitPtr = &GlxExtensionInitPtr; */
/*     glxsym.GlxInitVisualsPtr = &GlxInitVisualsPtr; */
   glxsym.FakeClientID = &FakeClientID;
   glxsym.FindWindowWithOptional = &FindWindowWithOptional;
   glxsym.LocalClient = &LocalClient;
   glxsym.Xalloc = &Xalloc;
   glxsym.Xfree = &Xfree;
   glxsym.PixmapWidthPaddingInfo = &PixmapWidthPaddingInfo;
   glxsym.screenInfo = &screenInfo;
   glxsym.StandardMinorOpcode = &StandardMinorOpcode;
}

static int mylog2( unsigned int foo )
{
   int val = 0;
   while (foo>>=1) val++;
   return val;
}

static ClientRec fake_client;

ClientPtr glxInitDirectClient( void *dpy, int scr, int first_visual,
			       int server, int chipset )
{
   int nvi = 0;
   XVisualInfo viproto;
   XVisualInfo *vip;
   Screen *screen = &((Display *)dpy)->screens[scr];
   XPixmapFormatValues *pmf;
   int i, n;

   __glx_is_server = 0;
   __glx_first_visual = first_visual;
   glx_server = server;
   glx_chipset = chipset;

   InstallLocalXFree86Symbols();

   clientDisplay = (Display *)dpy;
   clientScreenNum = scr;

   direct_client = &fake_client;
   direct_client->index = 1;
          
   /* Basic info
    */
   directScreen.myNum = scr;	/* index into Screens, but Screens not ref'd */
   directScreen.id = scr;	/* i suppose  */
   directScreen.width = screen->width;
   directScreen.height = screen->height;
   directScreen.mmWidth = screen->mwidth;
   directScreen.mmHeight = screen->mheight;
   

   /* Build the list of depths.
    */
   directScreen.numDepths = screen->ndepths;
   directScreen.allowedDepths = malloc( sizeof(DepthRec) * screen->ndepths );
   directScreen.rootDepth = screen->root_depth;


   for (i = 0 ; i < screen->ndepths ; i++) {
      int j;
      DepthPtr d = &directScreen.allowedDepths[i];

      d->depth = screen->depths[i].depth;
      d->numVids = screen->depths[i].nvisuals;
      d->vids = (VisualID *) malloc( sizeof(VisualID) *
				     screen->depths[i].nvisuals );

      for (j = 0 ; j < screen->depths[i].nvisuals ; j++) 
	 d->vids[j] = screen->depths[i].visuals[j].visualid;
   }   

   directScreen.rootVisual = screen->root_visual->visualid;
   directScreen.defColormap = screen->cmap;  /* same namespace? doubt it */
   directScreen.minInstalledCmaps = screen->min_maps;
   directScreen.maxInstalledCmaps = screen->max_maps;
   directScreen.backingStoreSupport = screen->backing_store;
   directScreen.saveUnderSupport = screen->save_unders;
   directScreen.whitePixel = screen->white_pixel;
   directScreen.blackPixel = screen->black_pixel;
   directScreen.rgf = 0;	/* hungarian??? wtf??? */
   directScreen.PixmapPerDepth[0] = 0; /* not used */
   directScreen.devPrivate = 0;	/* not used */

   /* build the list of visuals.
    */
   nvi = 0;
   viproto.screen = scr;
   vip = X11SYM(XGetVisualInfo) (dpy, VisualScreenMask, &viproto, &nvi);

   if (!vip) 
      fprintf(stderr, "no visuals!\n");
   else {
      directScreen.numVisuals = nvi;
      directScreen.visuals = (VisualPtr) malloc( sizeof(VisualRec) * nvi );

      for (i = 0; i < nvi; i++) {
	 int j,k;
	 VisualPtr v = &directScreen.visuals[i];
	 v->vid = vip[i].visualid;
	 v->class = vip[i].class;
	 v->bitsPerRGBValue = vip[i].bits_per_rgb;
	 v->ColormapEntries = vip[i].colormap_size;
	 v->redMask = vip[i].red_mask;
	 v->greenMask = vip[i].green_mask;
	 v->blueMask = vip[i].blue_mask;
	 
	 for (j = 0 ; j < screen->ndepths ; j++)
	    for (k = 0 ; k < screen->depths[j].nvisuals ; k++)
	       if (screen->depths[j].visuals[k].visualid == v->vid) {
		  v->nplanes = screen->depths[j].depth;
		  break;
	       }
      }

      free((char *) vip);
   }

   /* none of these are used: */
   directScreen.WindowPrivateLen = 0;
   directScreen.WindowPrivateSizes = 0;
   directScreen.totalWindowSize = 0;
   directScreen.GCPrivateLen = 0;
   directScreen.GCPrivateSizes = 0;
   directScreen.totalGCSize = 0;

   /* Build screenInfo... Fix this code for >1 screen
    */ 
   screenInfo.numScreens = 1;
   screenInfo.screens[0] = &directScreen;

   /* Build pixmap padding info.
    */
   pmf = X11SYM(XListPixmapFormats) (dpy, &n);
   if (pmf) {
      for (i = 0; i < n; i++) {
	 PaddingInfo *pi = &PixmapWidthPaddingInfo[i];
	 int pixels_per_pad = pmf[i].scanline_pad / pmf[i].bits_per_pixel;

	 pi->padRoundUp = pixels_per_pad - 1;
	 pi->padPixelsLog2 = mylog2(pixels_per_pad);
	 pi->padBytesLog2 = mylog2(pmf[i].scanline_pad / 8);
	 pi->notPower2 = ((1<<mylog2(pmf[i].bits_per_pixel)) != 
			  pmf[i].bits_per_pixel);
	 pi->bytesPerPixel = (pmf[i].bits_per_pixel + 7)>>4; /* maybe */
      }
      free(pmf);
   }

   /* This is ugly - the initvisuals code is incomprehensible.
    */
   {
      int numvis = 1;
      int numdep = directScreen.numDepths;
      int rootdep = directScreen.rootDepth;
      VisualID rootvis = directScreen.rootVisual;
      VisualPtr visp = (VisualPtr) malloc( sizeof(VisualRec) * numvis );
      DepthPtr depp = (DepthPtr) malloc( sizeof(DepthRec) * numdep );
      int i;

      for (i = 0 ; i < numdep ; i++) {
	 memcpy(&depp[i], &directScreen.allowedDepths[i], sizeof(DepthRec));

	 if (!depp[i].vids) continue;
	 
	 depp[i].vids = (VisualID *)malloc(depp[i].numVids * sizeof(VisualID));
	 memcpy(depp[i].vids, 
		directScreen.allowedDepths[i].vids,
		depp[i].numVids * sizeof(VisualID));
      }

      for (i = 0 ; i < numvis ; i++) 
	 memcpy( &visp[i], &directScreen.visuals[i], sizeof(VisualRec) );

      GlxInitVisuals( &visp, &depp, &numvis, &numdep, 
		      &rootdep, &rootvis, 0, 0 );	
   }

   fprintf(stderr, "glx extension init:\n");

   GlxExtensionInit();

   fprintf(stderr, "adding direct client: \n");

   __glXAddClient( direct_client );

   fprintf(stderr, "finished glxInitDirectClient:\n");
     
   return direct_client;
}
