# Copyright (C) 2007  Lars Wirzenius <liw@iki.fi>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.


"""Unperish plugin to handle Debian packages."""


import logging
import os
import shutil

import unperishlib


def is_native(version):
    """Does a full Debian version number indicate the package is native?
    
    Native packages don't have a Debian revision, that is, they don't
    have a dash in the version number.
    
    """
    
    return "-" not in version


class IsNotDebianPackage(unperishlib.UnperishException):

    def __init__(self, directory):
        self._str = "%s is not a .deb package" % directory


class IsDebianPackage(unperishlib.Operation):

    """Check that the project includes .deb packaging.
    
    Specifically, check for the debian/control file in the directory we
    operate in. This operation is mostly useful for centralized error
    checking: other operations can now only require the is_debian_package
    option.
    
    """

    name = "is-debian-package"
    
    required_options = ["directory"]
    
    provided_options = ["is_debian_package"]
    
    def add_options(self, parser):
        parser.add_option("--is-debian-package", action="store_true",
                          help="Assume this is a Debian package. "
                               "This is mostly for internal use.")

    def do_it(self, options):
        path = os.path.join(options.directory, "debian", "control")
        if not os.path.isfile(path):
            raise IsNotDebianPackage(options.directory)
        options.is_debian_package = True


class ProjectAndDebianVersionMismatch(unperishlib.UnperishException):

    def __init__(self, project_version, debian_upstream_version):
        self._str = (("Project version '%s' and "
                      "Debian upstream version '%s' do not match.") 
                      % (project_version, debian_upstream_version))


class DebianInfo(unperishlib.Operation):

    """Gather information about a .deb package.
    
    This operation extracts the Debian source package name, version number,
    and the upstream portion of the full Debian version number into the
    relevant options.
    
    """
    
    name = "debian-info"

    required_options = ["is_debian_package"]
    
    provided_options = ["debian_source_name", "debian_version",
                        "debian_upstream_version"]

    def add_options(self, parser):
        parser.add_option("--debian-source-name", metavar="NAME",
                          help="Use NAME as the Debian source package name. "
                               "By default it is read from debian/changelog.")
        parser.add_option("--debian-version", metavar="VERSION",
                          help="Use VERSION as the Debian package version. "
                               "By default it is read from debian/changelog.")
        parser.add_option("--debian-upstream-version", metavar="VERSION",
                          help="Use VERSION as the Debian upstream part of "
                               "of the Debian package version. "
                               "By default it is computed from "
                               "--debian-version.")

    def compute_upstream_part(self, debian_version):
        """Return the upstream part of a full Debian version string."""
        if "-" in debian_version:
            return "-".join(debian_version.split("-")[:-1])
        else:
            return debian_version

    def verify_project_and_upstream_versions_match(self, options):
        """Verify that the project and Debian upstream versions match.
        
        If not, raise an exception. Both values are retrieved from
        options. If the project version is not set, accept the match
        in any case.
        """
        
        if (hasattr(options, "project_version") and
            options.project_version is not None and
            options.project_version != options.debian_upstream_version):
            raise ProjectAndDebianVersionMismatch(options.project_version,
                                          options.debian_upstream_version)

    def do_it(self, options):
        out = unperishlib.run(["dpkg-parsechangelog"], cwd=options.directory)
        for line in out.splitlines():
            if line.startswith("Source:") and not options.debian_source_name:
                options.debian_source_name = line[len("Source:"):].strip()
            if line.startswith("Version:") and not options.debian_version:
                options.debian_version = line[len("Version:"):].strip()
        if (options.debian_upstream_version is None and
            options.debian_version is not None):
            options.debian_upstream_version = self.compute_upstream_part(
                options.debian_version)
        self.verify_project_and_upstream_versions_match(options)


class DebianOrigTarGz(unperishlib.Operation):

    """Create a Debian source package, if one doesn't already exist.
    
    The upstream tarball must already exist. We create it by using a
    hardlink, to save time and space. If the package is a native Debian
    package, we name the tarball with an ".tar.gz" suffix, otherwise with
    an ".orig.tar.gz" suffix.
    
    The debian_orig_tar_gz option is provided, and will include the full
    path to the tarball, not just the basename.
    
    """
    
    name = "debian-orig-tar-gz"

    required_options = ["debian_source_name", "debian_upstream_version",
                        "debian_version", "upstream_tarball",
                        "build_area_exists"]

    provided_options = ["debian_orig_tar_gz"]

    def add_options(self, parser):
        parser.add_option("--debian-orig-tar-gz", metavar="FILE",
                          help="Use FILE as the Debian .orig.tar.gz file. "
                               "Default is derived from debian/changelog.")

    def do_it(self, options):
        if options.debian_orig_tar_gz is None:
            if is_native(options.debian_version):
                suffix = "tar.gz"
            else:
                suffix = "orig.tar.gz"
            base = ("%s_%s.%s" % (options.debian_source_name,
                                  options.debian_upstream_version,
                                  suffix))
            options.debian_orig_tar_gz = os.path.join(options.build_area, 
                                                      base)
        if not os.path.exists(options.debian_orig_tar_gz):
            os.link(options.upstream_tarball, options.debian_orig_tar_gz)


class DebianSourceTreeFromBzr(unperishlib.Operation):

    """Create an unpacked Debian source tree from a bzr branch.
    
    The directory is assumed to be a bzr branch, and we export all files
    from it to a suitably named directory in the build area.
        
    """
    
    name = "debian-source-tree-from-bzr"
    
    required_options = ["debian_source_name", "debian_upstream_version",
                        "build_area_exists", "is_bzr_branch", "directory"]

    provided_options = ["debian_source_tree"]
    
    def add_options(self, parser):
        parser.add_option("--debian-source-tree", metavar="DIR",
                          help="Use DIR as the unpacked Debian source tree. "
                               "Default is computed from debian/changelog.")

    def do_it(self, options):
        if options.debian_source_tree is None:
            base = ("%s-%s" % (options.debian_source_name, 
                               options.debian_upstream_version))
            options.debian_source_tree = os.path.join(options.build_area, 
                                                      base)
        if not os.path.exists(options.debian_source_tree):
            unperishlib.run(["bzr", "export", "--quiet", "--format=dir",
                             options.debian_source_tree],
                            cwd=options.directory)


class DebianSourcePackage(unperishlib.Operation):

    """Create a Debian source package, if one doesn't already exist."""
    
    name = "dsc"

    required_options = ["debian_source_name", "debian_upstream_version",
                        "debian_version", "debian_orig_tar_gz", 
                        "upstream_tarball", "build_area_exists", 
                        "debian_source_tree"]

    provided_options = ["debian_dsc", "debian_source_changes"]
    
    def add_options(self, parser):
        parser.add_option("--debian-dsc", metavar="FILE",
                          help="Use FILE as the Debian .dsc file. "
                               "Default is computed from debian/changelog.")
        parser.add_option("--debian-source-changes", metavar="FILE",
                          help="Use FILE as the Debian _source.changes file. "
                               "Default is what dpkg-buildpackage creates.")

        parser.add_public_option("--source-buildpackage-options",
                                 metavar="OPTIONS",
                                 default="",
                                 help="""\
Give OPTIONS to dpkg-buildpackage when creating a source package.
Multiple options can be given in OPTIONS by separating them with spaces.

Note that this only applies to building of source packages, not
binary packages. See --debbuildopts for that.
""")

    def source_changes(self, options):
        """Return full pathname to the source.changes file."""
        basename = "%s_%s_source.changes" % (options.debian_source_name,
                                             options.debian_version)
        return os.path.join(options.build_area, basename)

    def do_it(self, options):
        if options.debian_dsc is None:
            base = "%s_%s.dsc" % (options.debian_source_name,
                                  options.debian_version)
            options.debian_dsc = os.path.join(options.build_area, base)
        if not os.path.exists(options.debian_dsc):
            if is_native(options.debian_version):
                origtgz = ""
            else:
                origtgz = options.debian_orig_tar_gz
                if os.path.dirname(origtgz) == options.build_area:
                    origtgz = os.path.basename(origtgz)

            args = ["dpkg-buildpackage", "-S", "-us", "-uc", "-rfakeroot"]
            args += options.source_buildpackage_options.split()

            unperishlib.run(args, cwd=options.debian_source_tree)

        if not options.debian_source_changes:
            options.debian_source_changes = self.source_changes(options)
                                                         

class DebianArchitecture(unperishlib.Operation):

    """Determine the Debian architecture of the build machine."""
    
    name = "debian-architecture"
    
    provided_options = ["debian_architecture"]
    
    def add_options(self, parser):
        parser.add_option("--debian-architecture", metavar="ARCH",
                          help="Use ARCH as the Debian architecture string "
                               "of the machine we run the build on.")

    def do_it(self, options):
        if options.debian_architecture is None:
            output = unperishlib.run(["dpkg", "--print-architecture"])
            options.debian_architecture = output.strip()


class DebianBinaryPackages(unperishlib.Operation): #pragma: no cover

    # This operation requires root, and that is not something I want to
    # give my unit tests, so I will exclude this entire class from unit
    # test coverage metering.

    """Create Debian binary packages."""
    
    name = "deb"
    
    required_options = ["build_area", "debian_dsc", "debian_architecture"]
    
    provided_options = ["debian_binary_changes"]
    
    def add_options(self, parser):
        parser.add_public_option("--pbuilder-basetgz", metavar="FILE",
                                 help="""\
Use FILE as the pbuilder basetgz file.
Default is to use whatever pbuilder has as its default.""")

        parser.add_public_option("--debbuildopts", 
                                 metavar="OPTIONS",
                                 help="""\
Give OPTIONS to pbuilder as the value of pbuilder's --debbuildopts
option. Pbuilder will then parse the string and give the options in
the string to dpkg-buildpackage when building the package. Multiple
options can be given in OPTIONS by separating them with spaces.

Note that this only applies to building of binary packages, not
source packages. See --source-buildpackage-options for that.
""")

        parser.add_option("--debian-binary-changes", metavar="FILE",
                          help="Use FILE as the Debian binary .changes file. "
                               "The default is computed from "
                               "debian/changelog.")

    def do_it(self, options):
        logging.debug("In debian-binary-packages")
        if options.debian_binary_changes is None:
            options.debian_binary_changes = \
                os.path.join(options.build_area,
                             "%s_%s_%s.changes" % 
                             (options.debian_source_name,
                              options.debian_version,
                              options.debian_architecture))
            logging.debug("Set debian_binary_changes to %s" % 
                          repr(options.debian_binary_changes))

        if not os.path.exists(options.debian_binary_changes):
            args = ["sudo", "-p", "Password (for sudo to run pbuilder): ",
                    "pbuilder", "--build", 
                    "--buildresult", options.build_area]
            if options.pbuilder_basetgz:
                args.append("--basetgz")
                args.append(options.pbuilder_basetgz)

            debbuildopts = "-b"
            if options.debbuildopts:
                debbuildopts += options.debbuildopts
            args.append("--debbuildopts")
            args.append(debbuildopts)

            args.append(options.debian_dsc)
            unperishlib.run(args, cwd=options.build_area)
        else:
            logging.debug(".changes file already exists, nothing to do")


class DebianChanges(unperishlib.Operation):

    """Provide a Debian .changes file."""
    
    name = "debian-changes"
    
    provided_options = ["debian_changes"]
    
    def add_options(self, parser):
        parser.add_option("--debian-changes", metavar="FILE",
                          help="Use FILE as the Debian .changes file, "
                               "either the source or binary one.")

    def do_it(self, options):
        if not options.debian_changes:
            if options.debian_binary_changes:
                logging.debug("Setting debian_changes to "
                              "debian_binary_changes: %s" % 
                              options.debian_binary_changes)
                options.debian_changes = options.debian_binary_changes
            elif options.debian_source_changes:
                logging.debug("Setting debian_changes to "
                              "debian_source_changes: %s" % 
                              options.debian_source_changes)
                options.debian_changes = options.debian_source_changes
            else:
                logging.debug("Neither source nor binary .changes set. "
                              "Can't set debian_changes.")
        else:
            logging.debug("debian_changes already set")


class DebsignChanges(unperishlib.Operation): #pragma: no cover

    # This operation requires a Debian package and .changes file, and
    # building that is awkward in unit tests, so I'm marking this to
    # be excluded from coverage measurements.

    """Run debsign on a .changes file."""
    
    name = "debsign"
    
    required_options = ["debian_changes", "directory"]
    
    def do_it(self, options):
        unperishlib.run(["debsign", options.debian_changes],
                        cwd=options.directory)
    

class Lintian(unperishlib.Operation): #pragma: no cover

    # We have no easy way of generating binary debs for unit testing, so
    # this class is excluded from coverage metering.

    """Check Debian packages with lintian."""
    
    name = "lintian"
    
    required_options = ["debian_changes"]

    def do_it(self, options):
        output = unperishlib.run(["lintian", options.debian_changes])
        if output:
            logging.error("Lintian found problems:\n%s" %
                          "\n".join("  " + line 
                                    for line in output.splitlines()))



class Piuparts(unperishlib.Operation): #pragma: no cover

    # We have no easy way of generating binary debs for unit testing, so
    # this class is excluded from coverage metering.

    """Check Debian packages with piuparts.
    
    This runs piuparts on all the .deb files (at once) in the build area
    directory.
    
    """
    
    name = "piuparts"
    
    required_options = ["build_area"]
    
    def do_it(self, options):
        debs = [os.path.join(options.build_area, x)
                for x in os.listdir(options.build_area) if x.endswith(".deb")]
        if debs:
            unperishlib.run(["sudo", 
                             "-p", "Password (for sudo to run piuparts): ", 
                             "piuparts"]
                            + debs)
    

class CreateAptRepository(unperishlib.Operation):

    """Create a new apt repository to be filled with reprepro."""

    name = "create-apt-repository-for-reprepro"
    
    required_options = ["apt_repository", "apt_architectures", 
                        "debian_source_name"]
    
    provided_options = ["apt_repository_exists"]
    
    def add_options(self, parser):
        parser.add_public_option("--apt-repository", metavar="DIR",
                                 help="""\
Use DIR as the root of an apt repository. There is no default.""")

        parser.add_option("--apt-repository-exists", action="store_true",
                          help="""\
Signal that the apt repository directory exists.""")

        parser.add_public_option("--apt-architectures", action="append",
                                 help="""\
Which architectures should be included in the apt repository.
Default (the full list of architectures in Debian): %default.""")
        parser.set_defaults(apt_architectures=["alpha", "amd64", "arm",
                                               "hppa", "hurd-i386", "i386",
                                               "ia64", "m68k", "mips",
                                               "mipsel", "powerpc", "s390",
                                               "sparc", "source"])

    def new_file(self, filename, contents):
        """Write a file, unless it exists."""
        if not os.path.exists(filename):
            f = file(filename, "w")
            f.write(contents)
            f.close()

    def do_it(self, options):
        if not os.path.exists(options.apt_repository):
            os.mkdir(options.apt_repository)
            
            conf = os.path.join(options.apt_repository, "conf")
            os.mkdir(conf)

            distributions = os.path.join(conf, "distributions")
            self.new_file(distributions, """\
Origin: %(debian_source_name)s
Label: %(debian_source_name)s
Suite: unstable
Codename: sid
Architectures: %(apt_architectures)s
Components: main
Description: %(debian_source_name)s
Uploaders: uploaders
""" % {
    "apt_architectures": " ".join(options.apt_architectures),
    "debian_source_name": options.debian_source_name,
})

            uploaders = os.path.join(conf, "uploaders")
            self.new_file(uploaders, "allow * by unsigned\n")

        options.apt_repository_exists = True


class UnknownDistribution(unperishlib.UnperishException): #pragma: no cover

    def __init__(self, changes):
        self._str = "%s does not contain a Distribution: header" % changes


class UpdateAptRepository(unperishlib.Operation): #pragma: no cover

    # Writing tests for this class is too tedious for me right now.
    # I am a bad boy and need to be spanked for this.

    """Update apt repository with new packages."""
    
    name = "update-apt-repository-with-reprepro"
    
    required_options = ["apt_repository", "apt_repository_exists",
                        "debian_changes", "build_area"]

    def do_it(self, options):
        f = file(options.debian_changes)
        lines = f.readlines()
        f.close()
        distros = [x for x in lines if x.startswith("Distribution: ")]
        if len(distros) != 1:
            raise UnknownDistribution(options.debian_changes)
        distro = distros[0].split()[1]
        
        unperishlib.run(["reprepro", "-b", options.apt_repository, 
                         "include", distro, options.debian_changes],
                         cwd=options.build_area)


class Dput(unperishlib.Operation):

    """Upload a Debian source or binary package with dput."""
    
    name = "dput"
    
    required_options = ["debian_changes", "dput_host", "debian_source_name",
                        "debian_version", "debian_architecture"]
    
    def add_options(self, parser):
        parser.add_public_option("--dput-host", metavar="HOST",
                                 help="Upload to HOST when using dput.")

    def upload_name(self, options):
        """Return full pathname to the .upload file."""
        basename = "%s_%s_%s.upload" % (options.debian_source_name,
                                        options.debian_version,
                                        options.debian_architecture)
        return os.path.join(options.build_area, basename)

    def do_it(self, options): #pragma: no cover
        upload = self.upload_name(options)
        if os.path.exists(upload):
            logging.debug("Upload file %s exists already" % upload)
        else:
            logging.debug("Uploading Debian package with dput to %s" %
                          options.dput_host)
            args = ["dput", options.dput_host, options.debian_changes]
            logging.debug("dput: %s" % args)
            unperishlib.run(args, cwd=options.build_area)
