# Copyright (C) 2007  Lars Wirzenius <liw@iki.fi>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.


"""Unperish plugin for build area for working files and results."""


import os
import shutil

import unperishlib


class CreateBuildArea(unperishlib.Operation):

    """Create the build area.
    
    The build area is where we put output files, such as tarballs and
    binary packages. The idea is to put them in a place that is easy
    to access, and then have the user test the results, and finally
    publish them somewhere somehow (possibly by renaming the build_area
    directory).
    
    If the build area directory does not exist, this operation creates
    it.
    
    """

    name = "create-build-area"
    
    required_options = ["build_area"]
    
    provided_options = ["build_area_exists"]

    def add_options(self, parser):
        parser.add_public_option("--build-area", metavar="DIR",
                                 help="""\
Put results into DIR (default: %default).""")
        parser.set_defaults(build_area="../build-area")

        parser.add_option("--build-area-exists", action="store_true",
                          help="Indicate that the build area directory "
                               "exists. For internal use mainly.")

    def do_it(self, options):
        if not os.path.exists(options.build_area):
            os.mkdir(options.build_area)
        options.build_area_exists = True


class CleanBuildArea(unperishlib.Operation):

    """Remove all files from the build area.
    
    Note that the build area directory itself is not removed.
    
    There are no safety checks against wrong files being removed. Beware.
    
    """
    
    name = "clean-build-area"
    
    required_options = ["build_area"]
    
    def do_it(self, options):
        if os.path.exists(options.build_area):
            for name in os.listdir(options.build_area):
                pathname = os.path.join(options.build_area, name)
                if os.path.isdir(pathname):
                    shutil.rmtree(pathname)
                else:
                    os.remove(pathname)
