/*
 * UnixCW - Unix CW (Morse code) training program
 * Copyright (C) 2001  Simon Baldwin (simonb@caldera.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 * cwlibtest.c - Self-test functions for cwlib library.
 *
 */

/* Include files. */
#include <stdio.h>
#include <ctype.h>
#include <limits.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>

#if defined(HAVE_STRING_H)
#	include <string.h>
#endif /* HAVE_STRING_H */
#if defined(HAVE_STRINGS_H)
#	include <strings.h>
#endif /* HAVE_STRINGS_H */

/* Include definitions of CW library functions. */
#include "cwlib.h"

/* True/False macros, for use with int-type (lazy person) "booleans". */
#define	FALSE		(0)
#define	TRUE		(!FALSE)

/* Success/Error macros for return codes. */
#define	RC_SUCCESS	0
#define	RC_ERROR	-1


/*
 * cw_self_test_admin()
 */
static int
cw_self_test_admin ()
{
	int		failures = 0;		/* count of tests that fail */
	int		flags, index;		/* debug flags, loop count */

	/*
	 * Test the cw_version and cw_license functions.
	 */
	printf ("cwlib: version %d.%d\n", cw_version () >> 16,
		       			  cw_version () & 0xFF);
	cw_license ();

	/*
	 * Test getting and setting of debug flags.
	 */
	flags = cw_get_debug_flags ();
	for (index = 0; index <= CW_DEBUG_MASK; index++)
	    {
		cw_set_debug_flags (index);
		if (cw_get_debug_flags () != index)
		    {
			printf ("cwlib: ERROR: cw_get/set_debug flags\n");
			failures++;
		       	break;
		    }
	    }
	cw_set_debug_flags (flags);
	printf ("cwlib: cw_get/set_debug flags tests complete\n");

	/*
	 * If stdout cannot do sound, proceed with tests silently.
	 */
	if (cw_set_file_descriptor (1) != 0)
	    {
		printf ("cwlib: stdout cannot do sound, proceeding silently\n");
		cw_set_console_sound (0);
	    }

	return failures;
}


/*
 * cw_self_test_limits()
 */
static int
cw_self_test_limits ()
{
	int		failures = 0;		/* count of tests that fail */
	int		cw_min_speed;		/* parameter limit locations */
	int		cw_max_speed;
	int		cw_min_frequency;	/* parameter limit locations */
	int		cw_max_frequency;
	int		cw_min_gap;		/* parameter limit locations */
	int		cw_max_gap;
	int		cw_min_tolerance;	/* parameter limit locations */
	int		cw_max_tolerance;

	/*
	 * Ensure that we can obtain the main parameter limits.
	 */
	cw_get_speed_limits (&cw_min_speed, &cw_max_speed);
	printf ("cwlib: cw_get_speed_limits=%d,%d\n",
					cw_min_speed, cw_max_speed);
	cw_get_frequency_limits (&cw_min_frequency, &cw_max_frequency);
	printf ("cwlib: cw_get_frequency_limits=%d,%d\n",
					cw_min_frequency, cw_max_frequency);
	cw_get_gap_limits (&cw_min_gap, &cw_max_gap);
	printf ("cwlib: cw_get_gap_limits=%d,%d\n",
					cw_min_gap, cw_max_gap);
	cw_get_tolerance_limits (&cw_min_tolerance, &cw_max_tolerance);
	printf ("cwlib: cw_get_tolerance_limits=%d,%d\n",
					cw_min_tolerance, cw_max_tolerance);
	printf ("cwlib: cw_get_limits tests complete\n");

	return failures;
}


/*
 * cw_self_test_ranges()
 */
static int
cw_self_test_ranges ()
{
	int		failures = 0;		/* count of tests that fail */
	int		error;			/* general error return code */
	int		index;			/* loop count */
	int		cw_min;			/* parameter limit locations */
	int		cw_max;

	/*
	 * Set the main parameters to out-of-range values, and through their
	 * complete valid ranges.
	 */
	cw_get_speed_limits (&cw_min, &cw_max);
	errno = 0;
	error = cw_set_send_speed (cw_min - 1);
	if (error != -1 || errno != EINVAL)
	    {
		printf ("cwlib: ERROR: cw_set_send_speed(cw_min_speed-1)\n");
		failures++;
	    }
	errno = 0;
	error = cw_set_send_speed (cw_max + 1);
	if (error != -1 || errno != EINVAL)
	    {
		printf ("cwlib: ERROR: cw_set_send_speed(cw_max_speed+1)\n");
		failures++;
	    }
	for (index = cw_min; index <= cw_max; index++)
	    {
		cw_set_send_speed (index);
		if (cw_get_send_speed () != index)
		    {
			printf ("cwlib: ERROR: cw_get/set_send_speed\n");
			failures++;
		       	break;
		    }
	    }
	errno = 0;
	error = cw_set_receive_speed (cw_min - 1);
	if (error != -1 || errno != EINVAL)
	    {
		printf ("cwlib: ERROR: cw_set_receive_speed(cw_min_speed-1)\n");
		failures++;
	    }
	errno = 0;
	error = cw_set_receive_speed (cw_max + 1);
	if (error != -1 || errno != EINVAL)
	    {
		printf ("cwlib: ERROR: cw_set_receive_speed(cw_max_speed+1)\n");
		failures++;
	    }
	for (index = cw_min; index <= cw_max; index++)
	    {
		cw_set_receive_speed (index);
		if (cw_get_receive_speed () != index)
		    {
			printf ("cwlib: ERROR: cw_get/set_receive_speed\n");
			failures++;
		       	break;
		    }
	    }
	printf ("cwlib: cw_set/get_send/receive_speed tests complete\n");

	cw_get_frequency_limits (&cw_min, &cw_max);
	errno = 0;
	error = cw_set_frequency (cw_min - 1);
	if (error != -1 || errno != EINVAL)
	    {
		printf ("cwlib: ERROR: cw_set_frequency(cw_min_frequency-1)\n");
		failures++;
	    }
	errno = 0;
	error = cw_set_frequency (cw_max + 1);
	if (error != -1 || errno != EINVAL)
	    {
		printf ("cwlib: ERROR: cw_set_frequency(cw_max_frequency+1)\n");
		failures++;
	    }
	for (index = cw_min; index <= cw_max; index++)
	    {
		cw_set_frequency (index);
		if (cw_get_frequency () != index)
		    {
			printf ("cwlib: ERROR: cw_get/set_frequency\n");
			failures++;
			break;
		    }
	    }
	printf ("cwlib: cw_set/get_frequency tests complete\n");

	cw_get_gap_limits (&cw_min, &cw_max);
	errno = 0;
	error = cw_set_gap (cw_min - 1);
	if (error != -1 || errno != EINVAL)
	    {
		printf ("cwlib: ERROR: cw_set_gap(cw_min_gap-1)\n");
		failures++;
	    }
	errno = 0;
	error = cw_set_gap (cw_max + 1);
	if (error != -1 || errno != EINVAL)
	    {
		printf ("cwlib: ERROR: cw_set_gap(cw_max_gap+1)\n");
		failures++;
	    }
	for (index = cw_min; index <= cw_max; index++)
	    {
		cw_set_gap (index);
		if (cw_get_gap () != index)
		    {
			printf ("cwlib: ERROR: cw_get/set_gap\n");
			failures++;
			break;
		    }
	    }
	printf ("cwlib: cw_set/get_gap tests complete\n");

	cw_get_tolerance_limits (&cw_min, &cw_max);
	errno = 0;
	error = cw_set_tolerance (cw_min - 1);
	if (error != -1 || errno != EINVAL)
	    {
		printf ("cwlib: ERROR: cw_set_tolerance(cw_min_tolerance-1)\n");
		failures++;
	    }
	errno = 0;
	error = cw_set_tolerance (cw_max + 1);
	if (error != -1 || errno != EINVAL)
	    {
		printf ("cwlib: ERROR: cw_set_tolerance(cw_max_tolerance+1)\n");
		failures++;
	    }
	for (index = cw_min; index <= cw_max; index++)
	    {
		cw_set_tolerance (index);
		if (cw_get_tolerance () != index)
		    {
			printf ("cwlib: ERROR: cw_get/set_tolerance\n");
			failures++;
			break;
		    }
	    }
	printf ("cwlib: cw_set/get_tolerance tests complete\n");

	return failures;
}


/*
 * cw_self_test_tone_parameters()
 */
static int
cw_self_test_tone_parameters ()
{
	int		failures = 0;		/* count of tests that fail */
	int		error;			/* general error return code */
	int		cw_min;			/* parameter limit locations */
	int		cw_max;

	/*
	 * Test the limits of the parameters to the tone queue routine.
	 */
	cw_get_frequency_limits (&cw_min, &cw_max);
	errno = 0;
	error = cw_queue_tone (-1, cw_min);
	if (error != -1 || errno != EINVAL)
	    {
		printf ("cwlib: ERROR: cw_queue_tone(-1, cw_min_frequency)\n");
		failures++;
	    }
	errno = 0;
	error = cw_queue_tone (0, cw_min - 1);
	if (error != -1 || errno != EINVAL)
	    {
		printf ("cwlib: ERROR: cw_queue_tone(0, cw_min_frequency-1)\n");
		failures++;
	    }
	errno = 0;
	error = cw_queue_tone (0, cw_max + 1);
	if (error != -1 || errno != EINVAL)
	    {
		printf ("cwlib: ERROR: cw_queue_tone(0, cw_max_frequency+1)\n");
		failures++;
	    }
	printf ("cwlib: cw_queue_tone argument tests complete\n");

	return failures;
}


/*
 * cw_self_test_simple_tones()
 */
static int
cw_self_test_simple_tones ()
{
	int		failures = 0;		/* count of tests that fail */

	/*
	 * Ensure we can generate a few simple tones, and wait for them
	 * to end.
	 */
	if (cw_queue_tone (100000, 8000) != 0
			|| cw_queue_tone (100000, 4000) != 0
			|| cw_queue_tone (100000, 2000) != 0)
	    {
		printf ("cwlib: ERROR: cw_queue_tone(10000, 8000|4000|2000)\n");
		failures++;
	    }
	if (cw_tone_wait () != 0
			&& cw_get_tone_queue_length () != 2)
	    {
		printf ("cwlib: ERROR: cw_tone_wait()\n");
		failures++;
	    }
	if (cw_tone_wait () != 0
			&& cw_get_tone_queue_length() != 1)
	    {
		printf ("cwlib: ERROR: cw_tone_wait()\n");
		failures++;
	    }
	if (cw_tone_wait () != 0
		       	&& cw_get_tone_queue_length () != 0)
	    {
		printf ("cwlib: ERROR: cw_tone_wait()\n");
		failures++;
	    }
	if (cw_queue_tone (100000, 1000) != 0
			|| cw_queue_tone (100000, 500) != 0)
	    {
		printf ("cwlib: ERROR: cw_queue_tone(10000, 1000|500)\n");
		failures++;
	    }
	if (cw_tone_queue_wait () != 0)
	    {
		printf ("cwlib: ERROR: cw_tone_queue_wait()\n");
		failures++;
	    }
	printf ("cwlib: cw_queue_tone single tone test complete\n");

	return failures;
}


/*
 * cw_self_test_complex_tones()
 */
static int
cw_self_test_complex_tones ()
{
	int		failures = 0;		/* count of tests that fail */
	int		index;			/* loop count */
	int		cw_min;			/* parameter limit locations */
	int		cw_max;

	/*
	 * Run the complete range of tone generation, at 100Hz intervals,
	 * first up the octaves, and then down.  If the queue fills, though
	 * it shouldn't with this amount of data, then pause until it isn't
	 * so full.
	 */
	cw_get_frequency_limits (&cw_min, &cw_max);
	for (index = cw_min; index < cw_max; index += 100)
	    {
		while (cw_tone_queue_full ())
		    {
			if (cw_tone_wait () != 0)
			    {
				printf ("cwlib: ERROR: cw_tone_wait()\n");
				failures++;
			       	break;
			    }
		    }
		if (cw_queue_tone (10000, index) != 0)
		    {
			printf ("cwlib: ERROR: cw_queue_tone()\n");
			failures++;
		       	break;
		    }
	    }
	for (index = cw_max; index > cw_min; index -= 100)
	    {
		while (cw_tone_queue_full ())
		    {
			if (cw_tone_wait () != 0)
			    {
				printf ("cwlib: ERROR: cw_tone_wait()\n");
				failures++;
			       	break;
			    }
		    }
		if (cw_queue_tone (10000, index) != 0)
		    {
			printf ("cwlib: ERROR: cw_queue_tone()\n");
			failures++;
		       	break;
		    }
	    }
	if (cw_tone_queue_wait () != 0)
	    {
		printf ("cwlib: ERROR: cw_tone_queue_wait()\n");
		failures++;
	    }
	cw_queue_tone (0, 0);
	cw_tone_queue_wait ();
	printf ("cwlib: cw_queue_tone/cw_tone_queue_wait tests complete\n");

	return failures;
}


/*
 * cw_self_test_tone_queue()
 */
static int
cw_self_test_tone_queue ()
{
	int		failures = 0;		/* count of tests that fail */
	int		error;			/* general error return code */

	/*
	 * Test the tone queue manipulations, ensuring that we can fill the
	 * queue, that it looks full when it is, and that we can flush it all
	 * again afterwards, and recover.
	 */
	printf ("cwlib: cw_get_tone_queue_capacity=%d\n",
						cw_get_tone_queue_capacity ());
	printf ("cwlib: empty cw_get_tone_queue_length=%d\n",
						cw_get_tone_queue_length ());
	while (!cw_tone_queue_full ())
		cw_queue_tone (1000000, 100);
	printf ("cwlib: full cw_get_tone_queue_length=%d\n",
						cw_get_tone_queue_length ());
	errno = 0;
	error = cw_queue_tone (1000000, 100);
	if (error != -1 || errno != EAGAIN)
	    {
		printf ("cwlib: ERROR: full cw_queue_tone()\n");
		failures++;
	    }
	cw_flush_tone_queue ();
	if (cw_get_tone_queue_length () != 0)
	    {
		printf ("cwlib: ERROR: cw_get_tone_queue_length()\n");
		failures++;
	    }
	printf ("cwlib: cw_flush_tone_queue/length/capacity tests complete\n");

	return failures;
}


/*
 * cw_self_test_lookups()
 */
static int
cw_self_test_lookups ()
{
	int		failures = 0;		/* count of tests that fail */
	unsigned int	index;			/* loop count */
	unsigned char	c;			/* general character variable */
	unsigned char	charlist[ UCHAR_MAX+1 ];/* character list string */
	char		representation[ 256 ];	/* generous representation */

	/*
	 * Collect and print out a list of characters in the table.
	 */
	printf ("cwlib: cw_get_character_count %d\n",
						cw_get_character_count ());
	cw_list_characters (charlist);
	printf ("cwlib: cw_list_characters\n");
	printf ("cwlib:     %s\n", charlist);

	/*
	 * For each character, look up it's representation, the look up each
	 * representation in the opposite direction.
	 */
	printf ("cwlib: cw_get_maximum_representation_length %d\n",
				cw_get_maximum_representation_length ());
	for (index = 0; index < strlen (charlist); index++)
	    {
		if (cw_lookup_character (charlist[index], representation) != 0)
		    {
			printf ("cwlib: ERROR: cw_lookup_character()\n");
			failures++; break;
		    }
		if (cw_lookup_representation (representation, &c) != 0)
		    {
			printf ("cwlib: ERROR: cw_lookup_representation()\n");
			failures++; break;
		    }
		if (charlist[index] != c)
		    {
			printf ("cwlib: ERROR: cw_lookup_() mapping wrong\n");
			failures++; break;
		    }
	    }
	printf ("cwlib: cw list and lookup tests complete\n");

	return failures;
}


/*
 * cw_self_test_dot_dash()
 */
static int
cw_self_test_dot_dash ()
{
	int		failures = 0;		/* count of tests that fail */

	/*
	 * Send basic dot and dash using the library primitives.
	 */
	cw_set_send_speed (30); cw_set_frequency (800); cw_set_gap (0);
	if (cw_send_dot () != 0)
	    {
		printf ("cwlib: ERROR: cw_send_dot()\n");
		failures++;
	    }
	if (cw_send_dash () != 0)
	    {
		printf ("cwlib: ERROR: cw_send_dash()\n");
		failures++;
	    }
	if (cw_send_character_space () != 0)
	    {
		printf ("cwlib: ERROR: cw_send_character_space()\n");
		failures++;
	    }
	if (cw_send_word_space () != 0)
	    {
		printf ("cwlib: ERROR: cw_send_word_space()\n");
		failures++;
	    }
	cw_tone_queue_wait ();
	printf ("cwlib: cw_send_dot/dash tests complete\n");

	return failures;
}


/*
 * cw_self_test_representations()
 */
static int
cw_self_test_representations ()
{
	int		failures = 0;		/* count of tests that fail */

	/*
	 * Check just a couple of basic representations, and send the valid
	 * ones as tones.
	 */
	if (cw_check_representation (".-.-.-") != 0
			|| cw_check_representation ("INVALID") != -1)
	    {
		printf ("cwlib: ERROR: cw_check_representation()\n");
		failures++;
	    }
	if (cw_send_representation_partial (".-.-.-") != 0)
	    {
		printf ("cwlib: ERROR: cw_send_representation_partial()\n");
		failures++;
	    }
	if (cw_send_representation (".-.-.-") != 0)
	    {
		printf ("cwlib: ERROR: valid cw_send_representation()\n");
		failures++;
	    }
	if (cw_send_representation ("INVALID") != -1)
	    {
		printf ("cwlib: ERROR: invalid cw_send_representation()\n");
		failures++;
	    }
	cw_tone_queue_wait ();
	printf ("cwlib: cw_send_representation tests complete\n");

	return failures;
}


/*
 * cw_self_test_characters()
 */
static int
cw_self_test_characters ()
{
	int		failures = 0;		/* count of tests that fail */
	int		index;			/* loop count */
	unsigned char	charlist[ UCHAR_MAX+1 ];/* character list string */

	/*
	 * Check all the single characters we can, up to UCHAR_MAX.
	 */
	cw_list_characters (charlist);
	for (index = 0; index < UCHAR_MAX; index++)
	    {
		if (index == ' '
			|| (index != 0
				&& strchr (charlist, toupper (index)) != NULL))
		    {
			if (cw_check_character (index) != 0)
			    {
				printf (
				"cwlib: ERROR: valid cw_check_character()\n");
				failures++;
			       	break;
			    }
		    }
		else
		    {
			if (cw_check_character (index) != -1)
			    {
				printf (
				"cwlib: ERROR: invalid cw_check_character()\n");
				failures++;
			       	break;
			    }
		    }
	    }


	/*
	 * Check the whole charlist item as a single string, then check a
	 * known invalid string.
	 */
	cw_list_characters (charlist);
	if (cw_check_string (charlist) != 0)
	    {
		printf ("cwlib: ERROR: cw_check_string()\n");
		failures++;
	    }
	if (cw_check_string ("~INVALID~") != -1)
	    {
		printf ("cwlib: ERROR: invalid cw_check_string()\n");
		failures++;
	    }
	printf ("cwlib: cw_check_character/string tests complete\n");

	return failures;
}


/*
 * cw_self_test_full_send()
 */
static int
cw_self_test_full_send ()
{
	int		failures = 0;		/* count of tests that fail */
	unsigned int	index;			/* loop count */
	unsigned char	c;			/* general character variable */
	unsigned char	charlist[ UCHAR_MAX+1 ];/* character list string */

	/*
	 * Send all the characters from the charlist individually.
	 */
	cw_list_characters (charlist);
	printf ("cwlib: cw_send_character\n");
	printf ("cwlib:     ");
	for (index = 0; index < strlen (charlist); index++)
	    {
		printf ("%c", charlist[index]); fflush (stdout);
		if (cw_send_character (charlist[index]) != 0)
		    {
			printf ("cwlib: ERROR: cw_send_character()\n");
			failures++;
		    }
		cw_tone_queue_wait ();
	    }
	printf ("\n");
	if (cw_send_character (0) != -1)
	    {
		printf ("cwlib: ERROR: invalid cw_send_character()\n");
		failures++;
	    }


	/* 
	 * Now send the complete charlist as a single string, and do one of
	 * those cute rotating lines that you see around from time to time
	 * while we do it.
	 */
	printf ("cwlib: cw_send_string\n");
	printf ("cwlib:     %s\n", charlist);
	if (cw_send_string (charlist) != 0)
	    {
		printf ("cwlib: ERROR: cw_send_string()\n");
		failures++;
	    }
	c = '-';
	while (cw_get_tone_queue_length () > 0)
	    {
		printf ("cwlib: tone queue length %-6d %c\r",
					cw_get_tone_queue_length (), c);
		fflush (stdout);
		if      (c == '-' )	c = '\\';
		else if (c == '\\')	c = '|';
		else if (c == '|' )	c = '/';
		else			c = '-';
		cw_tone_wait ();
	    }
	printf ("cwlib: tone queue length %-6d %c\n",
					cw_get_tone_queue_length (), ' ');
	cw_tone_queue_wait ();
	if (cw_send_string ("~INVALID~") != -1)
	    {
		printf ("cwlib: ERROR: invalid cw_send_string()\n");
		failures++;
	    }
	printf ("cwlib: cw_send_character/string tests complete\n");

	return failures;
}


/*
 * cw_self_test_fixed_receive()
 */
static int
cw_self_test_fixed_receive ()
{
	int		failures = 0;		/* count of tests that fail */
	int		word_flag;		/* word receive flag */
	int		rxerror_flag;		/* receive error flag */
	struct timeval	tv;			/* general timestamp variable */
	unsigned char	c;			/* general character variable */
	char		representation[ 256 ];	/* generous representation */

	/*
	 * Test receive functions by spoofing them with a timestamp.
	 * Getting the test suite to generate reliable timing events is
	 * a little too much work.  Add just a little jitter to the time-
	 * stamps.  This is a _very_ minimal test, omitting all error
	 * states.
	 */
	printf ("cwlib: cw_get_receive_buffer_capacity=%d\n",
					cw_get_receive_buffer_capacity ());

	cw_set_receive_speed (60); cw_set_gap (0); cw_set_tolerance (35);
	cw_disable_adaptive_receive ();
	tv.tv_sec = 0;
	tv.tv_usec  = 0;	cw_start_receive_tone (&tv);
	tv.tv_usec += 63456;	cw_end_receive_tone (&tv);	/* dash */
	tv.tv_usec += 20000;	cw_start_receive_tone (&tv);
	tv.tv_usec += 63456;	cw_end_receive_tone (&tv);	/* dash */
	tv.tv_usec += 20000;	cw_start_receive_tone (&tv);
	tv.tv_usec += 23456;	cw_end_receive_tone (&tv);	/* dot */
	tv.tv_usec += 20000;	cw_start_receive_tone (&tv);
	tv.tv_usec += 63456;	cw_end_receive_tone (&tv);	/* dash */
	tv.tv_usec += 60000;
	if (cw_get_receive_buffer_length () != 4)
	    {
		printf ("cwlib: ERROR: incorrect receive_buffer_length()\n");
		failures++;
	    }
	if (cw_receive_representation (&tv, representation,
						&word_flag, &rxerror_flag) != 0)
	    {
		printf ("cwlib: ERROR: cw_receive_representation()\n");
		failures++;
	    }
	if (strcmp (representation, "--.-") != 0)
	    {
		printf ("cwlib: ERROR: incorrect cw_receive_representation\n");
		failures++;
	    }
	if (word_flag)
	    {
		printf ("cwlib: ERROR: cw_receive_representation not char\n");
		failures++;
	    }
	if (rxerror_flag)
	    {
		printf ("cwlib: ERROR: cw_receive_representation rxerror\n");
		failures++;
	    }
	if (cw_receive_character (&tv, &c, &word_flag, &rxerror_flag) != 0)
	    {
		printf ("cwlib: ERROR: cw_receive_character()\n");
		failures++;
	    }
	if (c != 'Q')
	    {
		printf ("cwlib: ERROR: incorrect cw_receive_character\n");
		failures++;
	    }
	printf ("cwlib: cw_receive_representation/character <%s>,<%c>\n",
							representation, c);
	cw_clear_receive_buffer ();
	if (cw_get_receive_buffer_length () != 0)
	    {
		printf ("cwlib: ERROR: incorrect receive_buffer_length()\n");
		failures++;
	    }

	tv.tv_sec++;
	tv.tv_usec  = 0;	cw_start_receive_tone (&tv);
	tv.tv_usec += 17654;	cw_end_receive_tone (&tv);	/* dot */
	tv.tv_usec += 20000;	cw_start_receive_tone (&tv);
	tv.tv_usec += 57654;	cw_end_receive_tone (&tv);	/* dash */
	tv.tv_usec += 20000;	cw_start_receive_tone (&tv);
	tv.tv_usec += 17654;	cw_end_receive_tone (&tv);	/* dot */
	tv.tv_usec += 60000;
	if (cw_get_receive_buffer_length () != 3)
	    {
		printf ("cwlib: ERROR: incorrect receive_buffer_length()\n");
		failures++;
	    }
	if (cw_receive_representation (&tv, representation,
						&word_flag, &rxerror_flag) != 0)
	    {
		printf ("cwlib: ERROR: cw_receive_representation()\n");
		failures++;
	    }
	if (strcmp (representation, ".-.") != 0)
	    {
		printf ("cwlib: ERROR: incorrect cw_receive_representation\n");
		failures++;
	    }
	if (word_flag)
	    {
		printf ("cwlib: ERROR: cw_receive_representation not char\n");
		failures++;
	    }
	if (rxerror_flag)
	    {
		printf ("cwlib: ERROR: cw_receive_representation rxerror\n");
		failures++;
	    }
	if (cw_receive_character (&tv, &c, &word_flag, &rxerror_flag) != 0)
	    {
		printf ("cwlib: ERROR: cw_receive_character()\n");
		failures++;
	    }
	if (c != 'R')
	    {
		printf ("cwlib: ERROR: incorrect cw_receive_character\n");
		failures++;
	    }
	printf ("cwlib: cw_receive_representation/character <%s>,<%c>\n",
							representation, c);
	cw_clear_receive_buffer ();

	tv.tv_sec++;
	tv.tv_usec  = 0;
	tv.tv_usec += 23456;	cw_receive_buffer_dot (&tv);	/* dot */
	tv.tv_usec += 20000;
	tv.tv_usec += 63456;	cw_receive_buffer_dash (&tv);	/* dash */
	tv.tv_usec += 20000;
	tv.tv_usec += 63456;	cw_receive_buffer_dash (&tv);	/* dash */
	tv.tv_usec += 20000;
	tv.tv_usec += 23456;	cw_receive_buffer_dot (&tv);	/* dot */
	tv.tv_usec += 140000;
	if (cw_get_receive_buffer_length () != 4)
	    {
		printf ("cwlib: ERROR: incorrect receive_buffer_length()\n");
		failures++;
	    }
	if (cw_receive_representation (&tv, representation,
						&word_flag, &rxerror_flag) != 0)
	    {
		printf ("cwlib: ERROR: cw_receive_representation()\n");
		failures++;
	    }
	if (strcmp (representation, ".--.") != 0)
	    {
		printf ("cwlib: ERROR: incorrect cw_receive_representation\n");
		failures++;
	    }
	if (!word_flag)
	    {
		printf ("cwlib: ERROR: cw_receive_representation not word\n");
		failures++;
	    }
	if (rxerror_flag)
	    {
		printf ("cwlib: ERROR: cw_receive_representation rxerror\n");
		failures++;
	    }
	if (cw_receive_character (&tv, &c, &word_flag, &rxerror_flag) != 0)
	    {
		printf ("cwlib: ERROR: cw_receive_character()\n");
		failures++;
	    }
	if (c != 'P')
	    {
		printf ("cwlib: ERROR: incorrect cw_receive_character\n");
		failures++;
	    }
	printf ("cwlib: cw_receive_representation/character <%s>,<%c>\n",
							representation, c);
	cw_clear_receive_buffer ();
	printf ("cwlib: cw_receive_representation/character tests complete\n");
	printf ("cwlib: cw fixed speed receive tests complete\n");

	return failures;
}


/*
 * cw_self_test_adaptive_receive()
 */
static int
cw_self_test_adaptive_receive ()
{
	int		failures = 0;		/* count of tests that fail */
	int		word_flag;		/* word receive flag */
	int		rxerror_flag;		/* receive error flag */
	struct timeval	tv;			/* general timestamp variable */
	unsigned char	c;			/* general character variable */
	char		representation[ 256 ];	/* generous representation */

	/*
	 * Test adaptive receive functions in much the same sort of way.
	 * Again, this is a _very_ minimal test, omitting all error states.
	 */
	cw_set_receive_speed (45); cw_set_gap (0); cw_set_tolerance (35);
	cw_enable_adaptive_receive ();
	tv.tv_sec = 0;
	tv.tv_usec  = 0;	cw_start_receive_tone (&tv);
	tv.tv_usec += 60000;	cw_end_receive_tone (&tv);	/* 60wpm dash */
	tv.tv_usec += 20000;	cw_start_receive_tone (&tv);
	tv.tv_usec += 60000;	cw_end_receive_tone (&tv);	/* 60wpm dash */
	tv.tv_usec += 20000;	cw_start_receive_tone (&tv);
	tv.tv_usec += 20000;	cw_end_receive_tone (&tv);	/* 60wpm dot */
	tv.tv_usec += 20000;	cw_start_receive_tone (&tv);
	tv.tv_usec += 60000;	cw_end_receive_tone (&tv);	/* 60wpm dash */
	tv.tv_usec += 60000;
	if (cw_get_receive_buffer_length () != 4)
	    {
		printf ("cwlib: ERROR: incorrect receive_buffer_length()\n");
		failures++;
	    }
	if (cw_receive_representation (&tv, representation,
						&word_flag, &rxerror_flag) != 0)
	    {
		printf ("cwlib: ERROR: cw_receive_representation()\n");
		failures++;
	    }
	if (strcmp (representation, "--.-") != 0)
	    {
		printf ("cwlib: ERROR: incorrect cw_receive_representation\n");
		failures++;
	    }
	if (word_flag)
	    {
		printf ("cwlib: ERROR: cw_receive_representation not char\n");
		failures++;
	    }
	if (rxerror_flag)
	    {
		printf ("cwlib: ERROR: cw_receive_representation rxerror\n");
		failures++;
	    }
	if (cw_receive_character (&tv, &c, &word_flag, &rxerror_flag) != 0)
	    {
		printf ("cwlib: ERROR: cw_receive_character()\n");
		failures++;
	    }
	if (c != 'Q')
	    {
		printf ("cwlib: ERROR: incorrect cw_receive_character\n");
		failures++;
	    }
	printf ("cwlib: adaptive speed tracking reports %d wpm\n",
						cw_get_receive_speed ());
	printf ("cwlib: cw_receive_representation/character <%s>,<%c>\n",
							representation, c);
	cw_clear_receive_buffer ();
	if (cw_get_receive_buffer_length () != 0)
	    {
		printf ("cwlib: ERROR: incorrect receive_buffer_length()\n");
		failures++;
	    }

	tv.tv_sec++;
	tv.tv_usec  = 0;	cw_start_receive_tone (&tv);
	tv.tv_usec += 30000;	cw_end_receive_tone (&tv);	/* 40wpm dot */
	tv.tv_usec += 30000;	cw_start_receive_tone (&tv);
	tv.tv_usec += 90000;	cw_end_receive_tone (&tv);	/* 40wpm dash */
	tv.tv_usec += 30000;	cw_start_receive_tone (&tv);
	tv.tv_usec += 30000;	cw_end_receive_tone (&tv);	/* 40wpm dot */
	tv.tv_usec += 90000;
	if (cw_get_receive_buffer_length () != 3)
	    {
		printf ("cwlib: ERROR: incorrect receive_buffer_length()\n");
		failures++;
	    }
	if (cw_receive_representation (&tv, representation,
						&word_flag, &rxerror_flag) != 0)
	    {
		printf ("cwlib: ERROR: cw_receive_representation()\n");
		failures++;
	    }
	if (strcmp (representation, ".-.") != 0)
	    {
		printf ("cwlib: ERROR: incorrect cw_receive_representation\n");
		failures++;
	    }
	if (word_flag)
	    {
		printf ("cwlib: ERROR: cw_receive_representation not char\n");
		failures++;
	    }
	if (rxerror_flag)
	    {
		printf ("cwlib: ERROR: cw_receive_representation rxerror\n");
		failures++;
	    }
	if (cw_receive_character (&tv, &c, &word_flag, &rxerror_flag) != 0)
	    {
		printf ("cwlib: ERROR: cw_receive_character()\n");
		failures++;
	    }
	if (c != 'R')
	    {
		printf ("cwlib: ERROR: incorrect cw_receive_character\n");
		failures++;
	    }
	printf ("cwlib: adaptive speed tracking reports %d wpm\n",
						cw_get_receive_speed ());
	printf ("cwlib: cw_receive_representation/character <%s>,<%c>\n",
							representation, c);
	cw_clear_receive_buffer ();

	tv.tv_sec++;
	tv.tv_usec  = 0;	cw_start_receive_tone (&tv);
	tv.tv_usec += 40000;	cw_end_receive_tone (&tv);	/* 30wpm dot */
	tv.tv_usec += 40000;	cw_start_receive_tone (&tv);
	tv.tv_usec += 120000;	cw_end_receive_tone (&tv);	/* 30wpm dash */
	tv.tv_usec += 40000;	cw_start_receive_tone (&tv);
	tv.tv_usec += 120000;	cw_end_receive_tone (&tv);	/* 30wpm dash */
	tv.tv_usec += 40000;	cw_start_receive_tone (&tv);
	tv.tv_usec += 40000;	cw_end_receive_tone (&tv);	/* 30wpm dot */
	tv.tv_usec += 280000;
	if (cw_get_receive_buffer_length () != 4)
	    {
		printf ("cwlib: ERROR: incorrect receive_buffer_length()\n");
		failures++;
	    }
	if (cw_receive_representation (&tv, representation,
						&word_flag, &rxerror_flag) != 0)
	    {
		printf ("cwlib: ERROR: cw_receive_representation()\n");
		failures++;
	    }
	if (strcmp (representation, ".--.") != 0)
	    {
		printf ("cwlib: ERROR: incorrect cw_receive_representation\n");
		failures++;
	    }
	if (!word_flag)
	    {
		printf ("cwlib: ERROR: cw_receive_representation not word\n");
		failures++;
	    }
	if (rxerror_flag)
	    {
		printf ("cwlib: ERROR: cw_receive_representation rxerror\n");
		failures++;
	    }
	if (cw_receive_character (&tv, &c, &word_flag, &rxerror_flag) != 0)
	    {
		printf ("cwlib: ERROR: cw_receive_character()\n");
		failures++;
	    }
	if (c != 'P')
	    {
		printf ("cwlib: ERROR: incorrect cw_receive_character\n");
		failures++;
	    }
	printf ("cwlib: adaptive speed tracking reports %d wpm\n",
						cw_get_receive_speed ());
	printf ("cwlib: cw_receive_representation/character <%s>,<%c>\n",
							representation, c);
	cw_clear_receive_buffer ();
	printf ("cwlib: cw_receive_representation/character tests complete\n");
	printf ("cwlib: cw adaptive speed receive tests complete\n");

	return failures;
}


/*
 * cw_self_test_keyer()
 */
static int
cw_self_test_keyer ()
{
	int		failures = 0;		/* count of tests that fail */
	int		index;			/* loop count */
	int		dot_paddle, dash_paddle;/* paddle state values */

	/*
	 * Perform some tests on the iambic keyer.  The latch finer timing
	 * points are not tested here, just the basics - dots, dashes, and
	 * alternating dots and dashes.
	 */
	cw_set_send_speed (30); cw_set_gap (0); cw_set_frequency (800);
	if (cw_keyer_paddle_event (TRUE,  FALSE) != 0)
	    {
		printf ("cwlib: ERROR: cw_keyer_paddle_event\n");
		failures++;
	    }
	printf ("cwlib: testing iambic keyer dots   "); fflush (stdout);
	for  (index = 0; index < 30; index++)
	    {
		cw_keyer_element_wait ();
		printf ("#"); fflush (stdout);
	    }
	printf ("\n");
	cw_get_keyer_paddles (&dot_paddle, &dash_paddle);
	if (!dot_paddle || dash_paddle)
	    {
		printf ("cwlib: ERROR: cw_keyer_get_paddles mismatch\n");
		failures++;
	    }

	if (cw_keyer_paddle_event (FALSE, TRUE) != 0)
	    {
		printf ("cwlib: ERROR: cw_keyer_paddle_event\n");
		failures++;
	    }
	printf ("cwlib: testing iambic keyer dashes "); fflush (stdout);
	for  (index = 0; index < 30; index++)
	    {
		cw_keyer_element_wait ();
		printf ("#"); fflush (stdout);
	    }
	printf ("\n");
	cw_get_keyer_paddles (&dot_paddle, &dash_paddle);
	if (dot_paddle || !dash_paddle)
	    {
		printf ("cwlib: ERROR: cw_keyer_get_paddles mismatch\n");
		failures++;
	    }

	if (cw_keyer_paddle_event (TRUE, TRUE) != 0)
	    {
		printf ("cwlib: ERROR: cw_keyer_paddle_event\n");
		failures++;
	    }
	printf ("cwlib: testing iambic alternating  "); fflush (stdout);
	for  (index = 0; index < 30; index++)
	    {
		cw_keyer_element_wait ();
		printf ("#"); fflush (stdout);
	    }
	printf ("\n");
	cw_get_keyer_paddles (&dot_paddle, &dash_paddle);
	if (!dot_paddle || !dash_paddle)
	    {
		printf ("cwlib: ERROR: cw_keyer_get_paddles mismatch\n");
		failures++;
	    }

	cw_keyer_paddle_event (FALSE, FALSE);
	cw_keyer_wait ();
	printf ("cwlib: cw_keyer_paddle_event tests complete\n");

	return failures;
}


/*
 * cw_self_test_straightkey()
 */
static int
cw_self_test_straightkey ()
{
	int		failures = 0;		/* count of tests that fail */
	int		index;			/* loop count */

	/*
	 * Finally, a nice simple set of tests.
	 */
	cw_set_send_speed (30); cw_set_gap (0); cw_set_frequency (800);
	for (index = 0; index < 10; index++)
	    {
		if (cw_straightkey_event (FALSE) != 0)
		    {
			printf ("cwlib: ERROR: cw_straightkey_event FALSE\n");
			failures++;
		    }
		if (cw_get_straightkey_state ())
		    {
			printf ("cwlib: ERROR: cw_get_straightkey_state\n");
			failures++;
		    }
		if (cw_straightkey_busy ())
		    {
			printf ("cwlib: ERROR: cw_straightkey_busy\n");
			failures++;
		    }
	    }
	for (index = 0; index < 10; index++)
	    {
		if (cw_straightkey_event (TRUE) != 0)
		    {
			printf ("cwlib: ERROR: cw_straightkey_event TRUE\n");
			failures++;
		    }
		if (!cw_get_straightkey_state ())
		    {
			printf ("cwlib: ERROR: cw_get_straightkey_state\n");
			failures++;
		    }
		if (!cw_straightkey_busy ())
		    {
			printf ("cwlib: ERROR: cw_straightkey_busy\n");
			failures++;
		    }
	    }
	sleep (1);
	for (index = 0; index < 10; index++)
	    {
		if (cw_straightkey_event (FALSE) != 0)
		    {
			printf ("cwlib: ERROR: cw_straightkey_event FALSE\n");
			failures++;
		    }
	    }
	if (cw_get_straightkey_state ())
	    {
		printf ("cwlib: ERROR: cw_get_straightkey_state\n");
		failures++;
	    }
	printf ("cwlib: cw_straight_key_event/busy tests complete\n");

	return failures;
}


/*
 * cw_self_test()
 *
 * Perform a series of self-tests on library public interfaces.
 */
int
cw_self_test ()
{
	int		tests		= 0;	/* test count */
	int		failures	= 0;	/* cumulative failure count */
	int		testset		= -1;	/* bitmask of tests to run */
	char		*testset_value;		/* value of CWLIB_TESTSET */
	int		(*test_function []) () = {
		cw_self_test_admin,		cw_self_test_limits,
		cw_self_test_ranges,		cw_self_test_tone_parameters,
		cw_self_test_simple_tones,	cw_self_test_complex_tones,
		cw_self_test_tone_queue,	cw_self_test_lookups,
		cw_self_test_dot_dash,		cw_self_test_representations,
		cw_self_test_characters,	cw_self_test_full_send,
		cw_self_test_fixed_receive,	cw_self_test_adaptive_receive,
		cw_self_test_keyer,		cw_self_test_straightkey,
		NULL
	};					/* array of test functions */
	int		index;			/* index into test array */

	/*
	 * Obtain a bitmask of the tests to run from the environment variable
	 * CWLIB_TESTSET.  If the variable is not set, then default to -1,
	 * which effectively requests all tests.
	 */
	testset_value = getenv ("CWLIB_TESTSET");
	if (testset_value != NULL)
		testset = atoi (testset_value);

	/*
	 * If admin test not selected, set silent mode if necessary here.
	 */
	if (testset && !(testset & 1)
			&& cw_set_file_descriptor (1) != 0)
	    {
		printf ("cwlib: stdout cannot do sound, proceeding silently\n");
		cw_set_console_sound (0);
	    }

	/*
	 * Run each test specified in the testset bit mask, and add up the
	 * errors that the tests report.
	 */
	for (index = 0; test_function[index] != NULL; index++)
	    {
		if (testset & (1 << index))
		    {
			tests++;
			failures += (*test_function[index]) ();
		    }
	    }

	/*
	 * All tests done; return success if no failures, otherwise return
	 * an error status code.
	 */
	if (failures == 0)
	    {
		printf ("cwlib: %d test%c completed SUCCESSFULLY\n",
						tests,
						(tests == 1)?' ':'s');
		return RC_SUCCESS;
	    }
	else
	    {
		printf ("cwlib: %d test%c completed with %d ERROR%c\n",
						tests,
						(tests == 1)?' ':'s',
						failures,
						(failures == 1)?' ':'S');
		return RC_ERROR;
	    }
}
