/*
 * Copyright (c) 2003-2005 Erez Zadok
 * Copyright (c) 2003-2005 Charles P. Wright
 * Copyright (c) 2003-2005 Mohammad Nayyer Zubair
 * Copyright (c) 2003-2005 Puja Gupta
 * Copyright (c) 2003-2005 Harikesavan Krishnan
 * Copyright (c) 2003-2005 Stony Brook University
 * Copyright (c) 2003-2005 The Research Foundation of State University of New York
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package.
 *
 * This Copyright notice must be kept intact and distributed with all sources.
 */
/*
 *  $Id: file.c,v 1.100 2005/03/24 03:26:46 dquigley Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#include "fist.h"
#include "unionfs.h"
#include <linux/dcache.h>

/*******************
 * File Operations *
 *******************/

static int unionfs_file_revalidate(struct file *file) {
	struct super_block *sb;
	struct dentry *dentry;
	int sbgen, fgen, dgen;
	int bindex, bstart, bend;
	struct file *hidden_file;
	struct dentry *hidden_dentry;

	int err = 0;

	print_entry(" file = %p", file);

	PASSERT(file);
	PASSERT(ftopd(file));

	dentry = file->f_dentry;
	PASSERT(dentry);
	PASSERT(dentry->d_op);
	PASSERT(dentry->d_op->d_revalidate);
	if (!(dentry->d_op->d_revalidate(dentry, 0))) {
		err = -ESTALE;
		goto out;
	}

	sb = dentry->d_sb;
	PASSERT(sb);
	sbgen = atomic_read(&stopd(sb)->usi_generation);
	dgen = atomic_read(&dtopd(dentry)->udi_generation);
	fgen = atomic_read(&ftopd(file)->ufi_generation);
	if (sbgen > dgen) {
		FISTBUG("The dentry is not up to date!\n");
	}
	if (sbgen > fgen) {
		/* First we throw out the existing files. */
		bstart = fbstart(file);
		bend = fbend(file);
		for (bindex = bstart; bindex <= bend; bindex++) {
			if (ftohf_index(file, bindex)) {
				branchput(dentry->d_sb, bindex);
				fput(ftohf_index(file, bindex));
			}
		}
		if (ftohf_ptr(file)) {
			KFREE(ftohf_ptr(file));
		}

		/* Now we reopen the file(s) as in unionfs_open. */
		bstart = fbstart(file) = dbstart(dentry);
		bend = fbend(file) = dbend(dentry);

		ftohf_ptr(file) = KMALLOC(sizeof(file_t *) * sbmax(sb), GFP_UNIONFS);
		if (!ftohf_ptr(file)) {
			err = -ENOMEM;
			goto out;
		}

		init_ftohf_ptr(file, sbmax(sb));


		if (S_ISDIR(dentry->d_inode->i_mode)) {
			/* We need to open all the files. */
			for (bindex = bstart; bindex <= bend; bindex++) {
				hidden_dentry = dtohd_index(dentry, bindex);
				if (!hidden_dentry) {
					continue;
				}

				dget(hidden_dentry);
				mntget(stohiddenmnt_index(sb, bindex));
				branchget(sb, bindex);

				hidden_file = dentry_open(hidden_dentry, stohiddenmnt_index(sb, bindex), file->f_flags);
				if (IS_ERR(hidden_file)) {
					err = PTR_ERR(hidden_file);
					goto out;
				} else {
					ftohf_index(file, bindex) = hidden_file;
				}
			}
		} else {
			/* We only open the highest priority branch. */
			hidden_dentry = dtohd(dentry);
			PASSERT(hidden_dentry);
			PASSERT(hidden_dentry->d_inode);
			if (IS_WRITE_FLAG(file->f_flags) && is_robranch(dentry)) {
				for (bindex = bstart - 1; bindex >= 0; bindex--) {
					err = unionfs_copyup_file(dentry->d_parent->d_inode, file, bstart, bindex, file->f_dentry->d_inode->i_size);
					if (!err) {
						break;
					}
				}
				goto out;
			}

			dget(hidden_dentry);
			mntget(stohiddenmnt_index(sb, bstart));
			branchget(sb, bstart);
			hidden_file = dentry_open(hidden_dentry, stohiddenmnt_index(sb, bstart), file->f_flags);
			if (IS_ERR(hidden_file)) {
				err = PTR_ERR(hidden_file);
				goto out;
			}
			ftohf(file) = hidden_file;
			/* Fix up the position. */
			hidden_file->f_pos = file->f_pos;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
			/* update readahead information, all the time, because the old file
			 * could have read-ahead information that doesn't match.
			 */
			if(file->f_reada)
			{
				hidden_file->f_reada = file->f_reada;
				hidden_file->f_ramax = file->f_ramax;
				hidden_file->f_raend = file->f_raend;
				hidden_file->f_ralen = file->f_ralen;
				hidden_file->f_rawin = file->f_rawin;
			}
#else
			memcpy(&(hidden_file->f_ra), &(file->f_ra), sizeof(struct file_ra_state));
#endif
		}
		atomic_set(&ftopd(file)->ufi_generation, atomic_read(&itopd(dentry->d_inode)->uii_generation));
	}

out:
	print_exit_status(err);
	return err;
}

	STATIC loff_t
unionfs_llseek(file_t *file, loff_t offset, int origin)
{
	loff_t err;
	file_t *hidden_file = NULL;

	print_entry_location();

	if ((err = unionfs_file_revalidate(file))) {
		goto out;
	}

	if (ftopd(file) != NULL) {
		hidden_file = ftohf(file);
	}

	fist_dprint(6, "unionfs_llseek: file=%p, offset=0x%llx, origin=%d\n",
			file, offset, origin);

	/* always set hidden position to this one */
	hidden_file->f_pos = file->f_pos;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	/* update readahead information, all the time, because the old file
	 * could have read-ahead information that doesn't match.
	 */
	if(file->f_reada)
	{
		hidden_file->f_reada = file->f_reada;
		hidden_file->f_ramax = file->f_ramax;
		hidden_file->f_raend = file->f_raend;
		hidden_file->f_ralen = file->f_ralen;
		hidden_file->f_rawin = file->f_rawin;
	}
#else
	memcpy(&(hidden_file->f_ra), &(file->f_ra), sizeof(struct file_ra_state));
#endif

	if (hidden_file->f_op && hidden_file->f_op->llseek)
		err = hidden_file->f_op->llseek(hidden_file, offset, origin);
	else
		err = generic_file_llseek(hidden_file, offset, origin);

	if (err < 0)
		goto out;
	if (err != file->f_pos) {
		file->f_pos = err;
		// ION maybe this?
		// 	file->f_pos = hidden_file->f_pos;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
		file->f_reada = 0;
#endif
		file->f_version++;
	}
out:
	print_exit_status((int) err);
	return err;
}


	STATIC ssize_t
unionfs_read(file_t *file, char *buf, size_t count, loff_t *ppos)
{
	int err = -EINVAL;
	file_t *hidden_file = NULL;
	loff_t pos = *ppos;

	print_entry_location();

	if ((err = unionfs_file_revalidate(file))) {
		goto out;
	}

	fist_print_file("entering read()", file);

	if (ftopd(file) != NULL) {
		hidden_file = ftohf(file);
	}

	if (!hidden_file->f_op || !hidden_file->f_op->read)
		goto out;

	err = hidden_file->f_op->read(hidden_file, buf, count, &pos);
	*ppos = pos;
	if (err >= 0) {
		/* atime should also be updated for reads of size zero or more */
		fist_copy_attr_atime(file->f_dentry->d_inode,
				hidden_file->f_dentry->d_inode);
	}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	/*
	 * because pread() does not have any way to tell us that it is
	 * our caller, then we don't know for sure if we have to update
	 * the file positions.  This hack relies on read() having passed us
	 * the "real" pointer of its struct file's f_pos field.
	 */
	if (ppos == &file->f_pos)
		hidden_file->f_pos = *ppos = pos;
	if (hidden_file->f_reada) { /* update readahead information if needed */
		file->f_reada = hidden_file->f_reada;
		file->f_ramax = hidden_file->f_ramax;
		file->f_raend = hidden_file->f_raend;
		file->f_ralen = hidden_file->f_ralen;
		file->f_rawin = hidden_file->f_rawin;
	}
#else
	memcpy(&(file->f_ra),&(hidden_file->f_ra),sizeof(struct file_ra_state));
#endif

out:
	fist_print_file("leaving read()", file);
	print_exit_status(err);
	return err;
}


/* this unionfs_write() does not modify data pages! */
static ssize_t unionfs_write(file_t *file, const char *buf, size_t count, loff_t *ppos)
{
	int err = -EINVAL;
	file_t *hidden_file = NULL;
	inode_t *inode;
	inode_t *hidden_inode;
	loff_t pos = *ppos;
	int bstart, bend;

	print_entry_location();

	if ((err = unionfs_file_revalidate(file))) {
		goto out;
	}

	inode = file->f_dentry->d_inode;

	bstart = fbstart(file);
	bend = fbend(file);

	ASSERT(bstart != -1);
	PASSERT(ftopd(file));
	PASSERT(ftohf(file));

	hidden_file = ftohf(file);
	hidden_inode = hidden_file->f_dentry->d_inode;

	if (!hidden_file->f_op || !hidden_file->f_op->write) {
		goto out;
	}

	/* adjust for append -- seek to the end of the file */
	if (file->f_flags & O_APPEND)
		pos = inode->i_size;

	err = hidden_file->f_op->write(hidden_file, buf, count, &pos);

	/*
	 * copy ctime and mtime from lower layer attributes
	 * atime is unchanged for both layers
	 */
	if (err >= 0)
		fist_copy_attr_times(inode, hidden_inode);

	/*
	 * XXX: MAJOR HACK
	 *
	 * because pwrite() does not have any way to tell us that it is
	 * our caller, then we don't know for sure if we have to update
	 * the file positions.  This hack relies on write() having passed us
	 * the "real" pointer of its struct file's f_pos field.
	 */
	*ppos = pos;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	if (ppos == &file->f_pos)
		hidden_file->f_pos = *ppos = pos;
#endif

	/* update this inode's size */
	if (pos > inode->i_size)
		inode->i_size = pos;

out:
	print_exit_status(err);
	return err;
}

struct unionfs_getdents_callback {
	struct unionfs_dir_state *rdstate;
	void *dirent;
	int entries_written;
	int filldir_called;
	int filldir_error;
	filldir_t filldir;
};

/* copied from generic filldir in fs/readir.c */
static int unionfs_filldir(void *dirent, const char *name, int namelen, loff_t offset, ino_t ino, unsigned int d_type)
{
	struct unionfs_getdents_callback *buf = (struct unionfs_getdents_callback *) dirent;
	struct filldir_node *found = NULL;
	int err = 0;
	int is_wh_entry = 0;

	fist_dprint(9, "unionfs_filldir name=%*s\n", namelen, name);

	PASSERT(buf);
	buf->filldir_called++;

	if (!strncmp(name, ".wh.", 4)) {
		name += 4;
		namelen -= 4;
		is_wh_entry = 1;
	}
	found = find_filldir_node(buf->rdstate, name, namelen);

	if (!found) {
		/* if 'name' isn't a whiteout filldir it. */
		if (!is_wh_entry) {
			off_t pos = rdstate2offset(buf->rdstate);
			err = buf->filldir(buf->dirent, name, namelen, pos, ino, d_type);
			buf->rdstate->uds_offset++;
			verify_rdstate_offset(buf->rdstate);
		}
		/* If we did fill it, stuff it in our hash, otherwise return an error */
		if (err) {
			buf->filldir_error = err;
			goto out;
		} else {
			buf->entries_written++;
			if ((err = add_filldir_node(buf->rdstate, name, namelen, buf->rdstate->uds_bindex, is_wh_entry))) {
				buf->filldir_error = err;
			}
		}
	}

out:
	return err;
}

static int unionfs_readdir(file_t *file, void *dirent, filldir_t filldir)
{
	int err = 0;
	file_t *hidden_file = NULL;
	inode_t *inode = NULL;
	struct unionfs_getdents_callback buf;
	struct unionfs_dir_state *uds;
	int bend;
	loff_t offset;

	print_entry("file = %p, pos = %llx", file, file->f_pos);

	fist_print_file("In unionfs_readdir()", file);

	if ((err = unionfs_file_revalidate(file))) {
		goto out;
	}

	inode = file->f_dentry->d_inode;
	fist_checkinode(inode, "unionfs_readdir");

	uds = ftopd(file)->rdstate;
	if(!uds) {
		if (file->f_pos == DIREOF) {
			goto out;
		} else if (file->f_pos > 0) {
			uds = find_rdstate(inode, file->f_pos);
			if (!uds) {
				err = -ESTALE;
				goto out;
			}
			ftopd(file)->rdstate = uds;
		} else {
			init_rdstate(file);
			uds = ftopd(file)->rdstate;
		}
	}
	bend = fbend(file);

	while (uds->uds_bindex <= bend) {
		hidden_file = ftohf_index(file, uds->uds_bindex);
		if (!hidden_file) {
			fist_dprint(7, "Incremented bindex to %d of %d because hidden file is NULL.\n", uds->uds_bindex, bend);
			uds->uds_bindex++;
			uds->uds_dirpos = 0;
			continue;
		}

		/* prepare callback buffer */
		buf.filldir_called = 0;
		buf.filldir_error = 0;
		buf.entries_written = 0;
		buf.dirent = dirent;
		buf.filldir = filldir;
		buf.rdstate = uds;

		/* Read starting from where we last left off. */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
		if(hidden_file->f_op->llseek)
			offset = hidden_file->f_op->llseek(hidden_file,uds->uds_dirpos,0);
		else
			offset = generic_file_llseek(hidden_file,uds->uds_dirpos,0);
#else
		offset = vfs_llseek(hidden_file,uds->uds_dirpos,0);
#endif
		if(offset < 0){
			err = offset;
			goto out;
		}
		fist_dprint(7, "calling readdir for %d.%lld\n", uds->uds_bindex, hidden_file->f_pos);
		err = vfs_readdir(hidden_file, unionfs_filldir, (void *) &buf);
		fist_dprint(7, "readdir on %d.%lld = %d (entries written %d, filldir called %d)\n", uds->uds_bindex, (long long)uds->uds_dirpos, err, buf.entries_written, buf.filldir_called);
		/* Save the position for when we continue. */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
		if(hidden_file->f_op->llseek)
			offset = hidden_file->f_op->llseek(hidden_file,0,1);
		else
			offset = generic_file_llseek(hidden_file,0,1);
#else
		offset = vfs_llseek(hidden_file,0,1);
#endif
		if(offset < 0){
			err = offset;
			goto out;
		} else {
			uds->uds_dirpos = offset;
		}
		/* Copy the atime. */
		fist_copy_attr_atime(inode, hidden_file->f_dentry->d_inode);

		if (err < 0) {
			goto out;
		}

		if (buf.filldir_error) {
			break;
		}

		if (!buf.entries_written) {
			uds->uds_bindex++;
			uds->uds_dirpos = 0;
		}
	}

	if (!buf.filldir_error && uds->uds_bindex >= bend) {
		fist_dprint(3, "Discarding rdstate because readdir is over.\n");
		/* Save the number of hash entries for next time. */
		itopd(inode)->uii_hashsize = uds->uds_hashentries;
		free_rdstate(uds);
		ftopd(file)->rdstate = NULL;
		file->f_pos = DIREOF;
	} else {
		PASSERT(ftopd(file)->rdstate);
		file->f_pos = rdstate2offset(uds);
		fist_dprint(3, "rdstate now has a cookie of %u (err = %d)\n", uds->uds_cookie, err);
	}

out:
	fist_checkinode(inode, "post unionfs_readdir");
	print_exit_status(err);
	return err;
}

/* This is not meant to be a generic repositioning function.  If you do
 * things that aren't supported, then we return EINVAL.
 *
 * What is allowed:
 *  (1) seeking to the same position that you are currently at
 *	This really has no effect, but returns where you are.
 *  (2) seeking to the end of the file, if you've read everything
 *	This really has no effect, but returns where you are.
 *  (3) seeking to the beginning of the file
 *	This throws out all state, and lets you begin again.
 */
static loff_t unionfs_dir_llseek(file_t *file, loff_t offset, int origin)
{
	struct unionfs_dir_state *rdstate;
	loff_t err;

	print_entry(" file=%p, offset=0x%llx, origin = %d", file, offset, origin);

	if ((err = unionfs_file_revalidate(file))) {
		goto out;
	}

	rdstate = ftopd(file)->rdstate;

	/* We let users seek to their current position, but not anywhere else. */
	if (!offset) {
		switch(origin) {
			case SEEK_SET:
				if(rdstate){
					free_rdstate(rdstate);
					ftopd(file)->rdstate = NULL;
				}
				init_rdstate(file);
				err = 0;
				break;
			case SEEK_CUR:
				if(file->f_pos){	
					if(file->f_pos == DIREOF)
						err = DIREOF;
					else
						ASSERT(file->f_pos == rdstate2offset(rdstate));
						err = file->f_pos;
				} else {
					err = 0;
				}
				break;
			case SEEK_END:
				/* Unsupported, because we would break everything.  */
				err = -EINVAL;
				break;
		}
	} else {
		switch(origin) {
			case SEEK_SET:
				if(rdstate){
					if (offset == rdstate2offset(rdstate)) {
						err = offset;
					} else if (file->f_pos == DIREOF) {
						err = DIREOF;
					} 
					else {
						err = -EINVAL;	
					}
				}
				else {
					if ((rdstate = find_rdstate(file->f_dentry->d_inode, offset))) {
						ftopd(file)->rdstate = rdstate;
						err = rdstate->uds_offset;
					} else {
						err = -EINVAL;
					}
				}
				break;
			case SEEK_CUR:
			case SEEK_END:
				/* Unsupported, because we would break everything.  */
				err = -EINVAL;
				break;
		}
	}

out:
	print_exit_status((int) err);
	return err;
}

static int unionfs_file_readdir(file_t *file, void *dirent, filldir_t filldir)
{
	int err = -ENOTDIR;
	print_entry_location();
	print_exit_status(err);
	return err;
}


	STATIC unsigned int
unionfs_poll(file_t *file, poll_table *wait)
{
	unsigned int mask = DEFAULT_POLLMASK;
	file_t *hidden_file = NULL;

	print_entry_location();

	if (unionfs_file_revalidate(file)) {
		/* We should pretend an error happend. */
		mask = POLLERR|POLLIN|POLLOUT;
		goto out;
	}

	if (ftopd(file) != NULL)
		hidden_file = ftohf(file);

	if (!hidden_file->f_op || !hidden_file->f_op->poll)
		goto out;

	mask = hidden_file->f_op->poll(hidden_file, wait);

out:
	print_exit_status(mask);
	return mask;
}

	STATIC int
unionfs_ioctl(inode_t *inode, file_t *file, unsigned int cmd, unsigned long arg)
{
	int err = 0;		/* don't fail by default */
	file_t *hidden_file = NULL;
	vfs_t *this_vfs;
	vnode_t *this_vnode;
	int val;

	print_entry_location();

	if ((err = unionfs_file_revalidate(file))) {
		goto out;
	}

	this_vfs = inode->i_sb;
	this_vnode = inode;

	/* check if asked for local commands */
	switch (cmd) {
		case FIST_IOCTL_GET_DEBUG_VALUE:
			if (!capable(CAP_SYS_ADMIN)) {
				err = -EACCES;
				goto out;
			}
			val = fist_get_debug_value();
			err = put_user(val, (int *) arg);
			break;

		case FIST_IOCTL_SET_DEBUG_VALUE:
			if (!capable(CAP_SYS_ADMIN)) {
				err = -EACCES;
				goto out;
			}
			err = get_user(val, (int *) arg);
			if (err)
				break;
			fist_dprint(6, "IOCTL SET: got arg %d\n", val);
			if (val < 0 || val > 20) {
				err = -EINVAL;
				break;
			}
			fist_set_debug_value(val);
			break;

			/* add non-debugging fist ioctl's here */

		case UNIONFS_IOCTL_BRANCH_COUNT:
			if (!capable(CAP_SYS_ADMIN)) {
				err = -EACCES;
				goto out;
			}
			err = unionfs_ioctl_branchcount(inode, file, cmd, arg);
			break;

		case UNIONFS_IOCTL_INCGEN:
			if (!capable(CAP_SYS_ADMIN)) {
				err = -EACCES;
				goto out;
			}
			err = unionfs_ioctl_incgen(inode, file, cmd, arg);
			break;

		case UNIONFS_IOCTL_ADDBRANCH:
			if (!capable(CAP_SYS_ADMIN)) {
				err = -EACCES;
				goto out;
			}
			err = unionfs_ioctl_addbranch(inode, file, cmd, arg);
			break;

		case UNIONFS_IOCTL_DELBRANCH:
			if (!capable(CAP_SYS_ADMIN)) {
				err = -EACCES;
				goto out;
			}
			err = unionfs_ioctl_delbranch(inode, file, cmd, arg);
			break;

		case UNIONFS_IOCTL_RDWRBRANCH:
			if (!capable(CAP_SYS_ADMIN)) {
				err = -EACCES;
				goto out;
			}
			err = unionfs_ioctl_rdwrbranch(inode, file, cmd, arg);
			break;

		case UNIONFS_IOCTL_SUPERDUPER:
			if (!capable(CAP_SYS_ADMIN)) {
				err = -EACCES;
				goto out;
			}
			err = unionfs_ioctl_superduper(inode, file, cmd, arg);
			break;

		default:
			if (ftopd(file) != NULL) {
				hidden_file = ftohf(file);
			}
			/* pass operation to hidden filesystem, and return status */
			if (hidden_file && hidden_file->f_op && hidden_file->f_op->ioctl)
				err = hidden_file->f_op->ioctl(itohi(inode), hidden_file, cmd, arg);
			else
				err = -ENOTTY;	/* this is an unknown ioctl */
	} /* end of outer switch statement */

out:
	print_exit_status(err);
	return err;
}


/* FIST-LITE special version of mmap */
	STATIC int
unionfs_mmap(file_t *file, vm_area_t *vma)
{
	int err = 0;
	file_t *hidden_file = NULL;

	print_entry_location();

	if ((err = unionfs_file_revalidate(file))) {
		goto out;
	}

	if (ftopd(file) != NULL)
		hidden_file = ftohf(file);

	if (!file->f_op || !file->f_op->mmap) {
		err = -ENODEV;
		goto out;
	}

	PASSERT(hidden_file);
	PASSERT(hidden_file->f_op);
	PASSERT(hidden_file->f_op->mmap);

	vma->vm_file = hidden_file;
	err = hidden_file->f_op->mmap(hidden_file, vma);
	get_file(hidden_file); /* make sure it doesn't get freed on us */
	fput(file);			/* no need to keep extra ref on ours */

out:
	print_exit_status(err);
	return err;
}


	STATIC int
unionfs_open(inode_t *inode, file_t *file)
{
	int err = 0;
	int hidden_flags;
	file_t *hidden_file = NULL;
	dentry_t *hidden_dentry = NULL;
	dentry_t *dentry = NULL;
	super_block_t *sb;
	int bindex = 0, bstart = 0, bend = 0;

	print_entry_location();
	ftopd_lhs(file) = KMALLOC(sizeof(struct unionfs_file_info), GFP_UNIONFS);
	if (!ftopd(file)) {
		err = -ENOMEM;
		goto out;
	}
	init_ftopd(file);
	atomic_set(&ftopd(file)->ufi_generation, atomic_read(&itopd(inode)->uii_generation));
	ftohf_ptr(file) = KMALLOC((sizeof(file_t *) * sbmax(inode->i_sb)), GFP_UNIONFS);
	if (!ftohf_ptr(file)) {
		err = -ENOMEM;
		goto out;
	}
	init_ftohf_ptr(file, sbmax(inode->i_sb));

	hidden_flags = file->f_flags;

	dentry = file->f_dentry;
	PASSERT(dentry);

	bstart = fbstart(file) = dbstart(dentry);
	bend = fbend(file) = dbend(dentry);

	sb = inode->i_sb;

	/* open all directories and make the unionfs file struct point to these hidden file structs */
	if (S_ISDIR(inode->i_mode)) {

		for (bindex = bstart; bindex <= bend; bindex++) {

			hidden_dentry = dtohd_index(dentry, bindex);
			if (!hidden_dentry) {
				continue;
			}

			dget(hidden_dentry);
			/*
			 * dentry_open will decrement mnt refcnt if err.
			 * otherwise fput() will do an mntput() for us upon file close.
			 */
			mntget(stohiddenmnt_index(sb, bindex));
			branchget(sb, bindex);
			hidden_file = dentry_open(hidden_dentry, stohiddenmnt_index(sb, bindex), hidden_flags);
			if (IS_ERR(hidden_file)) {
				err =  PTR_ERR(hidden_file);
				goto out;
			}

			ftohf_index(file, bindex) = hidden_file;	/* link two files */
		}
	} else {
		/* open a file */
		hidden_dentry = dtohd(dentry);
		PASSERT(hidden_dentry);

		/* check for the permission for hidden file.  If the error is COPYUP_ERR,
		 * copyup the file.
		 */
		if (hidden_dentry->d_inode) {
			/* if this is file is on RO branch, copyup the file */
			if (IS_WRITE_FLAG(hidden_flags) && is_robranch(dentry)) {
				int size = file->f_dentry->d_inode->i_size;
				if (hidden_flags & O_TRUNC) {
					size = 0;
				}
				/* copyup the file */
				for (bindex = bstart - 1; bindex >= 0; bindex--) {
					err = unionfs_copyup_file(dentry->d_parent->d_inode, file, bstart, bindex, size);
					if (!err) {
						break;
					}
				}
				goto out;
			}
		}

		dget(hidden_dentry);
		/* dentry_open will decrement mnt refcnt if err.
		 * otherwise fput() will do an mntput() for us upon file close.
		 */
		mntget(stohiddenmnt_index(sb, bstart));
		branchget(sb, bstart);
		hidden_file = dentry_open(hidden_dentry, stohiddenmnt_index(sb, bstart), hidden_flags);
		if (IS_ERR(hidden_file)) {
			err =  PTR_ERR(hidden_file);
			goto out;
		} else {
			ftohf(file) = hidden_file;	/* link two files */
		}
	}

out:
	/* freeing the allocated resources, and fput the opened files */
	if (err < 0 && ftopd(file)) {
		for (bindex = bstart; bindex <= bend; bindex++) {
			hidden_file  = ftohf_index(file, bindex);
			if (hidden_file) {
				branchput(file->f_dentry->d_sb, bindex);
				/* fput calls dput for hidden_dentry */
				fput(hidden_file);
			}
		}
		if (ftohf_ptr(file)) {
			KFREE(ftohf_ptr(file));
		}
		KFREE(ftopd(file));
	}

	fist_print_file("OUT: unionfs_open", file);
	print_exit_status(err);
	return err;
}


	STATIC int
unionfs_flush(file_t *file)
{
	int err = 0;		/* assume ok (see open.c:close_fp) */
	file_t *hidden_file = NULL;
	int bindex, bstart, bend;

	print_entry_location();

	if ((err = unionfs_file_revalidate(file))) {
		goto out;
	}

	bstart = fbstart(file);
	bend = fbend(file);
	for (bindex = bstart; bindex <= bend; bindex++) {
		hidden_file  = ftohf_index(file, bindex);
		if (hidden_file && hidden_file->f_op && hidden_file->f_op->flush) {
			err = hidden_file->f_op->flush(hidden_file);
			if (err) {
				goto out;
			}
		}
	}


out:
	print_exit_status(err);
	return err;
}


static int unionfs_release(inode_t *inode, file_t *file)
{
	int err = 0;
	file_t *hidden_file = NULL;
	int bindex, bstart, bend;

	print_entry_location();

	fist_checkinode(inode, "unionfs_release");

	/* fput all the hidden files */
	bstart = fbstart(file);
	bend = fbend(file);

	for (bindex = bstart; bindex <= bend; bindex++) {
		hidden_file = ftohf_index(file, bindex);

		if (hidden_file) {
			/*
			 * will decrement file refcount, and if 0, destroy the file,
			 * which will call the lower file system's file release function.
			 */
			fput(hidden_file);
		}

		branchput(inode->i_sb, bindex);
	}
	KFREE(ftohf_ptr(file));

	if (ftopd(file)->rdstate) {
		fist_dprint(1, "Saving rdstate with cookie %u [%d.%lld]\n", ftopd(file)->rdstate->uds_cookie, ftopd(file)->rdstate->uds_bindex, (long long)ftopd(file)->rdstate->uds_dirpos);
		spin_lock(&itopd(inode)->uii_rdlock);
		itopd(inode)->uii_rdcount++;
		list_add_tail(&ftopd(file)->rdstate->uds_cache, &itopd(inode)->uii_readdircache);
		mark_inode_dirty(inode);
		spin_unlock(&itopd(inode)->uii_rdlock);
		ftopd(file)->rdstate = NULL;
	}
	KFREE(ftopd(file));

	fist_checkinode(inode, "post unionfs_release");

	print_exit_status(err);
	return err;
}


static int unionfs_fsync(file_t *file, dentry_t *dentry, int datasync)
{
	int err = -EINVAL;
	file_t *hidden_file = NULL;
	dentry_t *hidden_dentry;

	print_entry_location();

	if ((err = unionfs_file_revalidate(file))) {
		goto out;
	}

	if (ftopd(file) != NULL)
		hidden_file = ftohf(file);
	hidden_dentry = dtohd(dentry);

	if (hidden_file->f_op && hidden_file->f_op->fsync) {
		down(&hidden_dentry->d_inode->i_sem);
		err = hidden_file->f_op->fsync(hidden_file, hidden_dentry, datasync);
		up(&hidden_dentry->d_inode->i_sem);
	}

out:
	print_exit_status(err);
	return err;
}


	STATIC int
unionfs_fasync(int fd, file_t *file, int flag)
{
	int err = 0;
	file_t *hidden_file = NULL;

	print_entry_location();

	if ((err = unionfs_file_revalidate(file))) {
		goto out;
	}

	if (ftopd(file) != NULL)
		hidden_file = ftohf(file);

	if (hidden_file->f_op && hidden_file->f_op->fasync)
		err = hidden_file->f_op->fasync(fd, hidden_file, flag);

out:
	print_exit_status(err);
	return err;
}

STATIC int unionfs_lock(file_t *file, int cmd, struct file_lock *fl)
{
	int err = 0;
	file_t *lower_file = NULL;

	print_entry_location();
	if ((err = unionfs_file_revalidate(file))) {
		goto out;
	}

	if (ftopd(file) != NULL){   
	// XXX: ZH: this is wrong?:in this case ASSERT is not in ifndef... Put ASSERT before the if statement, kill if statement
			lower_file = ftohf(file);
	}
	ASSERT(lower_file != NULL);

	err = -EINVAL;
	if (!fl)
	{
		goto out;
	}
	fl->fl_file = lower_file;
	//Since we are being send a posix lock and not a flock or flock64 we can
	//treat the 64bit and regular calls the same.
	switch(cmd) {
		case F_GETLK:
		case F_GETLK64:
			err = unionfs_getlk(lower_file, fl);
			break;

		case F_SETLK:
		case F_SETLKW:
		case F_SETLK64:
		case F_SETLKW64:
			err = unionfs_setlk(lower_file, cmd, fl);
			break;
		default:
			err = -EINVAL;
	}
	fl->fl_file = file;

out:
	print_exit_status(err);
	return err;
}

/* Trimmed directory options, we shouldn't pass everything down since
 * we don't want to operate on partial directories.
 */
struct file_operations unionfs_dir_fops =
{
llseek:	unionfs_dir_llseek,
read:	generic_read_dir,
readdir:	unionfs_readdir,
ioctl:	unionfs_ioctl,
open:	unionfs_open,
release:	unionfs_release,
flush:	unionfs_flush,
};

struct file_operations unionfs_main_fops =
{
llseek:	unionfs_llseek,
read:	unionfs_read,
write:	unionfs_write,
readdir:	unionfs_file_readdir,
poll:	unionfs_poll,
ioctl:	unionfs_ioctl,
mmap:	unionfs_mmap,
open:	unionfs_open,
flush:	unionfs_flush,
release:	unionfs_release,
fsync:	unionfs_fsync,
fasync:	unionfs_fasync,
lock:	unionfs_lock,
		/* not needed: readv */
		/* not needed: writev */
		/* not implemented: sendpage */
		/* not implemented: get_unmapped_area */
};

/*
 * Local variables:
 * c-basic-offset: 8
 * End:
 * vim:shiftwidth=4
 */
