/* ==================================================== ======== ======= *
 *
 *  uubox.cpp
 *  Ubit Project [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uubox.cpp	ubit:03.06.04"
#include <iostream>
#include <udefs.hpp>
#include <uerror.hpp>
#include <ubrick.hpp>
#include <ucall.hpp>
#include <ustr.hpp>
#include <ucond.hpp>
#include <uctrl.hpp>
#include <ubox.hpp>
#include <uboxImpl.hpp>
#include <uwin.hpp>
#include <uview.hpp>
#include <uviewImpl.hpp>
#include <uevent.hpp>
#include <ustyle.hpp>
#include <ucolor.hpp>  
#include <uborder.hpp>
#include <ugraph.hpp>
#include <umenu.hpp>
#include <update.hpp>
using namespace std;

UStyle* UGroup::style = null;
const UStyle& UGroup::makeStyle() {
  if (!style) style = new UStyle();
  return *style;
}

const UStr* UGroup::getTextSeparator() const {
  return getStyle(null).textSeparator;
}

UGroup::UGroup(const UArgs& a0) : UCtrl() {
  setCmodes(UMode::CAN_SHOW | UMode::GROUP, true);

  // call add-initialization procedure of child (VOIR aussi: addlist)
  // !MODIF 19feb: duplication des liens
  for (ULink *la0 = a0.children->first(); la0 != null; la0 = la0->getNext()) {
    UBrick* child = la0->getChild();
    ULink* link = child->makeLink();
    link->setCond(la0);      //ne pas paumer les conditions!
    children.add(link);
    child->addingTo(link, this);  //!!init
  }
}

// NB: addlist() appelle addImpl comme add() ce qui permet d'eviter de 
// serieuses erreurs d'initialisation!

UGroup& UGroup::addlist(const UArgs& a0 /*, bool update*/) {
  bool should_upd = false;

  // !MODIF 19feb: duplication des liens (VOIR aussi: UGroup::UGroup)
  for (ULink *la0 = a0.children->first(); la0 != null; la0 = la0->getNext()) {
    UBrick* child = la0->getChild();
    ULink* link = child->makeLink();
    link->setCond(la0);      //ne pas paumer les conditions!
    addImpl(ELEM_LIST, child, link, -1, false, &should_upd);
  }

  if (/*update &&*/ should_upd) {
    UBox* box = this->boxCast();
    if (box && box->getViews() != null) box->update();
    // et pour les groups .....
  }

  return *this;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UStyle* UBox::style = null;

const UStyle& UBox::makeStyle() {
  if (!style) {
    style = new UStyle();
    style->textSeparator  = &ustr(" ");
    //style->orient         = UOrient::HORIZONTAL;
    style->orient         = UOrient::INHERIT;
    // abndonne: marche pas: NEW 30dec: UBox a la fois hflex et vflex
    style->local.halign   = UHalign::LEFT;
    style->local.valign   = UValign::FLEX;
    style->local.hspacing = 1;
    style->local.vspacing = 1;
    //style->local.vmargin  = 0;  style->local.hmargin  = 0;
    style->local.padding.set(0,0);
  }
  return *style;
}

UBox::UBox(const UArgs& a) : UGroup(a) {
  setCmodes(UMode::BOX, true);
  setCmodes(UMode::CAN_SELECT_TEXT, true);   // TST!
  setCmodes(UMode::GROUP, false);
}

ULink* UBox::makeLink() {return new UBoxLink(this);}

//UWhen& UBox::when(const UOn& cd) {
//  UWhen *wh = new UWhen(this, cd);
//  add(wh);
//  return *wh;
//}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// UHbox = horizontal box

UStyle* UHbox::style = null;
const UStyle& UHbox::makeStyle() {
  if (!style) {
    style = new UStyle();
    style->textSeparator  =  &ustr(" ");
    style->orient         = UOrient::HORIZONTAL;
    style->local.halign   = UHalign::LEFT;
    style->local.valign   = UValign::FLEX;
    style->local.hspacing = 1;
    style->local.vspacing = 1;
    //style->local.vmargin  = 0;  style->local.hmargin  = 0;
    style->local.padding.set(0,0);
  }
  return *style;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UStyle *UVbox::style = null;
const UStyle& UVbox::makeStyle() {
  if (!style) {
    style = new UStyle();
    style->textSeparator  = &ustr("\n");
    style->orient         = UOrient::VERTICAL;
    style->local.halign   = UHalign::FLEX;
    style->local.valign   = UValign::TOP;
    style->local.hspacing = 1;
    style->local.vspacing = 1;
    //style->local.vmargin  = 0;  style->local.hmargin  = 0;
    style->local.padding.set(0,0);
  }
  return *style;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UStyle *UBar::style = null;
const UStyle& UBar::makeStyle() {
  if (!style) {
    style = new UStyle();
    style->textSeparator  = &ustr("\t");
    style->orient         = UOrient::HORIZONTAL;
    style->local.halign   = UHalign::LEFT;
    style->local.valign   = UValign::FLEX;
    style->local.padding.set(2, 2); // ex 1
    style->local.hspacing = 6;  // ex 7
    style->local.vspacing = 6;  // ex 7
    style->local.border   = &UBorder::shadowOut;
  }
  return *style;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UStyle *UFlowbox::style = null;

const UStyle& UFlowbox::makeStyle() {
  if (!style) {
    style = new UStyle();
    style->viewStyle      = &UFlowView::style;
    style->textSeparator  = &ustr("\n");
    style->orient         = UOrient::HORIZONTAL;
    style->local.halign   = UHalign::FLEX;
    style->local.valign   = UValign::FLEX;
    style->local.hspacing = 1;
    style->local.vspacing = 1;
    style->local.padding.set(0,0);

    // keep initial size and don't change it when children are added/removed
    // (but size can be changed by user interaction and/or uvflex())
    style->local.width = UWidth::KEEP_SIZE;
    // relelated to canWarp()
  }
  return *style;
}

UFlowbox::UFlowbox(const UArgs& a): UBox(a) {
  setCmodes(UMode::CAN_SELECT_TEXT, true); 
}

/* ==================================================== ======== ======= */
/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// NB: les views ne sont plus detruites par ~BoxLink (supprime) mais par
// UGroup::remove (car sinon: mega-bug: les vues ne sont pas detruites 
// dans les enfants des UGroups)

void UBoxLink::addView(UView* v) {
  v->setNext(views);
  views = v;
}

UView* UBoxLink::getView(UDisp* disp) const {
  if (disp == null) return views; // default is the first view

  for (UView* v = views; v!= null; v = v->getNext()) {
    if (v->getDisp() == disp) return v;
  }
  return null; // not found
}

/* ==================================================== ======== ======= */
// NB: normal wins don't have a parent view (this field == null) 
// but UIncrust wins have one because they are laid out as normal boxes

UView* UBoxLink::getViewInside(UView* parview) const {
  for (UView* v = views; v != null; v = v->getNext()) {
    if (v->getParentView() == parview 
	//|| view->isDef(UView::WIN_SHARED))  //window view
	//|| hasWinChild()
	)
      return v;
  }
  return null;   //not found
}

UView* UBoxLink::getViewInsideSize(UView* parview, u_dim *w, u_dim *h) const {
  UView *v = getViewInside(parview);
  if (v) {*w = v->width; *h = v->height;}
  else {*w = 0; *h = 0;}
  return v;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UView* UBox::getView(int pos) const {
  int nn = 0;
  UView* last_view = null;

  for (ULinkLink* ll = parents.first(); ll != null; ll = ll->getNext()) {
    // UBoxLink* par construction
    UBoxLink *link = static_cast<UBoxLink*>(ll->link());
    for (UView* v = link->getViews(); v != null; v = v->getNext()) {
      if (nn == pos) return v;
      last_view = v;
      nn++;  //no de la vue (pas du parent comme ds fct ci-apres)
    }
  }

  if (pos == -1) return last_view;
  else return null;	// not found
}

/* ==================================================== ======== ======= */

int UBox::getViewCount() const  {
  int nn = 0;
  for (ULinkLink *ll = parents.first(); ll != null; ll = ll->getNext()) {
    // UBoxLink* par construction
    UBoxLink *link = static_cast<UBoxLink*>(ll->link());
    for (UView* v = link->getViews(); v != null; v = v->getNext())
      nn++;
  }
  return nn;
}

/* ==================================================== ======== ======= */

UView** UBox::getViews() const {
  int count;
  return getViews(count);
}

/* ==================================================== ======== ======= */

UView** UBox::getViews(int &count) const  {
  count = getViewCount();
  if (count <= 0) return null;

  UView **tab = new UView*[count+1];
  int nn = 0;

  for (ULinkLink *ll = parents.first(); ll != null; ll = ll->getNext()) {
    // UBoxLink* par construction
    UBoxLink *link = static_cast<UBoxLink*>(ll->link());
    for (UView* v = link->getViews(); v != null; v = v->getNext())
      tab[nn++] = v;
  }

  tab[nn] = null;  // null terminated
  return tab;
}

/* ==================================================== ======== ======= */

int UBox::getViews(std::vector<UView*>& tab) const {
  int count = 0;
  for (ULinkLink *ll = parents.first(); ll != null; ll = ll->getNext()) {
    // UBoxLink* par construction
    UBoxLink *link = static_cast<UBoxLink*>(ll->link());
    for (UView* v = link->getViews(); v != null; v = v->getNext()) {
      tab.push_back(v);
      count++;
    }
  }
  return count;
}

/* ==================================================== ======== ======= */

int UGroup::getViews(std::vector<UView*>& tab) const {
  for (ULinkLink *ll = parents.first(); ll != null; ll = ll->getNext()) {
    UGroup* parent = ll->getParent();
    parent->getViews(tab);
  }
  return tab.size();
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// Note: the box may be included several times in 'parview'.
// returns the 1st view

UView* UBox::getFirstViewInside(UView* parview) const {
  if (!parview) return null;

  for (ULinkLink *ll = parents.first(); ll != null; ll = ll->getNext()) {
    // UBoxLink* par construction
    UBoxLink *link = static_cast<UBoxLink*>(ll->link());

    for (UView* v = link->getViews(); v != null; v = v->getNext()) {
      if (v->getParentView() == parview
	  //window view comme pour UBoxLink::getViewInside ??
	  // (xx || view->isDef(UView::WIN_SHARED)) 
	  )
	return v;
    }
  }
  return null;	// not found
}

UView* UBox::getViewContaining(UView *childview) const { 
  for (UView *v = childview; v != null; v = v->getParentView()) {
    if (v->getBox() == this) return v;
  }
  return null; // not found
}

UView* UBox::getViewContaining(UEvent& e) const {
  return getViewContaining(e.getView());
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UGroup::close(int status) {
  if (status < 0) status = 0;
  show(false);

  if (isBmode(UMode::CONTAINER_CB)) {
    UEvent e(UEvent::close, null, null, NULL);
    e.setSource(this);
    fire(e, UOn::close);
  }
}

void UGroup::closeWin(int status /*,const UClass cl*/) {
  for (ULinkLink* ll = getParentList().first(); ll!=null; ll = ll->getNext()){
    UGroup* par = ll->getParent();
    // par == null if no other parent (UFrame or similar)
    if (par) {
      //UWin* win = dynamic_cast<UWin*>(par);
      UWin* win = par->winCast();
      if (win) win->close(status);
      else par->closeWin(status);
    }
  } 
}

// NB: ne ferme que la VIEW du dialog qui contient le bouton clique

void UGroup::closeWin(UEvent& e, int status /*,const UClass cl*/) {
  for (UView *v = e.getView(); v != null; v = v->getParentView()) {
    UBox *box = v->getBox();
    if (box) {
      UWin *win = null;
      //if (cl == null)
      //win = dynamic_cast<UWin*>(box);
      win = box->winCast();
      if (win) {
	win->close(status);
	break;
      }
    }
  }
}

/* ==================================================== ======== ======= */
/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

// att: parview n'est pas la view du parent mais du 1er box!
void UGroup::initView(ULink *selflink, UView *parview) {
  // Propager aux children
  for (ULink *ch = getChildLinks(); ch != null; ch = ch->getNext()) {
    // ? QUESTION: init pour tous ou que ugroup/ubox ?
    UGroup* gchild = ch->getChild()->groupCast();
    if (gchild) gchild->initView(ch, parview);
  }
}

// att: parview n'est pas la view du parent mais du 1er box!
void UBox::initView(ULink *selflink, UView *parview) {
  if (!parview) {
    error("internal@Ubox::initView", "No parent view!");
    return;
  }
  // par construction
  UBoxLink *bl = static_cast<UBoxLink*>(selflink);

  const UViewStyle *render;
  if (isDef(0,UMode::HAS_RENDERER)) {
    // si renderer defini dynamiquement dans la childlist
    render = (UViewStyle*) getAnyChild(&UBrick::isInstance<UViewStyle>);
  }
  else {  // default: rendre le renderer defini par le style de la Win
    //NB: closest_parent: ne sert en fait pas a grand chose (sauf dans des
    //cas de styles contextuels ou le renderer changerait suivant parent)
    /*22sept03
    UBox *closest_parent = parview->getBox();
    if (!closest_parent)
      error("internal@UBox::initView","No closest parent");
    render = getStyle(closest_parent).viewStyle;
    */
    render = getStyle(null).viewStyle;
  }

  UView *view = null;
  if (render)
    view = (render->makeView)(bl, parview, &(parview->wg()));
  else {
    error("warning@UBox::initView", UError::Cant_retreive_style);
    view = new UView(bl, parview, &(parview->wg()));
  }

  bl->addView(view);
  // Propager aux children
  UGroup::initView(selflink, view); // view, pas parview !
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// ATTENTION:
// les appels aux fcts vituelles ne marchent pas normalement dans les
// destructeurs (l'appel est fait avec la classe du destructeur,
// pas avec la classe effective de this)
//
// ==> appeler cleanup() DANS LE DESTRUCTEUR de la CLASSE de l'objet
//     pour que cleanup() soit appele avec ce type
//     adequate pour qu'il soit applique avec le type effectif de l'objet
// ==> CECI IMPOSE DE DEFINIR DES DESTRUCTEURS DANS LES SOUS-CLASSES

void UGroup::destructs() {
  // superclasses' cleanup() methods wont be called
  if (isBmode(UMode::DESTRUCTED)) return;

  // this object is now being destructed
  setBmodes(UMode::DESTRUCTING, true);
  
  // mod 4aug03 (n'etait jamais appele car dans UGroup::~UGroup)
  // doit etre avant sinon ne fera rien car CBs enleves
  if (isBmode(UMode::DESTRUCT_CB)) {
    UEvent e(UEvent::destruct, null, null, NULL);
    e.setSource(this);
    fire(e, UOn::destruct);
  }

  // removes and destroys all children (when applicable)
  removeAll(true, false);
  removeAllAttr(true, false);

  // detaches from parents (this is done after removing children
  // for improving view management)
  if (parents.first()) removeFromParents(false);

  // must be at the end (callbacks wont be called otherwise)
  setBmodes(UMode::DESTRUCTED, true);
}


UGroup::~UGroup() {  
  destructs();
  // tells the appli the object is being deleted
  UAppli::deleteNotifyAll(this);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

//class UPostChildEventNotify : public UFlag, public UMode {
class UPostChildEventNotify : public UFlagdef {
  UGroup firing_list;
public:
  static const UFlag FLAG;
  //static const char* FLAGNAME;

  UPostChildEventNotify(const UArgs& args) :
    //: UFlag(null), firing_list(args) {
    // name = FLAGNAME; // !!ATT: pas de strdup!
    UFlagdef(FLAG),
    firing_list(args) {
  }

  UGroup& getFiringList() {return firing_list;}
    
  void flagEvents(UEvent& e) {
    e.setPostChildEventNotify(true);
    e.addFlagdef(this);
  }
};

const UFlag UPostChildEventNotify::FLAG;

// NOTE IMPORTANTE:
// event flagging is meaningless in 'searchedView' mode:
// mouse coords must be defined in order to determine which objects
// are traversed by events
// ==> ceci implique que seules les UBox peuvent flagger les events
//     (car les UGroup n'ont pas de coords)
//
// ET DONC: onChildEvent n'existe pas pour les UGroup !

UBox& UBox::onPostChildEvent(const UArgs& args) {
  //NB: UFrom cree par cette methode et partage ensuite
  UPostChildEventNotify* propflag = new UPostChildEventNotify(args);

  // will flag events so that they will fire the firing_list
  // after being processed by the object's children
  addAttr(UOn::preChildEvent / ucall(propflag, 
				     &UPostChildEventNotify::flagEvents));

  //but: destruction auto de cet objet
  addAttr(this);
  return *this;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UGroup::fire(UEvent &e, const UOn &cond) const{

  // dont call callback if the object has already been destructed
  if (isBmode(UMode::DESTRUCTED)) return;

  // optimisation pour eviter de scanner la liste quand c'est inutile
  // principe: 
  // les bmodes de l'objet doivent (au moins) verifier un des bmodes du cond 
  // (c'est un un OU, on ne demande pas que toutes les conditions soient 
  // verifiees, une suffit), sinon on est sur qu'aucun callback correspondant
  // ne sera dans la liste
  // note: on exclut ce qui ne concerne pas les CALLBACKS

  u_modes cond_modes = cond.bmodes & UMode::CALLBACKS;
  if (!cond_modes || (bmodes & cond_modes) != 0) {

    // tester d'abord les callbacks dans le cache PUIS dans les children
    ULink *l; UCall *c;

    if (cache) {
      for (l = cache->first(); l != null; l = l->getNext()) {
        if (l->match(cond) && (c = dynamic_cast<UCall*>(l->getChild()))) {
        //if (l->match(cond) && (c = l->getChild()->callCast())) {
	  e.setCond(&cond); 
	  c->call(e);
          // return now if the object has been destructed by c->call()
          if (isBmode(UMode::DESTRUCTED)) return;
	}
      }
    }

    for (l = getChildLinks(); l != null; l = l->getNext()) {
      if (l->match(cond) && (c = dynamic_cast<UCall*>(l->getChild()))) {
      //if (l->match(cond) && (c = l->getChild()->callCast())) {
	e.setCond(&cond); 
	c->call(e);
        // return now if the object has been destructed by c->call()
        if (isBmode(UMode::DESTRUCTED)) return;
      }
    }
  }

  // detect events that *have already* been received by children.
  // dont call on cond UOn::preChildEvent othwerwise the parent
  // would call back its own setFlag callback and we would get
  // an infinite loop 

  if (e.isPostChildEventNotify() && &cond != &UOn::preChildEvent) {
    const UFlagdef** flagdefs = e.getFlagdefs();

    // temporarily set to false to avoid an infinite loop
    e.setPostChildEventNotify(false);

    for (u_count k = 0; k < e.getFlagdefCount(); k++) {

      // we verify that this flag is UPostChildEventNotify
      if (flagdefs[k]->getFlag() == &UPostChildEventNotify::FLAG) {
        
	UPostChildEventNotify* propflag = (UPostChildEventNotify*)flagdefs[k];
        //cerr << "Post fire "  << kk++ << endl;
	propflag->getFiringList().fire(e, cond);
        
        // return now if the object has been destructed by c->call()
        if (isBmode(UMode::DESTRUCTED)) return;
      }
    }
    
    // restore value 
    e.setPostChildEventNotify(true);
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

ULink* UGroup::getChildLinks() const {
  return children.first();
}

int UGroup::getChildCount()const {
  int kk = 0;
  for (ULink *l = children.first(); l!=null; l = l->getNext()) 
    kk++;   //k++ mis ici pour eviter les bugs... des compilateurs!
  return kk;
}

UBrick** UGroup::getChildren() const {
  int count;
  return getChildren(count);
}

UBrick** UGroup::getChildren(int& count) const {
  count = getChildCount();

  if (count <= 0) return null;
  else {
    UBrick **tab = new UBrick*[count+1];
    int kk = 0;
    for (ULink *l = children.first(); l!=null; l = l->getNext(), kk++) {
      tab[kk] = l->getChild();
    }
    tab[kk] = null;  // null terminated
    return tab;
  }
}

int UGroup::getChildren(std::vector<UBrick*>& chvect) const {
  int count = getChildCount();

  if (count <= 0) return 0;
  else {
    chvect.reserve(count);
    int kk = 0;
    for (ULink *l = children.first(); l!=null; l = l->getNext(), kk++) {
      chvect.push_back(l->getChild());
    }
    return count;
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

ULink* UGroup::getAttrLinks() const {
  return cache ? cache->first() : null;
}

int UGroup::getAttrCount() const {
  if (!cache) return 0;
  int kk = 0;
  for (ULink *l = cache->first(); l!=null; l = l->getNext()) 
    kk++;   //k++ mis ici pour eviter les bugs... des compilateurs!
  return kk;
}

UBrick** UGroup::getAttrs() const {
  int count;
  return getAttrs(count);
}

UBrick** UGroup::getAttrs(int &count) const {
  count = getAttrCount();
  
  if (count <= 0) return null;
  else {
    UBrick **tab = new UBrick*[count+1];
    int kk = 0;
    for (ULink *l = cache->first(); l!=null; l = l->getNext(), kk++) {
      tab[kk] = l->getChild();
    }
    tab[kk] = null;  // null terminated
    return tab;
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UListPos UListPos::none;

UListPos::UListPos()   {link = null; pos = -1;}
void UListPos::reset() {link = null; pos = -1;}
int UListPos::getPos() const {return pos;}

/* ==================================================== ======== ======= */

ULink* UGroup::getChildImpl(ChildListType chtype, UListPos& from) const {
  ULink* l = null;
  const UChain* chlist = (chtype == ATTR_LIST) ? cache : &children;
  if (!chlist) return null;

  if (&from == &UListPos::none) 
    l = chlist->first();
  else {
    if (!from.link) {
      if (from.pos != -1) return null;  // end of list
      from.pos = 0;
      l = chlist->first();
      if (l) from.link = l->getNext();
    }
    else {
      l = from.link;
      (from.pos)++;
      from.link = l->getNext();
    }
  }
  return l;
}

/* ==================================================== ======== ======= */

UBrick* UGroup::getChild(UListPos& from) const {
  ULink* l = getChildImpl(ELEM_LIST, from);
  return l ? l->getChild() : null;
}

UBrick* UGroup::getAttrChild(UListPos& from) const {
  ULink* l = getChildImpl(ATTR_LIST, from);
  return l ? l->getChild() : null;
}

UBrick* UGroup::getAnyChild(UListPos& from) const {
  ULink* l = getChildImpl(ATTR_LIST, from);
  if (!l) getChildImpl(ELEM_LIST, from);
  return l ? l->getChild() : null;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

ULink* UGroup::getChildImpl(ChildListType chtype, UListCall& predc,
                             bool& stat, UListPos& from) const {
  ULink* l = null;
  const UChain* chlist = (chtype == ATTR_LIST) ? cache : &children;
  if (!chlist) return null;

  if (&from == &UListPos::none) 
    l = chlist->first();
  else {
    if (!from.link) {
      if (from.pos != -1) return null;  // end of list
      from.pos = 0;
      l = chlist->first();
     }
    else {
      l = from.link;
    }
  }

  int k = 0;
  for ( ; l != null; l = l->getNext(), k++) {
    predc.call(*l->getChild());
    if (stat) {stat = false; break;}
  }

  if (&from != &UListPos::none) {
    from.pos += k; 
    if (l) from.link = l->getNext(); else from.link = null;
  }
  return l;
}


/* ==================================================== ======== ======= */

UBrick* UGroup::getChild(UListCall& predc, bool& stat, UListPos& from) const {
  ULink* l = getChildImpl(ELEM_LIST, predc, stat, from);
  return l ? l->getChild() : null;
}

UBrick* UGroup::getAttrChild(UListCall& predc, bool& stat, UListPos& from) const {
  ULink* l = getChildImpl(ATTR_LIST, predc, stat, from);
  return l ? l->getChild() : null;
}

UBrick* UGroup::getAnyChild(UListCall& predc, bool& stat, UListPos& from) const {
  ULink* l = getChildImpl(ATTR_LIST, predc, stat, from);
  if (!l) getChildImpl(ELEM_LIST, predc, stat, from);
  return l ? l->getChild() : null;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

static void sub_type(UBrick& obj, bool(predicate)(const UBrick*), bool* status) {
  *status = (predicate)(&obj);
}

UBrick* UGroup::getChild(bool(predicate)(const UBrick*), UListPos& from) const {
  bool stat = false;
  UListCall& predcall = ucall(predicate, &stat, sub_type);
  
  ULink* l = getChildImpl(ELEM_LIST, predcall, stat, from);
  delete& predcall;
  return l ? l->getChild() : null;
}

UBrick* UGroup::getAttrChild(bool(predicate)(const UBrick*), UListPos& from) const {
  bool stat = false;
  UListCall& predcall = ucall(predicate, &stat, sub_type);
  
  ULink* l = getChildImpl(ATTR_LIST, predcall, stat, from);
  delete& predcall;
  return l ? l->getChild() : null;
}

UBrick* UGroup::getAnyChild(bool(predicate)(const UBrick*), UListPos& from) const {
  bool stat = false;
  UListCall& predcall = ucall(predicate, &stat, sub_type);

  ULink* l = getChildImpl(ATTR_LIST, predcall, stat, from);
  if (!l) l = getChildImpl(ELEM_LIST, predcall, stat, from);

  delete& predcall;
  return l ? l->getChild() : null;
}

/* ==================================================== ======== ======= */

static void eq_obj_ptr(UBrick& obj, const UBrick* searched_obj, bool* status) {
  *status = (&obj == searched_obj);
}

int UGroup::getChildPos(const UBrick* obj, UListPos& from) const {
  bool stat = false;
  UListCall& predcall = ucall(obj, &stat, eq_obj_ptr);

  ULink* l = getChildImpl(ELEM_LIST, predcall, stat, from);
  delete& predcall;

  if (l) return from.getPos();
  else return -1;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// retourne l'element cherche ou null
// 'prevlink' pointe sur l'element PRECEDANT l'element cherche'
// NOTE: 'prevlink' vaut null si elt pas trouve OU si c'est le 1er
//       element de la liste (prevlink pointe alors sur "debut-de-liste")

ULink* UGroup::getChildImpl(ChildListType chtype, const UBrick* child,
                            int nth, ULink*& prevlink, int* ppos) const {
  prevlink = null;
  const UChain* chlist = (chtype == ATTR_LIST) ? cache : &children;
  if (!chlist) return null;

  int pos = 0;
  int n = 0;
  ULink *prevl = null;

  for (ULink* l = chlist->first(); l!=null; prevl = l, l = l->getNext(), pos++) 
    if (l->getChild() == child) {
      if (n != nth) n++;
      else {
	prevlink = prevl;   // NB: prevlink=null means 'first child'
	if (ppos) *ppos = pos; 
	return l;
      }
    }

  if (ppos) *ppos = pos; 
  return null; // not found
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// retourne l'element cherche ou null
// 'prevlink' pointe sur l'element PRECEDANT l'element cherche'
// !ATT: 'prevlink' EST INITIALISE si l'elt cherche' n'existe pas 
//       MAIS si son predecesseur existe (c'est utile pour add)
// NOTE: 'prevlink' vaut null si elt pas trouve OU si c'est le 1er
//       element de la liste (prevlink pointe alors sur "debut-de-liste")

ULink* UGroup::getChildImpl(ChildListType chtype, int pos,
                            ULink*& prevlink) const {
  prevlink = null;
  const UChain* chlist = (chtype == ATTR_LIST) ? cache : &children;
  if (!chlist) return null;

  ULink* l = null;

  if (pos == -1) {  
    for (l = chlist->first(); l!=null && l->getNext()!=null; 
	 prevlink = l, l = l->getNext());
    return l;
  }

  else {
    int kk = 0;
    ULink *prevl = null;

    for (l = children.first(); l!=null; prevl = l, l = l->getNext(), kk++) {
      if (kk == pos-1) prevlink = l;  // init prevlink meme si l pas trouve
      else if (kk == pos) return l;
    }
  }

  return null; // pos out of range
}

/* ==================================================== ======== ======= */

UBrick* UGroup::getChild(int pos) const {
  ULink* prevlink = null;
  ULink* link = getChildImpl(ELEM_LIST, pos, prevlink);
  return link ? link->getChild() : null;
}

int UGroup::getChildPos(const UBrick& child, int nth) const {
  ULink* prevlink = null;
  int pos;
  ULink* link = getChildImpl(ELEM_LIST, &child, nth, prevlink, &pos);
  return link ? pos : -1;
}

int UGroup::getChildPos(const UBrick* child, int nth) const {
  if (!child) {
    error("UGroup::getChildPos",UError::Null_argument);
    return -1;
  }
  else return getChildPos(*child, nth);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UGroup::add(UBrick& b, bool update) {
  addImpl(ELEM_LIST, &b, b.makeLink(), -1, update, null);
}

void UGroup::add(UBrick* b, bool update) {
  if (!b) error("UGroup::add", UError::Null_argument);
  else addImpl(ELEM_LIST, b, b->makeLink(), -1, update, null);
}

void UGroup::add(ULink* childlink, bool update) {
  UBrick* b;
  if (!childlink || !(b = childlink->getChild())) 
    error("UGroup::add", UError::Null_argument);
  else addImpl(ELEM_LIST, b, childlink, -1, update, null);
}

void UGroup::add(ULink& childlink, bool update) {
  UBrick* b;
  if (!(b = childlink.getChild())) 
    error("UGroup::add", UError::Null_argument);
  else addImpl(ELEM_LIST, b, &childlink, -1, update, null);
}

/* ==================================================== ======== ======= */

void UGroup::addOnce(UBrick& b, bool update) {
  ULink* prevlink = null;
  if (!getChildImpl(ELEM_LIST, &b, 0, prevlink, null)) {
    addImpl(ELEM_LIST, &b, b.makeLink(), -1, update, null);
  }
}

void UGroup::addOnce(UBrick* b, bool update) {
  if (!b) error("UGroup::addOnce", UError::Null_argument);
  else addOnce(*b, update);
}

/* ==================================================== ======== ======= */

void UGroup::insert(int pos, UBrick& b, bool update) {
  addImpl(ELEM_LIST, &b, b.makeLink(), pos, update, null);
}

void UGroup::insert(int pos, UBrick* b, bool update) {
  if (!b) error("UGroup::add", UError::Null_argument);
  else addImpl(ELEM_LIST, b, b->makeLink(), pos, update, null);
}

void UGroup::insert(int pos, ULink* childlink, bool update) {
  UBrick* b;
  if (!childlink || !(b = childlink->getChild())) 
    error("UGroup::add", UError::Null_argument);
  else addImpl(ELEM_LIST, b, childlink, pos, update, null);
}

void UGroup::insert(int pos, ULink& childlink, bool update) {
  UBrick* b;
  if (!(b = childlink.getChild())) 
    error("UGroup::add", UError::Null_argument);
  else addImpl(ELEM_LIST, b, &childlink, pos, update, null);
}

/* ==================================================== ======== ======= */

void UGroup::addAttr(UBrick* b, bool update) {
  if (!b) error("UGroup::addAttr", UError::Null_argument);
  else addImpl(ATTR_LIST, b, b->makeLink(), -1, update, null);
}

void UGroup::addAttr(UBrick& b, bool update) {
  addImpl(ATTR_LIST, &b, b.makeLink(), -1, update, null);
}

void UGroup::addAttr(ULink& childlink, bool update) {
  UBrick* b;
  if (!(b = childlink.getChild())) 
    error("UGroup::add", UError::Null_argument);
  else addImpl(ATTR_LIST, b, &childlink, -1, update, null);
}

/* ==================================================== ======== ======= */

void UGroup::insertAttr(int pos, UBrick* b, bool update) {
  if (!b) error("UGroup::insertAttr", UError::Null_argument);
  else addImpl(ATTR_LIST, b, b->makeLink(), pos, update, null);
}

void UGroup::insertAttr(int pos, UBrick& b, bool update) {
  addImpl(ATTR_LIST, &b, b.makeLink(), pos, update, null);
}

void UGroup::insertAttr(int pos, ULink& childlink, bool update) {
  UBrick* b;
  if (!(b = childlink.getChild())) 
    error("UGroup::add", UError::Null_argument);
  else addImpl(ATTR_LIST, b, &childlink, pos, update, null);
}

/* ==================================================== ======== ======= */

void UGroup::addImpl(ChildListType chtype, UBrick* child, ULink* childlink, 
		     int pos, bool update, bool* should_update) {
  UChain* chlist;
  if (chtype == ATTR_LIST) {
    if (!cache) cache = new UChain();  // creer cache si pas deja fait
    chlist = cache;
  }
  else chlist = &children;

  // add(.,-1) a un sens particulier: end-of-list et non last-child !
  if (pos == -1)
    // add at the end of the list
    chlist->add(childlink);
  else {
    ULink* prevlink = null;
    getChildImpl(chtype, pos, prevlink);
    // on ne peut ajouter que si l'element precedent existe OU si pos = 0
    if (prevlink || pos == 0) {
      // OK: add after prevlink (NB: if prevlink=null then beginning-of-list)
      chlist->insertAfter(childlink, prevlink);
    }
    else {
      error("UGroup::add", UError::Warning_out_of_range, pos);
      delete childlink;  //??
      return;
    }
  }

  // call add-initialization procedure of child
  child->addingTo(childlink, this);

  UGroup* gchild = child->groupCast();
  if (gchild) {
    // new: CAN_SELECT_CHILDREN means that this objects can select
    // its ARMable children (in the same way as a listbox)
    if (isCmode(UMode::CAN_SELECT_CHILDREN) && gchild->isCmode(UMode::CAN_ARM)) {
      gchild->setCmodes(UMode::CAN_SELECT, true);
    }

    initChildViews(gchild, childlink);
  }

  // !!NEW: faire aussi update() (mais jamais pour les ucall!)
  bool should_upd = 
    (bmodes & UMode::NO_AUTO_UPDATE) == 0
    && (child->groupCast() || child->elemCast() || child->propCast());

  // permettra eventuellement de faire update ensuite (dans le cas des listes
  // ou on veut faire un seul update)
  if (should_update) *should_update = should_upd;

  if (update && should_upd) {
    UBox* box = this->boxCast();
    if (box && box->getViews() != null) box->update();
    // et pour les groups .....
  }

  // a la fin
  if (isBmode(UMode::CONTAINER_CB)) {
    UEvent e(UEvent::add, null, null, NULL);
    e.setSource(this);
    e.setAux(child);
    fire(e, UOn::add);
  }
}

void UGroup::initChildViews(UGroup* gchild, ULink *childlink) {
  // il faut remonter dans l'arborescence pour retrouver les Views
  // (c'est fait par appel recursif de cette fct)
  for (ULinkLink *ll = parents.first(); ll != null; ll = ll->getNext()) {
    UGroup* grp = ll->getParent();
    if (grp) grp->initChildViews(gchild, childlink);
  }
}

void UBox::initChildViews(UGroup* gchild, ULink *childlink) {
  // retrouver toutes les vues pour tous les parents de box
  for (ULinkLink *ll = parents.first(); ll != null; ll = ll->getNext()) {

    //NB: UBoxLink par construction
    UBoxLink *link = static_cast<UBoxLink*>(ll->link());
    
    for (UView* view = link->getViews(); 
	 view != null;
	 view = view->getNext()) {
      if (view && view->isRealized())   // initialiser b et ses descendants
	// ? QUESTION: init pour tous ou que ugroup/ubox ?
	gchild->initView(childlink, view);
    }
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// Removes or deletes 'child' from the object's child list

void UGroup::remove(UBrick* child, bool remove_mode, bool update) {
  if (!child) {
    error("UGroup::remove", UError::Null_argument);
    //return null;
  }
  else /*return*/ remove(*child, (RemoveMode)remove_mode, update);
}

void UGroup::remove(UBrick& child, bool remove_mode, bool update) {
  ULink* prevlink = null;

  if (!getChildImpl(ELEM_LIST, &child, 0, prevlink, null)) {
    //error("UGroup::remove",  UError::Child_not_found, child.cname());
    //return null;
  }
  // NB: prevlink=null means remove first child 
  else /*return*/ removeImpl(ELEM_LIST, &child, prevlink, 
			     (RemoveMode)remove_mode, update, null);
}

void UGroup::remove(int pos, bool remove_mode, bool update) {
  ULink* prevlink = null;
  ULink* link = getChildImpl(ELEM_LIST, pos, prevlink);
  if (!link) {
    //error("UGroup::remove", UError::Warning_out_of_range, pos);
    //return null;
  }
  else /*return*/ removeImpl(ELEM_LIST, link->getChild(), prevlink, 
			     (RemoveMode)remove_mode, update, null);
}

void UGroup::removeAll(bool remove_mode, bool update) {
  bool should_upd = false;
  // on detruit le premier jusqu'a ce qu'il n'y en ait plus
  // cette version est compatible avec les multi-occurences :
  // il n'y aura pas de probleme si 'child' est detruit alors qu'il est 
  // present plusieurs fois dans la liste
  ULink *l = null;
  while ((l = children.first())) {
    // prevlink=null means remove first child 
    removeImpl(ELEM_LIST, l->getChild(), null, (RemoveMode)remove_mode,
	       /*update*/false, &should_upd);
  }

  if (update && should_upd) {
    UBox* box = this->boxCast();
    if (box && box->getViews() != null) box->update();
    // et pour les groups .....
  }
}

/* ==================================================== ======== ======= */

void UGroup::removeAttr(UBrick* child, bool remove_mode, bool update) {
  if (!child) {
    error("UGroup::removeAttr", UError::Null_argument);
    //return null;
  }
  else removeAttr(*child, remove_mode, update);
}

void UGroup::removeAttr(UBrick& child, bool remove_mode, bool update) {
  ULink* prevlink = null;

  if (!getChildImpl(ATTR_LIST, &child, 0, prevlink, null)) {
    //if (remove_mode != QUIET_DEL)
     // error("UGroup::remove",UError::Child_not_found, child.cname());
    //return null;
  }
  // NB: prevlink=null means remove first child 
  else /*return*/ removeImpl(ATTR_LIST, &child, prevlink, 
			     (RemoveMode)remove_mode, update, null);
}

void UGroup::removeAllAttr(bool remove_mode, bool update) {
  if (!cache) return;
  bool should_upd = false;
  // on detruit le premier jusqu'a ce qu'il n'y en ait plus
  // cette version est compatible avec les multi-occurences :
  // il n'y aura pas de probleme si 'child' est detruit alors qu'il est 
  // present plusieurs fois dans la liste
  ULink *l;
  while ((l = cache->first())) {
    // prevlink=null means remove first child 
    removeImpl(ATTR_LIST, l->getChild(), null, (RemoveMode)remove_mode,
	       /*update*/false, &should_upd);
  }

  if (update && should_upd) {
    UBox* box = this->boxCast();
    if (box && box->getViews() != null) box->update();
    // et pour les groups .....
  }
}
/* ==================================================== ======== ======= */
//NB: et si child pas dans l'obj ? faudrait pas le detruire et renvoyer null!

// ATT: prevlink : pas le lien qui pointe sur brick mais le precedent!
// si prevlink=null on enleve le 1er child

void UGroup::removeImpl(ChildListType chtype, UBrick *child, 
			ULink *prevlink, RemoveMode rem_mode, 
			bool update, bool* should_update) {

  UChain* chlist = (chtype == ATTR_LIST) ? cache : &children;
  if (!chlist) return; // null;

  UBrick* res = null;

  // mod: faire aussi update() (mais jamais pour les ucall!)
  // !att: a faire avant tout delete!
  bool should_upd =
    (bmodes & UMode::NO_AUTO_UPDATE) == 0
    && (child->groupCast() || child->elemCast() || child->propCast());

  // permettra eventuellement de faire update ensuite (dans le cas 
  // des listes ou on veut faire un seul update)
  if (should_update) *should_update = should_upd;

  // detacher child de ce parent (!ATT: PREVlink not selflink!)
  ULink *selflink = chlist->removeAfter(prevlink);
  child->removingFrom(selflink, this);

  // ~BoxLink ne detruit plus les views, car les vues
  // des descendants des UGroup n'etaient pas detruites
  if (child->groupCast()) {
    child->groupCast()->deleteRelatedViews(null, selflink);
  }

  // les UBox peuvent avoir des Active UBorders (typiquement les Scrollpane)
  // et il faut egalement detruire les vues des elements contenus dedans
  if (child->getSubGroup()) {
    child->getSubGroup()->deleteRelatedViews(null, selflink);
  }

  delete selflink; selflink = null;

  bool fake_or_deleted_par;
  bool valid_par = child->hasValidParent(fake_or_deleted_par);

  // si plus de vrais parents, il faut detacher les autres
  // ATT: il ne faut pas faire cet appel si removeImpl est appelee
  // de removeFromParents sinon bouclage/plantage !

  if (rem_mode != REMOVE_FROM_PARENTS && !valid_par && fake_or_deleted_par)
    child->removeFromParents(false);

  // child n'est pas detruit dans les cas suivants: 
  // 1) auto_del est faux
  // 2) il a au moins un VRAI parent PAS en cours de destruction
  // 3) il est reference
  // 4) il n'a pas ete cree par new (ou a ete rendu indeletable)
  if ((rem_mode != true /*&& rem_mode != QUIET_DEL*/)
      || valid_par
      || child->getRefCount() > 0
      || !child->isBmode(UMode::CAN_DEL)
      ) {
    res = child;    // child pas detruit => le retourner
  }
  else {
    delete child;
    res = null;
   }

  if (update && should_upd) {
    UBox* box = this->boxCast();
    if (box && box->getViews() != null) box->update();
    // et pour les groups .....
  }

  if (isBmode(UMode::CONTAINER_CB)) {  //deplace apres
    UEvent e(UEvent::remove, null, null, NULL);
    e.setSource(this);
    e.setAux(child);   // !ATT child may have been deleted
    fire(e, UOn::remove);
  }
  // return res;
}

/* ==================================================== ======== ======= */
// supprimer les descendants de cette vue dans les enfants et (descendants)
// ATTENTION: il faut passer au travers des UGroup pour detruire les vues
// de ses descendants de type UBox 

void UGroup::deleteRelatedViews(UView* parview, ULink* link) {
  UGroup* parent;
  if (!link || !(parent = link->getParent())) return;

  std::vector<UView*> viewlist;
  // a tous les coups il manque les vues des Active Borders....
  parent->getViews(viewlist);

  for (unsigned int k = 0; k < viewlist.size(); k++)
    deleteRelatedViews2(viewlist[k]);
}

/* ==================================================== ======== ======= */

void UBox::deleteRelatedViews(UView* parview, ULink* link) {
  if (!link) return;
  UBoxLink* boxlink = static_cast<UBoxLink*>(link);
  UView* nextview = null;
  UView* prevview = null;

  UView* view_list = boxlink->getViews();
  for (UView* view = view_list; view!=null; view = nextview) {
 
    nextview = view->getNext();

    if (!parview  // cas appel initial par removeImpl()
        || view->getParentView() == parview) {

      deleteRelatedViews2(view);
      
      if (prevview) prevview->setNext(nextview);
      else boxlink->views = nextview;

      delete view;
      view = null;
    }
    
    if (view) prevview = view;
  }
}

/* ==================================================== ======== ======= */
// NOTE: a la diffrence des UBox, les Views des UWin sont partagees sauf
// dans le cas ou elles appartiennent a des UDisp diffrents
// La regle de destruction est donc differente: on ne detruit les vues
// que cette UWin est en cours de destruction (UMode::DESTRUCTING)
// ou si c'est une softwin qui depend d'une hardwin en cours de destr.

void UWin::deleteRelatedViews(UView* parview, ULink* link) {
  //if (!link) return;
  //UBoxLink* boxlink = static_cast<UBoxLink*>(link);
  UView* nextview = null;
  UView* prevview = null;

  // BUG GENIAL: win->getWinViews() et les boxlink->getViews()   !!!!!
  // ne correspondent pas toujours !!!
  UView* view_list = getWinViews();
  for (UView* view = view_list; view!=null; view = nextview) {

    nextview = view->getNext();

    bool do_del = true;
    if (isHardwin()) {
      if (isBmode(UMode::DESTRUCTING)) do_del = true;
      else do_del = false;
    }
    else {
      UWin* hardwin = view->getHardwin();
      if (hardwin->isBmode(UMode::DESTRUCTING)) do_del = true;
      else do_del = false;
    }

    if (do_del) {
      deleteRelatedViews2(view);
      if (prevview) prevview->setNext(nextview);
      else {
        // BUG GENIAL: win->getWinViews() et les boxlink->getViews()   !!!!!
        // ne correspondent pas toujours !!!
        //boxlink->views = nextview;
        winviews = nextview;

        // !! horreur!
        for (ULinkLink* ll = getParentList().first();
             ll != null; ll = ll->getNext()) {
          UBoxLink *link = static_cast<UBoxLink*>(ll->link());
          link->views = nextview;
        }
      }
      delete view;
      view = null;
    }

    if (view) prevview = view;
  }
}

/* ==================================================== ======== ======= */
// NOTES:
// - ne pas oublier que les UGroup peuvent avoir des enfants de type UBox
//   et qu'il faut donc aussi detruire ces vues
// - les UBox peuvent avoir des Active UBorders (typiquement les Scrollpane)
//   et il faut egalement detruire les vues des elements contenus dedans
//   (cas du getSubGroup()). De plus ces UBorders peuvent etre dans l'ATTR_LIST

void UGroup::deleteRelatedViews2(UView* parview) {
  
  for (ULink *ch = getAttrLinks(); ch != null; ch = ch->getNext()) {
    UBrick* child = ch->getChild();
    /* NE PAS FAIRE CELA:
    if (child->boxCast())
      child->boxCast()->deleteRelatedViews(parview, ch);
    else if (child->groupCast())
      child->groupCast()->deleteRelatedViews2(parview);
    else
     */ if (child->getSubGroup())
      child->getSubGroup()->deleteRelatedViews2(parview);
  }

  for (ULink *ch = getChildLinks(); ch != null; ch = ch->getNext()) {
    UBrick* child = ch->getChild();
    
    if (child->boxCast()) 
      child->boxCast()->deleteRelatedViews(parview, ch);
    else if (child->groupCast())
      child->groupCast()->deleteRelatedViews2(parview);
    else if (child->getSubGroup())
      child->getSubGroup()->deleteRelatedViews2(parview);
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

int UGroup::getTextLength(bool recursive) const {
  int len = 0;
  UStr *str = null;
  UGroup *grp = null;
  //bool grp_found = false;

  const UStr* sep = getTextSeparator();
  if (sep && sep->length() <= 0) sep = null;

  for (ULink *ch = children.first(); ch != null; ch = ch->getNext()) {
    UBrick *b = ch->getChild();

    if ((str = b->strCast())) { 
      len += str->length();
    }
    
    // new:14sep03: ne pas entrer dans les UWin 
    else if (recursive && (grp = b->groupCast()) && !grp->winCast()) {
      if (sep /*&& grp_found*/)	len += sep->length();
      //grp_found = true;
      len += grp->getTextLength(recursive);
    }    
  }
  return len;
}

char *UGroup::getTextData(char* ptrpos, bool recursive) const{
  UStr *str = null;
  UGroup *grp = null;
  //bool grp_found = false;

  const UStr* sep = getTextSeparator();
  if (sep && sep->length() <= 0) sep = null;

  for (ULink *ch = children.first(); ch != null; ch = ch->getNext()) {
    UBrick *b = ch->getChild();

    if ((str = b->strCast()) && str->chars()) {
      strcpy(ptrpos, str->chars());
      ptrpos += str->length();
    }

    // new:14sep03: ne pas entrer dans les UWin 
    else if (recursive && (grp = b->groupCast()) && !grp->winCast()) {
      if (sep /*&& grp_found*/) {
	strcpy(ptrpos, sep->chars());
	ptrpos += sep->length();
      }
      //grp_found = true;
      ptrpos = grp->getTextData(ptrpos, recursive);
    }
  }

  return ptrpos;
}

/* ==================================================== ======== ======= */

void UGroup::copyText(UStr& res, bool recursive) const {
  int len = getTextLength(recursive);
  if (len <= 0) {res = null; return;}
  
  char *data = (char*)malloc((len+1)*sizeof(char));
  if (!data) {
    error("UGroup::copyText", UError::No_more_memory);
    return;
  }

  char *ptrpos = getTextData(data, recursive);
  if (ptrpos != data + len) {
    error("internal@UGroup::copyText","erroneous length");
    res = null; return;
  }
  *ptrpos = '\0';  // null terminated!

  res.setImpl(data, len, true);
}

UStr UGroup::copyText(bool recursive) const {
  UStr res;
  copyText(res, recursive);
  return res;
}

/* ==================================================== ======== ======= */
/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UGroup::highlight(bool st) {
}
/*
  // mettre a jour cette Box virtuelle pour toutes ses realisations 
  // a l'ecran (ie. pour tous ses parents)
  ULinkLink *llk;

  for (llk = parents.first(); llk != null; llk = llk->getNext()) {
    //NB: au moins un UBoxLink * par construction
    UBoxLink *link = llk->boxLinkCast(); !!!FAUX: c'est un UGroup maintenant!
    for (int kv = 0; kv < link->viewCount; kv++) {

      UView *view = link->views[kv];

      // check views[k]!=null (some views may have been deleted)
      if (view && view->isRealized()) {
	// BUG: fait le changeAction N+1 fois !!!
	if (st > 0)
	  link->b->boxCast()->setState(UOn::HIGHLIGHTED);
	else 
	  link->b->boxCast()->setState(UOn::IDLE);

	UEvent e(UEvent::highlight, view->getWinView(), null);
	e.locateSource(view);
	updateView(&e, view, UUpdate::paint);
      }
    }
  }
}
*/
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */

