//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.0.1, Copyright (C) Peter A. Buhr 1994
// 
// uSocket.h -- Nonblocking UNIX Socket I/O
// 
// Author           : Peter Buhr
// Created On       : Tue Mar 29 17:04:36 1994
// Last Modified By : Peter A. Buhr
// Last Modified On : Sun Aug  8 08:18:46 2004
// Update Count     : 295
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 


#ifndef __U_SOCKET_H__
#define __U_SOCKET_H__


#include <uFile.h>
//#include <uDebug.h>


#pragma __U_NOT_USER_CODE__


#include <cstdio>
#include <netdb.h>					// MAXHOSTNAMELEN on Solaris
#include <sys/param.h>					// MAXHOSTNAMELEN on BSD


//######################### uSocket #########################


class uSocket {
    friend class uSocketServer;				// access: access, uSocket, ~uSocket
    friend class uSocketAccept;				// access: access
    friend class uSocketClient;				// access: access, uSocket, ~uSocket

    uIOaccess access;

    const int domain, type, protocol;			// copies for information purposes

    uSocket( uSocket & );				// no copy
    uSocket &operator=( uSocket & );			// no assignment

    void uCreateSocket( int domain, int type, int protocol );
    uSocket( int domain, int type, int protocol );	// creation restricted to friends
    virtual ~uSocket();
  public:
    uDualEvent uFailure;
    friend uDualEvent uFailure;				// access: access
    uDualEvent uOpenFailure;
    uDualEvent uCloseFailure;
    friend uDualEvent uCloseFailure;			// access: access
}; // uSocket


//######################### uSocketIO #########################


class uSocketIO : public uFileIO {
    int _send( char *buf, int len, int flags );
    int _sendto( char *buf, int len, int flags, struct sockaddr *to, socklen_t tolen );
    int _sendmsg( const struct msghdr *msg, int flags );
    int _recv( char *buf, int len, int flags );
    int _recvfrom( char *buf, int len, int flags, struct sockaddr *from, socklen_t *fromlen );
    int _recvmsg( struct msghdr *msg, int flags );
  protected:
    struct inetAddr : sockaddr_in {
	int code;
	char _name[MAXHOSTNAMELEN+1];
	void getinetAddr( unsigned short port, const char *name );
	inetAddr( unsigned short port, const char *name = "" );
    }; // inetAddr

    struct sockaddr *saddr;				// default send/receive address
    socklen_t saddrlen;					// size of send address
    socklen_t baddrlen;					// size of address buffer (UNIX/INET)

    uSocketIO( uIOaccess &acc, struct sockaddr *saddr );
  public:
    uMutex const struct sockaddr *getsockaddr();	// must cast result to sockaddr_in or sockaddr_un
    uMutex int getsockname( struct sockaddr *name, socklen_t *len );
    uMutex int getpeername( struct sockaddr *name, socklen_t *len );
    int send( char *buf, int len, int flags = 0, uDuration *timeout = NULL );
    int sendto( char *buf, int len, int flags = 0, uDuration *timeout = NULL );
    int sendto( char *buf, int len, struct sockaddr *to, socklen_t tolen, int flags = 0, uDuration *timeout = NULL );
    int sendmsg( const struct msghdr *msg, int flags = 0, uDuration *timeout = NULL );
    int recv( char *buf, int len, int flags = 0, uDuration *timeout = NULL );
    int recvfrom( char *buf, int len, int flags = 0, uDuration *timeout = NULL );
    int recvfrom( char *buf, int len, struct sockaddr *from, socklen_t *fromlen, int flags = 0, uDuration *timeout = NULL );
    int recvmsg( struct msghdr *msg, int flags = 0, uDuration *timeout = NULL );
}; // uSocketIO


//######################### uSocketServer #########################


uMonitor uSocketServer : public uSocketIO {
    friend class uSocketAccept;				// access: socket, _accept, acceptor, unacceptor, ~uSocketAccept

    int acceptorCnt;					// number of simultaneous acceptors using server
    uSocket socket;					// one-to-one correspondance between server and socket

    uMutex void acceptor();
    uMutex void unacceptor();
    int _accept( struct sockaddr *adr, socklen_t *len );
    void uCreateSocketServer1( const char *name, int type, int protocol, int backlog );
    void uCreateSocketServer2( unsigned short port, int type, int protocol, int backlog );
    void uCreateSocketServer3( unsigned short *port, int type, int protocol, int backlog );
  public:
    // AF_UNIX
    uSocketServer( const char *name, int type = SOCK_STREAM, int protocol = 0, int backlog = 10 );
    // AF_INET, local host
    uSocketServer( unsigned short port, int type = SOCK_STREAM, int protocol = 0, int backlog = 10 );
    uSocketServer( unsigned short *port, int type = SOCK_STREAM, int protocol = 0, int backlog = 10 );
    virtual ~uSocketServer();

    void uSetClient( struct sockaddr *addr, socklen_t len );
    void uGetClient( struct sockaddr *addr, socklen_t *len );

    uDualEvent uFailure;
    friend uDualEvent uFailure;				// access: socket
    uDualEvent uOpenFailure;
    uDualEvent uCloseFailure;
    uDualEvent uReadFailure;
    uDualEvent uReadTimeout;
    uDualEvent uWriteFailure;
    uDualEvent uWriteTimeout;
}; // uSocketServer


//######################### uSocketAccept #########################


uMonitor uSocketAccept : public uSocketIO {
    uSocketServer &socketserver;			// many-to-one correspondence among acceptors and server
    uIOaccess access;
    bool openAccept;					// current open accept for socket ?
    uDuration *timeout;
    struct sockaddr *adr;
    socklen_t *len;

    const uSocket &sock() const;
    void uCreateSocketAcceptor( uDuration *timeout, struct sockaddr *adr, socklen_t *len );
    int _close();
  public:
    uSocketAccept( uSocketServer &s, struct sockaddr *adr = NULL, socklen_t *len = NULL );
    uSocketAccept( uSocketServer &s, uDuration *timeout, struct sockaddr *adr = NULL, socklen_t *len = NULL );
    uSocketAccept( uSocketServer &s, bool doAccept, struct sockaddr *adr = NULL, socklen_t *len = NULL );
    uSocketAccept( uSocketServer &s, uDuration *timeout, bool doAccept, struct sockaddr *adr = NULL, socklen_t *len = NULL );
    virtual ~uSocketAccept();

    void accept();
    void accept( uDuration *timeout );
    void close();

    uDualEvent uFailure;
//    friend uDualEvent uFailure;
    uDualEvent uOpenFailure;
    uDualEvent uOpenTimeout;
    uDualEvent uCloseFailure;
    uDualEvent uReadFailure;
    uDualEvent uReadTimeout;
    uDualEvent uWriteFailure;
    uDualEvent uWriteTimeout;
}; // uSocketAccept


//######################### uSocketClient #########################


uMonitor uSocketClient : public uSocketIO {
    uSocket socket;					// one-to-one correspondence between client and socket
    char *tmpnm;					// temporary file for communicate with UNIX datagram

    void uCreateSocketClient1( const char *name, uDuration *timeout, int type, int protocol );
    void uCreateSocketClient2( unsigned short port, const char *name, uDuration *timeout, int type, int protocol );
  public:
    // AF_UNIX
    uSocketClient( const char *name, int type = SOCK_STREAM, int protocol = 0 );
    uSocketClient( const char *name, uDuration *timeout, int type = SOCK_STREAM, int protocol = 0 );
    // AF_INET, local host
    uSocketClient( unsigned short port, int type = SOCK_STREAM, int protocol = 0 );
    uSocketClient( unsigned short port, uDuration *timeout, int type = SOCK_STREAM, int protocol = 0 );
    // AF_INET, other host
    uSocketClient( unsigned short port, const char *name, int type = SOCK_STREAM, int protocol = 0 );
    uSocketClient( unsigned short port, const char *name, uDuration *timeout, int type = SOCK_STREAM, int protocol = 0 );
    virtual ~uSocketClient();

    void uSetServer( struct sockaddr *addr, socklen_t len );
    void uGetServer( struct sockaddr *addr, socklen_t *len );

    uDualEvent uFailure;
    friend uDualEvent uFailure;
    uDualEvent uOpenFailure;
    uDualEvent uOpenTimeout;
    uDualEvent uCloseFailure;
    uDualEvent uReadFailure;
    uDualEvent uReadTimeout;
    uDualEvent uWriteFailure;
    uDualEvent uWriteTimeout;
}; // uSocketClient


//######################### uSocket (cont) #########################


uDualEvent uSocket::uFailure : public uIOFailure {
    const uSocket &_socket;
  public:
    uFailure( const uSocket &s, const char *const msg );
    const uSocket &socket() const;
    virtual void defaultTerminate() const;
}; // uSocket::uFailure


uDualEvent uSocket::uOpenFailure : public uSocket::uFailure {
  protected:
    const int domain;
    const int type;
    const int protocol;
  public:
    uOpenFailure( const uSocket &socket, const int domain, const int type, const int protocol, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocket::uOpenFailure


uDualEvent uSocket::uCloseFailure : public uSocket::uFailure {
  public:
    uCloseFailure( const uSocket &socket, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocket::uCloseFailure


//######################### uSocketServer (cont) #########################


uDualEvent uSocketServer::uFailure : public uSocket::uFailure {
    const uSocketServer &_server;
  public:
    uFailure( const uSocketServer &s, const char *const msg );
    const uSocketServer &server() const;
    int fileDescriptor() const;
    virtual void defaultTerminate() const;
}; // uSocketServer::uFailure


uDualEvent uSocketServer::uOpenFailure : public uSocketServer::uFailure {
  protected:
    const unsigned short port;
    char _name[FILENAME_MAX];
    const int domain;
    const int type;
    const int protocol;
    const int backlog;
  public:
    uOpenFailure( const uSocketServer &server, const unsigned short port, const char *const name, int domain, int type, int protocol, int backlog, const char *const msg );
    const char *const name() const;			// return socket name, or NULL when hostname is used
    virtual void defaultTerminate() const;
}; // uSocketServer::uOpenFailure


uDualEvent uSocketServer::uCloseFailure : public uSocketServer::uFailure {
    const int acceptorCnt;
  public:
    uCloseFailure( const uSocketServer &server, const int acceptorCnt, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketServer::uCloseFailure


uDualEvent uSocketServer::uReadFailure : public uSocketServer::uFailure {
  protected:
    const char *buf;
    const int len;
    const int flags;
    const struct sockaddr *from;
    const socklen_t *fromlen;
    const uDuration *timeout;
  public:
    uReadFailure( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketServer::uReadFailure


uDualEvent uSocketServer::uReadTimeout : public uSocketServer::uReadFailure {
  public:
    uReadTimeout( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketServer::uReadTimeout


uDualEvent uSocketServer::uWriteFailure : public uSocketServer::uFailure {
  protected:
    const char *buf;
    const int len;
    const int flags;
    const struct sockaddr *to;
    const int tolen;
    const uDuration *timeout;
  public:
    uWriteFailure( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketServer::uWriteFailure


uDualEvent uSocketServer::uWriteTimeout : public uSocketServer::uWriteFailure {
  public:
    uWriteTimeout( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketServer::uWriteTimeout


//######################### uSocketAccept (cont) #########################


uDualEvent uSocketAccept::uFailure : public uSocket::uFailure {
    const uSocketAccept &_acceptor;
  public:
    uFailure( const uSocketAccept &acceptor, const char *const msg );
    const uSocketAccept &acceptor() const;
    const uSocketServer &server() const;
    int fileDescriptor() const;
    virtual void defaultTerminate() const;
}; // uSocketAccept::uFailure


uDualEvent uSocketAccept::uOpenFailure : public uSocketAccept::uFailure {
  protected:
    const uDuration *timeout;
    const struct sockaddr *adr;
    const socklen_t *len;
  public:
    uOpenFailure( const uSocketAccept &acceptor, const uDuration *timeout, const struct sockaddr *adr, const socklen_t *len, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketAccept::uOpenFailure


uDualEvent uSocketAccept::uOpenTimeout : public uSocketAccept::uOpenFailure {
  public:
    uOpenTimeout( const uSocketAccept &acceptor, const uDuration *timeout, const struct sockaddr *adr, const socklen_t *len, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketAccept::uOpenTimeout


uDualEvent uSocketAccept::uCloseFailure : public uSocketAccept::uFailure {
  public:
    uCloseFailure( const uSocketAccept &acceptor, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketAccept::uCloseFailure


uDualEvent uSocketAccept::uReadFailure : public uSocketAccept::uFailure {
  protected:
    const char *buf;
    const int len;
    const int flags;
    const struct sockaddr *from;
    const socklen_t *fromlen;
    const uDuration *timeout;
  public:
    uReadFailure( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketAccept::uReadFailure


uDualEvent uSocketAccept::uReadTimeout : public uSocketAccept::uReadFailure {
  public:
    uReadTimeout( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketAccept::uReadTimeout


uDualEvent uSocketAccept::uWriteFailure : public uSocketAccept::uFailure {
  protected:
    const char *buf;
    const int len;
    const int flags;
    const struct sockaddr *to;
    const int tolen;
    const uDuration *timeout;
  public:
    uWriteFailure( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketAccept::uWriteFailure


uDualEvent uSocketAccept::uWriteTimeout : public uSocketAccept::uWriteFailure {
  public:
    uWriteTimeout( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketAccept::uWriteTimeout


//######################### uSocketClient (cont) #########################


uDualEvent uSocketClient::uFailure : public uSocket::uFailure {
    const uSocketClient &_client;
  public:
    uFailure( const uSocketClient &client, const char *const msg );
    const uSocketClient &client() const;
    int fileDescriptor() const;
    virtual void defaultTerminate() const;
}; // uSocketClient::uFailure


uDualEvent uSocketClient::uOpenFailure : public uSocketClient::uFailure {
  protected:
    unsigned short port;
    char _name[FILENAME_MAX];
    const uDuration *timeout;
    const int domain;
    const int type;
    const int protocol;
  public:
    uOpenFailure( const uSocketClient &client, const unsigned short port, const char *const name, uDuration *timeout, const int domain, const int type, const int protocol, const char *const msg );
    const char *const name() const;
    virtual void defaultTerminate() const;
}; // uSocketClient::uOpenFailure


uDualEvent uSocketClient::uOpenTimeout : public uSocketClient::uOpenFailure {
  public:
    uOpenTimeout( const uSocketClient &client, const unsigned short port, const char *const name, uDuration *timeout, const int domain, const int type, const int protocol, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketClient::uOpenTimeout


uDualEvent uSocketClient::uCloseFailure : public uSocketClient::uFailure {
  public:
    uCloseFailure( const uSocketClient &acceptor, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketClient::uCloseFailure


uDualEvent uSocketClient::uReadFailure : public uSocketClient::uFailure {
  protected:
    const char *buf;
    const int len;
    const int flags;
    const struct sockaddr *from;
    const socklen_t *fromlen;
    const uDuration *timeout;
  public:
    uReadFailure( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketClient::uReadFailure


uDualEvent uSocketClient::uReadTimeout : public uSocketClient::uReadFailure {
  public:
    uReadTimeout( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketClient::uReadTimeout


uDualEvent uSocketClient::uWriteFailure : public uSocketClient::uFailure {
  protected:
    const char *buf;
    const int len;
    const int flags;
    const struct sockaddr *to;
    const int tolen;
    const uDuration *timeout;
  public:
    uWriteFailure( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketClient::uWriteFailure


uDualEvent uSocketClient::uWriteTimeout : public uSocketClient::uWriteFailure {
  public:
    uWriteTimeout( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketClient::uWriteTimeout


#pragma __U_USER_CODE__

#endif // __U_SOCKET_H__


// Local Variables: //
// compile-command: "gmake install" //
// End: //
