/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_canvas.h 
 * @brief Generic canvas main class.
 * 
 */

#ifndef __GC_CANVAS_H__
#define __GC_CANVAS_H__

#ifdef MAKEDLL
  #define GENERIC_CANVAS_API  __declspec(dllexport)
#else
  #define GENERIC_CANVAS_API  __declspec(dllimport)
#endif

#include <map>
#include <vector>

#include "tree.h"   // libxml
#include "parser.h" // libxml
#include "myx_gc_utilities.h"
#include "myx_gc_layer.h"

//----------------------------------------------------------------------------------------------------------------------

// An opaque handle to a rendering context. Must be provided by the viewer.
#ifdef _WINDOWS
  typedef HDC GCContext;
#else
  typedef GLXContext GCContext;
#endif // ifdef _WINDOWS

using namespace std;

class CFigure;
class CLayer;

#pragma warning(disable: 4251) // Disable warning about DLL interface for template classes.

// The list of templates is an associated list of names and display lists, which are usually loaded by CGenericCanvas::AddTemplatesFromFile.
typedef map<string, int, CStringLessThan> CTemplates;
typedef map<string, int, CStringLessThan>::iterator CTemplateIterator;

// A list of layers.
typedef vector<CLayer*> CLayers;
typedef vector<CLayer*>::iterator CLayerIterator;

//----------------- General listeners ----------------------------------------------------------------------------------

class CGCListener;

typedef vector<CGCListener*> CListeners;
typedef vector<CGCListener*>::iterator CListenerIterator;

/**
 * The general listener class is used to notify users of the canvas about general events like repaints and errors.
 * This class is only an abstract class and must get a concrete implemention in the application.
 * All Listener classes are meant to be a means for calling back the application. They are implemented and instanciated
 * in the application and must be freed there. Don't forget to remove the listener class before you free it!
 */
class GENERIC_CANVAS_API CGCListener
{
public:
  virtual void __cdecl OnChange(void* AObject, TGCChangeReason Reason) = 0; // Called for changes in the canvas.
  virtual void __cdecl OnError(const char* Message) = 0;  // Called by the canvas when an error occured.
  virtual void __cdecl OnInvalidate(void) = 0;             // Called by the canvas when the viewer needs a refresh.
};

//----------------- Hit testing structures -----------------------------------------------------------------------------

typedef struct tagHitEntry
{
  CFigureInstance* Instance;
  double ZMin;
  double ZMax;
} THitEntry;

typedef vector<THitEntry> THitEntries;

/**
 * The CHitResult class is used to collect a number of figures that are located at a given point in the canvas.
 * Instances are created by calling CGenericCanvas::GetHitTestInfo
 */
class GENERIC_CANVAS_API CHitResults
{
  friend class CGenericCanvas;
private:
  THitEntries FEntries;
protected:
  void AddHit(CFigureInstance* Instance, double Min, double Max);
public:
  virtual int __cdecl Count(void);
  virtual THitEntry* __cdecl Get(int I);
  virtual void __cdecl Release(void);
};

//----------------- CGenericCanvas -------------------------------------------------------------------------------------

/**
 * CGenericCanvas is the main class of the library and is the base for all further functionality (e.g. it creates and 
 * maintains the model). Instances are created via the exported CreateGenericCanvas function (if called from non C++ 
 * languages). CGenericCanvas serves as the controller in the model-view-controller pattern, which is used here and 
 * communicates with the viewer via callbacks.
 * The viewer is platform specific and must be implemented individually. It is responsible to create a canvas controller class.
 *
 * @see CreateGenericCanvas
 */
class GENERIC_CANVAS_API CGenericCanvas 
{
  friend class CGCModel;
private:
  GCContext FContext;                  // The always active OpenGL rendering cntext.
  COLORREF FBackgroundColor;           // The background color of the context/viewer.
  TGCViewport FViewport;               // Coordinates of the output area within the viewer window.
  CTemplates FTemplates;               // A list of display lists associated with a name, so they can be used in figures.
  CGCModel* FModel;                    // The model this canvas is controlling.
  bool FIsPicking;                     // true if the model is currently picking figures.
  CLayers FLayers;                     // A list of layers currently in this canvas.
  CSelectionLayer* FSelectionLayer;    // The selection layer is special.
  int FUpdateCount;                    // If > 0 then the canvas is currently updating internal structures. Display will not update.
  double FZoomX;                       // The current horizontal zoom factor. It's a normal scaling factor but for the 
                                       // whole canvas (scales everything).
  double FZoomY;                       // The current vertical zoom factor.
  double FOffsetX;                     // A translation offset to enable scrolling (panning) in a zoomed scene.
  double FOffsetY;                     // The vertical translation offset.

  // Listeners
  CListeners FListeners;               // A list of listeners, which want to get notfied about general things happening in the canvas.
protected:
  void ApplyViewport(void);
  void ClearBuffers();
  GLuint FindTemplate(const string& Name);
public:
  CGenericCanvas(GCContext Context);
  ~CGenericCanvas(void);

  virtual void __cdecl AddLayer(CLayer* Layer);
  virtual void __cdecl AddListener(CGCListener* Listener);
  virtual TGCError __cdecl AddTemplatesFromFile(const char* FileName);
  virtual void __cdecl AddToSelection(CFigureInstance* Instance);
  virtual void __cdecl BeginUpdate(void);
  virtual void __cdecl Change(void* AObject, TGCChangeReason Reason);
  virtual void __cdecl CheckError(void);
  virtual void __cdecl ClearSelection(void);
  virtual void __cdecl ClearTemplates(void);
  virtual CLayer* __cdecl CreateLayer(const char* Name, TGCLayerType Type);
  virtual void __cdecl EndUpdate(void);
  virtual void __cdecl Error(const char* Message);
  virtual CHitResults* __cdecl GetHitTestInfoAt(const int X, const int Y);
  virtual CGCModel* __cdecl GetModel(void);
  virtual bool __cdecl GetProperty(TProperty Property, double& Value);
  virtual bool __cdecl GetProperty(TProperty Property, int& Value);
  virtual TGCSelectionInfo __cdecl GetSelectionInfo(const int X, const int Y);
  virtual void __cdecl Invalidate(void);
  virtual void __cdecl InvalidateSelectionBounds(CFigureInstance* Instance);
  virtual bool __cdecl IsUpdating(void);
  virtual void __cdecl Release(void);
  virtual void __cdecl RemoveFromSelection(CFigureInstance* Instance);
  virtual void __cdecl RemoveLayer(CLayer* Layer);
  virtual void __cdecl RemoveListener(CGCListener* Listener);
  virtual void __cdecl Render(void);
  virtual void __cdecl SetBackgroundColor(COLORREF NewColor);
  virtual void __cdecl SetOffset(double X, double Y);
  virtual bool __cdecl SetProperty(TProperty Property, double Value);
  virtual bool __cdecl SetProperty(TProperty Property, int Value);
  virtual void __cdecl SetViewport(int Left, int Top, int Width, int Height);
  virtual void __cdecl SetViewportV(TGCViewport* NewViewport);
  virtual void __cdecl SetZoom(double X, double Y);
  virtual void __cdecl ShowSelection(bool Visible);
};

//----------------------------------------------------------------------------------------------------------------------

// Factory function to create a generic canvas. This function is exported and must be used by the viewer implementations
// to actually create a canvas instance. This is the only way to get hold of a generic canvas instance.
extern "C" GENERIC_CANVAS_API CGenericCanvas* CreateGenericCanvas(GCContext Context);

#endif // __GC_CANVAS_H__
