// osx_dac.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

#ifdef MACOSX

#ifdef __GNUG__
#pragma implementation
#endif

#include <assert.h>
#include <unistd.h>
#include <errno.h>
#include <InterViews/action.h>
#include "application.h"
#include "osx_dac.h"
#include "typeconvert.h"
#include "progressaction.h"

#undef DEBUG

class MyTimer {
public:
	MyTimer(OSXConverter *obj) : _obj(obj) {}
	~MyTimer() {}
	void run(ProgressAction *action);
private:
	OSXConverter	*_obj;
};

void
MyTimer::run(ProgressAction *action)
{
#ifdef DEBUG
	fprintf(stderr, "MyTimer::run() starting\n");
#endif
	while(_obj->running())
	{
		float percent = _obj->_bytesDone / (float) _obj->_bytesToDo;
#ifdef DEBUG
		fprintf(stderr, "MyTimer::run() before action - percent %g\n", percent);
#endif
		if ((*action)(percent)) {
			_obj->stop();
			break;
		}
		usleep(50000);
	}
}

OSXConverter::OSXConverter()
	: _recDeviceID(kAudioDeviceUnknown), _playDeviceID(kAudioDeviceUnknown),
	  _audioBufferSize(0), _timer(new MyTimer(this)) {
}

OSXConverter::~OSXConverter() {
	stop();
	delete _timer;
}

int
OSXConverter::currentPlayLevel() const {
	return 0;
}

int
OSXConverter::currentRecordLevel() const {
	return 0;
}

boolean
OSXConverter::setPlayLevel(int volume) {
	return true;
}

boolean
OSXConverter::setRecordLevel(int volume) {
	return true;
}

boolean
OSXConverter::setSpeakerOutput(boolean toOn) {
	return true;
}

boolean
OSXConverter::isPlayableFormat(DataType type) {
	return type != MuLawData;
}

int
OSXConverter::checkDataType(DataType type) {
    switch(type) {
    case FloatData:
    case DoubleData:
    case IntData:
    case ShortData:
    case SignedCharData:
    case UnsignedCharData:
		return true;
        break;
	default:
		return false;
    }
}

int
OSXConverter::checkSampleRate(int sampleRate) {
    return true;
}

int
OSXConverter::checkChannels(int chans) {
    return chans == 1 || chans == 2;
}
int
OSXConverter::doConversion(ProgressAction* progress) {
	_bytesDone = 0;
	// Start the device.
	OSStatus err = AudioDeviceStart(_playDeviceID, audioPlayProc);
	int status = (err == kAudioHardwareNoError);
	if (status == true) {
		_timer->run(progress);
	}
	else {
		char string[32];
		int errval = err;
		sprintf(string, "AudioDeviceStart returned token '%.4s'",
				(char *) &errval);
		errno = 0;
		error(string);
		stop();
	}
	return status;
}

int
OSXConverter::doRecording(ProgressAction* progress) {
	_bytesDone = 0;
	// Start the device.
	OSStatus err = AudioDeviceStart(_recDeviceID, audioRecordProc);
	int status = (err == kAudioHardwareNoError);
	if (status == true) {
		_timer->run(progress);
	}
	else {
		char string[32];
		int errval = err;
		errno = 0;
		sprintf(string, "AudioDeviceStart returned token '%.4s'", 
				(char *) &errval);
		error(string);
		stop();
	}
	return status;
}

int
OSXConverter::pause() {
	return true;
}

int
OSXConverter::resume() {
	return true;
}

int
OSXConverter::stop() {
	OSStatus err;
	if (recording()) {
		err = AudioDeviceStop(_recDeviceID, audioRecordProc);
		err = AudioDeviceRemoveIOProc(_recDeviceID, audioRecordProc);
	}
	else {
		err = AudioDeviceStop(_playDeviceID, audioPlayProc);
		err = AudioDeviceRemoveIOProc(_playDeviceID, audioPlayProc);
	}
	return (err == kAudioHardwareNoError) ? Super::stop() : false;
}

int
OSXConverter::doConfigure() {
	AudioDeviceID devID;
	UInt32 size = sizeof(devID);
	OSStatus err = AudioHardwareGetProperty(
					willRecord() ? 
						kAudioHardwarePropertyDefaultInputDevice :
						kAudioHardwarePropertyDefaultOutputDevice,
					&size,
				   (void *) &devID);
	if (err != kAudioHardwareNoError || devID == kAudioDeviceUnknown) {
		return error("Cannot find default audio device.");
	}
	if (willRecord())
		_recDeviceID = devID;
	else
		_playDeviceID = devID;
	// Get current output format	
	size = sizeof(_deviceFormat);
	err = AudioDeviceGetProperty(devID, 
								  0,
								  willRecord(),
								  kAudioDevicePropertyStreamFormat, 
								  &size, 
								  &_deviceFormat);
	if (err != kAudioHardwareNoError) {
		return error("Can't get audio device format");
	}
#ifdef DEBUG
	printf("doConfigure: stream format:\n\tsr %g\n\tfmtID 0x%x\n\tfmtFlags 0x%x\n\tbytesperpckt %u\n\tframesperpckt %u\n\tbytesperframe %u\n\tchannelsperframe %u\n\tbitsperchannel %u\n",
			_deviceFormat.mSampleRate, _deviceFormat.mFormatID,
			 _deviceFormat.mFormatFlags, _deviceFormat.mBytesPerPacket,
			_deviceFormat.mFramesPerPacket, _deviceFormat.mBytesPerFrame,
			_deviceFormat.mChannelsPerFrame, _deviceFormat.mBitsPerChannel);
#endif
	Boolean writeable;
	// Set sample format
	size = sizeof(writeable);
	err = AudioDeviceGetPropertyInfo(devID, 
   									0,
								    willRecord(),
								    kAudioDevicePropertyStreamFormat,
									&size,
									&writeable);
	if (err != kAudioHardwareNoError) {
		return error("Can't get audio device property");
	}
#ifdef DEBUG
	printf("doConfigure: stream format property is %s\n",
		   writeable ? "writeable" : "not writeable");
#endif
	if (writeable)
	{
		AudioStreamBasicDescription sFormat = _deviceFormat;
		sFormat.mSampleRate = sampleRate();
		sFormat.mChannelsPerFrame = 2;	// always open in stereo
		size = sizeof(sFormat);
		err = AudioDeviceSetProperty(devID,
									 NULL,
									 0,
								     willRecord(),
								     kAudioDevicePropertyStreamFormat,
									 size,
									 (void *)&sFormat);
		if (err != kAudioHardwareNoError) {
			return error("Can't set audio stream format");
		}
	}
	size = sizeof(writeable);
	// Set buffer size
	err = AudioDeviceGetPropertyInfo(devID, 
   									0,
								    willRecord(),
								    kAudioDevicePropertyBufferSize, 
									&size,
									&writeable);
	if (err != kAudioHardwareNoError) {
		return error("Can't get audio device property");
	}
	// default buffer size is 0.1 second's worth of sound
	float bufferTime = 0.1;
	const char *res = Application::getGlobalResource("OSXBufferTime");
	if (res)
		bufferTime = atof(res);
	// We cannot set the buffer size larger than the segment we wish to play.
	if (duration() < bufferTime)
		bufferTime = duration();
	// Audio buffer is always floating point
	unsigned int bufferSize = sizeof(float) *
					iround(float(channels()) * sampleRate() * bufferTime);
#ifdef DEBUG
	fprintf(stderr, "requesting buffer size %d\n", bufferSize);
#endif

	if (writeable) {
		size = sizeof(bufferSize);
		err = AudioDeviceSetProperty(devID,
									 NULL,
									 0,
								     willRecord(),
								     kAudioDevicePropertyBufferSize,
									 size,
									 (void *)&bufferSize);
		// NB: even if err != kAudioHardwareNoError, we continue and use
		//  the default buffer size.
	}
	// Get the actual buffer size.  (The device may not want to change.)
	size = sizeof(_audioBufferSize);
	err = AudioDeviceGetProperty(devID,
								 0,
								 0,
								 kAudioDevicePropertyBufferSize,
								 &size,
								 &_audioBufferSize);
	if (err != kAudioHardwareNoError) {
		return error("Can't get audio output device buffer size");
	}
	// Register our callback function with the HAL.
	err = AudioDeviceAddIOProc(devID,
							   willRecord() ? audioRecordProc : audioPlayProc, 
							   (void *) this);
	if (err != kAudioHardwareNoError) {
		return error("Cannot register callback function with device.");
	}
	_bytesToDo = dataSize();
	return true;
}

OSStatus
OSXConverter::audioRecordProc(AudioDeviceID			inDevice,
							  const AudioTimeStamp	*inNow,
							  const AudioBufferList	*inInputData,
							  const AudioTimeStamp	*inInputTime,
							  AudioBufferList		*outOutputData,
							  const AudioTimeStamp	*inOutputTime,
							  void					*object)
{
	OSXConverter *converter = (OSXConverter *) object;
	unsigned inBytes = converter->_audioBufferSize;
	const DataType dataType = converter->dataType();
	const int sampSize = type_to_sampsize(dataType);
	const int sndChans = converter->channels();
	const int outFrames = (converter->_bytesToDo - converter->_bytesDone) / (sampSize * sndChans);
	int frames = inBytes / (2 * sizeof(float));	// always stereo
	unsigned offset = converter->_bytesDone;

	if (outFrames < frames)
		frames = outFrames;

#ifdef DEBUG
	printf("audioRecordProc: aud buf sz: %d, bytesdone = %u bytestotal = %u\n",
			inBytes, offset, converter->_bytesToDo);
#endif

	if (frames <= 0) {
#ifdef DEBUG
		printf("audioRecordProc: calling stop()\n");
#endif
		converter->stop();
		return kAudioHardwareNoError;
	}

	char *sound = (char *)converter->pointerToData();
	char *toBuf = sound + offset;
	float *inbuf = (float *) inInputData->mBuffers[0].mData;
	double scale;
	int in;

	switch (dataType) {
	case DoubleData:
	{
		double *to = (double *) toBuf;
		if (sndChans == 1) {
			scale = 32767.0 * 0.707;
			for (in=0; in < frames; ++in) {
				to[in] = (double) (inbuf[0] + inbuf[1]) * scale;
				inbuf += 2;
			}
		}
		else {
			scale = 32767.0;
			for (in=0; in < frames; ++in) {
				to[0] = (double)(inbuf[0] * scale);	
				to[1] = (double)(inbuf[1] * scale);	
				inbuf += 2;
				to += 2;
			}
		}
	}
		break;
	case FloatData:
	{
		float *to = (float *) toBuf;
		if (sndChans == 1) {
			scale = 32767.0 * 0.707;
			for (in=0; in < frames; ++in) {
				to[in] = (float) ((inbuf[0] + inbuf[1]) * scale);
				inbuf += 2;
			}
		}
		else {
			scale = 32767.0;
			for (in=0; in < frames; ++in) {
				to[0] = (float)(inbuf[0] * scale);	
				to[1] = (float)(inbuf[1] * scale);	
				inbuf += 2;
				to += 2;
			}
		}
	}
		break;
	case IntData:
	{
		int *to = (int *) toBuf;
		if (sndChans == 1) {
			scale = MAXINT * 0.707;
			for (in=0; in < frames; ++in) {
				to[in] = (int) ((inbuf[0] + inbuf[1]) * scale);
				inbuf += 2;
			}
		}
		else {
			scale = MAXINT;
			for (in=0; in < frames; ++in) {
				to[0] = (int)(inbuf[0] * scale);	
				to[1] = (int)(inbuf[1] * scale);	
				inbuf += 2;
				to += 2;
			}
		}
	}
		break;
	case ShortData:
	{
		short *to = (short *) toBuf;
		if (sndChans == 1) {
			scale = 32767.0 * 0.707;
			for (in=0; in < frames; ++in) {
				to[in] = (short) ((inbuf[0] + inbuf[1]) * scale);
				inbuf += 2;
			}
		}
		else {
			scale = 32767.0;
			for (in=0; in < frames; ++in) {
				to[0] = (short)(inbuf[0] * scale);	
				to[1] = (short)(inbuf[1] * scale);	
				inbuf += 2;
				to += 2;
			}
		}
	}
		break;
	case SignedCharData:
	{
		signed char *to = (signed char *) toBuf;
		if (sndChans == 1) {
			scale = 127.0 * 0.707;
			for (in=0; in < frames; ++in) {
				to[in] = (signed char) ((inbuf[0] + inbuf[1]) * scale);
				inbuf += 2;
			}
		}
		else {
			scale = 127.0;
			for (in=0; in < frames; ++in) {
				to[0] = (signed char)(inbuf[0] * scale);	
				to[1] = (signed char)(inbuf[1] * scale);	
				inbuf += 2;
				to += 2;
			}
		}
	}
		break;
	case UnsignedCharData:
	{
		signed char *to = (signed char *) toBuf;
		if (sndChans == 1) {
			scale = 127.0 * 0.707;
			for (in=0; in < frames; ++in) {
				to[in] = (unsigned char) ((inbuf[0] + inbuf[1]) * scale) + 0x80;
				inbuf += 2;
			}
		}
		else {
			scale = 127.0;
			for (in=0; in < frames; ++in) {
				to[0] = (unsigned char)(inbuf[0] * scale) + 0x80;
				to[1] = (unsigned char)(inbuf[1] * scale) + 0x80;	
				inbuf += 2;
				to += 2;
			}
		}
	}
		break;
	default:
		exit(1);
	}

	unsigned outBytes = frames * sndChans * sampSize;
//	inInputData->mBuffers[0].mDataByteSize = outBytes;
	converter->_bytesDone += outBytes;
	return kAudioHardwareNoError;
}

OSStatus
OSXConverter::audioPlayProc(AudioDeviceID			inDevice,
						  const AudioTimeStamp	*inNow,
						  const AudioBufferList	*inInputData,
						  const AudioTimeStamp	*inInputTime,
						  AudioBufferList		*outOutputData,
						  const AudioTimeStamp	*inOutputTime,
						  void					*object)
{
	OSXConverter *converter = (OSXConverter *) object;
	unsigned outBytes = converter->_audioBufferSize;
	const DataType dataType = converter->dataType();
	const int sampSize = type_to_sampsize(dataType);
	const int sndChans = converter->channels();
	const int inFrames = (converter->_bytesToDo - converter->_bytesDone) / (sampSize * sndChans);
	int frames = outBytes / (2 * sizeof(float));	// always stereo
	unsigned offset = converter->_bytesDone;

	if (inFrames < frames)
		frames = inFrames;

#ifdef DEBUG
	printf("audioPlayProc: aud buf sz: %d, bytesdone = %u bytestotal = %u\n",
			outBytes, offset, converter->_bytesToDo);
#endif

	if (frames <= 0) {
#ifdef DEBUG
		printf("audioPlayProc: calling stop()\n");
#endif
		converter->stop();
		return kAudioHardwareNoError;
	}

	char *sound = (char *)converter->pointerToData();
	char *frombuf = sound + offset;
	float *outbuf = (float *) outOutputData->mBuffers[0].mData;
	double scale;
	int in;

	switch (dataType) {
	case DoubleData:
	{
		scale = 1.0/converter->peakAmplitude();
		double *from = (double *) frombuf;
		if (sndChans == 1)
			for (in=0; in < frames; ++in) {
				outbuf[1] = outbuf[0] = (float)(from[in] * scale);	 
				outbuf += 2;
			}
		else
			for (in=0; in < frames; ++in) {
				outbuf[0] = (float)(from[0] * scale);	
				outbuf[1] = (float)(from[1] * scale);	
				from += 2;
				outbuf += 2;
			}
	}
		break;
	case FloatData:
	{
		scale = 1.0/converter->peakAmplitude();
		float *from = (float *) frombuf;
		if (sndChans == 1)
			for (in=0; in < frames; ++in) {
				outbuf[1] = outbuf[0] = (float)(from[in] * scale);	 
				outbuf += 2;
			}
		else
			for (in=0; in < frames; ++in) {
				outbuf[0] = (float)(from[0] * scale);	
				outbuf[1] = (float)(from[1] * scale);	
				from += 2;
				outbuf += 2;
			}
	}
		break;
	case IntData:
	{
		scale = 1.0/MAXINT;
		int *from = (int *) frombuf;
		if (sndChans == 1)
			for (in=0; in < frames; ++in) {
				outbuf[1] = outbuf[0] = (float)(from[in] * scale);	 
				outbuf += 2;
			}
		else
			for (in=0; in < frames; ++in) {
				outbuf[0] = (float)(from[0] * scale);	
				outbuf[1] = (float)(from[1] * scale);	
				from += 2;
				outbuf += 2;
			}
	}
		break;
	case ShortData:
	{
		scale = 1.0/MAXSHORT;
		short *from = (short *) frombuf;
		if (sndChans == 1)
			for (in=0; in < frames; ++in) {
				outbuf[1] = outbuf[0] = (float)(from[in] * scale);	 
				outbuf += 2;
			}
		else
			for (in=0; in < frames; ++in) {
				outbuf[0] = (float)(from[0] * scale);	
				outbuf[1] = (float)(from[1] * scale);	
				from += 2;
				outbuf += 2;
			}
	}
		break;
	case SignedCharData:
	{
		scale = 1.0/128;
		signed char *from = (signed char *) frombuf;
		if (sndChans == 1)
			for (in=0; in < frames; ++in) {
				outbuf[1] = outbuf[0] = (float)(from[in] * scale);	 
				outbuf += 2;
			}
		else
			for (in=0; in < frames; ++in) {
				outbuf[0] = (float)(from[0] * scale);	
				outbuf[1] = (float)(from[1] * scale);	
				from += 2;
				outbuf += 2;
			}
	}
		break;
	case UnsignedCharData:
	{
		scale = 1.0/128;
		signed char *from = (signed char *) frombuf;
		if (sndChans == 1)
			for (in=0; in < frames; ++in) {
				outbuf[1] = outbuf[0] = (float)((from[in]+0x80) * scale);	 
				outbuf += 2;
			}
		else
			for (in=0; in < frames; ++in) {
				outbuf[0] = (float)((from[0]+0x80) * scale);	
				outbuf[1] = (float)((from[1]+0x80) * scale);	
				from += 2;
				outbuf += 2;
			}
	}
		break;
	default:
		exit(1);
	}

	unsigned inBytes = frames * sndChans * sampSize;
	outOutputData->mBuffers[0].mDataByteSize = frames * 2 * sizeof(float);
	converter->_bytesDone += inBytes;
	return kAudioHardwareNoError;
}

#endif	/* MACOSX */
