// lowpassfilter.c

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <math.h>
#include "application.h"
#include "localdefs.h"
#include "lowpassfilter.h"
#include "request.h"
#include "valuerequester.h"

class LowPassRequester : public ValueRequester<double> {
public:
	LowPassRequester(LowPassFilter *);
protected:
	void configureRequest(Request *);
private:
	LowPassFilter* client;
};

LowPassRequester::LowPassRequester(LowPassFilter* l)
	:  ValueRequester<double>("Low Pass Filter Selected Region:",
				  "Gain Factor:", l->gain),
	   client(l) {}

void
LowPassRequester::configureRequest(Request* request) {
	Range allFreqs(0.0, client->sampRate()/2.0);
	request->appendValue("Cutoff Frequency:", &client->cutoff, allFreqs);
	ValueRequester<double>::configureRequest(request);
}

//********

double LowPassFilter::_savedCutoff = 1000.0;
double LowPassFilter::_savedGain = 1.0;

LowPassFilter::LowPassFilter(Data* output)
	: SimpleFunction(output, true),
	  cutoff(_savedCutoff), gain(_savedGain), g1(0), g2(0) {}

LowPassFilter::LowPassFilter(Data* output, double feedback)
		: SimpleFunction(output, true),
		  gain(1), g1(1.0 - feedback), g2(feedback) {
	initialize();
}

Requester *
LowPassFilter::createRequester() {
	return new LowPassRequester(this);
}

void
LowPassFilter::initialize() {
	clear();
	firstTime = true;
	setCoefficients(&g1, &g2);
	SimpleFunction::initialize();
}

void
LowPassFilter::saveConfig() {
	_savedCutoff = cutoff;
	_savedGain = gain;
}

void
LowPassFilter::restoreState() {
	clear();
	firstTime = true;
}

void
LowPassFilter::setCoefficients(double *cf1, double *cf2) {
	if(!*cf1 && !*cf2) {        // if these were not set in constructor
		double x = 2.0 - cos(cutoff * M_PI_2 / sampRate());
		*cf2 = x - sqrt(x * x - 1.0);	// feedback coeff.
		*cf1 = 1.0 - *cf2;		// gain coeff.
		*cf1 *= gain;			// gain multiplier
	}
}

void
LowPassFilter::operator () (double *input, int count) {
	if(firstTime) {
		past = input[0];
		firstTime = false;
	}
	for (int n = 0; n < count; n++) {
	    past = g1 * input[n] + g2 * past;
	    input[n] = past;
	}
}
