// interpolater.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "editor.h"
#include "ellipfilt.h"
#include "interpolater.h"
#include "localdefs.h"
#include "data.h"
#include "request.h"
#include "pitchconv.h"
#include "valuerequester.h"

class InterpolateRequester : public ValueRequester<double> {
	friend class Interpolater;
protected:
	InterpolateRequester(Interpolater* i)
		: ValueRequester<double>("Interpolate Selected Region to New Length:",
		                 "Gain Factor:", i->gain),
		  client(i) {}
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
protected:
	Interpolater* client;
};

void
InterpolateRequester::configureRequest(Request* request) {
	request->appendLabel("(Use either a Factor or Enter Desired Length)");
	request->appendValue(
		"Interpolation Factor:",
		&client->theInterpFactor,
		PositiveNumbers
	);
	request->appendValue("New Length:", &client->theNewLength,
	                     PositiveIntegers);
	ValueRequester<double>::configureRequest(request);
}

boolean
InterpolateRequester::confirmValues() {
	double ifactor = client->interpFactor();
	if(ifactor == 0.0) {
		Application::alert(
			"Zero is an illegal value for interpolation factor.");
		return false;
	}
	int oldLength = client->getInput()->length();
	ifactor = (ifactor != 1.0) ?
		1.0/ifactor : double(oldLength)/client->theNewLength;
	client->setInterpFactor(ifactor);
	return true;
}

//********

class TransposeRequester : public ValueRequester<double> {
	friend class Interpolater;
protected:
	TransposeRequester(Interpolater* i)
		: ValueRequester<double>("Transpose Selected Region:", "Gain Factor:", i->gain),
		  client(i) {}
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
private:
	enum TransposeMode { Ratio = 1, OctavePoint = 2, LinearOctave = 4 };
	Interpolater* client;
	static double theTransposeFactor;
	static ChoiceValue theTransposeMode;
};

double TransposeRequester::theTransposeFactor = 1.0;
ChoiceValue TransposeRequester::theTransposeMode = TransposeRequester::OctavePoint;

void
TransposeRequester::configureRequest(Request* request) {
	request->appendValue("Transposition Interval/Ratio:",
	                     &theTransposeFactor);
	request->appendChoice(
		"Transposition mode:",
		"|Ratio|Octave Pt. P.C.|Linear Octave P.C.|",
		&theTransposeMode,
		true
	);
	request->appendChoice(
		"Apply Tracking Filter:",
		"|Use Filter|",
		&client->useFilter,
		false);
	ValueRequester<double>::configureRequest(request);
}

boolean
TransposeRequester::confirmValues() {
	double factor = theTransposeFactor;
	double ifactor = 0;
	switch(theTransposeMode) {
	case Ratio:
		ifactor = factor;
		break;
	case OctavePoint:
		factor = Frequency::octaveFromPitch(factor);
		ifactor = Frequency::cpsFromOctave(10.0+factor) / Frequency::cpsFromOctave(10.0);  
		break;
	case LinearOctave:
		ifactor = Frequency::cpsFromOctave(10.0+factor) / Frequency::cpsFromOctave(10.0);  
		break;
	default:
		Application::die("TransposeRequester::confirmValues: invalid choice");
		break;
	}
	if(ifactor <= 0.0) {
		Application::alert("Ratio mode value must be a positive number.");
		return false;
	}
	client->setInterpFactor(ifactor);
	return true;
}

//********

double Interpolater::_savedInterpFactor = 1.0;
double Interpolater::_savedGain = 1.0;
ChoiceValue Interpolater::_savedUseFilter = false;

Interpolater::Interpolater(Data* input, Data* output, double factor,
			   boolean filt, double inGain)
	: InputOutputFunction(input, output, InputQLen, OutputQLen),
		processMode(None), theNewLength(input->length()),
	  	theInterpFactor(factor), gain(inGain), useFilter(filt),
		filter(nil) {
	initialize();
}

Interpolater::Interpolater(Data* input, Data* output, int newlen,
			   boolean filt, double inGain)
	: InputOutputFunction(input, output, InputQLen, OutputQLen),
		processMode(None),
		theNewLength(newlen),
		theInterpFactor(double(input->length())/newlen),
		gain(1.0), useFilter(filt), filter(nil) {
	initialize();
}

Interpolater::Interpolater(Data* input, Data* output, boolean transposing)
	: InputOutputFunction(input, output, InputQLen, OutputQLen),
		processMode(None),
		theNewLength(input->length()),
		theInterpFactor(_savedInterpFactor),
		gain(_savedGain), 
		useFilter(transposing ? _savedUseFilter : false),
		transpose(transposing),
		filter(nil) {}

Interpolater::~Interpolater() { delete filter; }

Requester *
Interpolater::createRequester() {
	if(transposing())
		return new TransposeRequester(this);
	else
		return new InterpolateRequester(this);
}

#define GAIN_ADJUST .99117

void
Interpolater::initialize() {
	clear();
	Super::initialize();
	// to avoid overflow in short int samples due to 2nd order interp
	double ifactor = interpFactor();
	gain *= (ifactor >= 1.0) ? 1.0 : GAIN_ADJUST;
	if(useFilter) {
		double cutoff = 0.0;
		if(ifactor > 1.0) {		// transposing upwards
			cutoff = sampRate() / (2.0 * ifactor);
			processMode = Pre;	// do processing before interpolation
		}
		else if(ifactor < 1.0) {	// transposing downwards
			cutoff = sampRate()  * ifactor / 2.0;
			processMode = Post;	// do processing after interpolation
		}
		if(processMode != None)
			filter = new EllipticalFilter(
				target(), .95 * cutoff, cutoff, 0.0, 1.0, 90.0
			);
	}
}

void
Interpolater::saveConfig() {
    _savedInterpFactor = theInterpFactor;
    // only save gain if user has altered it - dont save GAIN_ADJUST
    _savedGain = (gain == GAIN_ADJUST) ? 1.0 : gain;
    if (transposing())
        _savedUseFilter = useFilter;	// only saved if in transp mode
}

const char *
Interpolater::message() {
	return transposing() ? 
		(filtering() ? "Transposing and filtering..." : "Transposing...")
		: "Interpolating...";
}

Modifier *
Interpolater::create(DataEditor *ed) {
//	return new Interpolater(ed->currentSelection(), ed->copyBuffer());
	return nil;
}

boolean
Interpolater::areCompatibleLengths(int outlen, int newlen) {
	return outlen == newlen;
}

double
Interpolater::interpolate(double y_0, double y_1, double y_2, double fraction) {
	register double hy_2, hy_0, a, b, c;
	a = y_0;
	hy_0 = y_0/2.0;
	hy_2 =  y_2/2.0;
	b = (-3.0 * hy_0) + (2.0 * y_1) - hy_2;
	c = hy_0 - y_1 + hy_2;
	return(a + b * fraction + c * fraction * fraction);
}

void
Interpolater::restoreState() {
	Super::restoreState();
	clear();
	if(filter)
		filter->restoreState();
}

inline void
Interpolater::preProcess(double *input, int len) {
    if (processMode == Pre)
        (*filter)(input, len);
}

inline void
Interpolater::postProcess(double *input, int len) {
    if (processMode == Post)
        (*filter)(input, len);
}

int
Interpolater::processInput(boolean flushing) {
        static const int count = 1;	// XXX finish buffering code
	BUG("Interpolater::processInput()");
	// if interpolating counter is approaching input counter, get new input
	if((fractionCounter - double(incount)) >= -0.5) {
		voldsig = oldsig;
		oldsig = sig;
		double input = 0.0;
		if (!flushing) {
		    takeFromInQueue(&input, count);
		    input *= gain;
		}
		preProcess(&input, count);
		sig = input;	// XXX remove when buffering finished
		increment(count);
	}
	// while fractioncounter is lagging behind, do interp and increment
	while((double(incount) - fractionCounter) > 0.5) {
		double fraction = fractionCounter - incount + 2.0;
		double output = interpolate(voldsig, oldsig, sig, fraction);
		fractionCounter += interpFactor();
		postProcess(&output, count);
		if(addToOutQueue(&output, count) != count)
			return false;	// not done writing to output queue
	}
	return true;			// done writing to output queue
}
