/*
 *   mwmcmd.c -- Mwave Modem AT Command Parser
 *
 *  Written By: Paul Schroeder IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
#include <mwmparse.h>
#include <mwmparsi.h>

USHORT mwmCmdHostEqualCommand(STATEINFO *psi);
USHORT mwmCmdHostQuestionCommand(STATEINFO *psi);


/*****************************************************************************/
/* mwmCmdParseATcmd                                                          */
/*****************************************************************************/
/* This function is called to start parsing the remainder of the AT command  */
/* string.                                                                   */
/*                                                                           */
/* Returns usParserStatus (Combination of Error,Buffer Full, and Last Cmd    */
/* Information)                                                              */
/*****************************************************************************/
USHORT mwmCmdParseATcmd(STATEINFO *psi)
{
  USHORT  usParserStatus = 0;
  USHORT  usParm = 0;
  USHORT  usCommandComplete = 0;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdParseATcmd entry\n");
  do
  {
    /*************************************************************************/
    /* If the buffer overflows, we need to know what type of command we were */
    /* processing.  We will set the default here.  Some Commands are special */
    /* and might require more than one interrupt while processing a single   */
    /* command.  These commands will use this flag to tell the parser to     */
    /* continue a specific command rather than to continue to parse the AT   */
    /* buffer.                                                               */
    /*************************************************************************/
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;

    /*************************************************************************/
    /* Look at this character in AT Command.                                 */
    /* Then advance index to point to next character.                        */
    /*************************************************************************/

    switch(psi->achCommandBuffer[psi->usNextATIndex++])
    {
      /***********************************************************************/
      /***********************************************************************/
      case 'A' :
        usParserStatus = mwmCmdACommand(psi);
        break;

      case 'B' :
        usParserStatus = mwmCmdBCommand(psi);
        break;

      case 'C' :
        /*********************************************************************/
        /* Don't write any PP Commands for this one, so we don't need to     */
        /* check the buffer.                                                 */
        /* C1 is the only allowed combination.  Check to make sure that      */
        /* the parm passed is a 1.  No other parameter is allowed.           */
        /*********************************************************************/
        usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                       &psi->usNextATIndex);
        if (usParm != 1)
          usParserStatus = MWM_ATCMD_ERROR;
        break;

      case 'D' :
        usParserStatus = mwmCmdDCommand(psi);
        break;

      case 'E' :
        usParserStatus = mwmCmdECommand(psi);
        break;

      case 'F' :
        /*********************************************************************/
        /* Don't write any PP Commands for this one, so we don't need to     */
        /* check the PP buffer for overflow.                                 */
        /* F1 is the only allowed combination.  Check to make sure that      */
        /* the parm passed is a 1.  No other parameter is allowed.           */
        /*********************************************************************/
        usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                       &psi->usNextATIndex);
        if (usParm != 1)
          usParserStatus = MWM_ATCMD_ERROR;
        break;

      case 'H' :
        usParserStatus = mwmCmdHCommand(psi);
        break;

      case 'I' :
        psi->usSubCommandInfo = 0;
        usParserStatus = mwmCmdICommand(psi);
        break;

      case 'L' :
        usParserStatus = mwmCmdLCommand(psi);
        break;

      case 'M' :
        usParserStatus = mwmCmdMCommand(psi);
        break;

      case 'O' :
        usParserStatus = mwmCmdOCommand(psi);
        break;

      case 'P' :
        usParserStatus = mwmCmdPCommand(psi);
        break;

      case 'Q' :
        usParserStatus = mwmCmdQCommand(psi);
        break;

      case 'S' :
        /*********************************************************************/
        /* This command doesn't actually set anything in the modem.  It      */
        /* simply sets a flag in the psi structure signalling which S reg    */
        /* To use in the next AT? or AT= command.                            */
        /*********************************************************************/
        usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                       &psi->usNextATIndex);
/*        if (usParm < 30)*/
        if (usParm < MWM_NUM_S_REGISTERS)
          psi->usSelectedSRegister = usParm;
        /* MTS6384 re-work for more that just 37*/
        else
        {
           switch (usParm)
           {
           case 30:
           case 32:
           case 33:
           case 37:
           case 49:
           case 50:
           case 51:
           case 52:
           case 53:
           case 54:
           case 55:
           case 60:
              psi->usSelectedSRegister = usParm;
              break;
           default:
              usParserStatus = MWM_ATCMD_ERROR;
              break;
           } /* endswitch */
        }
/*        else if (usParm == 37)*/
/*          psi->usSelectedSRegister = usParm;*/
/*        else*/
/*          usParserStatus = MWM_ATCMD_ERROR;*/
        break;

      case 'T' :
        usParserStatus = mwmCmdTCommand(psi);
        break;

      case 'V' :
        usParserStatus = mwmCmdVCommand(psi);
        break;

      case 'W' :
        usParserStatus = mwmCmdWCommand(psi);
        break;

      case 'X' :
        usParserStatus = mwmCmdXCommand(psi);
        break;

      case 'Y' :
        usParserStatus = mwmCmdYCommand(psi);
        break;

      case 'Z' :
        usParserStatus = mwmCmdZCommand(psi);
        break;

      case ',' :
        /*********************************************************************/
        /* If the PP results of this command will fit into the buffer.       */
        /*********************************************************************/
        if ( (psi->usNextPPIndex + PP_COMMA_CMD_SPACE) < PP_BUFFER_THRESHOLD)
        {
          psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x0d | psi->usParserMode);
          break;
        }
        else
        {
          usParserStatus = MWM_GET_MORE_BUFFERS;
          psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
        }
        break;

      case '=' :
        if (psi->usSelectedSRegister < MWM_NUM_S_REGISTERS)
        {
          usParserStatus = mwmCmdEqualCommand(psi);
        }
        else
          usParserStatus = mwmCmdHostEqualCommand(psi);
        break;

      case '?' :
        /*********************************************************************/
        /* If the PP results of this command will fit into the buffer.       */
        /*********************************************************************/
/*        if (psi->usSelectedSRegister >= MWM_NUM_S_REGISTERS)*/
/*        {*/
          usParserStatus = mwmCmdHostQuestionCommand(psi);
/*        }*/
/*        else*/
/*        if ( (psi->usNextPPIndex + PP_QUESTION_CMD_SPACE) < PP_BUFFER_THRESHOLD)*/
/*        {*/
/*          psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x0a | psi->usParserMode);*/
/*          psi->ausPPcmdBuffer[psi->usNextPPIndex++] = 2*psi->usSelectedSRegister;*/
/*          break;*/
/*        }*/
/*        else*/
/*        {*/
/*          usParserStatus = MWM_GET_MORE_BUFFERS;*/
/*          psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;*/
/*        }*/
        break;

      case '&' :
        usParserStatus = mwmAmpCommand(psi);
        break;

      case '-' :                                                    /* DCR 2823*/
        usParserStatus = mwmDashCommand(psi);                       /* DCR 2823*/
        break;                                                      /* DCR 2823*/

      case '\\' :
        usParserStatus = mwmSlashCommand(psi);
        break;

      case '%' :
        usParserStatus = mwmPercentCommand(psi);
        break;

      case '"' :
        usParserStatus = mwmQuoteCommand(psi);
        break;

      case '\0':
        usCommandComplete = 1;
        break;

      case '@' :
        /*********************************************************************/
        /* For debug purposes only.  This sends commands directly as         */
        /* preprocessed commands.                                            */
        /* For example AT@1@25 would create the following PP_CMD buffer:     */
        /*  [0x0001] [0x0019]                                                */
        /*********************************************************************/
        usParserStatus = mwmCmdDirectPPCommand(psi);
        break;
      case '+' :
        /*********************************************************************/
        /* 11/01/93                                                          */
        /* Start looking for a Class 2 Fax command.                          */
        /*********************************************************************/
        usParserStatus = mwmPlusCommand(psi);

        break;
      case '*' :
        /*********************************************************************/
        /* 03/08/95                                                          */
        /* Add *TH command.                                                  */
        /*********************************************************************/
        usParserStatus = mwmStarCommand(psi);

        break;
      case '#' :
        /*********************************************************************/
        /* 05/11/95                                                          */
        /* Add #cid command.                                                 */
        /*********************************************************************/
        usParserStatus = mwmPoundCommand(psi);
        break;

      case 0x10 :  /* DLE*/
        usParserStatus = mwmDLECommand(psi);
        break;
      default:
        usParserStatus = MWM_ATCMD_ERROR;

    }


  } while (!usParserStatus && !usCommandComplete);

  if (usParserStatus & MWM_GET_MORE_BUFFERS)
    /*************************************************************************/
    /* Move the AT Command Index back to this command to be parsed           */
    /* on the next interrupt.                                                */
    /*************************************************************************/
    psi->usNextATIndex--;


 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdParseATcmd exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}




USHORT mwmCmdHostEqualCommand(STATEINFO *psi)
{
  USHORT usParm;
  /* MTS6384 handle new S30, S49-S53*/
  REGISTERS Registers;
  ULONG  ulAddress;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdHostEqualCommand entry\n");
  usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                     &psi->usNextATIndex);


  switch (psi->usSelectedSRegister)
  {
    case 37:
      if( usParm>255 )   /* Range check*/
      {
         return MWM_ATCMD_ERROR;
      }
      psi->usS37Value = usParm;
      /***********************************************************************/
      /* Now, set S28 according to this parameter as best we can.            */
      /***********************************************************************/
      switch (usParm)
      {
        case 0:
          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                               &psi->usNextPPIndex,
                               28,0,0);
          break;
        case 1:
        case 2:
        case 3:
          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                               &psi->usNextPPIndex,
                               28,6,6);
          break;
        case 5:
        case 7:
          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                               &psi->usNextPPIndex,
                               28,8,8);
          break;
        case 6:
          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                               &psi->usNextPPIndex,
                               28,9,9);
          break;
        case 8:
          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                               &psi->usNextPPIndex,
                               28,10,10);
          break;
        case 9:
          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                               &psi->usNextPPIndex,
                               28,12,12);
          break;
        case 10:
          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                               &psi->usNextPPIndex,
                               28,13,13);
          break;
        case 11:
          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                               &psi->usNextPPIndex,
                               28,14,14);
          break;
        case 12:
          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                               &psi->usNextPPIndex,
                               28,11,11);
          break;
        default:
          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                               &psi->usNextPPIndex,
                               28,0,0);
          break;
      }
      break;

    case 30:    /* MTS6384 S30 handled like \T*/
      if( usParm>540 )   /* Allow S30 to have the range of \T*/
      {
         return MWM_ATCMD_ERROR;
      }
      mwmParseSetDisconnectTime(psi, (USHORT)(usParm*10));
      break;

    /************************************************************
    ** S32 and S33 are the XON/XOFF characters for software
    ** flow control. Their values are maintained in the PC,
    ** and any changes to them are handled via PP Cmd 0x2B,
    ** with parameter 1 = 0x10 and parameter 2 = XON/XOFF
    ** characters the upper and lower bytes, respectively
    **
    ** MTS 6555 - Added 05/20/96
    *************************************************************/
    case 32:
      if( usParm>255 )  /* ASCII values only*/
      {
         return MWM_ATCMD_ERROR;
      }
      psi->usS32Value = usParm;
      if ( (psi->usNextPPIndex + PP_XON_XOFF_CMD_SPACE) < PP_BUFFER_THRESHOLD)
      {
        psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x2B | psi->usParserMode);
        psi->ausPPcmdBuffer[psi->usNextPPIndex++] = 0x10;
        psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (usParm << 8) | psi->usS33Value;
      } else {
        /*******************************************************************/
        /* Set the flag to request a new buffer from MDMCTL, and return.   */
        /*******************************************************************/
        return(MWM_GET_MORE_BUFFERS);
      }
      break;
    case 33:
      if( usParm>255 )  /* ASCII values only*/
      {
         return MWM_ATCMD_ERROR;
      }
      psi->usS33Value = usParm;
      if ( (psi->usNextPPIndex + PP_XON_XOFF_CMD_SPACE) < PP_BUFFER_THRESHOLD)
      {
        psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x2B | psi->usParserMode);
        psi->ausPPcmdBuffer[psi->usNextPPIndex++] = 0x10;
        psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (psi->usS32Value << 8) | usParm;
      } else {
        /*******************************************************************/
        /* Set the flag to request a new buffer from MDMCTL, and return.   */
        /*******************************************************************/
        return(MWM_GET_MORE_BUFFERS);
      }
      break;

    case 49:    /* MTS6384 handle S49-S53 - now we send to modem control*/
    case 50:    /*    This may be altered to handle in PC only*/
    case 51:
    case 52:
    case 53:
      if( usParm>255 )   /* Range register*/
      {
         return MWM_ATCMD_ERROR;
      }
      mwmParseQueryModemRegisters(&Registers);
      Registers.SV34[(psi->usSelectedSRegister)-49] = usParm;
      mwmParseSetModemRegisters(&Registers);
      break;

    case 54:    /* CMVC 1411 - add undocumented regs for test purposes*/
    case 55:
      if( usParm>255 )   /* Range register*/
      {
         return MWM_ATCMD_ERROR;
      }

      if(!dspLabelToAddress(psi->dsp.hMCTL, "SREG35", &ulAddress))
      {
         dspMemTransfer(psi->dsp.hDSP,
         ulAddress + ((psi->usSelectedSRegister-54) * 2),
         &usParm,
         1,
         DSP_MEMXFER_DATA_WRITE);
      }
      break;

    /************************************************************
    ** S60 is the a-law/mu-law flag. Its value is maintained in
    ** the PC,
    **
    ** CMVC #915 - Added 11/24/97
    *************************************************************/
    case 60:
      if( usParm > 1 )
      {
         return MWM_ATCMD_ERROR;
      }
      psi->usS60Value = usParm;
    break;

  }
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdHostEqualCommand exit ulRC 0\n");
  return 0;
}


USHORT mwmCmdHostQuestionCommand(STATEINFO *psi)
{
  USHORT usParserStatus;
  char   achSRegisterSetting[12] = "\0";
  REGISTERS Registers;
  USHORT usParm;
  ULONG  ulAddress;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdHostQuestionCommand entry\n");
  switch (psi->usSelectedSRegister)
  {
    case 37:
      sprintf(achSRegisterSetting,"%03d\n",psi->usS37Value);
      usParserStatus = mwmParseEchoString(psi,achSRegisterSetting);
      break;
   case 30:    /* MTS6384 S30 handled like \T - use current setting*/
               /*    Note: \T=90=5400 sec, so S30 could print > 255*/
               /*    I can block that here*/
      mwmParseQueryModemRegisters(&Registers);
      sprintf(achSRegisterSetting,"%03d\n",Registers.SlashT/10);
      usParserStatus = mwmParseEchoString(psi,achSRegisterSetting);
      break;

    /*****************************************************************
    ** S32 and S33 (XON and XOFF characters) are maintained in the PC
    ** MTS 6555 - Added 05/20/96
    ******************************************************************/
    case 32:
      sprintf(achSRegisterSetting,"%03d\n",psi->usS32Value);
      usParserStatus = mwmParseEchoString(psi,achSRegisterSetting);
      break;
    case 33:
      sprintf(achSRegisterSetting,"%03d\n",psi->usS33Value);
      usParserStatus = mwmParseEchoString(psi,achSRegisterSetting);
      break;

    case 49:    /* MTS6384 handle S49-S53 - now we send to modem control*/
    case 50:    /*    This may be altered to handle in PC only*/
    case 51:
    case 52:
    case 53:
      mwmParseQueryModemRegisters(&Registers);
      sprintf(achSRegisterSetting,"%03d\n",Registers.SV34[(psi->usSelectedSRegister)-49]);
      usParserStatus = mwmParseEchoString(psi,achSRegisterSetting);
      break;

    case 54:    /* CMVC 1411 - add undocumented regs for test purposes*/
    case 55:

      if(!dspLabelToAddress(psi->dsp.hMCTL, "SREG35", &ulAddress))
      {
         dspMemTransfer(psi->dsp.hDSP,
         ulAddress + ((psi->usSelectedSRegister-54) * 2),
         &usParm,
         1,
         DSP_MEMXFER_DATA_READ);
      }

      sprintf(achSRegisterSetting,"%03d\n",usParm);
      usParserStatus = mwmParseEchoString(psi,achSRegisterSetting);
      break;

    /*****************************************************************
    ** S60 (a-law/mu-law) is maintained in the PC.
    ** CMVC #915 - Added 11/24/97
    ******************************************************************/
    case 60:
      sprintf(achSRegisterSetting,"%03d\n",psi->usS60Value);
      usParserStatus = mwmParseEchoString(psi,achSRegisterSetting);
      break;

    default:
      /***********************************************************************/
      /* Regular S Register Query Command....Query the registers,            */
      /* Then, format the report.                                            */
      /***********************************************************************/
      mwmParseQueryModemRegisters(&Registers);
      sprintf(achSRegisterSetting,"%03d\n",Registers.S[psi->usSelectedSRegister]);
      usParserStatus = mwmParseEchoString(psi,achSRegisterSetting);
      break;

  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdHostQuestionCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}

USHORT mwmCmdACommand(STATEINFO *psi)
{
  USHORT usParserStatus = 0;
  REGISTERS    Registers;
  USHORT usHandsetStatus = 0;
  USHORT usHookStatus = 0;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdACommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_A_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {
    /*************************************************************************/
    /* MTS 4473                                                              */
    /* Logic for determining if ATA is valid..                               */
    /* ATA is valid in the following conditions.                             */
    /*                                                                       */
    /* 1.  Bit 8 in WTTAF2 is not set (This is used for all countries        */
    /*                                 except Germany at this time.)         */
    /* 2.  Rings are coming in...                                            */
    /* 3.  The handset is up, therefore, this is considered a "manual" answer*/
    /* MTS 7054                                                              */
    /* 4.  Modem is already offhook, this is not causing us to go offhook    */
    /*************************************************************************/

    /*************************************************************************/
    /* MTS 4408                                                              */
    /* Query S1 to see if a ring is coming in...                             */
    /*************************************************************************/
    if (mwmParseQueryModemRegisters(&Registers))
      return MWM_ATCMD_ERROR;

    if (mwmParseQueryHookStatus(&usHandsetStatus,&usHookStatus))
      return MWM_ATCMD_ERROR;

    if ( !(psi->wt.ulWTTAF2 & 0x0100)  ||                 /* ATA Allowed anytime...*/
          (Registers.S[1] > 0)      ||                    /* Ringcount > 0, so ATA is for sure valid.*/
          ((psi->wt.ulWTTAF2 & 0x0100) && usHandsetStatus)||/* Handset is up..."Manual Answer" ATA is valid.*/
          ((psi->wt.ulWTTAF2 & 0x0100) && usHookStatus)   /* Already offhook, another ATA is valid.MTS7054*/
       )
    {
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x24 | psi->usParserMode);
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x140);
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (3 | psi->usParserMode);
      usParserStatus = MWM_NO_MORE_COMMANDS;
    }
    else
      usParserStatus = MWM_ATCMD_ERROR;
  }
  else
  {
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdACommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}







USHORT mwmCmdBCommand(STATEINFO *psi)
{
  USHORT usParserStatus = 0;
  USHORT usParm;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdBCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_B_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {

    /*************************************************************************/
    /* MTS 3444  Query the Allowed Functions 1 word.                         */
    /*************************************************************************/
    usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                   &psi->usNextATIndex);

    switch (usParm)
    {
      case 0:
        /*********************************************************************/
        /* MTS 3444                                                          */
        /* If bell protocol is NOT allowed, allow the B0, otherwise, set     */
        /* B1                                                                */
        /* WTTAF2 & 0x8000    0 = CCITT    1 = BELL                          */
        /*********************************************************************/
/*        if ( !(psi->wt.ulWTTAF2 & 0x8000)  )*/
/*        {*/
          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                               &psi->usNextPPIndex,
                               27,0xffbf,0);
/*        }*/
/*        else*/
/*        {*/
/*          mwmParseSetSRegister(psi->ausPPcmdBuffer,*/
/*                               &psi->usNextPPIndex,*/
/*                               27,0xffff,0x40);*/
/*        }*/
        break;
      case 1:
        /*********************************************************************/
        /* MTS 3444                                                          */
        /* If bell protocol is allowed, allow the B1, otherwise, set         */
        /* B0                                                                */
        /* WTTAF2 & 0x8000    0 = CCITT    1 = BELL                          */
        /*********************************************************************/
/*        if ( (psi->wt.ulWTTAF2 & 0x8000) )*/
/*        {*/
          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                               &psi->usNextPPIndex,
                               27,0xffff,0x40);
/*        }*/
/*        else*/
/*        {*/
/*          mwmParseSetSRegister(psi->ausPPcmdBuffer,*/
/*                               &psi->usNextPPIndex,*/
/*                               27,0xffbf,0);*/
/*        }*/
        break;
      default:
        usParserStatus = MWM_ATCMD_ERROR;
    }
  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdBCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}




USHORT mwmCmdDCommand(STATEINFO *psi)
{
  USHORT usParserStatus = 0;
  BOOL   bEndCommand = FALSE;
  BOOL   bEndSearch = FALSE;
  BOOL   bDigitsFound = FALSE;
  USHORT usLastDigitIndex = 0;
  USHORT usNumDialedIndex = 0;
  USHORT usTempIndex;
  USHORT usHandsetStatus = 0;
  USHORT usHookStatus = 0;
  USHORT usSemicolonFound = 0;
  char   achValidDialDigits[20] = "\0";
  char   achDialString[AT_CMD_BUFFER_SIZE*2] = "\0";
  USHORT usNumberToUse = 0;
  USHORT usTempIndex2;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdDCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /* D Command can be as big as the AT Buffer.                               */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_D_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {

    /*************************************************************************/
    /* MTS 3444                                                              */
    /* Clear out the "Number Dialed" field so that we can save the number we */
    /* are about to dial.                                                    */
    /*************************************************************************/
    memset(psi->achNumberDialed,0,sizeof(psi->achNumberDialed));

    /*************************************************************************/
    /* MTS 3444 Integrate World Trade calls.                                 */
    /*   - Query the valid dial digits from the World Trade table access lib.*/
    /*************************************************************************/

    /*************************************************************************/
    /* MTS 3444 Integrate Blacklisting Server calls.                         */
    /*************************************************************************/
    switch (psi->wt.ulWTTDTADS)
    {
      case 9:
      /***********************************************************************/
      /* MTS 3549  Allow digits 0-9 in pulse only mode.                      */
      /***********************************************************************/
      case 0xFFFF:
        strcpy(achValidDialDigits,"0123456789");
        break;
      case 0x0B:
        strcpy(achValidDialDigits,"0123456789*#");
        break;
      case 0x0F:
        strcpy(achValidDialDigits,"0123456789ABCD*#");
        break;
    }



    /*************************************************************************/
    /* 08/22/95                                                              */
    /* Add a 3rd pass to the dial parsing.                                   */
    /* This pass will copy the dial string to a temporary buffer, and        */
    /* integrate any stored numbers (S=x) along the way.                     */
    /* This should allow stored numbers to be processed as a portion of the  */
    /* dial string, eliminating errors specific to stored numbers.           */
    /*************************************************************************/
    memset(achDialString,0,sizeof(AT_CMD_BUFFER_SIZE*2));

    usTempIndex = 0;
    while (psi->achCommandBuffer[psi->usNextATIndex] && !bEndSearch)
    {
      /***********************************************************************/
      /* Check to see if this is an 'S'                                      */
      /* if so, then we must go get the stored number...                     */
      /***********************************************************************/
      if (psi->achCommandBuffer[psi->usNextATIndex] == 'S')
      {
         /********************************************************************/
         /* MTS 4372                                                         */
         /* 12/08/94                                                         */
         /* Step over the S                                                  */
         /********************************************************************/
         psi->usNextATIndex++;

         /********************************************************************/
         /* Next item in string MUST be '='...if not, return ERROR.          */
         /* 12/08/94                                                         */
         /* Allow atds2 to work in addition to atds=2                        */
         /********************************************************************/
         if (psi->achCommandBuffer[psi->usNextATIndex] !=  '=')
         {
           /******************************************************************/
           /* 12/08/94                                                       */
           /* Assign the character immediately following the s to usNumberToUse.*/
           /* if this value is > 3, it will be reset to 0 below...           */
           /******************************************************************/
           usNumberToUse = psi->achCommandBuffer[psi->usNextATIndex] - '0';
           psi->usNextATIndex++;
         }
         else
         {
           psi->usNextATIndex++;
           usNumberToUse = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                                      &psi->usNextATIndex);
         }

         /********************************************************************/
         /* Now the next parameter must be 0-3                               */
         /********************************************************************/
         if  (usNumberToUse>3)
         {
           return MWM_ATCMD_ERROR;
         }

         strcpy(&achDialString[usTempIndex],psi->ModemMemory.aachStoredNumbers[usNumberToUse]);
         usTempIndex += strlen(psi->ModemMemory.aachStoredNumbers[usNumberToUse]);
      }
      else
      {
        /*********************************************************************/
        /* Simply copy the character to the temporary dial string buffer.    */
        /* - if this is a semicolon, then signal that this is the end of the */
        /* dial command.                                                     */
        /*********************************************************************/
        if (psi->achCommandBuffer[psi->usNextATIndex] == ';')
          bEndSearch = TRUE;
        achDialString[usTempIndex++] = psi->achCommandBuffer[psi->usNextATIndex++];
      }
    }



    /*************************************************************************/
    /* We must insert a 0x19 before the last dial digit in the string.       */
    /* This requires us to make 2 passes through the AT command.             */
    /* The first pass is to determine where the last dialing digit is        */
    /* located.                                                              */
    /*************************************************************************/
    bEndSearch = FALSE;
    usTempIndex = 0;
    /*************************************************************************/
    /* while the first digit in the dial string is a T or a P, step          */
    /* over it.                                                              */
    /*************************************************************************/
    while ( (achDialString[usTempIndex] == 'T') ||
            (achDialString[usTempIndex] == 'P') ||
            (achDialString[usTempIndex] == 'W') ||
            (achDialString[usTempIndex] == '!') ||
            (achDialString[usTempIndex] == '&') ||
            (achDialString[usTempIndex] == '$') ||      /* MTS 7337 07/10/96*/
            (achDialString[usTempIndex] == '@') )
      usTempIndex++;

    while (achDialString[usTempIndex] && !bEndSearch)
    {
      /***********************************************************************/
      /* MTS 3451 Allow mixed pulse/tone.                                    */
      /* Modify the meaning of usLastDigitIndex and bDigitsFound to mean     */
      /* whether their are digits found before the next 'T' or 'P'           */
      /***********************************************************************/
      if ( (achDialString[usTempIndex] == ';') ||
           (achDialString[usTempIndex] == 'T') ||
           (achDialString[usTempIndex] == 'P') ||
           (achDialString[usTempIndex] == 'W') ||
           (achDialString[usTempIndex] == '!') ||
           (achDialString[usTempIndex] == '$') ||      /* MTS 7337 07/10/96*/
           (achDialString[usTempIndex] == '&') ||
           (achDialString[usTempIndex] == '@') )
      {
        bEndSearch = TRUE;
        if (achDialString[usTempIndex] == ';')
          usSemicolonFound = 1;
      }
      else
      {
        if ( strchr(achValidDialDigits,achDialString[usTempIndex]) )
        {
           bDigitsFound = TRUE;
           usLastDigitIndex = usTempIndex;
        }
      }
      usTempIndex++;
    }


    /*************************************************************************/
    /* MTS 4476                                                              */
    /* Check to see if the handset is up.  If so, we won't allow the dial    */
    /* if it has digits without a semicolon.                                 */
    /*************************************************************************/
    if (mwmParseQueryHookStatus(&usHandsetStatus,&usHookStatus))
      return MWM_ATCMD_ERROR;


    if (usHandsetStatus)
    {
      if ( (bDigitsFound) && (!usSemicolonFound) )
        return MWM_ATCMD_BUSY;
    }
    else
    {
      if (psi->wt.ulWTTAF1 & 0x0001)
      {
        if ( !bDigitsFound && !usSemicolonFound )
          return MWM_ATCMD_ERROR;
      }
    }

    psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x24 | psi->usParserMode);
    psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x141);

    /*************************************************************************
    ** Handle the special case of ATD; -- If no digits and a semicolon are
    ** found, treat this as an ATH1 command.
    **************************************************************************/
    if ( !bDigitsFound && usSemicolonFound ) {
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (6 | psi->usParserMode);
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = 0xffff;
      return(0);
    } /* endif */


    /*************************************************************************/
    /* We know this is an ATD Command, so put the appropriate PP command     */
    /* into the buffer, and advance buffer pointer.                          */
    /*************************************************************************/
    psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (2 | psi->usParserMode);



    bEndCommand = FALSE;
    usTempIndex = 0;
    /*************************************************************************/
    /* while there are parameters...                                         */
    /*************************************************************************/
    while ((achDialString[usTempIndex]) &&
           (!bEndCommand) )
    {
      /***********************************************************************/
      /* If we are just about to process the last dialing digit, then insert */
      /* the "last dial digit" flag.                                         */
      /***********************************************************************/
      /* MTS 3451                                                            */
      /* This "last dial digit" now refers to the last dial digit before a   */
      /* 'P' or 'T' Command as well as the last dial digit of the string.    */
      /***********************************************************************/
      if (bDigitsFound && (usTempIndex == usLastDigitIndex) )
      {
        psi->ausPPcmdBuffer[psi->usNextPPIndex++] = 0x19;
        /*********************************************************************/
        /* Now, to reinitialize the usLastDigitIndex variable, lets start    */
        /* from this index, and search for last digit of the next portion of */
        /* the string.                                                       */
        /*********************************************************************/

        usTempIndex2 = usTempIndex + 1;
        if (achDialString[usTempIndex2] != ';')
        {
          /*******************************************************************/
          /* Find the last digit in the next section of the Dial String.     */
          /*******************************************************************/
          usLastDigitIndex = 0;
          bDigitsFound = FALSE;
          /*******************************************************************/
          /* while the next digit in the dial string is a T or a P, step     */
          /* over it.                                                        */
          /*******************************************************************/
          while ( (achDialString[usTempIndex2] == 'T') ||
                  (achDialString[usTempIndex2] == 'P') )
            usTempIndex2++;

          bEndSearch = FALSE;
          while (achDialString[usTempIndex2] && !bEndSearch)
          {
            /*****************************************************************/
            /* MTS 3451 Allow mixed pulse/tone.                              */
            /* Modify the meaning of usLastDigitIndex and bDigitsFound to mean*/
            /* whether their are digits found before the next 'T' or 'P'     */
            /*****************************************************************/
            if ( (achDialString[usTempIndex2] == ';') ||
                 (achDialString[usTempIndex2] == 'T') ||
                 (achDialString[usTempIndex2] == 'P') )
              bEndSearch = TRUE;
            else
            {
              if ( strchr(achValidDialDigits,achDialString[usTempIndex2]) )
              {
                bDigitsFound = TRUE;
                usLastDigitIndex = usTempIndex2;
              }
            }
            usTempIndex2++;
          }
        }
      }

      if ( strchr(achValidDialDigits,achDialString[usTempIndex]) )
      {

        /*********************************************************************/
        /* Add this digit to the "Number Dialed"                             */
        /*********************************************************************/
        psi->achNumberDialed[usNumDialedIndex++] =
                           achDialString[usTempIndex];

        switch(achDialString[usTempIndex])
        {
          case '0':
          case '1':
          case '2':
          case '3':  /* digits to be dialed */
          case '4':  /* convert ASCII to number */
          case '5':
          case '6':
          case '7':
          case '8':
          case '9':
            psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (USHORT)
                               achDialString[usTempIndex] - '0';
            break;

          case 'A' :
            psi->ausPPcmdBuffer[psi->usNextPPIndex++] =0x0c;
            break;

          case 'B' :
            psi->ausPPcmdBuffer[psi->usNextPPIndex++] =0x0d;
            break;

          case 'C' :
            psi->ausPPcmdBuffer[psi->usNextPPIndex++] =0x0e;
            break;

          case 'D' :
            psi->ausPPcmdBuffer[psi->usNextPPIndex++] =0x0f;
            break;

          case '*' :
            psi->ausPPcmdBuffer[psi->usNextPPIndex++] =0x0a;
            break;

          case '#' :
            psi->ausPPcmdBuffer[psi->usNextPPIndex++] =0x0b;
            break;

          case 0   :
            bEndCommand = TRUE;
            break;
        }
      } /* if Valid Dial Digit.*/
      else
      {  /* These are not in the list of valid dial digits. */
        switch(achDialString[usTempIndex])
        {
          case 'R' :
            psi->ausPPcmdBuffer[psi->usNextPPIndex++] =0x11;
            break;

          case 'T' :
            if (psi->wt.ulWTTDTADS != 0xFFFF)
              /***************************************************************/
              /* MTS 3444 Integrate World Trade support                      */
              /* Tone is allowed, add the PP Command for Tone.               */
              /***************************************************************/
              psi->ausPPcmdBuffer[psi->usNextPPIndex++] =0x12;
            else
              /***************************************************************/
              /* MTS 3444 Integrate World Trade support                      */
              /* Tone is not allowed...explicity send the modifier for pulse */
              /* so that MCTL is not confused about whether to use pulse or  */
              /* tone.                                                       */
              /***************************************************************/
              psi->ausPPcmdBuffer[psi->usNextPPIndex++] =0x15;
            break;

          case 'W' :
            psi->ausPPcmdBuffer[psi->usNextPPIndex++] =0x13;
            break;

          case ',' :
            psi->ausPPcmdBuffer[psi->usNextPPIndex++] =0x14;
            break;

          case 'P' :
            if (psi->wt.ulWTTDMODE)
              /***************************************************************/
              /* MTS 3444 Integrate World Trade support                      */
              /* Pulse is allowed, add the PP Command for pulse.             */
              /***************************************************************/
              psi->ausPPcmdBuffer[psi->usNextPPIndex++] =0x15;
            else
              /***************************************************************/
              /* MTS 3444 Integrate World Trade support                      */
              /* Pulse is not allowed...explicity send the modifier for tone */
              /* so that MCTL is not confused about whether to use pulse or  */
              /* tone.                                                       */
              /***************************************************************/
              psi->ausPPcmdBuffer[psi->usNextPPIndex++] =0x12;
            break;

          case '@' :
            psi->ausPPcmdBuffer[psi->usNextPPIndex++] =0x16;
            break;

          /*******************************************************************/
          /* MTS 4732 Add wait for calling card bong....                     */
          /*******************************************************************/
          case '&' :
          case '$' :    /* MTS7337 - 07/10/96*/
            psi->ausPPcmdBuffer[psi->usNextPPIndex++] =0x21;
            break;

          case ';' :
            psi->ausPPcmdBuffer[psi->usNextPPIndex++] =0x17;
            bEndCommand = TRUE;
            break;
          case '-' :
            break;
          case '!':
            /*****************************************************************/
            /* if the flash hook is allowed, then add the PP Command for     */
            /* flash hook.                                                   */
            /* Otherwise, just ignore it...                                  */
            /*****************************************************************/
            if (!(psi->wt.ulWTTAF1 & 0x0100)) /* if the flash hook is valid*/
            {
              psi->ausPPcmdBuffer[psi->usNextPPIndex++] =0x10;
            }
            break;
          case 0   :
            bEndCommand = TRUE;
            break;
        }
      } /* if Valid Dial Digit.*/

      usTempIndex++;
    }

    /*************************************************************************/
    /* Mark then end of the Dial string...                                   */
    /*************************************************************************/
    /* 11/3/93 If the command was not ended by a SemiColon, put the          */
    /* "go to handshaking" Dial Modifier on the end of the command string.   */
    /* PTR 2157                                                              */
    /* Previously, the 0x18 was always put on the end of the dial string.    */
    /*************************************************************************/
    if (psi->ausPPcmdBuffer[psi->usNextPPIndex-1] != 0x17)
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] =0x18;


  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdDCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}






USHORT mwmCmdDirectPPCommand(STATEINFO *psi)

{
  USHORT usParm;
  USHORT usParserStatus = 0;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdDirectPPCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + 1) < PP_BUFFER_THRESHOLD)
  {
     usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                    &psi->usNextATIndex);

     psi->ausPPcmdBuffer[psi->usNextPPIndex++] = usParm;

  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdDirectPPCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}









USHORT mwmCmdECommand(STATEINFO *psi)

{
  USHORT usParm;
  USHORT usParserStatus = 0;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdECommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_E_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {
     usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                    &psi->usNextATIndex);

     switch (usParm)
     {
       case 0:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                         &psi->usNextPPIndex,
                         14,0xfffd,0);
         break;
       case 1:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                         &psi->usNextPPIndex,
                         14,0xffff,0x02);
         break;
       default:
         usParserStatus = MWM_ATCMD_ERROR;
     }
  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdECommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}




USHORT mwmCmdEqualCommand(STATEINFO *psi)

{
  USHORT usParm;
  USHORT usParserStatus = 0;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdEqualCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_EQUAL_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {
     usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                    &psi->usNextATIndex);

     if (usParm<256)
     {
         USHORT usAndMask = 0xff00|usParm;
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                         &psi->usNextPPIndex,
                         psi->usSelectedSRegister,usAndMask,usParm);
     }
     else
       usParserStatus = MWM_ATCMD_ERROR;
  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdEqualCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}




USHORT mwmCmdHCommand(STATEINFO *psi)

{
  USHORT usParserStatus = 0;
  USHORT usParm;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdHCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_H_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {
     usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                    &psi->usNextATIndex);

     switch (usParm)
     {
       case 0:
         psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (6 | psi->usParserMode);
         psi->ausPPcmdBuffer[psi->usNextPPIndex++] = 0;
         break;
       case 1:
         if (!(psi->wt.ulWTTAF1 & 0x0001))
         {
           psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (6 | psi->usParserMode);
           psi->ausPPcmdBuffer[psi->usNextPPIndex++] = 0xffff;
         }
         else
           return MWM_ATCMD_ERROR;
         break;
       default:
         usParserStatus = MWM_ATCMD_ERROR;
     }
  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdHCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}



/*****************************************************************************/
/* mwmCmdICommand                                                            */
/*****************************************************************************/
/*****************************************************************************/
USHORT mwmCmdICommand(STATEINFO *psi)
{
  USHORT usParm;
  USHORT usParserStatus = 0;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdICommand entry\n");
  if (!psi->usSubCommandInfo)
  {
    /*************************************************************************/
    /* Get the Parameter out of the AT Command string.                       */
    /*************************************************************************/
    usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                        &psi->usNextATIndex);
    if (usParm < 8)
    {
      /***********************************************************************/
      /* Aptiva CMVC #43984 Add ATI 5,6,7 to be same as ATI3                 */
      /***********************************************************************/
      if ((usParm == 5) || (usParm == 6) || (usParm == 7))
      {
        usParm = 3;
      }
      psi->usSavedCommandParameter = usParm;
      psi->usSubCommandInfo++;
    }
    else
      return MWM_ATCMD_ERROR;
  }


  /***************************************************************************/
  /* This is one of the iterations                                           */
  /***************************************************************************/
  switch (psi->usSavedCommandParameter)
  {
    case 0:
      /***********************************************************************/
      /* I0 Command                                                          */
      /***********************************************************************/
      usParserStatus = mwmCmdI0Command(psi);
      break;
    case 1:
      break;
    case 2:
      /***********************************************************************/
      /* I2 Command                                                          */
      /***********************************************************************/
/*      usParserStatus = mwmCmdI2Command(psi);*/
      break;
    case 3:
      /***********************************************************************/
      /* I3 Command                                                          */
      /***********************************************************************/
      /***********************************************************************/
      /* MTS 4581 Get rid of ATI2 and ATI3 Commands.                         */
      /***********************************************************************/
      usParserStatus = mwmCmdI3Command(psi);
      break;
    case 4:
      /***********************************************************************/
      /* I4 Command                                                          */
      /***********************************************************************/
      usParserStatus = mwmCmdI4Command(psi);
      break;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdICommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}




/*****************************************************************************/
/* mwmCmdI0Command                                                           */
/*****************************************************************************/
USHORT mwmCmdI0Command(STATEINFO *psi)
{

  USHORT usParserStatus = 0;
  char achString[20];
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdIOCommand entry\n");
  /***************************************************************
  ** The speed values reported here come from the INI file
  ** entries which were read at init time.        02/27/97 DR
  ****************************************************************/
  if ( psi->ulInstalledFeatures & FEATURE_PCM )
    strcpy(achString, psi->achPCMString);
  else
  if ( psi->ulInstalledFeatures & FEATURE_V34EM )
    strcpy(achString, psi->ach336String);
  else
  if ( psi->ulInstalledFeatures & FEATURE_V34 )
    strcpy(achString, psi->ach288String);
  else
  if ( psi->ulInstalledFeatures & FEATURE_V32BIS )
    strcpy(achString, psi->ach144String);
  else
    strcpy(achString, psi->ach2400String);


  usParserStatus = mwmParseEchoString(psi,achString);
  if (usParserStatus & MWM_GET_MORE_BUFFERS)
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    psi->usCurrentCommand = MWM_I_COMMAND;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdIOCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;

}



/*****************************************************************************/
/* mwmCmdI3Command                                                           */
/*****************************************************************************/
USHORT mwmCmdI3Command(STATEINFO *psi)
{

  USHORT usParserStatus = 0;
  char   achVersionString[MWM_VERSION_STRING_LENGTH] = "IBM Mwave Modem ";

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdI3Command entry\n");
  /* 06/02/98 DR - Use the string pulled from the INI file. */
  strcpy( achVersionString, psi->achIDString );
  strcat( achVersionString, " " );

  /***************************************************************************/
  /* MTS 2944                                                                */
  /* Append the version information from the INI file to this text.          */
  /***************************************************************************/
  strcat(achVersionString,psi->achModemVersion);

  usParserStatus = mwmParseEchoString(psi,achVersionString);
  if (usParserStatus & MWM_GET_MORE_BUFFERS)
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    psi->usCurrentCommand = MWM_I_COMMAND;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdI3Command exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;

}




/*****************************************************************************/
/* mwmCmdI4Command                                                            */
/*****************************************************************************/
USHORT mwmCmdI4Command(STATEINFO *psi)
{
  USHORT usParserStatus = 0;
  char   achVersionString[MWM_VERSION_STRING_LENGTH] = "\0";
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdI4Command entry\n");
  /***************************************************************************/
  /* MTS 2944                                                                */
  /* Append the version information from the INI file to this text.          */
  /***************************************************************************/
  strcat(achVersionString,psi->achOEMVersion);


  usParserStatus = mwmParseEchoString(psi,achVersionString);
  if (usParserStatus & MWM_GET_MORE_BUFFERS)
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    psi->usCurrentCommand = MWM_I_COMMAND;
    psi->usSubCommandInfo = 1;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdI4Command exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;

}



/*****************************************************************************/
/* mwmCmdLCommand                                                            */
/*****************************************************************************/
/* This AT Command changes the VOLUME of the speaker control.                */
/* The ON/OFF setting of the speaker is defined by the 'M' Command.          */
/* The speaker has 4 volume settings 0 - 3.  We parse the arguments, and     */
/* if the parameter is between 0 and 3 (inclusive), we set the appropriate   */
/* S Register (S22)                                                          */
/*****************************************************************************/
USHORT mwmCmdLCommand(STATEINFO *psi)

{
  USHORT usParserStatus = 0;
  USHORT usParm;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdLCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_L_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {
     usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                    &psi->usNextATIndex);

     if (usParm<4)
     {
       /**********************************************************************/
       /* Need to either                                                     */
       /*   1) Notify the user interface here that the speaker needs to be   */
       /*      updated, or                                                   */
       /*   2) Design a new interrupt from MDMCTL to identify that the       */
       /*      speaker setting has been changed.                             */
       /**********************************************************************/
       USHORT usAndMask = 0xfffc | usParm;
       mwmParseSetSRegister(psi->ausPPcmdBuffer,
                       &psi->usNextPPIndex,
                       22,usAndMask,usParm);
#ifndef LINUX_BLD
       /* If we had an integrated UI we could post a SPEAKER CHANGE MESSAGE HERE */
#endif
     }
     else
       usParserStatus = MWM_ATCMD_ERROR;
  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdLCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}





/*****************************************************************************/
USHORT mwmCmdMCommand(STATEINFO *psi)

{
  USHORT usParserStatus = 0;
  USHORT usParm;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdMCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_M_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {
     usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                    &psi->usNextATIndex);

     if (usParm<4 )
     {
       /**********************************************************************/
       /* Need to either                                                     */
       /*   1) Notify the user interface here that the speaker needs to be   */
       /*      updated, or                                                   */
       /*   2) Design a new interrupt from MDMCTL to identify that the       */
       /*      speaker setting has been changed.                             */
       /**********************************************************************/

       /**********************************************************************/
       /* MTS 4665                                                           */
       /* Effectively disallow ATM2                                          */
       /* Treat it just like ATM1                                            */
       /**********************************************************************/
       /* 02/12/96                                                           */
       /* MTS 6374 ... Remove this mask...allow M2...MCTL still manually     */
       /* turns off speaker in data mode even if M2.                         */
       /**********************************************************************/
/*       if (usParm == 2)*/
/*         usParm = 1;*/

       usParm=4*usParm;
       {
       USHORT usAndMask = 0xfff3 | usParm;
       mwmParseSetSRegister(psi->ausPPcmdBuffer,
                       &psi->usNextPPIndex,
                       22,usAndMask,usParm);
       }
#ifndef LINUX_BLD
       /* If we had an integrated UI we could post a SPEAKER CHANGE MESSAGE HERE */
#endif
     }
     else
       usParserStatus = MWM_ATCMD_ERROR;
  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdMCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}



/*****************************************************************************/
USHORT mwmCmdOCommand(STATEINFO *psi)

{
  USHORT usParserStatus = 0;
  USHORT usParm;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdOCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_O_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {
     usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                    &psi->usNextATIndex);


     switch (usParm)
     {
       case 0:
         psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (4 | psi->usParserMode);
         break;
       case 1:
         break;
       default:
         usParserStatus = MWM_ATCMD_ERROR;
     }


    /*************************************************************************/
    /* Either way, this command has been processed, and it MUST be the last  */
    /* command in the string, so tell the parser that we won't parse any     */
    /* more commands...                                                      */
    /*************************************************************************/
    usParserStatus = usParserStatus | MWM_NO_MORE_COMMANDS;
  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdOCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}



USHORT mwmCmdPCommand(STATEINFO *psi)

{
  USHORT usParserStatus = 0;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdPCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_P_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {
    /*************************************************************************/
    /* MTS 3444 Integrate World Trade Details.                               */
    /* If Pulse dialing is allowed, then set the dialing mode to pulse.      */
    /* Otherwise, explicitly set it to tone to avoid confusion.              */
    /*************************************************************************/
    if (psi->wt.ulWTTDMODE)
    {
      /***********************************************************************/
      /* MTS 3444  World Trade Support.                                      */
      /* Set the dial mode to pulse.                                         */
      /***********************************************************************/
      mwmParseSetSRegister(psi->ausPPcmdBuffer,
                           &psi->usNextPPIndex,
                           14,0xffff,0x20);
    }
    else
    {
      /***********************************************************************/
      /* MTS 3444  World Trade Support.                                      */
      /* Explicitly set the dial mode to tone.                               */
      /* I believe there is a risk of confusion here....How did the dial mode*/
      /* get set to tone...I never even called the TCommand routine.         */
      /* Still, this appears to be the best way to gaurantee that the modem  */
      /* never tries to dial using the wrong method.                         */
      /***********************************************************************/
      mwmParseSetSRegister(psi->ausPPcmdBuffer,
                           &psi->usNextPPIndex,
                           14,0xffdf,0x0);
    }

  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdPCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}



USHORT mwmCmdQCommand(STATEINFO *psi)
{
  USHORT usParserStatus = 0;
  USHORT usParm;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdQCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_Q_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {
     usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                    &psi->usNextATIndex);

     switch (usParm)
     {
       case 0:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              14,0xfffb,0);
         break;
       case 1:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              14,0xffff,0x04);
         break;
       default:
         usParserStatus = MWM_ATCMD_ERROR;
     }
  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdQCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}




USHORT mwmCmdTCommand(STATEINFO *psi)

{
  USHORT usParserStatus = 0;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdTCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_T_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {
    /*************************************************************************/
    /* MTS 3444 Integrate World Trade Details.                               */
    /* If Pulse dialing is allowed, then set the dialing mode to pulse.      */
    /* Otherwise, explicitly set it to tone to avoid confusion.              */
    /*************************************************************************/
    if (psi->wt.ulWTTDTADS != 0xFFFF)
    {
      /***********************************************************************/
      /* MTS 3444  World Trade Support.                                      */
      /* Set the dial mode to tone.                                          */
      /***********************************************************************/
      mwmParseSetSRegister(psi->ausPPcmdBuffer,
                           &psi->usNextPPIndex,
                           14,0xffdf,0x0);
    }
    else
    {
      /***********************************************************************/
      /* MTS 3444  World Trade Support.                                      */
      /* Explicitly set the dial mode to pulse.                              */
      /* I believe there is a risk of confusion here....How did the dial mode*/
      /* get set to pulse...I never even called the PCommand routine!        */
      /* Still, this appears to be the best way to gaurantee that the modem  */
      /* never tries to dial using the wrong method.                         */
      /***********************************************************************/
      mwmParseSetSRegister(psi->ausPPcmdBuffer,
                           &psi->usNextPPIndex,
                           14,0xffff,0x20);
    }
  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdTCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}




USHORT mwmCmdVCommand(STATEINFO *psi)
{
  USHORT usParserStatus = 0;
  USHORT usParm;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdVCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_V_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {
     usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                    &psi->usNextATIndex);

     switch (usParm)
     {
       case 0:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              14,0xfff7,0);
         break;
       case 1:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              14,0xffff,0x08);
         break;
       default:
         usParserStatus = MWM_ATCMD_ERROR;
     }
  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdVCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}


USHORT mwmCmdWCommand(STATEINFO *psi)
{
  USHORT     usParserStatus = 0;
  USHORT     usParm;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdWCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_X_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {

     usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                    &psi->usNextATIndex);
     switch (usParm)
     {
       case 0:
       case 1:
       case 2:
         psi->usWValue = usParm;
         break;
       default:
         usParserStatus = MWM_ATCMD_ERROR;
     }
  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdWCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}

USHORT mwmCmdXCommand(STATEINFO *psi)
{
  USHORT usParserStatus = 0;
  USHORT usParm;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdXCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_X_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {

     usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                    &psi->usNextATIndex);

     switch (usParm)
     {
       case 0:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              22,0xff8f,0);
         break;
       case 1:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              22,0xff8f|0x40,0x40);
         break;
       case 2:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              22,0xff8f|0x50,0x50);
         break;
       case 3:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              22,0xff8f|0x60,0x60);
         break;
       case 4:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              22,0xff8f|0x70,0x70);
         break;

       default:
         usParserStatus = MWM_ATCMD_ERROR;
     }
  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdXCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}


USHORT mwmCmdYCommand(STATEINFO *psi)
{
  USHORT usParserStatus = 0;
  USHORT usParm;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdYCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_Y_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {
     usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                    &psi->usNextATIndex);

     switch (usParm)
     {
       case 0:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              21,0xff7f,0);
         break;
       case 1:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              21,0xffff,0x80);
         break;
       default:
         usParserStatus = MWM_ATCMD_ERROR;
     }
  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdYCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}




USHORT mwmCmdZCommand(STATEINFO *psi)
{
  USHORT usParserStatus = 0;
  USHORT usParm;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmcmd::mwmCmdZCommand entry\n");
  usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                      &psi->usNextATIndex);

  psi->usCurrentCommand = MWM_LOAD_SPECIFIC_PROFILE;
  switch (usParm)
  {
    case 0:
      psi->usSavedCommandParameter = usParm;
      psi->usSubCommandInfo = 1;
      usParserStatus = mwmParseInitializeModem(psi,
                                               usParm);
      break;
    case 1:
      psi->usSavedCommandParameter = usParm;
      psi->usSubCommandInfo = 1;
      usParserStatus = mwmParseInitializeModem(psi,
                                               usParm);
      break;
    case 2:
      psi->usSavedCommandParameter = usParm;
      psi->usSubCommandInfo = 1;
      usParserStatus = mwmParseInitializeModem(psi,
                                               usParm);
      break;
    default:
    {
      usParserStatus = MWM_ATCMD_ERROR;
      psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
    }
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmcmd::mwmCmdZCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}



