/***************************************************************************
                          gnudownload.cpp  -  description
                             -------------------
    begin                : Tue May 29 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/

// the original version of this file was taken from Gnucleus (http://gnucleus.sourceforge.net)

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"
#include <unistd.h>
//#include <stropts.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "preferences.h"

#include "gnucache.h"
#include "gnuhash.h"
#include "gnudirector.h"

#include "gnudownload.h"
#include "gnusearch.h"

#include "asyncfile.h"
#include "event.h"

/////////////////////////////////////////////////////////////////////////////
// service: we need this set for thread-safe callbacks

static MMutex s_mutexDownloadSet(true); // recursive to enable 'delete this' from callbacks
static set<MGnuDownload*> s_setDownloads;
//
static inline void RegisterDownload(MGnuDownload* p)
{
	s_mutexDownloadSet.lock();
	ASSERT(s_setDownloads.find(p)==s_setDownloads.end());
	s_setDownloads.insert(p);
	s_mutexDownloadSet.unlock();
}
static inline void UnregisterDownload(MGnuDownload* p)
{
	s_mutexDownloadSet.lock();
	ASSERT(s_setDownloads.find(p)!=s_setDownloads.end());
	s_setDownloads.erase(p);
	s_mutexDownloadSet.unlock();
}
static inline bool IsRegistred(MGnuDownload* p)
{
	//MLock lock(s_mutexDownloadSet); it is always called with mutex locked
	return s_setDownloads.find(p)!=s_setDownloads.end();
}

//////////////////////////////////////////////////////////////////////////////

class MDownloadFile : public MAsyncFile {
public:
	MDownloadFile(MGnuDownload*);
	virtual void OnSuccess();
	virtual void OnError();
	void SetMoveTarget(CString s){
		m_bMove = s.length();
		m_sDest = s;
	}
	void SetDeleteFlag(){
		m_bDelete = true;
	}
	void GiveAway(){
		s_mutexDownloadSet.lock();
		m_pBuddy = NULL;
		s_mutexDownloadSet.unlock();
	}
private:
	virtual ~MDownloadFile();
	MGnuDownload* m_pBuddy;
	bool    m_bMove;
	bool    m_bDelete;
	CString m_sDest;
	void DoFileMove();
	void DoFileDelete();
};

MDownloadFile::MDownloadFile(MGnuDownload* pBuddy) : MAsyncFile(AFM_READWRITE|AFM_CREATEPATH), m_pBuddy(pBuddy)
{
	m_bMove = false;
	m_bDelete = false;
}

MDownloadFile::~MDownloadFile()
{
}

void MDownloadFile::OnSuccess()
{
	//TRACE2("MDownloadFile::OnSuccess; request=", m_pRequest->type_compl);
	s_mutexDownloadSet.lock();
	//
	switch (m_pRequest->type_compl)
	{
		case AFR_OPEN:
			if (m_pBuddy && IsRegistred(m_pBuddy))
				m_pBuddy->OnFileOpen();
		break;
		case AFR_WRITE:
			if (m_pBuddy && IsRegistred(m_pBuddy))
				m_pBuddy->OnFileWritten(m_pRequest->size, m_pRequest->nReturn, m_pRequest->nErrNo);
		break;
		case AFR_READ:
			if (m_pBuddy && IsRegistred(m_pBuddy))
				m_pBuddy->OnFileRead(m_pRequest->size, m_pRequest->nReturn, m_pRequest->nErrNo);
		break;
		case AFR_CLOSE:
			if (m_pBuddy && IsRegistred(m_pBuddy))
				m_pBuddy->OnFileClose();
			if (m_bMove)
				DoFileMove();
			else if (m_bDelete)
				DoFileDelete();
			break;
		case AFR_NONE:
			TRACE("MDownloadFile::OnSuccess: AFR_NONE - should not happen");
		break;
	}
	//
	s_mutexDownloadSet.unlock();
}

void MDownloadFile::OnError()
{
	//TRACE2("MDownloadFile::OnError; request=", m_pRequest->type_compl);
	//
	s_mutexDownloadSet.lock();
	//
	if (m_pBuddy && IsRegistred(m_pBuddy))
		m_pBuddy->OnFileError(m_pRequest->nErrNo);
	//
	if (m_pRequest->type_compl == AFR_CLOSE)
	{
		if (m_bMove)
			DoFileMove();
		else if (m_bDelete)
			DoFileDelete();
		delete this;
	}
	//
	s_mutexDownloadSet.unlock();
}

void MDownloadFile::DoFileMove()
{
	//TRACE2("MDownloadFile::DoFileMove to ", m_sDest);
	CString path = m_sDest;
	// make sure we dont overwrite anything
	for (int n = 1; FileExists(path) && n<1000; n++)
	{
		 CString sName = m_sDest;
		 CString sExt;
		 int nDotPos = m_sDest.rfind(".");
		 if (nDotPos != -1)
		 {
		 	sName = m_sDest.substr(0,nDotPos);
		 	sExt = m_sDest.substr(nDotPos);
		 }
		 path = sName + "." + DWrdtoStr(n) + sExt;
	}		
	if(!MoveFile(m_szName, path))
	{
		// TODO: messageID
		POST_ERROR(ES_IMPORTANT, CString("Failed to move completed file `") + m_szName + "' to download folder");
	}
	else
	{
		// TODO: messageID
		POST_MESSAGE(ES_GOODTOKNOW, CString("Downloading of `") + path + "' completed");
	}
}

void MDownloadFile::DoFileDelete()
{
	if (!DeleteFile(m_szName))
	{
		// TODO: messageID
		POST_ERROR(ES_IMPORTANT, CString("Failed to delete file `") + m_szName);
	}
}

/////////////////////////////////////////////////////////////////////////////
// MGnuDownload

MGnuDownload::MGnuDownload(MGnuDirector* pComm) : m_mutex(true)
{
	RegisterDownload(this);
	//
	m_pDirector  = pComm;
	m_pPrefs = m_pDirector->GetPrefs();

	m_bActive = false;
	m_nQueuePos		  = 0;

	m_dwBytesCompleted = 0;
	m_dwResumeStart    = 0;
    // timers
	m_nSecCounter     = 0;
	m_nSecsUnderLimit = 0;
	m_nSecsDead       = 0;
	m_nSecInactive    = 0;
	m_nWaitDelay      = 0;
	// buffers
	m_nBuffSize = 8192;
	m_pBuffer = new BYTE[m_nBuffSize];
	m_nAvgBytesAvailable = m_nBuffSize; // we need this for smart buffer resize
	// Bandwidth
	for(int i = 0; i < 60; i++)
		m_dwAvgBytes[i] = 0;
	m_dwBytes60  = 0;
	m_dwSecBytes = 0;
	m_nSecPos    = 0;
	m_dRate      = 0;
	
	//m_hFile = -1;
	m_pAFile = NULL;
	
	m_pDataToWriteOnOpen = NULL;
	m_nSizeToWriteOnOpen = 0;
	
	m_nDisconnectReason = REASON_UNDEFINED;
	m_nStatus = TRANSFER_NEW;
	
	m_dwSearchID = 0;
	m_bDeleteSearch = false;
	
	m_bReceiveBlocked = false;
	
	m_dwID = 0;
}

MGnuDownload::~MGnuDownload()
{
	UnregisterDownload(this);
	//
	m_mutex.lock();
	if (m_Queue.size()) // for any case disconnect if we can be connected to smth
	{
		//ASSERT(m_mutex.locked());
		ForceDisconnect(REASON_STOPPED);
	}
	// PROBLEM #3 - close on the fly - try to get rid of this. check who calls StartDownload
	CloseFile();
	// buffers
	if (m_pBuffer)
		delete [] m_pBuffer;
	m_mutex.unlock();
}

/////////////////////////////////////////////////////////////////////////////
// MGnuDownload member functions

LPCSTR SGnuDownload::GetErrorString(int nCode)
{
	switch(nCode)
	{
		case REASON_UNDEFINED:
			return "UNDEFINED";
		case REASON_STOPPED:
			return "STOPPED";
		case REASON_NO_DATA:
			return "NO DATA";
		case REASON_SOCKET_ERROR:
			return "BAD SOCKET";
		case REASON_NO_RESUME_SUPPORT:
			return "NO RESUME";
		case REASON_WRONG_FILE_SIZE:
			return "WRONG SIZE";
		case REASON_FILE_NOT_FOUND:
			return "NO FILE";
		case REASON_SERVER_BUSY:
			return "BUSY";
		case REASON_UNKNOWN_SERVER_ERROR:
			return "UNKNOWN";
		case REASON_WRONG_STATE:
			return "WRONG STATE";
		case REASON_CONNECTION_ERROR:
			return "CONN ERROR";
		case REASON_REMOTELY_CANCELED:
			return "REM. CANCELED";
		case REASON_FAILED_TO_OPEN_FILE:
			return "CANNOT OPEN";
		case REASON_PARTIAL_FILESIZE_DIFFERENT:
			return "PART. SIZE";
		case REASON_HOST_OCCUPIED:
			return "OCCUPIED";
		case REASON_FILES_ARE_DIFFERENT:
			return "PART. MISMATCH";
		case REASON_WRITE_ERROR:
			return "WRITE ERROR";
		case REASON_COMPLETED:
			return "COMPLETED";
		case REASON_NO_RESPONSE:
			return "NO RESPONSE";
		case REASON_BELOW_MINIMUM_SPEED:
			return "TOO SLOW";
	}
	return         "WRONG CODE";
}

LPCSTR SGnuDownload::GetStatusString(int nStatus)
{
	switch(nStatus)
	{
		case TRANSFER_QUEUED:
			return "QUEUED";
		case TRANSFER_CONNECTING:
			return "NEGOT";
		case TRANSFER_CONNECTED:
			return "CONN";
		case TRANSFER_SENDING:
			return "SEND";
		case TRANSFER_RECEIVING:
			return "RECV";
		case TRANSFER_PUSH:
		case TRANSFER_PUSH_CONNECTING:
		case TRANSFER_PUSH_CONNECTED:
			return "PUSH";
		case TRANSFER_CLOSED:
			return "CLOSED";
		case TRANSFER_COMPLETED:
			return "COMPL";
		case TRANSFER_COOLDOWN:
			return "DELAY";
		case TRANSFER_NEW:
			return "NEW";
		case TRANSFER_NO_RESULTS:
			return "EMPTY";
	}
	return         "WRONG";
}

void MGnuDownload::OnConnect(int nErrorCode)
{
	//TRACE("OnConnect");
	m_mutex.lock();
	ASSERT(m_nQueuePos < m_Queue.size());
	m_Queue[m_nQueuePos].nConErrors = 0;
	
	CheckFile(); // check file*+ will send request if open is successfull
	StatusUpdate(TRANSFER_CONNECTED);
	//SendRequest();
	m_mutex.unlock();
	MAsyncSocket::OnConnect(nErrorCode);
}

void MGnuDownload::OnReceive(int nErrorCode)
{
	//TRACE("OnReceive");
	// before we receive enything lets keep track of the ammount of data available
	// this will allow us to adjust buffer size later
	if (m_bReceiveBlocked)
		return;
	MLock lock(m_mutex);
#ifdef FIONREAD
	int nBytes = 0;
    if ( ::ioctl(m_hSocket, FIONREAD, (char*)&nBytes) >=0 )
    {
    	m_nAvgBytesAvailable *= 0x1f;
    	m_nAvgBytesAvailable += nBytes;
    	m_nAvgBytesAvailable >>=5;
    }
#else
    m_nAvgBytesAvailable = m_nBuffSize; //quick patch
#endif
	//
	if (m_nSizeToWriteOnOpen && m_nStatus == TRANSFER_RECEIVING)
	{
		Download(m_pDataToWriteOnOpen, m_nSizeToWriteOnOpen);
		m_nSizeToWriteOnOpen = 0;
		m_pDataToWriteOnOpen = NULL;
		// we have to return otherwise conflicts with SelectFlags occur
		return;
	}
	// Receive Data
	int BuffLength = Receive(m_pBuffer, m_nBuffSize);
    //printf("recieved %d bytes, errno = %d\n", BuffLength, errno);
	// Check for errors
	switch (BuffLength)
	{
	case 0:
		//ASSERT(m_mutex.locked());
		ForceDisconnect(REASON_NO_DATA);
		return;
		break;
	case SOCKET_ERROR:
		//ASSERT(m_mutex.locked());
		ForceDisconnect(REASON_SOCKET_ERROR);
		return;
		break;
	}
	
	// Connected and downloading
	if(m_nStatus == TRANSFER_RECEIVING)
	{
		Download(m_pBuffer, BuffLength);
	}

	// Server sending us download header
	else if(m_nStatus == TRANSFER_CONNECTED)
	{	
		// set file begin here, manually build, stop at end header
		
		CString Header;
		int     FileBegin = -1;

		for(int i = 0; i < BuffLength; i++)
		{
			if( memcmp(m_pBuffer+i, "\r\n\r\n", 4) == 0)
			{
				Header   += "\r\n\r\n";
				FileBegin = i + 4;
				break;
			}
			else
				Header += m_pBuffer[i];
		}
		
		m_sHeader			 += Header;
		//m_Current->Handshake += Header;

		// If the entire header is here
		if(m_sHeader.find("\r\n\r\n") != -1)
		{
			m_sHeader = m_sHeader.substr(0, m_sHeader.find("\r\n\r\n") + 4);

			// Extract HTTP code from first line
			int  Code       = 0;
			CString sLine = m_sHeader.substr(0, m_sHeader.find("\r\n"));
			if (sLine.find("HTTP")>=0)
			{
				CString sCode = sLine.substr(sLine.find("HTTP")+4);
				sCode = sLine.substr(sLine.find(" ")+1);
				Code = atoi(sCode.c_str());
			}
			
			// Success code
			if(200 <= Code && Code < 300)
			{
				// Redo this to be based on a download's content range
				MakeLower(m_sHeader);
				DWORD ServerSize = 0, ServerCompleted = 0, SmallSize = 0;
				int   NewDL     = m_sHeader.find("\r\ncontent-length:");
				int   ResumedDL = m_sHeader.find("\r\ncontent-range:");

				// New download
				if(NewDL >= 0 && ResumedDL == -1)
				{	
					//sscanf(m_Header.substr(NewDL).c_str(), "\r\ncontent-length:%ld\r\n", &ServerSize);
					CString sLen = m_sHeader.substr(NewDL+strlen("content-length:")+2);
					ServerSize = atoi(sLen.c_str());
					
					if(m_dwBytesCompleted != 0)
					{
						//ASSERT(m_mutex.locked());
						ForceDisconnect(REASON_NO_RESUME_SUPPORT);
					}

					else if(ServerSize == m_dwFileLength)
					{
						// PROBLEM #5 - write some data on open
						m_pDataToWriteOnOpen = NULL;
						m_nSizeToWriteOnOpen = 0;
						if(BuffLength - FileBegin > 0)
						{
							m_pDataToWriteOnOpen = m_pBuffer+FileBegin;
							m_nSizeToWriteOnOpen = BuffLength - FileBegin;
						}
						StatusUpdate(TRANSFER_RECEIVING);
						OpenFile();
					}
					else
					{
						//ASSERT(m_mutex.locked());
						ForceDisconnect(REASON_WRONG_FILE_SIZE);
					}
				}
				// Resumed download
				else if(ResumedDL >= 0)
				{
					//sscanf( m_Header.substr(ResumedDL + 16).c_str(), " bytes=%ld-%ld/%ld\r\n", &ServerCompleted, &SmallSize, &ServerSize);
					CString sHelp = m_sHeader.substr(ResumedDL+strlen("content-range:")+2);
					sHelp = sHelp.substr(0, sHelp.find("\r\n"));
					CString sRange = sHelp.substr(sHelp.find("bytes")+6);
					ServerCompleted = atoi(sRange.c_str());
					sHelp = sRange.substr(sRange.find("-")+1);
					SmallSize = atoi(sHelp.c_str());
					if (sHelp.find("/")>=0)
					{
						CString s = sHelp.substr(sHelp.find("/")+1);
						ServerSize = atoi(s.c_str());
					}
					else if (NewDL>=0)
					{
						CString sLen = m_sHeader.substr(NewDL+strlen("content-length:")+2);
						ServerSize = SmallSize;
						SmallSize = atoi(sLen.c_str());
					}
					if (ServerSize == 0)
						ServerSize = SmallSize;
					
					if(ServerCompleted != m_dwResumeStart)
					{
						//ASSERT(m_mutex.locked());
						ForceDisconnect(REASON_NO_RESUME_SUPPORT);
					}
					else if(ServerSize == m_dwFileLength)
					{
						// PROBLEM #5 - write some data on open
						m_pDataToWriteOnOpen = NULL;
						m_nSizeToWriteOnOpen = 0;
						if(BuffLength - FileBegin > 0)
						{
							m_pDataToWriteOnOpen = m_pBuffer+FileBegin;
							m_nSizeToWriteOnOpen = BuffLength - FileBegin;
						}
						StatusUpdate(TRANSFER_RECEIVING);
						OpenFile();
					}
					else
					{
						//ASSERT(m_mutex.locked());
						ForceDisconnect(REASON_WRONG_FILE_SIZE);
					}
				}
			}
			// Server error code
			else if(400 <= Code && Code < 500)
			{
				//ASSERT(m_mutex.locked());
				ForceDisconnect(REASON_FILE_NOT_FOUND);
			}
			// Server error code
			else if(500 <= Code && Code < 600)
			{
				//ASSERT(m_mutex.locked());
				ForceDisconnect(REASON_SERVER_BUSY);
			}
			else
			{
				//ASSERT(m_mutex.locked());
				ForceDisconnect(REASON_UNKNOWN_SERVER_ERROR);
			}
		}
	}
	// We are not in a connected or receiving state
	else
	{
		//ASSERT(m_mutex.locked());
		ForceDisconnect(REASON_WRONG_STATE);
	}
	
	MAsyncSocket::OnReceive(nErrorCode);
}

void MGnuDownload::OnClose(int nErrorCode)
{
	//TRACE("OnClose");
	MLock lock(m_mutex);
	if (m_nStatus == TRANSFER_CONNECTING)
	{
		// update Resilt counters
		ASSERT(m_nQueuePos<m_Queue.size());
		m_Queue[m_nQueuePos].nConErrors++;
		//ASSERT(m_mutex.locked());
		ForceDisconnect(REASON_CONNECTION_ERROR);
		//TRACE("OnClose: switching to push mode");
		StatusUpdate(TRANSFER_PUSH);
		if (!m_pDirector->Route_LocalPush( m_Queue[m_nQueuePos] ))
		{
			m_Queue[m_nQueuePos].nPushTimeouts++;
			//ASSERT(m_mutex.locked());
			ForceDisconnect(REASON_NO_RESPONSE); // there wont be a response
		}
		MAsyncSocket::OnClose(nErrorCode);
		return;
	}
	else if (m_nStatus == TRANSFER_RECEIVING)
		if(m_nDisconnectReason == REASON_UNDEFINED)
		{
			//ASSERT(m_mutex.locked());
			ForceDisconnect(REASON_REMOTELY_CANCELED);
		}

	StatusUpdate(TRANSFER_CLOSED);
	//ASSERT(m_mutex.locked());
	ForceDisconnect(m_nDisconnectReason);// just to ensure cleanup

	MAsyncSocket::OnClose(nErrorCode);
}

void MGnuDownload::CloseFile()
{
	if (m_pAFile)
	{
		((MDownloadFile*)m_pAFile)->GiveAway();
		m_pAFile->Close();
		m_pAFile->Destroy();// this will call 'delete this' upon completion
		// we are not anymore
		m_pAFile = NULL; // concerned about this file object
	}
}

void MGnuDownload::CheckFile()
{
	//TRACE("CheckFile");
	// Get plain file name with out directory crap
	CString FileName = m_sName;
	ReplaceSubStr(FileName,"\\", "/");
	ReplaceSubStr(FileName,"`", " ");
	ReplaceSubStr(FileName,"'", " ");
	ReplaceSubStr(FileName,"\"", " ");
	FileName = FileName.substr( FileName.rfind("/") + 1);
	//
	m_sName = FileName;
	// i really doubt
	MakeLower(FileName);

	// Set where the file will go upon completion
	m_sFilePath = ExpandPath(m_pPrefs->m_szDownloadPath) + "/" + FileName;
	ReplaceSubStr(m_sFilePath,"//", "/");
	//TRACE("file will be saved as ", m_FilePath);

	// Add tail to it so we can support partial downloads
	// CString Tail = ".[[" + DWrdtoStr(m_FileLength) + "," + m_Search + "]]";
	CString Tail = ".[[" + DWrdtoStr(m_dwFileLength) + "]]";
	FileName += Tail;
	// Create / Find the file in the partial directory
	CString PartialDir = ExpandPath(m_pPrefs->m_szDownloadPath) + "/part";
	m_sPartialPath = PartialDir + "/" + FileName;
	ReplaceSubStr(m_sPartialPath, "//", "/");
	//
	// we will enable OnReceive in OnFileOpen
	BlockOnReceive(true);
	// PROBLEM #1 - get file size
	if (m_pAFile == NULL)
	{
		m_pAFile = new MDownloadFile(this);
		VERIFY(m_pAFile->Open(m_sPartialPath.c_str())); // non-blocking call should never fail
	}
}

void MGnuDownload::OpenFile()
{
	//TRACE("OpenFile");
	// we will enable OnReceive in OnFileOpen
	BlockOnReceive(true);
	// PROBLEM #2 - open the file
	if (m_pAFile == NULL)
	{
		m_pAFile = new MDownloadFile(this);
		VERIFY(m_pAFile->Open(m_sPartialPath.c_str()));
	}
	else
	{
		ASSERT(m_pAFile->IsOpen());
		VERIFY(m_pAFile->Seek(0,SEEK_SET));
		//ASSERT(m_mutex.locked());
		OnFileOpen(false); // as if we've just opened it
	}
}

IP MGnuDownload::GetCurrentHost()
{
	MLock lock(m_mutex);
	ASSERT(m_nQueuePos < m_Queue.size());
	return m_Queue[m_nQueuePos].Host;
}

int MGnuDownload::GetCurrentPort()
{
	MLock lock(m_mutex);
	ASSERT(m_nQueuePos < m_Queue.size());
	return m_Queue[m_nQueuePos].Port;
}

void MGnuDownload::SendRequest()
{
	//TRACE("SendRequest");
	// this function is called just after the connection has been established -- good point to set up a flags
	ASSERT(m_nQueuePos < m_Queue.size());
	m_Queue[m_nQueuePos].bServerIsUp = true;
	m_Queue[m_nQueuePos].nPushTimeouts = 0;  // successful connection 'brakes' the sequence
	// At least 4096 bytes needed for resume verification
	m_dwResumeStart = 0;
	m_nVerifyPos   = 0;
	
	if(m_dwBytesCompleted > 4096)
		m_dwResumeStart = m_dwBytesCompleted - 4096;
	else
		m_dwBytesCompleted = 0;

	CString sGetRequest;
	sGetRequest.format("GET /get/%ld/%s HTTP/1.0\r\nUser-Agent: Mutella\r\nRange: bytes=%d-\r\n\r\n",
	                    m_Queue[m_nQueuePos].FileIndex, m_Queue[m_nQueuePos].Name.c_str(), m_dwResumeStart);
	
	Send(sGetRequest.c_str(), sGetRequest.length());
	
	StatusUpdate(TRANSFER_CONNECTED);
}

void MGnuDownload::StartDownload(bool bLock)
{
	//TRACE("StartDownload");
	MLock lock(m_mutex, bLock);

	m_bActive = true;
	m_nSecInactive = 0;
	m_nSecsDead = 0;
	m_nSecsUnderLimit = 0;
	m_sHeader = "";
	//m_Current->Handshake = "";
	ASSERT(m_nQueuePos < m_Queue.size());
	
	StatusUpdate(TRANSFER_CONNECTING);

	if(m_hSocket == INVALID_SOCKET)
		if(!Create(0, SOCK_STREAM, FD_READ | FD_CONNECT | FD_CLOSE))
		{
			POST_ERROR(ES_IMPORTANT, CString("Download Create Error: ") + DWrdtoStr(GetLastError()));
		}

	// PROBLEM #3 - close on the fly - try to get rid of this. check who calls StartDownload
	// we can try to use self-deleting object
	CloseFile();
	ASSERT(m_pAFile==NULL);
	// Check for downloads already from same host
	if( m_pDirector->GetPrefs()->m_nMaxPerHostDownloads > 0 && m_pDirector->CountDownloads(m_Queue[m_nQueuePos].Host, this) > m_pDirector->GetPrefs()->m_nMaxPerHostDownloads )
	{
		//ASSERT(m_mutex.locked());
		ForceDisconnect(REASON_HOST_OCCUPIED);
		//m_ServersAlive = true;
		NextInQueue();
		return;
	}
	// Attempt to connect
	bool bOkForConnect = m_Queue[m_nQueuePos].nConErrors<3 && m_pDirector->IsOkForDirectConnect(m_Queue[m_nQueuePos].Host); /* && !m_Queue[m_nQueuePos].Firewall*/
	if(!bOkForConnect || !Connect(IPtoStr(m_Queue[m_nQueuePos].Host).c_str(), m_Queue[m_nQueuePos].Port))
		if(!bOkForConnect || (GetLastError() != EWOULDBLOCK && GetLastError() != EINPROGRESS && GetLastError() != EAGAIN) )
		{
			/*if (!bOkForConnect)
				printf("start download: sending push request\n");
			else
				printf("start download: switching to push mode\n");*/
			StatusUpdate(TRANSFER_PUSH);
			if (!m_pDirector->Route_LocalPush( m_Queue[m_nQueuePos] ))
			{
				m_Queue[m_nQueuePos].nPushTimeouts++;
				//ASSERT(m_mutex.locked());
				ForceDisconnect(REASON_NO_RESPONSE); // there wont be a response
			}
			Close();
		}
}

void MGnuDownload::StopDownload(bool bDeleteFile, bool bLock)
{
	//TRACE("StopDownload");
	MLock lock(m_mutex, bLock);
	
	m_bActive = false;
	if (bDeleteFile && m_pAFile)
		((MDownloadFile*)m_pAFile)->SetDeleteFlag();
	m_bDeleteSearch = bDeleteFile;
	//ASSERT(m_mutex.locked());
	ForceDisconnect(REASON_STOPPED);
}

void MGnuDownload::Download(BYTE* pBuff, int nSize)
{
	//ASSERT(m_mutex.locked());
	//TRACE2("Download: enter ", rand());
	// PROBLEM #4 -- download
	ASSERT(m_nSelectFlags & FD_READ);
	if( m_pAFile )
	{
		ASSERT(m_pAFile->IsReady());
		m_dwSecBytes += nSize;
		if(m_dwResumeStart)
		{
			int VerSize   = min(nSize, (int)(m_dwBytesCompleted-m_dwResumeStart-m_nVerifyPos));
			ASSERT(m_nVerifyPos>=0);
			ASSERT(VerSize <= 4096 - m_nVerifyPos);
			ASSERT(m_nVerifyPos<4096);
			if(memcmp(m_Verification+m_nVerifyPos, pBuff, VerSize) != 0)
			{
				//ASSERT(m_mutex.locked());
				ForceDisconnect(REASON_FILES_ARE_DIFFERENT);
				return;
			}
			m_nVerifyPos += VerSize;
			if(m_nVerifyPos != 4096)
				return;
			
			// Compare succeeded
			ASSERT(m_nVerifyPos == 4096);
			
			ASSERT(m_pAFile->GetPos()==m_dwBytesCompleted);
			
			m_dwResumeStart = 0;
			nSize -= VerSize;
			pBuff += VerSize;
			
			if (nSize == 0)
				return; // we used up all buffer
		}
		// Tranfer has begun
		//ASSERT(m_mutex.locked());
		BlockOnReceive(true);
		ASSERT(m_pAFile->GetPos()==m_dwBytesCompleted);
		VERIFY(-1!=m_pAFile->Write((char*)pBuff, nSize, false /* dont memcpy the buffer */));
	}
	else
	{
		//ASSERT(m_mutex.locked());
		ForceDisconnect(REASON_WRITE_ERROR);
	}
	//TRACE("Download: leave");
}

void MGnuDownload::OnFileWritten(int nRequested, int nWritten, int nError)
{
	//TRACE("OnWritten: enter");
	MLock lock(m_mutex);
	//TRACE("OnWritten: past mutex");
	if (TRANSFER_RECEIVING != m_nStatus || m_pAFile == NULL)
		return;
	m_dwBytesCompleted += nWritten;
	
	if (nWritten != nRequested)
	{
		//ASSERT(m_mutex.locked());
		ForceDisconnect(REASON_WRITE_ERROR);
	}
	else if(m_dwFileLength == m_dwBytesCompleted)
	{
		((MDownloadFile*)m_pAFile)->SetMoveTarget(m_sFilePath);
		//ASSERT(m_mutex.locked());
		ForceDisconnect(REASON_COMPLETED);
		m_bDeleteSearch = true;
		m_bActive = false;
	}
	else if(m_dwFileLength < m_dwBytesCompleted)
	{
		//ASSERT(m_mutex.locked());
		ForceDisconnect(REASON_CORRUPT);
		//
		((MDownloadFile*)m_pAFile)->SetMoveTarget(m_sFilePath+".[possibly corrupt]");
		// TODO: messageID
		POST_ERROR(ES_IMPORTANT, CString("Received file `") + m_sName + "' is likely to be corrupt");
		m_bActive = false;
	}
		
	// now its time to think about buffer resize
	if ( TRANSFER_RECEIVING == m_nStatus &&
		 m_dwBytesCompleted-m_dwResumeStart >= 1048576 ) //1M
	{
		if (m_nAvgBytesAvailable > 1.5*m_nBuffSize && m_nBuffSize < MAX_BUFFERSIZE) // grow condition
		{
			delete [] m_pBuffer;
			m_nBuffSize = (int) 1.2*m_nAvgBytesAvailable;
			if (m_nBuffSize > MAX_BUFFERSIZE)
				m_nBuffSize = MAX_BUFFERSIZE;
			m_pBuffer = new BYTE[m_nBuffSize];
			//TRACE2("receive buffer has been resized to ", m_nBuffSize);
		} else if (m_nAvgBytesAvailable < 0.5*m_nBuffSize && m_nBuffSize > MIN_BUFFERSIZE) // shrink condition
		{
			delete [] m_pBuffer;
			m_nBuffSize = (int) 1.2*m_nAvgBytesAvailable;
			if (m_nBuffSize < MIN_BUFFERSIZE)
				m_nBuffSize = MIN_BUFFERSIZE;
			m_pBuffer = new BYTE[m_nBuffSize];
			//TRACE2("receive buffer has been resized to ", m_nBuffSize);
		}
	}
	// enable OnRecieve
	BlockOnReceive(false);
}

void MGnuDownload::OnFileClose()
{
	//TRACE("MGnuDownload::OnFileClose");
}

void MGnuDownload::BlockOnReceive(bool bBlock)
{
	m_bReceiveBlocked = bBlock;
	if (bBlock)
	{
		ModifySelectFlags(0, FD_READ);
	}
	else
	{
		ModifySelectFlags(FD_READ, 0);
	}
}

void MGnuDownload::OnFileRead(int nRequested, int nRead, int nError)
{
	//TRACE("OnFileRead");
	m_mutex.lock();
	//TRACE("OnFileRead 1");
	ASSERT(m_pAFile);
	// only called if resuming to notify us that now we can start download procedure
	ASSERT(nRequested==nRead);
	m_nVerifyPos = 0;
	ASSERT(m_pAFile->EoF());
	// enable OnReceive
	m_mutex.unlock();
	//TRACE("OnFileRead 2");
	BlockOnReceive(false);
	//TRACE("OnFileRead done!");
}

void MGnuDownload::OnFileOpen(bool bLock)
{
	//TRACE("OnFileOpen");
	MLock lock(m_mutex, bLock);
	//ASSERT(m_mutex.locked());
	ASSERT(m_pAFile);
	//TRACE("OnFileOpen 1");
	if (TRANSFER_CONNECTED == m_nStatus)
	{
		m_dwBytesCompleted = m_pAFile->GetSize();
		
		SendRequest();
		BlockOnReceive(false);
	}
	else if (TRANSFER_RECEIVING == m_nStatus)
	{
		if (m_dwBytesCompleted && m_dwBytesCompleted != m_pAFile->GetSize())
		{
			//ASSERT(m_mutex.locked());
			ForceDisconnect(REASON_PARTIAL_FILESIZE_DIFFERENT);
		}
		// if resuming -- load the resume buffer
		if(m_dwResumeStart)
		{
			m_nVerifyPos = -1;
			VERIFY(-1!=m_pAFile->ReadSeek(-4096, SEEK_END, (char*)m_Verification,4096));
			//TRACE("OnFileOpen 2");
		}
		else
		{
			// only called if resuming to notify us that now we can start download procedure
			ASSERT(0==m_pAFile->GetPos());
			// enable OnReceive
			BlockOnReceive(false);
		}
	}
}

void MGnuDownload::OnFileError(int nError)
{
	//TRACE2("MGnuDownload::OnFileError: errno=", nError);
	MLock lock(m_mutex);
	// TODO: check if it works
	if (TRANSFER_RECEIVING == m_nStatus)
	{
		//ASSERT(m_mutex.locked());
		ForceDisconnect(REASON_WRITE_ERROR);
	}
	else
	{
		//ASSERT(m_mutex.locked());
		ForceDisconnect(REASON_FAILED_TO_OPEN_FILE);
	}
}

void MGnuDownload::ForceDisconnect(int nReason)
{
	//TRACE4("ForceDisconnect: ID=",m_dwID," reason=",nReason);
	//ASSERT(m_mutex.locked());
	m_nDisconnectReason = nReason;
	if(m_hSocket != INVALID_SOCKET)
	{
		// TODO: check if this can cause deadlock
		AsyncSelect(FD_CLOSE);
		ShutDown(2);
	}
	//TRACE("ForceDisconnect: 1");
	m_nSizeToWriteOnOpen = 0;
	m_pDataToWriteOnOpen = NULL;
	m_bReceiveBlocked = false;
	
	// PROBLEM #3 async close again
	CloseFile();
	//TRACE("ForceDisconnect: 2");
	
	if (m_nQueuePos < m_Queue.size())
	{
		m_Queue[m_nQueuePos].ErrorCode = nReason;
	}

	Close();
	StatusUpdate(TRANSFER_CLOSED);
	
	// reset bandwidth
	for(int i = 0; i < 60; i++)
		m_dwAvgBytes[i] = 0;
	m_dwBytes60  = 0;
	m_dwSecBytes = 0;
	m_nSecPos    = 0;
	m_dRate      = 0;
}

void MGnuDownload::StatusUpdate(DWORD Status)
{
	//TRACE("StatusUpdate");
	m_nStatus = Status;
	
	m_nSecsDead = 0;

	ASSERT(m_nQueuePos < m_Queue.size());
	m_Queue[m_nQueuePos].Status = Status;
	m_Queue[m_nQueuePos].StatusUpdateTime = xtime();
	//if (Status != TRANSFER_CLOSED)
	//	m_Queue[m_nQueuePos].ErrorCode = REASON_UNDEFINED; // ???
	//
	m_nSecCounter = 0;
	//
	m_pDirector->TransferMessage(DOWNLOAD_UPDATE, (WPARAM) this);
}

void MGnuDownload::UpdateResults()
{
	//TRACE("UpdateResults");
	if (m_sSearch.length())
	{
		// first we have to look the appropriate search up
		MGnuSearch* pSearch = m_pDirector->LookUpSearch(m_sSearch, m_dwFileLength);
		if (!pSearch || !pSearch->m_bUpdated)
		{
			if (!pSearch)
				AddSearch(false);
			return;
		}
		// now we have to match and update
		// but first reset the 'updated' flag on the search
		pSearch->m_bUpdated = false;
		int nRes;
		for ( int i = 0; i<pSearch->m_CurrentList.size(); i++ )
		{
			Result& SR = pSearch->m_CurrentList[i];
			bool bFound = false;
			nRes = m_Queue.size();
			for ( int j = 0; j<nRes; j++ )
			{
				if (SR.Port        == m_Queue[j].Port        &&
		            SR.FileIndex   == m_Queue[j].FileIndex   &&
		            SR.Host.S_addr == m_Queue[j].Host.S_addr &&
		            SR.NameLower   == m_Queue[j].NameLower   )
		  		{
		  			// update current result
		  			//TRACE("UpdateResults: updating existing");
		  			if (m_Queue[j].ChangeTime != SR.ChangeTime)
		  			{
		  				m_Queue[j].ChangeTime = SR.ChangeTime;
		  				// the result was found again -- give it another chance
		  				m_Queue[j].nPushTimeouts = 0;
		  			}
					m_Queue[j].PushID = SR.PushID;
					m_Queue[j].Origin = SR.Origin;
					bFound = true;
					break;
		  		}
			}
			if (!bFound)
			{
				// look the result up in the m_BadResults list and see if it has been updated.
				nRes = m_BadResults.size();
				bool bShouldAdd = true;
				for ( int j = 0; j<nRes; j++ )
				{
					if (SR.Port        == m_BadResults[j].Port        &&
		            	SR.FileIndex   == m_BadResults[j].FileIndex   &&
		            	SR.Host.S_addr == m_BadResults[j].Host.S_addr &&
		            	SR.NameLower   == m_BadResults[j].NameLower   )
		  			{
		  				// check if it was updated
		  				if (m_BadResults[j].ChangeTime != SR.ChangeTime)
		  				{
		  					// it is not gona be bad anymore
		  					m_BadResults.erase(&m_BadResults[j]);
		  					break;
		  				}
		  				else
		  				{
		  					bShouldAdd = false;
		  					break;
		  				}
		  			}
		  		}
		  		if (bShouldAdd)
				{
					m_Queue.push_back(SR); //TODO: put new results in the beginning so that have sort of priority
				}
			}
		}
	}
	else
	{
		//
		AddSearch(false);
	}
}

void MGnuDownload::AddSearch(bool bAutoget)
{
	MGnuSearch* pSearch = NULL;
	if (0==m_sSearch.length())
	{
		m_sSearch = MakeSearchOfFilename(m_sName);
	}
	//
	if ( (0==m_dwSearchID) || NULL == (pSearch = m_pDirector->GetSearchByID(m_dwSearchID)) )
	{
		if (NULL == (pSearch = m_pDirector->LookUpSearch(m_sSearch, m_dwFileLength)) )
			pSearch = m_pDirector->AddSearch(m_sSearch, ST_ALTERNATIVE, m_dwFileLength, LIMIT_EXACTLY);
		//
		if (pSearch)
		{
			MLock lock(pSearch->m_mutex);
			pSearch->m_Filename = m_sName;
			m_dwSearchID = pSearch->m_dwID;
		}
	}	
	if (bAutoget && pSearch)
	{
		pSearch->m_bAutoget = true;
		// clear the search for any case
		pSearch->Clear();
	}	
}

void MGnuDownload::NextInQueue()
{
	StatusUpdate(TRANSFER_CLOSED);

	// first check if the current Result is good enough to stay in the queue
	ASSERT(m_nQueuePos<m_Queue.size());
	if (/*m_Queue[m_nQueuePos].nConErrors>=3 ||*/ m_Queue[m_nQueuePos].nPushTimeouts>=25)
	{
		// this is not going to anywhere
		if (m_Queue.size()>1)
		{
			// remove this result
			// and put it to the BAD list
			m_BadResults.push_back(m_Queue[m_nQueuePos]);
			m_Queue.erase(&m_Queue[m_nQueuePos]);
			// note that counter should not be incremented
			m_nQueuePos = m_nQueuePos % m_Queue.size();
		}
		else
			m_bActive = false; // stop this download
	}
	else
	{
		// switch to the next
		m_nQueuePos = (m_nQueuePos+1) % m_Queue.size();
	}
	//
	if (m_bActive)
	{
		if (m_nQueuePos==0)
			UpdateResults(); // we have tried all the current results, but didn't succeed
		if(m_Queue[m_nQueuePos].StatusUpdateTime+m_pPrefs->m_nRetryWait <  xtime())
		{
			StatusUpdate(TRANSFER_QUEUED);
		}
		else
		{
			m_nWaitDelay = m_pPrefs->m_nRetryWait - (xtime() - m_Queue[m_nQueuePos].StatusUpdateTime);
			StatusUpdate(TRANSFER_COOLDOWN);
		}
		m_sHeader  = "";
		m_nDisconnectReason = m_Queue[m_nQueuePos].ErrorCode;
	}
}

void MGnuDownload::BandwidthTimer()
{
	//TRACE("BandwidthTimer");
	MLock lock(m_mutex);
	m_nSecCounter++;
	if (m_bActive)
		m_nSecInactive = 0;
	else
		m_nSecInactive++;
	// Bytes
	m_dwBytes60 			-= m_dwAvgBytes[m_nSecPos];
	m_dwAvgBytes[m_nSecPos]	 = m_dwSecBytes;
	m_dwBytes60 			+= m_dwSecBytes;
	if (m_nSecCounter)
		if (m_nSecCounter<60)
			m_dRate = m_dwBytes60 / m_nSecCounter;
		else
			m_dRate = m_dwBytes60 / 60;
	else
		m_dRate = 0;

	ASSERT(m_nQueuePos < m_Queue.size());
	
	switch (m_nStatus)
	{
		case TRANSFER_QUEUED: //check if its time to start
			{
				bool Ready = true;
				if(m_pPrefs->m_nMaxDownloads >=0 &&  m_pDirector->CountDownloading() >= m_pPrefs->m_nMaxDownloads)
					Ready = false;
				if(Ready)
				{
					// TODO: check for maxperhost downloads
					StartDownload(false);
				}
			}
			break;
		case TRANSFER_CONNECTING: // check for timeouts
			m_nSecsDead++;
			if(m_nSecsDead > m_pPrefs->m_dwConnectTimeout)
			{
				// update Resilt counters
				ASSERT(m_nQueuePos<m_Queue.size());
				m_Queue[m_nQueuePos].nConErrors++;
				//ASSERT(m_mutex.locked());
				ForceDisconnect(REASON_NO_RESPONSE);
				StatusUpdate(TRANSFER_PUSH);
				if (!m_pDirector->Route_LocalPush(m_Queue[m_nQueuePos]));
				{
					m_Queue[m_nQueuePos].nPushTimeouts++;
					//ASSERT(m_mutex.locked());
					ForceDisconnect(REASON_NO_RESPONSE); // there wont be a response
				}
				m_nSecsDead = 0;
			}
			break;
		case TRANSFER_PUSH: // waiting for a push -- check for timeout
			m_nSecsDead++;
			if(m_nSecsDead > m_pPrefs->m_dwPushTimeout)
			{
				// update Resilt counters
				ASSERT(m_nQueuePos<m_Queue.size());
				m_Queue[m_nQueuePos].nPushTimeouts++;
				//ASSERT(m_mutex.locked());
				ForceDisconnect(REASON_NO_RESPONSE);
				NextInQueue();
			}
			break;
		case TRANSFER_RECEIVING: // check for min speed limit
			if(m_pPrefs->m_dMinDownSpeed > 0 && m_nSecCounter>m_pPrefs->m_dwSpeedTimeout)
			{
				// Check if its under the bandwidth limit
				if(m_dRate < m_pPrefs->m_dMinDownSpeed)
					m_nSecsUnderLimit++;
				else
					m_nSecsUnderLimit = 0;
				if(m_nSecsUnderLimit > 15)
				{
					//ASSERT(m_mutex.locked());
					ForceDisconnect(REASON_BELOW_MINIMUM_SPEED);
					NextInQueue();
				}
			}
			// continue here
		case TRANSFER_CONNECTED:  // Check for dead transfer
			if(m_dwSecBytes == 0 && m_nSecCounter > 60 ) // 1 min is reasonable start tolerance
			{
				m_nSecsDead++;
				if(m_nSecsDead > m_pPrefs->m_dwTransferTimeout)
				{
					//ASSERT(m_mutex.locked());
					ForceDisconnect(REASON_NO_RESPONSE);
					NextInQueue();
				}
			}
			else
				m_nSecsDead = 0;
			break;
		case TRANSFER_COOLDOWN:
			if(m_nWaitDelay == 0)
				StatusUpdate(TRANSFER_QUEUED);
			else
				m_nWaitDelay--;
			break;
		case TRANSFER_CLOSED:
			// PROBLEM #3 async close
			CloseFile();
			if(m_bActive && m_dwBytesCompleted < m_dwFileLength)
			{
				NextInQueue();
			}
			break;
	}
	m_dwSecBytes = 0;
	m_nSecPos = (m_nSecPos+1)%60;
}


