#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"
#include "dir.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <ctype.h>

bool SinglePatternMatch(const char* string, const char* pattern, bool bCaseSensitive)
{
    if (*pattern == 0)
        return *string == 0;
    if (*string == 0)
        return false;
    if (*pattern == '?')
    {
        return SinglePatternMatch(string+1, pattern+1, bCaseSensitive);
    }
    if (*pattern == '*')
    {
        // probably its not the most efficient way to do it...
        // i'll think about it later...
        for (int i=0; string[i]; ++i)
        {
            if (SinglePatternMatch(string+i, pattern+1, bCaseSensitive))
                return true;
        }
        return false;
    }
    if (bCaseSensitive)
    {
        return *pattern == *string && SinglePatternMatch(string+1, pattern+1, bCaseSensitive);
    }
    else
    {
        bool bMatch = (*pattern) == (*string);
        if ( !bMatch && isalpha(*pattern) && isalpha(*string) )
        {
            bMatch = (*pattern)+'a'-'A' == (*string) ||
                     (*pattern)-'a'+'A' == (*string);
        }
        return  bMatch && SinglePatternMatch(string+1, pattern+1, bCaseSensitive);
    }
}

bool MultiPatternMatch(const CString& string, const CString& patterns, bool bCaseSensitive)
{
    // isolate each pattern and try to match it against the string
    CString patts = StripWhite(patterns);
    CString pat;
    int nSpace;
    if (!patts.length())
        return true; // special case
    patts += " "; // to make life easier
    while (patts.length())
    {
        nSpace = patts.find(" ");
        pat = patts.substr(0,nSpace);
        patts = patts.substr(nSpace+1);
        if (pat.length() && SinglePatternMatch(string.c_str(), pat.c_str(), bCaseSensitive))
            return true;
    }
    return false;
}

int sortByName(const void * l, const void * r){
    return (*(DirEntry**)l)->Name > (*(DirEntry**)r)->Name ?  1 :
           (*(DirEntry**)l)->Name < (*(DirEntry**)r)->Name ? -1 : 0;
}

bool ScanDir( DirEntryVec& result, const CString &path, const CString &nameFilter, int typeFilter, int sortSpec )
{
    DIR	     *dir;
    dirent   *file;

    g_LibcMutex.lock();

    dir = opendir( path.c_str() );
    if ( !dir )
    {
        g_LibcMutex.unlock();
        return false;
    }
    // ready to rock
    DirEntryVec res;

    while ( (file = readdir(dir)) )
    {
        DirEntry de;
        de.Name = file->d_name;
        //
        if (de.Name == "." || de.Name == "..")
            continue;
        // time to unlock
        g_LibcMutex.unlock();
        //
        de.Path = path + "/" + de.Name; // TODO: clean path
        struct stat st;
        if (0==stat(de.Path.c_str(), &st))
        {
            // fill in the flags first
            de.Type = ( S_ISLNK(st.st_mode)  ? DirEntry::link    : 0 ) |
                      ( S_ISREG(st.st_mode)  ? DirEntry::regular : 0 ) |
                      ( S_ISDIR(st.st_mode)  ? DirEntry::dir     : 0 ) |
                      ( S_ISCHR(st.st_mode)  ? DirEntry::chr_dev : 0 ) |
                      ( S_ISBLK(st.st_mode)  ? DirEntry::blk_dev : 0 ) |
                      ( S_ISFIFO(st.st_mode) ? DirEntry::fifo    : 0 ) |
                      ( S_ISSOCK(st.st_mode) ? DirEntry::socket  : 0 );
            if (de.Type & (DirEntry::regular | DirEntry::dir))
            {
                de.Size = st.st_size;
                de.AccessTime = st.st_atime;
                de.ModifyTime = st.st_mtime;
                de.ChangeTime = st.st_ctime;
            }
            if (de.Type & typeFilter)
            {
                if (MultiPatternMatch(de.Name, nameFilter, false))
                    res.push_back(de);
            }
        }
        // time to lock back
        g_LibcMutex.lock();
    }
    g_LibcMutex.unlock();
    //
    result.clear();
    // Sort...
    if(res.size())
    {
        DirEntry** adi = new DirEntry*[res.size()];
        int i;
        for (i=0;i<res.size(); ++i)
            adi[i] = &res[i];

        qsort( adi, res.size(), sizeof(adi[0]), sortByName);

        for (i=0;i<res.size(); ++i)
            result.push_back(*adi[i]);
        delete [] adi;
    }
    // lock again
    g_LibcMutex.lock();
    closedir(dir);
    g_LibcMutex.unlock();
    return true;
}


