/*
 * Implementation of the proxy around Muine's D-Bus interface.
 *
 * Music Applet
 * Copyright (C) 2006 Paul Kuliniewicz <paul.kuliniewicz@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02111-1301, USA.
 *
 */

#include <config.h>

#include "ma-muine-proxy.h"

#include <stdlib.h>
#include <string.h>


#define GET_PRIVATE(o) 		(G_TYPE_INSTANCE_GET_PRIVATE ((o), MA_TYPE_MUINE_PROXY, MaMuineProxyPrivate))


typedef struct _MaMuineProxyPrivate	MaMuineProxyPrivate;


struct _MaMuineProxyPrivate
{
	DBusGProxy *player_proxy;

	DBusGProxyCall *call_playing;
	DBusGProxyCall *call_song;
	DBusGProxyCall *call_position;
};

static MaDBusProxyClass *parent_class;


/*********************************************************************
 *
 * Function declarations
 *
 *********************************************************************/

static void ma_muine_proxy_class_init (MaMuineProxyClass *klass);
static void ma_muine_proxy_init (MaMuineProxy *mproxy);

static void ma_muine_proxy_dispose (GObject *object);
static void ma_muine_proxy_finalize (GObject *object);

static void ma_muine_proxy_toggle_playback (MaProxy *proxy);
static void ma_muine_proxy_previous (MaProxy *proxy);
static void ma_muine_proxy_next (MaProxy *proxy);

static void ma_muine_proxy_rate_song (MaProxy *proxy, gdouble rating);

static void ma_muine_proxy_prepare_prefs (MaProxy *proxy, GladeXML *xml);

static void ma_muine_proxy_launch (MaProxy *proxy);

static void ma_muine_proxy_poll_connected (MaProxy *proxy);

static void ma_muine_proxy_connect (MaDBusProxy *dproxy, DBusGConnection *connection);
static void ma_muine_proxy_disconnect (MaDBusProxy *dproxy);

static void parse_song_data (MaMuineProxy *mproxy, const gchar *song_data);

static void state_changed_cb (DBusGProxy *player_proxy,
			      gboolean playing,
			      MaMuineProxy *mproxy);

static void get_playing_cb (DBusGProxy *player_proxy,
			    DBusGProxyCall *call,
			    gpointer proxy);

static void song_changed_cb (DBusGProxy *player_proxy,
			     const gchar *song_data,
			     MaMuineProxy *mproxy);

static void get_current_song_cb (DBusGProxy *player_proxy,
				 DBusGProxyCall *call,
				 gpointer proxy);

static void get_position_cb (DBusGProxy *player_proxy,
			     DBusGProxyCall *call,
			     gpointer proxy);


/*********************************************************************
 *
 * GType stuff
 *
 *********************************************************************/

GType
ma_muine_proxy_get_type (void)
{
	static GType type = 0;

	if (type == 0)
	{
		static const GTypeInfo info = {
			sizeof (MaMuineProxyClass),			/* class_size */
			NULL,						/* base_init */
			NULL,						/* base_finalize */
			(GClassInitFunc) ma_muine_proxy_class_init,	/* class_init */
			NULL,						/* class_finalize */
			NULL,						/* class_data */
			sizeof (MaMuineProxy),				/* instance_size */
			0,						/* n_preallocs */
			(GInstanceInitFunc) ma_muine_proxy_init,	/* instance_init */
			NULL
		};

		type = g_type_register_static (MA_TYPE_DBUS_PROXY, "MaMuineProxy", &info, 0);
	}

	return type;
}

static void
ma_muine_proxy_class_init (MaMuineProxyClass *klass)
{
	GObjectClass *object_class = (GObjectClass *) klass;
	MaProxyClass *proxy_class = (MaProxyClass *) klass;
	MaDBusProxyClass *dproxy_class = (MaDBusProxyClass *) klass;
	parent_class = g_type_class_peek_parent (klass);

	object_class->dispose = ma_muine_proxy_dispose;
	object_class->finalize = ma_muine_proxy_finalize;

	proxy_class->toggle_playback = ma_muine_proxy_toggle_playback;
	proxy_class->previous = ma_muine_proxy_previous;
	proxy_class->next = ma_muine_proxy_next;

	proxy_class->rate_song = ma_muine_proxy_rate_song;

	proxy_class->prepare_prefs = ma_muine_proxy_prepare_prefs;

	proxy_class->launch = ma_muine_proxy_launch;

	proxy_class->poll_connected = ma_muine_proxy_poll_connected;

	dproxy_class->connect = ma_muine_proxy_connect;
	dproxy_class->disconnect = ma_muine_proxy_disconnect;

	g_type_class_add_private (klass, sizeof (MaMuineProxyPrivate));
}

static void
ma_muine_proxy_init (MaMuineProxy *mproxy)
{
	MaMuineProxyPrivate *priv = GET_PRIVATE (mproxy);

	priv->player_proxy = NULL;

	priv->call_playing = NULL;
	priv->call_song = NULL;
	priv->call_position = NULL;
}


/*********************************************************************
 *
 * Public interface
 *
 *********************************************************************/

MaProxy *
ma_muine_proxy_new (MaConf *conf)
{
	MaProxy *proxy;

	g_return_val_if_fail (conf != NULL, NULL);

	proxy = g_object_new (MA_TYPE_MUINE_PROXY,
			      "name", "Muine",
			      "icon-name", "muine",
			      "conf", conf,
			      "dbus-name", "org.gnome.Muine",
			      NULL);

	return proxy;
}


/*********************************************************************
 *
 * GObject overrides
 *
 *********************************************************************/

static void
ma_muine_proxy_dispose (GObject *object)
{
	ma_muine_proxy_disconnect (MA_DBUS_PROXY (object));

	G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
ma_muine_proxy_finalize (GObject *object)
{
	G_OBJECT_CLASS (parent_class)->finalize (object);
}


/*********************************************************************
 *
 * MaProxy overrides
 *
 *********************************************************************/

static void
ma_muine_proxy_toggle_playback (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_MUINE_PROXY (proxy));

	dbus_g_proxy_call_no_reply (GET_PRIVATE (proxy)->player_proxy, "SetPlaying",
				    G_TYPE_BOOLEAN, !ma_proxy_get_playing (proxy),
				    G_TYPE_INVALID);
}

static void
ma_muine_proxy_previous (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_MUINE_PROXY (proxy));

	dbus_g_proxy_call_no_reply (GET_PRIVATE (proxy)->player_proxy, "Previous",
				    G_TYPE_INVALID);
}

static void
ma_muine_proxy_next (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_MUINE_PROXY (proxy));

	dbus_g_proxy_call_no_reply (GET_PRIVATE (proxy)->player_proxy, "Next",
				    G_TYPE_INVALID);
}

static void
ma_muine_proxy_rate_song (MaProxy *proxy, gdouble rating)
{
	g_warning ("Muine does not support changing the current song's rating");
}

static void
ma_muine_proxy_prepare_prefs (MaProxy *proxy, GladeXML *xml)
{
	MaConf *conf;

	GtkWidget *page;
	GtkWidget *command;
	GtkWidget *browse;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_MUINE_PROXY (proxy));

	conf = _ma_proxy_get_conf (proxy);

	page = glade_xml_get_widget (xml, "muine");
	command = glade_xml_get_widget (xml, "muine-command");
	browse = glade_xml_get_widget (xml, "muine-browse");

	ma_conf_bind_string_entry (conf, "muine_command", GTK_ENTRY (command));
	ma_conf_bind_string_browse (conf, "muine_command", GTK_BUTTON (browse));

	gtk_widget_show (page);
}

static void
ma_muine_proxy_launch (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_MUINE_PROXY (proxy));

	_ma_proxy_launch_command (proxy, "muine_command");
}

static void
ma_muine_proxy_poll_connected (MaProxy *proxy)
{
	MaMuineProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_MUINE_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	if (priv->call_position == NULL)
	{
		priv->call_position = dbus_g_proxy_begin_call (priv->player_proxy, "GetPosition",
							       get_position_cb, proxy, NULL,
							       G_TYPE_INVALID);
	}
}


/*********************************************************************
 *
 * MaDBusProxy overrides
 *
 *********************************************************************/

static void
ma_muine_proxy_connect (MaDBusProxy *dproxy, DBusGConnection *connection)
{
	MaMuineProxyPrivate *priv;

	g_return_if_fail (dproxy != NULL);
	g_return_if_fail (MA_IS_MUINE_PROXY (dproxy));

	priv = GET_PRIVATE (dproxy);

	priv->player_proxy = dbus_g_proxy_new_for_name (connection,
							"org.gnome.Muine",
							"/org/gnome/Muine/Player",
							"org.gnome.Muine.Player");

	dbus_g_proxy_add_signal (priv->player_proxy, "StateChanged",
				 G_TYPE_BOOLEAN,
				 G_TYPE_INVALID);

	dbus_g_proxy_connect_signal (priv->player_proxy, "StateChanged",
				     G_CALLBACK (state_changed_cb), dproxy, NULL);

	dbus_g_proxy_add_signal (priv->player_proxy, "SongChanged",
				 G_TYPE_STRING,
				 G_TYPE_INVALID);

	dbus_g_proxy_connect_signal (priv->player_proxy, "SongChanged",
				     G_CALLBACK (song_changed_cb), dproxy, NULL);

	priv->call_playing = dbus_g_proxy_begin_call (priv->player_proxy, "GetPlaying",
						      get_playing_cb, dproxy, NULL,
						      G_TYPE_INVALID);

	priv->call_song = dbus_g_proxy_begin_call (priv->player_proxy, "GetCurrentSong",
						   get_current_song_cb, dproxy, NULL,
						   G_TYPE_INVALID);
}

static void
ma_muine_proxy_disconnect (MaDBusProxy *dproxy)
{
	MaMuineProxyPrivate *priv;

	g_return_if_fail (dproxy != NULL);
	g_return_if_fail (MA_IS_MUINE_PROXY (dproxy));

	priv = GET_PRIVATE (dproxy);

	_ma_dbus_proxy_cancel_call (priv->player_proxy, &priv->call_playing);
	_ma_dbus_proxy_cancel_call (priv->player_proxy, &priv->call_song);
	_ma_dbus_proxy_cancel_call (priv->player_proxy, &priv->call_position);

	if (priv->player_proxy != NULL)
	{
		g_object_unref (priv->player_proxy);
		priv->player_proxy = NULL;
	}
}


/*********************************************************************
 *
 * Internal functions
 *
 *********************************************************************/

static void
parse_song_data (MaMuineProxy *mproxy, const gchar *song_data)
{
	gchar *artist = NULL;

	gchar **lines;
	gchar **tokens;
	gchar **iter;

	if (song_data != NULL && song_data[0] != '\0')
	{
		lines = g_strsplit (song_data, "\n", -1);

		for (iter = lines; *iter != NULL; iter++)
		{
			tokens = g_strsplit (*iter, ": ", 2);

			if (tokens[0][0] != '\0' && tokens[1][0] != '\0')
			{
				if (strcmp(tokens[0], "title") == 0)
				{
					_ma_proxy_set_title (MA_PROXY (mproxy), tokens[1]);
				}
				else if (strcmp(tokens[0], "artist") == 0 || strcmp(tokens[0], "performer") == 0)
				{
					if (artist != NULL)
					{
						gchar *tmp = g_strconcat(artist, ", ", tokens[1], NULL);
						g_free (artist);
						artist = tmp;
					}
					else
						artist = g_strdup (tokens[1]);
				}
				else if (strcmp(tokens[0], "album") == 0)
				{
					_ma_proxy_set_album (MA_PROXY (mproxy), tokens[1]);
				}
				else if (strcmp(tokens[0], "duration") == 0 && g_ascii_isdigit (tokens[1][0]))
				{
					_ma_proxy_set_duration (MA_PROXY (mproxy), atol (tokens[1]));
				}
			}

			g_strfreev (tokens);
		}

		g_strfreev (lines);

		_ma_proxy_set_artist (MA_PROXY (mproxy), artist);
		g_free (artist);
	}
	else
		_ma_proxy_set_no_song (MA_PROXY (mproxy));
}


/*********************************************************************
 *
 * Callbacks
 *
 *********************************************************************/

static void
state_changed_cb (DBusGProxy *player_proxy,
		  gboolean playing,
		  MaMuineProxy *mproxy)
{
	_ma_proxy_set_playing (MA_PROXY (mproxy), playing);
}

static void
get_playing_cb (DBusGProxy *player_proxy,
		DBusGProxyCall *call,
		gpointer proxy)
{
	MaMuineProxyPrivate *priv = GET_PRIVATE (proxy);

	gboolean playing;
	GError *error = NULL;

	if (dbus_g_proxy_end_call (player_proxy, call, &error,
				   G_TYPE_BOOLEAN, &playing,
				   G_TYPE_INVALID))
	{
		_ma_proxy_set_playing (MA_PROXY (proxy), playing);
	}
	else
		_ma_dbus_proxy_report_error ("GetPlaying", &error);

	priv->call_playing = NULL;
}

static void
song_changed_cb (DBusGProxy *player_proxy,
		 const gchar *song_data,
		 MaMuineProxy *mproxy)
{
	parse_song_data (mproxy, song_data);
}

static void
get_current_song_cb (DBusGProxy *player_proxy,
		     DBusGProxyCall *call,
		     gpointer proxy)
{
	MaMuineProxyPrivate *priv = GET_PRIVATE (proxy);

	gchar *song_data;
	GError *error = NULL;

	if (dbus_g_proxy_end_call (player_proxy, call, &error,
				   G_TYPE_STRING, &song_data,
				   G_TYPE_INVALID))
	{
		parse_song_data (MA_MUINE_PROXY (proxy), song_data);
		g_free (song_data);
	}
	else
		_ma_dbus_proxy_report_error ("GetCurrentSong", &error);

	priv->call_song = NULL;
}

static void
get_position_cb (DBusGProxy *player_proxy,
		 DBusGProxyCall *call,
		 gpointer proxy)
{
	MaMuineProxyPrivate *priv = GET_PRIVATE (proxy);

	gint elapsed;
	GError *error = NULL;

	if (dbus_g_proxy_end_call (player_proxy, call, &error,
				   G_TYPE_INT, &elapsed,
				   G_TYPE_INVALID))
	{
		_ma_proxy_set_elapsed (MA_PROXY (proxy), elapsed);
	}
	else
		_ma_dbus_proxy_report_error ("GetPosition", &error);

	priv->call_position = NULL;
}
