//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: mrecord.cpp,v 1.2 2002/02/07 08:14:48 muse Exp $
//
//  (C) Copyright 1999-2001 Werner Schweer (ws@seh.de)
//=========================================================

#include <unistd.h>

#include "midithreadp.h"
#include "mtc.h"
#include "event.h"
#include "song.h"
#include "midithread.h"
#include "utils.h"
#include "synth.h"
#include "globals.h"
#include "mididev.h"
#include "midiitransform.h"
#include "utils.h"

extern void processMidiInputTransformPlugins(MidiEvent*);

static MTC mtcCurTime;
static int mtcState;    // 0-7 next expected quarter message
static bool mtcValid;
static int mtcLost;
static bool mtcSync;

static bool mcStart = false;
static int mcStartTick;

//---------------------------------------------------------
//   eventReceived
//---------------------------------------------------------

void MidiThread::eventReceived(int port, unsigned char channel, unsigned char c1, unsigned c2)
      {
      MidiEvent* event = new MidiEvent(
         port, channel & 0xf,
         0, MidiEvent::EventType(0), c1, c2, 0, 0);
      switch(channel & 0xf0) {
            case 0x80:
                  event->setType(MidiEvent::NoteOff);
                  break;
            case 0x90:
                  event->setType(c2 == 0 ? MidiEvent::NoteOff : MidiEvent::Note);
                  break;
            case 0xa0:
                  event->setType(MidiEvent::PAfter);  // Poly Pressure
                  break;
            case 0xb0:
                  event->setType(MidiEvent::Ctrl7);
                  break;
            case 0xc0:
                  event->setType(MidiEvent::Program);
                  break;
            case 0xd0:                            // After Touch
                  event->setType(MidiEvent::CAfter);
                  break;
            case 0xe0:                            // Pitch Bender
                  event->setType(MidiEvent::Pitch);
                  break;
            case 0xf0:
                  break;
            }
      eventReceived(event);
      }

//---------------------------------------------------------
//   filterEvent
//    return true if event filtered
//---------------------------------------------------------

bool MidiThread::filterEvent(const MidiEvent* event, int type, bool thru)
      {
      switch(event->type()) {
            case MidiEvent::Note:
            case MidiEvent::NoteOff:
                  if (type & MIDI_FILTER_NOTEON)
                        return true;
                  break;
            case MidiEvent::PAfter:
                  if (type & MIDI_FILTER_POLYP)
                        return true;
                  break;
            case MidiEvent::Ctrl7:
                  if (type & MIDI_FILTER_CTRL)
                        return true;
                  if (!thru && (midiFilterCtrl1 == event->dataA()
                     || midiFilterCtrl2 == event->dataA()
                     || midiFilterCtrl3 == event->dataA()
                     || midiFilterCtrl4 == event->dataA())) {
                        return true;
                        }
                  break;
            case MidiEvent::Program:
                  if (type & MIDI_FILTER_PROGRAM)
                        return true;
                  break;
            case MidiEvent::CAfter:
                  if (type & MIDI_FILTER_AT)
                        return true;
                  break;
            case MidiEvent::Pitch:
                  if (type & MIDI_FILTER_PITCH)
                        return true;
                  break;
            case MidiEvent::Sysex:
                  if (type & MIDI_FILTER_SYSEX)
                        return true;
                  break;
            default:
                  break;
            }
      return false;
      }

//---------------------------------------------------------
//   Sequencer::eventReceived
//---------------------------------------------------------

void MidiThread::eventReceived(MidiEvent* event)
      {
      if (midiInputTrace)
            event->dump();

      //
      //  special handling for software synthesizer events
      //
      int port = event->port();           // hack
      bool isSynth = (port & 0x100) == 0x100;
      if (!isSynth) {
            for (iSynthI i = synthiInstances.begin(); i != synthiInstances.end(); ++i) {
                  SynthI* si = *i;
                  MidiDevice* dev = si->mdev();
                  if (dev && (dev->port() == port)) {
                        (*i)->writeToGui(event);
                        isSynth = true;
                        break;
                        }
                  }
            }
      port &= 0xff;
      event->setPort(port);

      if (event->type() == MidiEvent::Sysex) {
            const unsigned char* p = event->data();
            int n = event->dataLen();
            if (n >= 4) {
                  if ((p[0] == 0x7f)
                     && ((p[1] == 0x7f) || (p[1] == deviceId))) {
                        if (p[2] == 0x06) {
                              mmcInput(p, n);
                              delete event;
                              return;
                              }
                        if (p[2] == 0x01) {
                              mtcInputFull(p, n);
                              delete event;
                              return;
                              }
                        }
                  else if (p[0] == 0x7e) {
                        nonRealtimeSystemSysex(p, n);
                        delete event;
                        return;
                        }
                  }
            }
      int tick = recTimeStamp();
      event->setPosTick(tick);

      //
      // filter midi remote control events
      //
      int c1 = event->dataA();
      if (rcEnable && (event->isNote() || event->isNoteOff())) {
            if (c1 == rcStopNote) {
                  if (event->isNote()) {
                        write(data->sigFd, "0", 1);
                        }
                  delete event;
                  return;
                  }
            else if (c1 == rcRecordNote) {
                  if (event->isNote()) {
                        write(data->sigFd, "2", 1);
                        }
                  delete event;
                  return;
                  }
            else if (c1 == rcGotoLeftMarkNote) {
                  if (event->isNote()) {
                        write(data->sigFd, "3", 1);
                        }
                  delete event;
                  return;
                  }
            else if (c1 == rcPlayNote) {
                  if (event->isNote()) {
                        write(data->sigFd, "1", 1);
                        }
                  delete event;
                  return;
                  }
            }

      processMidiInputTransformPlugins(event);

      if (filterEvent(event, midiRecordType, false)) {
            delete event;
            return;
            }

      if (!applyMidiInputTransformation(event)) {
            if (midiInputTrace)
                  printf("   midi input transformation: event filtered\n");
            delete event;
            return;
            }

      if (event->type() == MidiEvent::Note) {
            data->curPitch = event->pitch();
            data->curVelo  = event->velo();
            write(data->sigFd, "I", 1);
            }

      TrackList* tl = song->tracks();
      for (iTrack it = tl->begin(); it != tl->end(); ++it) {
            MidiTrack* mt = dynamic_cast<MidiTrack*>(*it);
            if (mt == 0 || !mt->recordFlag())
                  continue;
            event->setPort(mt->outPort());
            event->setChannel(mt->outChannel());
            if ((mt->inPortMask() & (1 << event->port())) && (mt->inChannelMask() & (1 << event->channel()))) {
                  if (data->state == PLAY && song->record()) {
                        if (data->loopPassed) {
                              // this is the first event in a new loop
                              //   - erase all events from previous loop
                              mt->events()->clear();
                              }
                        MidiEvent* ev = new MidiEvent(*event);
                        mt->events()->add(ev);
                        }

                  // echo event to current track output
                  // (if not from synth gui)

                  MidiPort* mport = &midiPorts[event->port()];
                  if (!isSynth && mt->midiThruFlag() && !filterEvent(event, midiThruType, true)) {
                        if (event->isNote() || event->isNoteOff()) {
                              MidiEvent e(*event);
                              int pitch = e.pitch() + mt->transposition;
                              if (pitch > 127)
                                    pitch = 127;
                              if (pitch < 0)
                                    pitch = 0;
                              e.setPitch(pitch);

                              //
                              // apply track values
                              //
                              if (!e.isNoteOff()) {
                                    int velo = e.velo() + mt->velocity;
                                    velo = (velo * mt->compression) / 100;
                                    if (velo > 127)
                                          velo = 127;
                                    if (velo < 1)
                                          velo = 1;
                                    e.setVelo(velo);
                                    }
                              mport->putEvent(&e);
                              }
                        else
                              mport->putEvent(event);
                        }
                  }
            }
      data->loopPassed = false;
      }

//---------------------------------------------------------
//  mmcInput
//    Midi Machine Control Input received
//---------------------------------------------------------

void MidiThread::mmcInput(const unsigned char* /*p*/, int /*n*/)
      {
#if 0
      if (!extSyncFlag.value())
            return;
//      printf("mmcInput: %x -- %02x %d %02x %02x %02x\n",
//         this, n, p[2], p[3], p[4]);

      switch(p[3]) {
            case 1:
                  printf("MMC: STOP\n");
                  if (_state == SSTATE_PLAY || _state == SSTATE_START_PLAY)
                        _state = SSTATE_STOP;
                  break;
            case 2:
                  printf("MMC: PLAY\n");
//                  break;
            case 3:
                  printf("MMC: DEFERRED PLAY\n");
                  mtcState = 0;
                  mtcValid = false;
                  mtcLost  = 0;
                  mtcSync  = false;
                  break;

            case 4:  printf("FF\n"); break;
            case 5:  printf("REWIND\n"); break;
            case 6:  printf("REC STROBE\n"); break;
            case 7:  printf("REC EXIT\n"); break;
            case 0xd:  printf("RESET\n"); break;
            case 0x44:
                  if (p[5] == 0) {
                        printf("LOCATE IF\n");
                        break;
                        }
                  else if (p[5] == 1) {
                        MTC mtc(p[6] & 0x1f, p[7], p[8], p[9], p[10]);
                        mmcPos = tempomap.time2tick(mtc.time());

                        printf("MMC: seek ");
                        mtc.print();
                        printf("\n");

                        write(data->sigFd, "G", 1);
                        break;
                        }
                  // fall through
            default:
                  printf("MMC %x %x\n", p[3], p[4]); break;
            }
#endif
      }

//---------------------------------------------------------
//   mtcInputQuarter
//    process Quarter Frame Message
//---------------------------------------------------------

void MidiThread::mtcInputQuarter(int, unsigned char c)
      {
      static int hour, min, sec, frame;

      if (!extSyncFlag.value())
            return;

      int valL = c & 0xf;
      int valH = valL << 4;

      int _state = (c & 0x70) >> 4;
      if (mtcState != _state)
            mtcLost += _state - mtcState;
      mtcState = _state + 1;

      switch(_state) {
            case 7:
                  hour  = (hour  & 0x0f) | valH;
                  break;
            case 6:
                  hour  = (hour  & 0xf0) | valL;
                  break;
            case 5:
                  min   = (min   & 0x0f) | valH;
                  break;
            case 4:
                  min   = (min   & 0xf0) | valL;
                  break;
            case 3:
                  sec   = (sec   & 0x0f) | valH;
                  break;
            case 2:
                  sec   = (sec   & 0xf0) | valL;
                  break;
            case 1:
                  frame = (frame & 0x0f) | valH;
                  break;
            case 0:  frame = (frame & 0xf0) | valL;
                  break;
            }
      frame &= 0x1f;    // 0-29
      sec   &= 0x3f;    // 0-59
      min   &= 0x3f;    // 0-59
      hour  &= 0x1f;

      if (mtcState == 8) {
            mtcValid = (mtcLost == 0);
            mtcState = 0;
            mtcLost  = 0;
            if (mtcValid) {
                  mtcCurTime.set(hour, min, sec, frame);
                  mtcSyncMsg(mtcCurTime, !mtcSync);
                  mtcSync = true;
                  }
            }
      else if (mtcValid && (mtcLost == 0)) {
            mtcCurTime.incQuarter();
            mtcSyncMsg(mtcCurTime, false);
            }
      }

//---------------------------------------------------------
//   mtcInputFull
//    process Frame Message
//---------------------------------------------------------

void MidiThread::mtcInputFull(const unsigned char* p, int n)
      {
      if (!extSyncFlag.value())
            return;

      if (p[3] != 1) {
            if (p[3] != 2) {   // silently ignore user bits
                  printf("unknown mtc msg subtype 0x%02x\n", p[3]);
                  dump(p, n);
                  }
            return;
            }
      int hour  = p[4];
      int min   = p[5];
      int sec   = p[6];
      int frame = p[7];

      frame &= 0x1f;    // 0-29
      sec   &= 0x3f;    // 0-59
      min   &= 0x3f;    // 0-59
//      int type = (hour >> 5) & 3;
      hour &= 0x1f;

      mtcCurTime.set(hour, min, sec, frame);
      mtcState = 0;
      mtcValid = true;
      mtcLost = 0;

//      printf("MTC Full ");
//      mtcCurTime.print();
//      printf("\n");
      }

//---------------------------------------------------------
//   nonRealtimeSystemSysex
//---------------------------------------------------------

void MidiThread::nonRealtimeSystemSysex(const unsigned char* p, int/* n*/)
      {
//      int chan = p[2];
      switch(p[3]) {
            case 4:
                  printf("NRT Setup\n");
                  break;
            default:
//                  printf("unknown NRT Msg 0x%02x\n", p[3]);
//                  dump(p, n);
                  break;
           }
      }

//---------------------------------------------------------
//   mtcSyncMsg
//    process received mtc Sync
//---------------------------------------------------------

void MidiThread::mtcSyncMsg(const MTC& /*mtc*/, bool /*seekFlag*/)
      {
#if 0
      double time = mtc.time();
      if (seekFlag && data->state == IDLE) {
            int tick = tempomap.time2tick(mtc.time());
            start(tick);
            data->state = PLAY;
            write(data->sigFd, "1", 1);
            return;
            }
      double curT = curTime();

      int newTempoSN;
      double cposTime = tempomap.tick2time(curTickPos, &newTempoSN);
      if (tempoSN != newTempoSN) {
            tempoSN   = newTempoSN;
            startTime = curT - cposTime;
            }

      //
      // diff is the time in sec MusE is out of sync
      //
      double diff = time - (curT - startTime);
      if (debugSync)
            printf("%d %f\n", mtcState, diff);
#endif
      }

//---------------------------------------------------------
//   setSongPosition
//    MidiBeat is a 14 Bit value. Each MidiBeat spans
//    6 MIDI Clocks. Inother words, each MIDI Beat is a
//    16th note (since there are 24 MIDI Clocks in a
//    quarter note).
//---------------------------------------------------------

void MidiThread::setSongPosition(int port, int midiBeat)
      {
      if (midiInputTrace)
            printf("set song position port:%d %d\n", port, midiBeat);
      if (!extSyncFlag.value())
            return;
      data->playTickPos = data->midiClick = (division * midiBeat) / 4;
      write(data->sigFd, "G", 1);
      }

//---------------------------------------------------------
//   realtimeSystemInput
//    real time message received
//---------------------------------------------------------

void MidiThread::realtimeSystemInput(int port, int c)
      {
      data->realtimeSystemInput(port, c);
      }

void MidiThreadPrivate::realtimeSystemInput(int port, int c)
      {
      if (midiInputTrace)
            printf("realtimeSystemInput port:%d 0x%x\n", port+1, c);

      if (midiInputTrace && (extSyncPort != port)) {
//            printf("extSyncPort %d != received sync port %d\n",
//               extSyncPort, port);
            return;
            }
      if (!extSyncFlag.value())
            return;
      switch(c) {
            case 0xf8:  // midi clock (24 ticks / quarter note)
                  {
                  double mclock0 = curTime();
                  processMidiTick();
                  if (state == PLAY) {
                        recTick  += division / 24;
                        int diff  = recTick - midiTick;
                        int tempo = tempomap.tempo(0);
                        tempo    -= diff*5;
                        tempomap.setTempo(0, tempo);
                        midiTick = recTick;  // brutal
                        break;
                        }
                  double tdiff0   = mclock0 - mclock1;
                  double tdiff1   = mclock1 - mclock2;
                  if (mclock1 == 0.0)
                        midiPorts[port].device()->discardInput();
                  if ((mclock2 != 0.0) && (tdiff0 > 0.0)) {
                        int tempo0 = int(24000000.0 * tdiff0 + .5);
                        int tempo1 = int(24000000.0 * tdiff1 + .5);
                        int tempo = tempomap.tempo(0);

                        int diff0 = tempo0 - tempo;
                        int diff1 = tempo1 - tempo0;
                        if (diff0) {
                              int newTempo = tempo + diff0/8 + diff1/16;
//                              printf("new tempo %d -> %d (%d + %d + %d)\n",
//                                tempo, tempo0,
//                                tempo, diff0/4, diff1/8);
                              tempomap.setTempo(0, newTempo);
                              }
                        }
                  mclock2 = mclock1;
                  mclock1 = mclock0;
                  }
                  break;
            case 0xf9:  // midi tick  (every 10 msec)
                  if (mcStart) {
                        song->setPos(0, mcStartTick);
                        mcStart = false;
                        return;
                        }
                  break;
            case 0xfa:  // start
printf("start\n");
                  if (state == IDLE) {
                        seek(0);
                        startPlay();
                        }
                  break;
            case 0xfb:  // continue
printf("continue\n");
                  if (state == IDLE) {
                        startPlay();
                        }
                  break;
            case 0xfc:  // stop
printf("stop\n");
                  if (state == PLAY)
                        stopPlay();
                  break;
            case 0xfd:  // unknown
            case 0xfe:  // active sensing
            case 0xff:  // system reset
                  break;
            }
      }


