/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <math.h>
#include <unistd.h>
#include <string.h>
#include <lcms.h>

#include "MRI.h"

extern char *progname;


/* Prepend a convertor to LINETYPE_DOUBLE.
 * If required, prepend a convertor from the camera to the internal color space.
 */
static struct link *
ConvertToFSpace (MRI *mri,
		 struct link *head,
		 MRI_balance balanceSpec,
		 int useLab,
		 const char *cameraProfileFile,
		 const char *iSpaceFileName,
		 FILE *tracefd)
{
    head = GenMRIStFConverter (head, balanceSpec, useLab);
    /* if (tracefd) head = MRI_GenPPMWriterTee (head, 8, FALSE, tracefd); */

    if (strcmp (cameraProfileFile, iSpaceFileName) != 0)
	head = MRI_GenColorSpacer (mri, cameraProfileFile, iSpaceFileName, INTENT_PERCEPTUAL, head);
    return head;
}

static char	*cameraProfileFile = NULL;
static char	*outputProfileFile = NULL;
static char	*iSpaceFileName;

void
InitColorProfiles(MRI *mri, int useLab, char *CPFile, char *OPFile)
{
	char	*iSpace;
        cameraProfileFile = CPFile;
	outputProfileFile = OPFile;
	iSpace = useLab ? "__LAB__" : cameraProfileFile;
	if (strcmp (iSpace, "__LAB__") == 0)
		iSpaceFileName = strdup ("__LAB__");
	else {
		iSpaceFileName = cameraProfileFile;
	}
}

struct link *
GenOutputProfileConverter (MRI *mri, struct link *head)
{
	if (iSpaceFileName != (char *)0 && outputProfileFile != (char *)0) {
		if (strcmp (iSpaceFileName, "__LAB__") != 0)
			head = MRI_GenColorSpacer (mri, iSpaceFileName, outputProfileFile, INTENT_PERCEPTUAL, head);
		else {
#if 0
			head = MRI_GenColorSpacer (mri, MRI_NATIVE_PROFILE, outputProfileFile, INTENT_PERCEPTUAL, head);
			head = MRI_GenColorSpacer (mri, iSpaceFileName, MRI_NATIVE_PROFILE, INTENT_ABSOLUTE_COLORIMETRIC, head);
			head = MRI_GenColorSpacer (mri, iSpaceFileName, outputProfileFile, INTENT_PERCEPTUAL, head);
#endif
#if 0
			head = MRI_GenColorSpacer (mri, iSpaceFileName, outputProfileFile, INTENT_ABSOLUTE_COLORIMETRIC, head);
#endif
			head = MRI_GenColorSpacer (mri, iSpaceFileName, outputProfileFile, INTENT_PERCEPTUAL, head);
		}
	}
	else
		fprintf (stderr, "%s: Info: colorspace conversion NOT done.\n", progname);
	return head;
}

void
FreeProfileFiles()
{
	if (cameraProfileFile) free (cameraProfileFile);
	if (outputProfileFile) free (outputProfileFile);
}

MRI_Link *
MRI_GenBayerInterpolator (MRI *mri, MRI_Link *head, const MRI_InterpolatorOptions *io, FILE *tracefd)
{
	MRI_balance	nobalanceSpec;

	/* If doing vector median interpolation, compute output pixels from the
	 * 8 surrounding pseudo-pixels.
	 */
	if (io->doInterpolate) {
		if (strcmp (io->imethod, "vmedian") == 0)
			head = GenVMedianCondenser (io->vmedian_tolerance, head);
		else if (strcmp (io->imethod, "svmedian") == 0)
			head = GenVMedianCondenser (io->vmedian_tolerance, head);
	}

	/* Convert from LINETYPE_SHORT to LINETYPE_DOUBLE.
	 */
	nobalanceSpec.rgain = 256;
	nobalanceSpec.ggain = 256;
	nobalanceSpec.bgain = 256;
	head = ConvertToFSpace (mri, head, nobalanceSpec, io->useLab,
				cameraProfileFile, iSpaceFileName, tracefd);

	/* Interpolate Bayer pattern pixel data.
	 * Except for vector median interpolation: in that case,
	 * generate pseudo-pixels based on neighboring pixels.
	 */
	if (io->doInterpolate == 0 || strcmp (io->imethod, "none") == 0) {
	}
	else if (strcmp (io->imethod, "bilinear") == 0) {
		head = MRI_GenInterpBilinearRGB (head, MRI_RGFirst (mri, io->rotate));
	}
	else if (strcmp (io->imethod, "mgbrb") == 0) {
		head = MRI_GenInterpMedianGBilinearRB (head, MRI_RGFirst (mri, io->rotate));
	}
	else if (strcmp (io->imethod, "mgmrb") == 0) {
		head = MRI_GenInterpMedianGHueRBilinearB (head, MRI_RGFirst (mri, io->rotate));
	}
	else if (strcmp (io->imethod, "vmedian") == 0) {
		head = MRI_GenPseudoPixelExpanderFloat (head, MRI_RGFirst (mri, io->rotate), MRI_PSEUDO_ORIG);
	}
	else if (strcmp (io->imethod, "svmedian") == 0) {
		head = MRI_GenPseudoPixelExpanderFloat (head, MRI_RGFirst (mri, io->rotate), MRI_PSEUDO_SMOOTH);
	}
	else if (strcmp (io->imethod, "pseudoexp") == 0) {
		head = MRI_GenPseudoPixelExpanderFloat (head, MRI_RGFirst (mri, io->rotate), MRI_PSEUDO_SMOOTH);
	}
	else {
		fprintf (stderr, "%s: Error: interpolation method %s is not implemented.\n", progname, io->imethod);
		exit (1);
	}

	return head;
}

struct link *
MRI_GenImageImprover (struct link *head, const MRI_ImproverOptions *opt,
		      MRI_ImproverInfo *info)
{
	if (fabs(opt->iSharpen) > 1.0e-5 ||
	    (opt->useLab && fabs(opt->cSharpen) > 1.0e-5)) {
		double blur = 1.0 / opt->blur;
		double scale;
		int i, j, k;
		int radius = opt->radius;
		int tworp1 = 2 * radius + 1;
		double *M = (double *)alloca(sizeof(double) * tworp1 * tworp1);
		double *M2 = (double *)alloca(sizeof(double) * tworp1 * tworp1);

		if (M == (double *)0 || M2 == (double *)0) {
			fprintf (stderr, "Warning: cannot allocate memory for Blue matrix. Struggling on.\n");
		}
		else {
			k = 0;
			scale = 0.0;
			for (i = 0; i < tworp1; i++) {
				int ri = i > radius ? i - radius : radius - i;
				for (j = 0; j < tworp1; j++) {
					int rj = j > radius ? j - radius : radius - j;
					M[k] = pow (blur, sqrt(ri*ri + rj*rj));
					scale += M[k++];
				}
			}
			scale = 1.0 / scale;

			k = 0;
			for (i = 0; i < tworp1; i++)
				for (j = 0; j < tworp1; j++)
					M[k++] *= scale;

#if 0
			k = 0;
			for (i = 0; i < tworp1; i++) {
				fprintf (stderr, i == 0 ? "M = [" : "     ");
				for (j = 0; j < tworp1; j++)
					fprintf (stderr, "%s%5g", j == 0 ? "  " : ", ", M[k++]);
				if (i == 2*radius)
					fprintf (stderr, " ]");
				fprintf (stderr, "\n");
			}
#endif

			if (fabs(opt->iSharpen) > 1.0e-5) {
				k = 0;
				for (i = 0; i < tworp1; i++)
					for (j = 0; j < tworp1; j++) {
						M2[k] = -opt->iSharpen * M[k];
						if (i == radius && j == radius)
							M2[k] += 1 + opt->iSharpen;
						k++;
					}
				head = MRI_GenConvolver (opt->useLab ? TONE_R : TONE_RGB, 0.0, 100.0, radius, M2, head);
			}
			if (opt->useLab && fabs(opt->cSharpen) > 1.0e-5) {
				k = 0;
				for (i = 0; i < tworp1; i++)
					for (j = 0; j < tworp1; j++) {
						M2[k] = -opt->cSharpen * M[k];
						if (i == radius && j == radius)
							M2[k] += 1 + opt->cSharpen;
						k++;
					}
				head = MRI_GenConvolver (TONE_G | TONE_B, -128.0, 128.0, radius, M2, head);
			}
		}
	}

	/* Apply a Vector Median filter (if requested).
	 */
	if (opt->applyVMF)
		head = GenVMedianFilter (head, opt->useLab, opt->vMedianTolerance);
	if (opt->applyMedianFilter)
		head = GenFMedianFilter (head, (opt->useLab && opt->filterColorOnly) ? (TONE_G|TONE_B) : TONE_RGB);

	if (info != NULL)
	    head = MRI_GenHistogramCounter (0, NULL, &info->Lmin, &info->Lmax, head);

	/* Apply tone curve adjustment (if requested). */
	if (opt->toneCurveFile != (char *)0) {
		MRI_SampledCurve *crv;
		extern MRI_SampledCurve *LoadSampledToneCurve(char *);
		crv = LoadSampledToneCurve (opt->toneCurveFile);
		head = MRI_GenToneCurver (crv, head, opt->useLab);

	}

	return head;
}

