/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001  CodeFactory AB
 * Copyright (C) 2001  Richard Hult <rhult@codefactory.se>
 * Copyright (C) 2001  Mikael Hallendal <micke@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult <rhult@codefactory.se>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <locale.h>
#include <math.h>
#include <stdlib.h>
#include <gal/widgets/e-unicode.h>
#include <bonobo.h>
#include <bonobo/bonobo-shlib-factory.h>
#include <liboaf/liboaf.h>
#include <libgnomevfs/gnome-vfs.h>
#include "gnome-xml/parser.h"
#include "gnome-xml/parserInternals.h"
#include "gnome-xml/xmlmemory.h"
#include "libmrproject/GNOME_MrProject.h"
#include "util/time-utils.h"
#include "util/type-utils.h"
#include "project-engine/project.h"

#include "file-filter.h"
#include "xml-file-filter.h"

static void xml_file_filter_init	(XmlFileFilter	      *filter);
static void xml_file_filter_class_init	(XmlFileFilterClass   *klass);
static void xff_destroy                 (GtkObject            *object);

GNOME_CLASS_BOILERPLATE (XmlFileFilter, xml_file_filter,
			 FileFilter,    file_filter);

struct _XmlFileFilterPriv {
        xmlDocPtr            doc;      /* Xml document. */
	xmlNsPtr             ns;       /* Main namespace. */

	xmlNodePtr           tasks_node;
	xmlNodePtr           resource_groups_node;
	xmlNodePtr           resources_node;
 	xmlNodePtr           allocations_node;
	
	GHashTable	    *id_hash;
	GSList              *delayed_dependencies;

	/* Keep track of the earliest/latest task. */
	time_t               t1, t2;
};

typedef struct {
	gint                 task_id;
	gint                 predecessor_id;
	gint                 dependency_id;
	GM_DependencyType    type;
} DelayedDependency;

/* static gboolean      xff_parse                (FileFilter          *filter, */
/*                                                const gchar         *buffer, */
/*                                                gint                 length, */
/*                                                CORBA_Environment   *ev); */

static gchar        *xml_get_value            (xmlNodePtr       node,
                                               const char      *name);
static gchar        *xml_get_string_utf8      (xmlNodePtr       node, 
                                               const char      *name);
static xmlNodePtr    xml_search_child         (xmlNodePtr       node, 
                                               const gchar     *name);
static void          read_project_properties  (XmlFileFilter   *filter,
                                               xmlNodePtr       node);
static void          read_start_end_time      (XmlFileFilter   *filter);
static void          write_project_properties (XmlFileFilter   *filter,
                                               xmlNodePtr       node);
static void
write_predecessor (XmlFileFilter   *filter,
		   xmlNodePtr       parent,
		   GM_Dependency   *dependency)
{
        XmlFileFilterPriv   *priv;
	xmlNodePtr           node;
	gchar               *str;

	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_XML_FILE_FILTER (filter));

        priv = filter->priv;
        
	node = xmlNewChild (parent, priv->ns, "predecessor", NULL);
	
	if (!node){
		g_warning ("Failed saving predecessor.\n");
		return;
	}

	str = g_strdup_printf ("%d", dependency->depId);
	xmlSetProp (node, "id", str);
	g_free (str);

	str = g_strdup_printf ("%d", dependency->predecessorId);
	xmlSetProp (node, "predecessor-id", str);
	g_free (str);

	switch (dependency->type) {
	case GNOME_MrProject_DEPENDENCY_FS:
		str = "FS";
		break;
	case GNOME_MrProject_DEPENDENCY_FF:
		str = "FF";
		break;
	case GNOME_MrProject_DEPENDENCY_SF:
		str = "SF";
		break;
	case GNOME_MrProject_DEPENDENCY_SS:
		str = "SS";
		break;
	default:
		str = "FS";
	}

	xmlSetProp (node, "type", str);
}

static void
write_task (XmlFileFilter *filter, GM_Task *task)
{
        XmlFileFilterPriv   *priv;
	xmlNodePtr           node, parent_node, predecessors_node;
	gint                 parent_id, *key;
	gchar               *str, *note;
	CORBA_Environment    ev;
	GM_DependencySeq    *predecessors;
	guint                i;
	
	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_XML_FILE_FILTER (filter));
	g_return_if_fail (task != NULL);

        priv = filter->priv;

	parent_id   = task->parentId;
	parent_node = g_hash_table_lookup (priv->id_hash, &parent_id);

	/* This is just a precaution, should never be needed. */
	if (parent_node == NULL) {
		parent_node = priv->tasks_node;
	}
		
	node = xmlNewChild (parent_node, priv->ns, "task", NULL);

	if (!node){
		g_warning ("Failed saving task.");
		return;
	}

	key = g_new (gint, 1);
	*key = task->taskId;
	g_hash_table_insert (priv->id_hash, key, node);

	str = g_strdup_printf ("%d", task->taskId);
	xmlSetProp (node, "id", str);
	g_free (str);

	str = e_utf8_to_locale_string (task->name);
	xmlSetProp (node, "name", str);
	g_free (str);

	str = isodate_from_time_t (task->start);
	xmlSetProp (node, "start", str);
	g_free (str);

	str = isodate_from_time_t (task->end);
	xmlSetProp (node, "end", str);
	g_free (str);

	str = g_strdup_printf ("%d", task->percentComplete);
	xmlSetProp (node, "percent-complete", str);
	g_free (str);

	CORBA_exception_init (&ev);

	note = GNOME_MrProject_TaskManager_getNote (FILE_FILTER(filter)->task_mgr,
						    task->taskId,
						    &ev);
	
	if (!BONOBO_EX (&ev) && note) {
		str = e_utf8_to_locale_string (note);
		xmlSetProp (node, "note", str);
		g_free (str);
		CORBA_free (note);
	}

	predecessors = GNOME_MrProject_TaskManager_getPredecessors (
		FILE_FILTER(filter)->task_mgr, task->taskId, &ev);
	
	if (!BONOBO_EX (&ev) && predecessors) {
		predecessors_node = xmlNewChild (node, priv->ns, 
						 "predecessors", 
						 NULL);
		for (i = 0; i < predecessors->_length; i++) {
			write_predecessor (filter, predecessors_node,
					   &predecessors->_buffer[i]);
		}
	}

	CORBA_exception_free (&ev);
}

static void
write_resource (XmlFileFilter   *filter,
		GM_Resource     *resource)
{
        XmlFileFilterPriv   *priv;
	xmlNodePtr           node;
	gchar      *str;
	
	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_XML_FILE_FILTER (filter));
	g_return_if_fail (resource != NULL);

        priv = filter->priv;

	node = xmlNewChild (priv->resources_node, priv->ns, "resource", NULL);

	if (!node){
		g_warning ("Failed saving resource.");
		return;
	}

	str = g_strdup_printf ("%d", resource->resourceId);
	xmlSetProp (node, "id", str);
	g_free (str);

	str = e_utf8_to_locale_string (resource->name);
	xmlSetProp (node, "name", str);
	g_free (str);

	str = g_strdup_printf ("%d", resource->groupId);
	xmlSetProp (node, "group", str);
	g_free (str);
	
	str = g_strdup_printf ("%d", resource->type);
	xmlSetProp (node, "type", str);
	g_free (str);

	str = g_strdup_printf ("%d", resource->units);
	xmlSetProp (node, "units", str);
	g_free (str);

	str = e_utf8_to_locale_string (resource->email);
	xmlSetProp (node, "email", str);
	g_free (str);

	str = g_strdup_printf ("%f", resource->stdRate);
	xmlSetProp (node, "std-rate", str);
	g_free (str);

	str = g_strdup_printf ("%f", resource->ovtRate);
	xmlSetProp (node, "ovt-rate", str);
	g_free (str);
}

static void
write_resource_group (XmlFileFilter      *filter, 
                      GM_ResourceGroup   *group)
{
        XmlFileFilterPriv   *priv;
	xmlNodePtr           node;
	gchar               *str;

	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_XML_FILE_FILTER (filter));
	g_return_if_fail (group != NULL);

        priv = filter->priv;

	node = xmlNewChild (priv->resource_groups_node, 
                            priv->ns, "group", NULL);

	if (!node){
		g_warning ("Failed saving resource group.");
		return;
	}

	str = g_strdup_printf ("%d", group->groupId);
	xmlSetProp (node, "id", str);
	g_free (str);
	
	str = e_utf8_to_locale_string (group->name);
	xmlSetProp (node, "name", str);	


	g_free (str);

	str = e_utf8_to_locale_string (group->adminName);
	xmlSetProp (node, "admin-name", str);
	g_free (str);

	str = e_utf8_to_locale_string (group->adminPhone);
	xmlSetProp (node, "admin-phone", str);
	g_free (str);

	str = e_utf8_to_locale_string (group->adminEmail);
	xmlSetProp (node, "admin-email", str);
	g_free (str);
}

static void
write_allocation (XmlFileFilter *filter, GM_Allocation *allocation)
{
        XmlFileFilterPriv   *priv;
	xmlNodePtr           node;
	gchar               *str;

	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_XML_FILE_FILTER (filter));
	g_return_if_fail (allocation != NULL);

        priv = filter->priv;

	node = xmlNewChild (priv->allocations_node, priv->ns,
			    "allocation", NULL);

	if (!node){
		g_warning ("Failed saving allocation.");
		return;
	}

	str = g_strdup_printf ("%d", allocation->taskId);
	xmlSetProp (node, "task-id", str);
	g_free (str);
	str = g_strdup_printf ("%d", allocation->resourceId);
	xmlSetProp (node, "resource-id", str);
	g_free (str);
}

static xmlNodePtr
xml_write_project (XmlFileFilter *filter)
{
        XmlFileFilterPriv     *priv;
	xmlNodePtr             cur;
	xmlNsPtr               ns;
	gint                  *key;
	gchar                 *old_locale;
	CORBA_Environment      ev;
	GM_TaskSeq            *tasks;
	GM_ResourceSeq        *resources;
	GM_ResourceGroupSeq   *resource_groups;
	GM_AllocationSeq      *allocations;
	guint                  i;
        GM_Id                  default_group;
        gchar                 *str;
        
	g_return_val_if_fail (filter != NULL, NULL);
	g_return_val_if_fail (IS_XML_FILE_FILTER (filter), NULL);

        priv = filter->priv;

	cur = xmlNewDocNode (priv->doc, priv->ns, "project", NULL);

	if (cur == NULL) {
		return NULL;
	}
	
	if (priv->ns == NULL) {
		ns = xmlNewNs (cur, "http://mrproject.codefactory.se/format/project/1", NULL);
		xmlSetNs (cur, ns);
		priv->ns = ns;
	}

	/* Project properties. */
	write_project_properties (filter, cur);

	/* Make sure we don't get locale specific floats. */
	old_locale = setlocale (LC_NUMERIC, "C");

	/* Write tasks */

	priv->tasks_node = xmlNewChild (cur, priv->ns, "tasks", NULL);
	key = g_new (gint, 1);
	*key = 0;
	g_hash_table_insert (priv->id_hash, key, priv->tasks_node);

	CORBA_exception_init (&ev);
	tasks = GNOME_MrProject_TaskManager_getAllTasks (
                FILE_FILTER(filter)->task_mgr, &ev);

	if (!BONOBO_EX (&ev) && tasks) {
		for (i = 0; i < tasks->_length; i++) {
			write_task (filter, &tasks->_buffer[i]);
		}
		CORBA_free (tasks);
	} else {
		CORBA_exception_free (&ev);
	}

	GNOME_MrProject_TaskManager__set_state (
		FILE_FILTER(filter)->task_mgr, 
                GNOME_MrProject_PROJECT_STATE_SAVED, &ev);

	if (BONOBO_EX (&ev)) {
		CORBA_exception_free (&ev);
	}

	/* Write resource groups. */

	priv->resource_groups_node = xmlNewChild (cur, priv->ns, 
						  "resource-groups", 
						  NULL);

	resource_groups = GNOME_MrProject_ResourceManager_getAllGroups (
		FILE_FILTER(filter)->resource_mgr,
		&ev);
        
        default_group = GNOME_MrProject_ResourceManager_getDefaultGroup (
                FILE_FILTER(filter)->resource_mgr,
                &ev);

        if (default_group != -1) {
                str = g_strdup_printf ("%d", default_group);
                xmlSetProp (priv->resource_groups_node, 
                            "default_group", str);
                g_free (str);
        }
        

	if (!BONOBO_EX (&ev) && resource_groups) {
		for (i = 0; i < resource_groups->_length; i++) {
			write_resource_group (filter, 
                                              &resource_groups->_buffer[i]);
		}
		CORBA_free (resource_groups);
	} else {
		CORBA_exception_free (&ev);
	}

	/* Write resources. */

	priv->resources_node = xmlNewChild (cur, priv->ns, "resources", NULL);
	resources = GNOME_MrProject_ResourceManager_getAllResources (
		FILE_FILTER(filter)->resource_mgr, &ev);

	if (!BONOBO_EX (&ev) && resources) {
		for (i = 0; i < resources->_length; i++) {
			write_resource (filter, &resources->_buffer[i]);
		}
		CORBA_free (resources);
	} else {
		CORBA_exception_free (&ev);
	}

	GNOME_MrProject_ResourceManager__set_state (
		FILE_FILTER(filter)->resource_mgr, 
                GNOME_MrProject_PROJECT_STATE_SAVED, &ev);

	if (BONOBO_EX (&ev)) {
		CORBA_exception_free (&ev);
	}

	/* Write allocations. */
	priv->allocations_node = xmlNewChild (cur, priv->ns,
					      "allocations", NULL);
	allocations = GNOME_MrProject_AllocationManager_getAllAllocations (
		FILE_FILTER(filter)->allocation_mgr, &ev);
        
	if (!BONOBO_EX (&ev) && allocations) {
		for (i = 0; i < allocations->_length; i++) {
			write_allocation (filter, &allocations->_buffer[i]);
		}
		CORBA_free (allocations);
	} else {
		CORBA_exception_free (&ev);
	}

	GNOME_MrProject_AllocationManager__set_state (
		FILE_FILTER(filter)->allocation_mgr, 
                GNOME_MrProject_PROJECT_STATE_SAVED, &ev);

	if (BONOBO_EX (&ev)) {
		CORBA_exception_free (&ev);
	}

	/* Restore locale. */
	setlocale (LC_NUMERIC, old_locale);

 	return cur;
}

static void
id_hash_destroy (gpointer key, gpointer value, gpointer user_data)
{
	g_free (key);
}

static void
xff_destroy (GtkObject *object)
{
        XmlFileFilter       *filter;
        XmlFileFilterPriv   *priv;
        
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_XML_FILE_FILTER (object));

        filter = XML_FILE_FILTER (object);
        priv   = filter->priv;

	g_hash_table_foreach (priv->id_hash, id_hash_destroy, NULL);
	g_hash_table_destroy (priv->id_hash);

	g_free (priv);
	filter->priv = NULL;

	if (GTK_OBJECT_CLASS (parent_class)->destroy) {
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
	}
}

/*
 * Read
 */

static void
xml_read_predecessor (XmlFileFilter        *filter,
		      GNOME_MrProject_Id    task_id,
		      xmlNodePtr            tree)
{
        XmlFileFilterPriv   *priv;
	gint                 id, predecessor_id;
	DelayedDependency   *dep;
	gchar               *str;

	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_XML_FILE_FILTER (filter));

        priv = filter->priv;

	if (strcmp (tree->name, "predecessor")){
		g_warning ("Got %s, expected 'predecessor'.", tree->name);
		return;
	}

	str = xml_get_value (tree, "id");
	if (str) {
		id = atoi (str);
		g_free (str);
	} else {
		id = -1;
	}

	str = xml_get_value (tree, "predecessor-id");
	if (str) {
		predecessor_id = atoi (str);
		g_free (str);
	} else {
		predecessor_id = -1;
	}

	if (id == -1 || predecessor_id == -1) {
		g_warning ("Invalid predecessor read.");
		return;
	}

	dep = g_new0 (DelayedDependency, 1);
	dep->task_id = task_id;
	dep->dependency_id = id;
	dep->predecessor_id = predecessor_id;

	str = xml_get_value (tree, "type");
	if (!strcmp (str, "FS")) {
		dep->type = GNOME_MrProject_DEPENDENCY_FS;
	}
	else if (!strcmp (str, "FF")) {
		dep->type = GNOME_MrProject_DEPENDENCY_FF;
	}
	else if (!strcmp (str, "SS")) {
		dep->type = GNOME_MrProject_DEPENDENCY_SS;
	}
	else if (!strcmp (str, "SF")) {
		dep->type = GNOME_MrProject_DEPENDENCY_SF;
	}
	else {
		g_warning ("Invalid dependency type.");
		dep->type = GNOME_MrProject_DEPENDENCY_FS;
	}

	priv->delayed_dependencies = g_slist_prepend (
		priv->delayed_dependencies,
		dep);
}

static void
xml_read_task (XmlFileFilter *filter, xmlNodePtr tree, gint parent_id)
{
        XmlFileFilterPriv   *priv;
	xmlNodePtr           tasks, predecessor;
	gchar               *str, *name;
	gint                 id;
	time_t               start, end;
	GM_Task             *task;
	CORBA_Environment    ev;
	guint                percent_complete = 0;
	gchar               *note;
	
	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_XML_FILE_FILTER (filter));

        priv = filter->priv;

	if (strcmp (tree->name, "task")){
		g_warning ("Got %s, expected 'task'.", tree->name);
		return;
	}

	name = xml_get_string_utf8 (tree, "name");

	str = xml_get_value (tree, "start");
	if (str) {
		start = time_from_isodate (str);
		g_free (str);
	} else {
		start = -1;
	}
	
	str = xml_get_value (tree, "end");
	if (str) {
		end = time_from_isodate (str);
		g_free (str);
	} else {
		end = -1;
	}
	
	str = xml_get_value (tree, "id");
	id = strtod (str, NULL);
	g_free (str);
	
	str = xml_get_value (tree, "percent-complete");
	if (str) {
		percent_complete = atoi (str);
		g_free (str);
	} 

	note = xml_get_string_utf8 (tree, "note");

	/* Create and insert a new row. */
	task = GNOME_MrProject_Task__alloc ();
	task->name = CORBA_string_dup (name);
	task->taskId = id;
	task->start = (GNOME_MrProject_Time) start;
	task->end = (GNOME_MrProject_Time) end;
	task->percentComplete = percent_complete;
	task->type = GNOME_MrProject_TASK_NORMAL;

	priv->t1 = MIN (priv->t1, start);
	priv->t2 = MAX (priv->t2, end);
	
	CORBA_exception_init (&ev);

	if (note) {
		GNOME_MrProject_TaskManager_setNote (
			FILE_FILTER(filter)->task_mgr, id, note, &ev);
		g_free (note);
	}
	
	GNOME_MrProject_TaskManager_insertTask (
		FILE_FILTER(filter)->task_mgr, task, parent_id, &ev);
	CORBA_exception_free (&ev);

	CORBA_free (task);
	g_free (name);
	
	for (tasks = tree->childs; tasks; tasks = tasks->next) {
		if (!strcmp (tasks->name, "task")) {
			xml_read_task (filter, tasks, id);
		}
		else if (!strcmp (tasks->name, "predecessors")) {
			for (predecessor = tasks->childs; predecessor; predecessor = predecessor->next) {
				xml_read_predecessor (filter, id, predecessor);
			}
		}
	}
}

static void
xml_read_resource (XmlFileFilter *filter, xmlNodePtr tree)
{
	GNOME_MrProject_Id                 id = 0;
	GNOME_MrProject_ResourceType       type = 0;
	gchar                             *str, *name, *email;
	gint                               gid = 0;
	glong                              units = 0;
	gfloat                             std_rate = 0.0, ovt_rate = 0.0;
	GNOME_MrProject_Resource          *resource;
	CORBA_Environment                  ev;

	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_XML_FILE_FILTER (filter));

	if (strcmp (tree->name, "resource")){
		g_warning ("Got %s, expected 'resource'.", tree->name);
		return;
	}

	str = xml_get_value (tree, "id");
	if (str) {
		id = atoi (str);
		g_free (str);
	} 

	name = xml_get_string_utf8 (tree, "name");

	str = xml_get_value (tree, "group");
	if (str) {
		gid = atoi (str);
		g_free (str);
	}

	str = xml_get_value (tree, "type");
	if (str) {
		type = atoi (str);
		g_free (str);
	}

	str = xml_get_value (tree, "units");
	if (str) {
		units = atol (str);
		g_free (str);
	}

	email = xml_get_string_utf8 (tree, "email");
	if (!email) {
		email = g_strdup ("");
	}

	str = xml_get_value (tree, "std-rate");
	if (str) {
		std_rate = atof (str);
		g_free (str);
	}

	str = xml_get_value (tree, "ovt-rate");
	if (str) {
		ovt_rate = atof (str);
		g_free (str);
	}

	/* Create and insert a new row. */
	resource             = GNOME_MrProject_Resource__alloc ();
	resource->resourceId = id;
	resource->name       = CORBA_string_dup (name);
	resource->groupId    = gid;
	resource->type       = type;
	resource->units      = units;
	resource->email      = CORBA_string_dup (email);
	resource->stdRate    = std_rate;
	resource->ovtRate    = ovt_rate;

	CORBA_exception_init (&ev);
	
	GNOME_MrProject_ResourceManager_insertResource (
                FILE_FILTER(filter)->resource_mgr, resource, &ev);

	CORBA_exception_free (&ev);
	
	CORBA_free (resource);
	g_free (name);
	g_free (email);
}

static void
xml_read_group (XmlFileFilter *filter, xmlNodePtr tree)
{
	gint                            id = 0;
	gchar                          *str;
	gchar                          *name, *a_name, *a_phone, *a_email;
	GNOME_MrProject_ResourceGroup  *group;
	CORBA_Environment               ev;
        
	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_XML_FILE_FILTER (filter));

	if (strcmp (tree->name, "group")){
		g_warning ("Got %s, expected 'group'.", tree->name);
		return;
	}

	str = xml_get_value (tree, "id");
	if (str) {
		id = atoi (str);
		g_free (str);
	} 

	name    = xml_get_string_utf8 (tree, "name");
	a_name  = xml_get_string_utf8 (tree, "admin-name");
	a_phone = xml_get_string_utf8 (tree, "admin-phone");
	a_email = xml_get_string_utf8 (tree, "admin-email");

        /* Create and insert a new row. */
	group             = GNOME_MrProject_ResourceGroup__alloc ();
	group->groupId    = id;
	group->name       = CORBA_string_dup (name);
	group->adminName  = CORBA_string_dup (a_name);
	group->adminPhone = CORBA_string_dup (a_phone);
	group->adminEmail = CORBA_string_dup (a_email);

	CORBA_exception_init (&ev);
	
	GNOME_MrProject_ResourceManager_insertGroup (
                FILE_FILTER(filter)->resource_mgr, group, &ev);

	CORBA_exception_free (&ev);
	
	CORBA_free (group);
	g_free (name);
	g_free (a_name);
	g_free (a_phone);
	g_free (a_email);
}

static void
xml_read_allocation (XmlFileFilter *filter, xmlNodePtr tree)
{
	gchar              *str;
	GNOME_MrProject_Id  task_id = -1, resource_id = -1;
	CORBA_Environment   ev;

	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_XML_FILE_FILTER (filter));

	if (strcmp (tree->name, "allocation")){
		g_warning ("Got %s, expected 'allocation'.", tree->name);
		return;
	}

	str = xml_get_value (tree, "task-id");
	if (str) {
		task_id = atoi (str);
		g_free (str);
	}

	str = xml_get_value (tree, "resource-id");
	if (str) {
		resource_id = atoi (str);
		g_free (str);
	}
	
	CORBA_exception_init (&ev);
	
	GNOME_MrProject_AllocationManager_allocate (
                FILE_FILTER(filter)->allocation_mgr, 
                task_id, resource_id, &ev);

	CORBA_exception_free (&ev);
}

static void
read_project_properties (XmlFileFilter *filter, xmlNodePtr node)
{
	Bonobo_PropertyBag  pb_co;
	CORBA_Environment   ev;
	gchar              *str;

	CORBA_exception_init (&ev);	

	pb_co = Bonobo_Unknown_queryInterface (FILE_FILTER(filter)->project,
					       "IDL:Bonobo/PropertyBag:1.0",
					       &ev);

	if (BONOBO_EX (&ev) || pb_co == CORBA_OBJECT_NIL) {
		g_warning ("Error '%s'", bonobo_exception_get_text (&ev));
		CORBA_exception_free (&ev);
		return;
 	}	

	str = xml_get_string_utf8 (node, "name");
	if (str != NULL) {
		bonobo_property_bag_client_set_value_string (
			pb_co, "Name", str, NULL);
		g_free (str);
	}

	str = xml_get_string_utf8 (node, "company");
	if (str != NULL) {
		bonobo_property_bag_client_set_value_string (
			pb_co, "Company", str, NULL);
		g_free (str);
	}

	str = xml_get_string_utf8 (node, "manager");
	if (str != NULL) {
		bonobo_property_bag_client_set_value_string (
			pb_co, "Manager", str, NULL);
		g_free (str);
	}
	
	bonobo_object_release_unref (pb_co, NULL);
	CORBA_exception_free (&ev);
}

/* This simply gets the start of the first task and sets
 * the Start property of the Project to that time.
 */
static void
read_start_end_time (XmlFileFilter *filter)
{
        XmlFileFilterPriv    *priv;
	Bonobo_PropertyBag    pb_co;
	CORBA_Environment     ev;

	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_XML_FILE_FILTER (filter));

        priv = filter->priv;

	if (priv->t1 != G_MAXLONG) {
		CORBA_exception_init (&ev);	
		pb_co = Bonobo_Unknown_queryInterface (
                        FILE_FILTER(filter)->project,
                        "IDL:Bonobo/PropertyBag:1.0",
                        &ev);

		if (BONOBO_EX (&ev) || pb_co == CORBA_OBJECT_NIL) {
			g_warning ("Error '%s'", bonobo_exception_get_text (&ev));
			CORBA_exception_free (&ev);
			return;
		}	
		
		bonobo_property_bag_client_set_value_glong (
			pb_co, "Start", priv->t1, NULL);

		bonobo_property_bag_client_set_value_glong (
			pb_co, "Finish", priv->t2, NULL);
	
		bonobo_object_release_unref (pb_co, NULL);
		CORBA_exception_free (&ev);
	}
}

static void
write_project_properties (XmlFileFilter *filter, xmlNodePtr node)
{
	gchar              *tmp, *str;
	Bonobo_PropertyBag  pb_co;
	CORBA_Environment   ev;

        g_return_if_fail (filter != NULL);
        g_return_if_fail (IS_XML_FILE_FILTER (filter));

	CORBA_exception_init (&ev);	
	
	pb_co = Bonobo_Unknown_queryInterface (FILE_FILTER(filter)->project,
					       "IDL:Bonobo/PropertyBag:1.0",
					       &ev);

	if (BONOBO_EX (&ev) || pb_co == CORBA_OBJECT_NIL) {
		g_warning ("Error '%s'", bonobo_exception_get_text (&ev));
		CORBA_exception_free (&ev);
		return;
 	}	
	
	tmp = bonobo_property_bag_client_get_value_string (pb_co, "Name", NULL);

	str = e_utf8_to_locale_string (tmp);
	xmlSetProp (node, "name", str);
	g_free (str);
	g_free (tmp);

	tmp = bonobo_property_bag_client_get_value_string (pb_co, "Company", NULL);
	str = e_utf8_to_locale_string (tmp);
	xmlSetProp (node, "company", str);
	g_free (str);
	g_free (tmp);

	tmp = bonobo_property_bag_client_get_value_string (pb_co, "Manager", NULL);
	str = e_utf8_to_locale_string (tmp);
	xmlSetProp (node, "manager", str);
	g_free (str);
	g_free (tmp);

	bonobo_object_release_unref (pb_co, NULL);
	CORBA_exception_free (&ev);
}

static void
process_delayed_predecessors (XmlFileFilter *filter)
{
        XmlFileFilterPriv   *priv;
	GSList              *list;
	CORBA_Environment    ev;

	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_XML_FILE_FILTER (filter));

        priv = filter->priv;

	CORBA_exception_init (&ev);
	for (list = priv->delayed_dependencies; list; list = list->next) {
		DelayedDependency *dep;

		dep = list->data;

		GNOME_MrProject_TaskManager_linkTasks (
                        FILE_FILTER(filter)->task_mgr, 
                        dep->task_id,
                        dep->predecessor_id,
                        dep->type,
                        &ev);

		if (BONOBO_EX (&ev)) {
			g_warning ("Could not add dependency.");
		}
		CORBA_exception_free (&ev);
		g_free (dep);
	}

	g_slist_free (priv->delayed_dependencies);
	priv->delayed_dependencies = NULL;
}

static gboolean
xml_read_project (XmlFileFilter *filter)
{
        XmlFileFilterPriv   *priv;
	xmlNodePtr           top, child, tasks, resources, groups, allocations;
	gchar               *old_locale;
	CORBA_Environment    ev;
        gchar               *str;
        GM_Id                default_group = -1;
	
	g_return_val_if_fail (filter != NULL, FALSE);
	g_return_val_if_fail (IS_XML_FILE_FILTER (filter), FALSE);
        
        priv = filter->priv;

	top = priv->doc->root;

	if (!top) {
		g_warning ("Could not find project");
		return FALSE;
	}
	
	CORBA_exception_init (&ev);
	
	read_project_properties (filter, top);
	
	/* Make sure we don't get locale specific floats. */
	old_locale = setlocale (LC_NUMERIC, "C");

	/* Load tasks. */
	child = xml_search_child (top, "tasks");

	if (child != NULL) {
		for (tasks = child->childs; tasks; tasks = tasks->next) {
			xml_read_task (filter, tasks, 0 /* parent_id */);
		}
	}
	
	read_start_end_time (filter);

	/* Now that all tasks are loaded, we can add the dependencies. */
	process_delayed_predecessors (filter);

	GNOME_MrProject_TaskManager__set_state (
		FILE_FILTER(filter)->task_mgr, 
                GNOME_MrProject_PROJECT_STATE_SAVED, &ev);
        
	if (BONOBO_EX (&ev)) {
		CORBA_exception_free (&ev);
                return FALSE;
	}

	/* Load resources. */
 	child = xml_search_child (top, "resources"); 
 
	if (child != NULL) {
		for (resources = child->childs; resources; resources = resources->next) {
			xml_read_resource (filter, resources); 
		}
	}
	
	/* Load resource groups. */
 	child = xml_search_child (top, "resource-groups"); 
        
        str = xml_get_value (child, "default_group");
        
        if (str) {
                default_group = atoi (str);
                g_free (str);
        }
        
        
        
 	if (child != NULL) {
		for (groups = child->childs; groups; groups = groups->next) {
			xml_read_group (filter, groups); 
		}
	}

        if (default_group > 0) {
                GNOME_MrProject_ResourceManager_setDefaultGroup (
                        FILE_FILTER(filter)->resource_mgr,
                        default_group,
                        &ev);
        }

	GNOME_MrProject_ResourceManager__set_state (
		FILE_FILTER(filter)->resource_mgr, 
                GNOME_MrProject_PROJECT_STATE_SAVED, &ev);

	if (BONOBO_EX (&ev)) {
		CORBA_exception_free (&ev);
                return FALSE;
	}

	/* Load allocations. */
	child = xml_search_child (top, "allocations");

	if (child != NULL) {
		for (allocations = child->childs; allocations; allocations = allocations->next) {
			xml_read_allocation (filter, allocations);
		}
	}

	GNOME_MrProject_AllocationManager__set_state (
		FILE_FILTER(filter)->allocation_mgr, 
                GNOME_MrProject_PROJECT_STATE_SAVED, &ev);

	if (BONOBO_EX (&ev)) {
		CORBA_exception_free (&ev);
                return FALSE;
	}

	/* Restore locale. */
	setlocale (LC_NUMERIC, old_locale);

	return TRUE;
}

/*
 * Get a value for a node either carried as an attibute or as
 * the content of a child.
 *
 * Returns a g_malloc'ed string.  Caller must free.
 */
static gchar *
xml_get_value (xmlNodePtr node, const char *name)
{
	char *ret, *val;
	xmlNodePtr child;

	g_return_val_if_fail (node != NULL, NULL);
	
	val = (char *) xmlGetProp (node, name);

	if (val != NULL) {
		ret = g_strdup (val);
		xmlFree (val);
		return ret;
	}

	child = node->childs;

	while (child != NULL) {
		if (!strcmp (child->name, name)) {
		        /*
			 * !!! Inefficient, but ...
			 */
			val = xmlNodeGetContent(child);
			if (val != NULL) {
				ret = g_strdup (val);
				xmlFree (val);
				return ret;
			}
		}
		child = child->next;
	}

	return NULL;
}

static gchar *
xml_get_string_utf8 (xmlNodePtr node, const char *name)
{
	gchar *value, *utf8;

	value = xml_get_value (node, name);
	utf8 = e_utf8_from_locale_string (value);
	g_free (value);
	return utf8;
}

/*
 * Search a child by name, if needed go down the tree to find it.
 */
static xmlNodePtr
xml_search_child (xmlNodePtr node, const gchar *name)
{
	xmlNodePtr ret;
	xmlNodePtr child;

	child = node->childs;
	while (child != NULL) {
		if (!strcmp (child->name, name))
			return child;
		child = child->next;
	}
	child = node->childs;
	while (child != NULL) {
		ret = xml_search_child (child, name);
		if (ret != NULL)
			return ret;
		child = child->next;
	}
	return NULL;
}



static gboolean
xff_probe (FileFilter        *filter,
	   const gchar       *uri,
	   CORBA_Environment *ev)
{
	g_return_val_if_fail (filter != NULL, FALSE);
	g_return_val_if_fail (IS_XML_FILE_FILTER (filter), FALSE);

	return TRUE;
}

static void
xff_load (FileFilter          *filter,
	  const gchar         *uri,
	  const GM_Project     project,
	  CORBA_Environment   *ev)
{
	xmlDoc              *doc;
	XmlFileFilter       *xml_filter;
        XmlFileFilterPriv   *priv;
        xmlParserCtxtPtr     push_priv;
	GnomeVFSResult	     result;
	GnomeVFSHandle      *handle;
	GnomeVFSFileSize     bytes_read;
	gchar                buffer[1024];

	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_XML_FILE_FILTER (filter));
        
        xml_filter = XML_FILE_FILTER (filter);
        priv       = xml_filter->priv;

	result = gnome_vfs_open	(&handle,
				 uri,
				 GNOME_VFS_OPEN_READ |
				 GNOME_VFS_OPEN_RANDOM);

	if (result != GNOME_VFS_OK) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_GNOME_MrProject_FileFilter_Failure,
				     NULL);
		return;
	}

	/* Load the file into an XML tree. */
	push_priv = xmlCreatePushParserCtxt (NULL, NULL, NULL, 0, NULL);
	
	while (1) {
		result = gnome_vfs_read	(handle,
					 buffer,
					 1024,
					 &bytes_read);

		if (result == GNOME_VFS_ERROR_EOF) {
			break;
		}			
		else if (result != GNOME_VFS_OK) {
			break;
		}
			
		if (xmlParseChunk (push_priv, buffer, bytes_read, (bytes_read == 0))) {
			g_warning ("Leak bits of tree everywhere");
			break;
		}
	}
	
	gnome_vfs_close (handle);
	
	if (push_priv->wellFormed) {
		doc = push_priv->myDoc;
	} else {
		doc = NULL;
		xmlFreeDoc (push_priv->myDoc);
		push_priv->myDoc = NULL;
	}
	
	xmlFreeParserCtxt (push_priv);

	/*doc = xmlParseFile (location);*/
	if (!doc) {
		g_warning ("Failed to parse file.");
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_GNOME_MrProject_FileFilter_Failure,
				     NULL);
		return;
	}
	if (!doc->root) {
		xmlFreeDoc (doc);
		g_warning ("Invalid xml file. Tree is empty?");
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_GNOME_MrProject_FileFilter_Failure,
				     NULL);
		return;
	}

	/* FIXME: Check the namespace here? */
	priv->doc = doc;

	if (!xml_read_project (xml_filter)) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_GNOME_MrProject_FileFilter_Failure,
				     NULL);
	}
	
/* 	gtk_object_destroy (GTK_OBJECT (xml_filter)); */
	xmlFreeDoc (doc);
}

static void
xff_load_summary (FileFilter        *filter,
		  const gchar       *uri,
		  GNOME_MrProject_FileSummary **summary,
		  CORBA_Environment *ev)
{
	xmlDoc              *doc;
	xmlParserCtxtPtr     push_priv;
	GnomeVFSResult	     result;
	GnomeVFSHandle      *handle;
	GnomeVFSFileSize     bytes_read;
	gchar                buffer[1024];
	gchar               *str;
	time_t               t;

	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_XML_FILE_FILTER (filter));

	result = gnome_vfs_open	(&handle,
				 uri,
				 GNOME_VFS_OPEN_READ |
				 GNOME_VFS_OPEN_RANDOM);

	if (result != GNOME_VFS_OK) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_GNOME_MrProject_FileFilter_Failure,
				     NULL);
		return;
	}

	/* Load the file into an XML tree. */
	push_priv = xmlCreatePushParserCtxt (NULL, NULL, NULL, 0, NULL);
	
	while (1) {
		result = gnome_vfs_read	(handle,
					 buffer,
					 1024,
					 &bytes_read);

		if (result == GNOME_VFS_ERROR_EOF) {
			break;
		}			
		else if (result != GNOME_VFS_OK) {
			break;
		}
			
		if (xmlParseChunk (push_priv, buffer, bytes_read, (bytes_read == 0))) {
			g_warning ("Leak bits of tree everywhere");
			break;
		}
	}

	gnome_vfs_close (handle);
	
	if (push_priv->wellFormed) {
		doc = push_priv->myDoc;
	} else {
		doc = NULL;
		xmlFreeDoc (push_priv->myDoc);
		push_priv->myDoc = NULL;
	}
	
	xmlFreeParserCtxt (push_priv);

	if (!doc) {
		g_warning ("Failed to parse file.");
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_GNOME_MrProject_FileFilter_Failure,
				     NULL);
		return;
	}
	if (!doc->root) {
		xmlFreeDoc (doc);
		g_warning ("Invalid xml file. Tree is empty?");
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_GNOME_MrProject_FileFilter_Failure,
				     NULL);
		return;
	}

	/* FIXME: Check the namespace here? */
	
	/* Ok, we have the file, now fetch the interesting pieces for the
	 * summary.
	 */

	*summary = GNOME_MrProject_FileSummary__alloc ();

	str = xml_get_string_utf8 (doc->root, "name");
	(**summary).name = CORBA_string_dup (str);
	g_free (str);

	str = xml_get_string_utf8 (doc->root, "company");
	(**summary).company = CORBA_string_dup (str);
	g_free (str);

	str = xml_get_string_utf8 (doc->root, "manager");
	(**summary).manager = CORBA_string_dup (str);
	g_free (str);

	str = xml_get_value (doc->root, "start");
	t = time_from_isodate (str);
	g_free (str);
	(**summary).start = t;

	str = xml_get_value (doc->root, "end");
	t = time_from_isodate (str);
	g_free (str);
	(**summary).end = t;

/*	{
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_GNOME_MrProject_FileFilter_Failure,
				     NULL);
	}
*/
	xmlFreeDoc (doc);
}

static void
xff_save (FileFilter          *filter,
	  const gchar         *uri,
	  const GM_Project     project,
	  CORBA_Environment   *ev)
{
	xmlDocPtr    xml;
	int          ret;
	gchar       *location;

	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_XML_FILE_FILTER (filter));

	location = gnome_vfs_get_local_path_from_uri (uri);

	if (!location) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_GNOME_MrProject_FileFilter_Failure,
				     NULL);
		return;
	}

	xml = xmlNewDoc ("1.0");

	if (!xml) {
		g_warning ("Error creating xml tree.");
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_GNOME_MrProject_FileFilter_Failure,
				     NULL);
		g_free (location);
		return;
	}

	xml->root = xml_write_project (XML_FILE_FILTER(filter));
	
	ret = xmlSaveFile (location, xml);
	if (ret == -1) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_GNOME_MrProject_FileFilter_Failure,
				     NULL);
	}
	
	xmlFreeDoc (xml);
	g_free (location);
}

static void
xml_file_filter_class_init (XmlFileFilterClass *klass)
{
        GtkObjectClass  *object_class      = (GtkObjectClass *) klass;
	FileFilterClass *file_filter_class = (FileFilterClass *) klass;

        object_class->destroy           = xff_destroy;

	file_filter_class->probe        = xff_probe;
	file_filter_class->load         = xff_load;
	file_filter_class->load_summary = xff_load_summary;
	file_filter_class->save         = xff_save;
}

static void
xml_file_filter_init (XmlFileFilter *filter)
{
	XmlFileFilterPriv   *priv;

	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_XML_FILE_FILTER (filter));
        
	priv = g_new0 (XmlFileFilterPriv, 1);

	priv->doc     = NULL;
	priv->t1      = G_MAXLONG;
	priv->t2      = -G_MAXLONG;
	priv->id_hash = g_hash_table_new (g_int_hash, g_int_equal);

	priv->delayed_dependencies = NULL;

        filter->priv  = priv;
}

static BonoboObject *
xff_factory (BonoboGenericFactory *this,
                         const char           *object_id,
                         void                 *data)
{
	g_return_val_if_fail (object_id != NULL, NULL);
	
	if (!strcmp (object_id, "OAFIID:GNOME_MrProject_XmlFileFilter")) {
		FileFilter *filter;
                
		filter = gtk_type_new (TYPE_XML_FILE_FILTER);
		return BONOBO_OBJECT (filter);
	} else {
		g_warning ("Failing to manufacture a '%s'", object_id);
	}
	
	return NULL;
}

BONOBO_OAF_SHLIB_FACTORY_MULTI ("OAFIID:GNOME_MrProject_XmlFileFilterFactory",
				"Mr Project XML file filter factory",
				xff_factory,
				NULL);

