/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 Ximian, Inc
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Michael Meeks <michael@ximian.com>
 *
 * Based on:
 *    http://support.microsoft.com/support/kb/articles/Q270/1/39.ASP
 */


#ifdef HAVE_CONFIG_H
  #include <config.h>
#endif

#include <locale.h>
#include <math.h>
#include <stdlib.h>
#include <gal/widgets/e-unicode.h>
#include <bonobo.h>
#include <bonobo/bonobo-shlib-factory.h>
#include <liboaf/liboaf.h>
#include <libgnomevfs/gnome-vfs.h>
#include "libmrproject/GNOME_MrProject.h"
#include "util/time-utils.h"
#include "util/type-utils.h"
#include "util/corba-utils.h"
#include "project-engine/project.h"
#include "file-filter.h"
#include "mpx-file-filter.h"

static void mpx_file_filter_init	(MpxFileFilter	    *mpx_file_filter);
static void mpx_file_filter_class_init	(MpxFileFilterClass *klass);

GNOME_CLASS_BOILERPLATE (MpxFileFilter, mpx_file_filter,
			 FileFilter, file_filter);


#undef MPX_DEBUG

typedef struct {
	GM_Project            project;
	GM_TaskManager 	      task_manager;
	GM_ResourceManager    resource_manager;
	GM_AllocationManager  allocation_manager;

	FILE                 *fh;
	int                   cur_line;
	GArray               *task_offsets;
	GArray               *resource_offsets;
	GSList               *delayed_mappings;
	GArray               *outlines;
	GHashTable           *map_task_ids;
	GHashTable           *map_resource_ids;
} MpxReadContext;

static GM_Id
mpx_id_from_mpx (GHashTable *mapping, guint mpx_id)
{
	return GPOINTER_TO_UINT (
		g_hash_table_lookup (mapping, GUINT_TO_POINTER (mpx_id)));
}

static void
mpx_add_id (GHashTable *mapping, guint mpx_id, GM_Id id)
{
	g_hash_table_insert (mapping,
			     GUINT_TO_POINTER (mpx_id),
			     GUINT_TO_POINTER (id));
}

static MpxReadContext *
mpx_new (const char *location)
{
	MpxReadContext *mpx;
	FILE           *fh;

	fh = fopen (location, "r");
	g_return_val_if_fail (fh != NULL, NULL);

	mpx = g_new0 (MpxReadContext, 1);
	mpx->fh = fh;

	mpx->cur_line         = 0;
	mpx->task_offsets     = g_array_new (TRUE, TRUE, sizeof (guint));
	mpx->resource_offsets = g_array_new (TRUE, TRUE, sizeof (guint));
	mpx->delayed_mappings = NULL;
	mpx->outlines         = g_array_new (TRUE, TRUE, sizeof (GM_Id));
	mpx->map_task_ids     = g_hash_table_new (NULL, NULL);
	mpx->map_resource_ids = g_hash_table_new (NULL, NULL);
	
	return mpx;
}

static void
mpx_free (MpxReadContext *mpx)
{
	if (mpx) {
		if (mpx->fh) {
			fclose (mpx->fh);
		}
		
		mpx->fh = NULL;
		mpx->project = NULL;

		if (mpx->task_offsets) {
			g_array_free (mpx->task_offsets, TRUE);
		}
		
		mpx->task_offsets = NULL;

		if (mpx->resource_offsets) {
			g_array_free (mpx->resource_offsets, TRUE);
		}
		
		mpx->resource_offsets = NULL;

		if (mpx->outlines) {
			g_array_free (mpx->outlines, TRUE);
		}
		
		mpx->outlines = NULL;

		if (mpx->map_task_ids) {
			g_hash_table_destroy (mpx->map_task_ids);
		}
		
		mpx->map_task_ids = NULL;

		if (mpx->map_resource_ids) {
			g_hash_table_destroy (mpx->map_resource_ids);
		}
		
		mpx->map_resource_ids = NULL;

		g_free (mpx);
	}
}

static int
mpx_peek (MpxReadContext *mpx)
{
	int c;

	g_return_val_if_fail (mpx != NULL, EOF);
	
	c = fgetc (mpx->fh);
	ungetc (c, mpx->fh);

	return c;
}

static char *
mpx_next_token (MpxReadContext *mpx)
{
	GString  *str = g_string_sized_new (8);
	char     *s;
	gboolean  quoted = FALSE;
	int       c;

	do {
		c = fgetc (mpx->fh);

		if (c == EOF) {
			break;
		}

		if (c == '"') {
			if (quoted) {
				if (mpx_peek (mpx) == '"') {
					c = fgetc (mpx->fh);
					g_string_append_c (str, c);
				} else {
					quoted = FALSE;
				}
			} else {
				quoted = TRUE;
			}
		} else {
			if ((!quoted && c == ',') ||
			    c == '\n' || c == '\r' || c == EOF) {
				break;
			} else {
				g_string_append_c (str, c);
			}
		}
	} while (1);

	if (str->len == 0 && (c == EOF || c == '\n' || c == '\r')) {

		if (c == '\n' && mpx_peek (mpx) == '\r') {
			fgetc (mpx->fh);
		}

		mpx->cur_line++;
			
		g_string_free (str, TRUE);

		return NULL;
	}

/*	fprintf (stderr, "mpx_next_token '%s'", str->str); */

	s = str->str;
	g_string_free (str, FALSE);

	return s;
}

static gboolean
mpx_eof (MpxReadContext *mpx)
{
	g_return_val_if_fail (mpx != NULL, TRUE);

	return feof (mpx->fh);
}

static gboolean
mpx_next_line (MpxReadContext *mpx)
{
	int c;

	while ((c = fgetc (mpx->fh)) != EOF && c != '\n') {
		;
	}

	if (c == '\n' && mpx_peek (mpx) == '\r') {
		c = fgetc (mpx->fh);
	}

	mpx->cur_line++;

	return c != EOF;
}

typedef void (*MpxFieldIdHandlerFn) (gpointer    ctx, 
				     int         task_id, 
				     const char *tok);

typedef struct {
	const char          *name;
	int                  id;
	MpxFieldIdHandlerFn  fn;
} MpxFieldDef;

typedef struct {
	GM_Id id;
	guint parent_id;
} MpxTaskDelayed;

typedef struct {
	MpxReadContext *mpx;
	GM_Task        *task;
	guint           id;
	guint           parent_id;
	guint           outline_level;
	guint           percent_complete;
} MpxTaskCtx;

enum {
	MPX_TASK_DEF_NAME             = 1,
	MPX_TASK_DEF_OUTLINE_LEVEL    = 3,
	MPX_TASK_DEF_START            = 50,
	MPX_TASK_DEF_FINISH           = 51,
	MPX_TASK_DEF_PREDECESSORS     = 70,
	MPX_TASK_DEF_ID               = 90,
	MPX_TASK_DEF_PERCENT_COMPLETE = 44
};

static void
mpx_task_handle_int (gpointer ctx, int task_id, const char *tok)
{
	MpxTaskCtx *tctx = ctx;

	switch (task_id) {
	case MPX_TASK_DEF_ID:
		tctx->id = atoi (tok);
		break;
	case MPX_TASK_DEF_PREDECESSORS:
		tctx->parent_id = atoi (tok);
#ifdef MPX_DEBUG
		if (tctx->parent_id) {
			g_warning ("Task '%s' child of %d",
				   tctx->task->name, tctx->parent_id);
		}
#endif
		break;
	case MPX_TASK_DEF_OUTLINE_LEVEL:
		tctx->outline_level = atoi (tok);
		break;
	case MPX_TASK_DEF_PERCENT_COMPLETE:
		tctx->task->percentComplete = atoi (tok);
		break;
	default:
		g_assert_not_reached ();
		break;
	}
}

static void
mpx_task_handle_name (gpointer ctx, int task_id, const char *tok)
{
	MpxTaskCtx *tctx = ctx;

	tctx->task->name = CORBA_string_dup (tok);
}

static void
mpx_task_handle_time (gpointer ctx, int task_id, const char *tok)
{
	MpxTaskCtx *tctx = ctx;
	GM_Time    *time;

	switch (task_id) {
	case MPX_TASK_DEF_START:
		time = &tctx->task->start;
		break;
	case MPX_TASK_DEF_FINISH:
		time = &tctx->task->end;
		break;
	default:
		time = NULL;
		g_assert_not_reached ();
		break;
	}

	*time = (GM_Time) time_from_msdate (tok);
}

static const MpxFieldDef
mpx_task_defs [] = {
	{ "% Complete", MPX_TASK_DEF_PERCENT_COMPLETE, mpx_task_handle_int },
	{ "% Work Complete", 25, NULL },
	{ "Actual Cost", 32, NULL },
	{ "Actual Duration", 42, NULL },
	{ "Actual Finish", 59, NULL },
	{ "Actual Start", 58, NULL },
	{ "Actual Work", 22, NULL },
	{ "Baseline Cost", 31, NULL },
	{ "Baseline Duration", 41, NULL },
	{ "Baseline Finish", 57, NULL },
	{ "Baseline Start", 56, NULL },
	{ "Baseline Work", 21, NULL },
	{ "BCWP", 86, NULL },
	{ "BCWS", 85, NULL },
	{ "Confirmed", 135, NULL },
	{ "Constraint Date", 68, NULL },
	{ "Constraint Type", 91, NULL },
	{ "Contact", 15, NULL },
	{ "Cost", 30, NULL },
	{ "Cost1", 36, NULL },
	{ "Cost2", 37, NULL },
	{ "Cost3", 38, NULL },
	{ "Cost Variance", 34, NULL },
	{ "Created", 125, NULL },
	{ "Critical", 82, NULL },
	{ "CV", 88, NULL },
	{ "Delay", 92, NULL },
	{ "Duration", 40, NULL },
	{ "Duration1", 46, NULL },
	{ "Duration2", 47, NULL },
	{ "Duration3", 48, NULL },
	{ "Duration Variance", 45, NULL },
	{ "Early Finish", 53, NULL },
	{ "Early Start", 52, NULL },
	{ "Finish", MPX_TASK_DEF_FINISH, mpx_task_handle_time },
	{ "Finish1", 61, NULL },
	{ "Finish2", 63, NULL },
	{ "Finish3", 65, NULL },
	{ "Finish4", 127, NULL },
	{ "Finish5", 129, NULL },
	{ "Finish Variance", 67, NULL },
	{ "Fixed", 80, NULL },
	{ "Fixed Cost", 35, NULL },
	{ "Flag1", 110, NULL },
	{ "Flag2", 111, NULL },
	{ "Flag3", 112, NULL },
	{ "Flag4", 113, NULL },
	{ "Flag5", 114, NULL },
	{ "Flag6", 115, NULL },
	{ "Flag7", 116, NULL },
	{ "Flag8", 117, NULL },
	{ "Flag9", 118, NULL },
	{ "Flag10", 119, NULL },
	{ "Free Slack", 93, NULL },
	{ "Hide Bar", 123, NULL },
	{ "ID", MPX_TASK_DEF_ID, mpx_task_handle_int },
	{ "Late Finish", 55, NULL },
	{ "Late Start", 54, NULL },
	{ "Linked Fields", 122, NULL },
	{ "Marked", 83, NULL },
	{ "Milestone", 81, NULL },
	{ "Name", MPX_TASK_DEF_NAME, mpx_task_handle_name },
	{ "Notes", 14, NULL },
	{ "Number1", 140, NULL },
	{ "Number2", 141, NULL },
	{ "Number3", 142, NULL },
	{ "Number4", 143, NULL },
	{ "Number5", 144, NULL },
	{ "Objects", 121, NULL },
	{ "Outline Level", MPX_TASK_DEF_OUTLINE_LEVEL, mpx_task_handle_int },
	{ "Outline Number", 99, NULL },
	{ "Predecessors", MPX_TASK_DEF_PREDECESSORS, mpx_task_handle_int },
	{ "Priority", 95, NULL },
	{ "Project", 97, NULL },
	{ "Remaining Cost", 33, NULL },
	{ "Remaining Duration", 43, NULL },
	{ "Remaining Work", 23, NULL },
	{ "Resource Group", 16, NULL },
	{ "Resource Initials", 73, NULL },
	{ "Resource Names", 72, NULL },
	{ "Resume", 151, NULL },
	{ "Resume No Earlier Than", 152, NULL },
	{ "Rollup", 84, NULL },
	{ "Start", MPX_TASK_DEF_START, mpx_task_handle_time },
	{ "Start1", 60, NULL },
	{ "Start2", 62, NULL },
	{ "Start3", 64, NULL },
	{ "Start4", 126, NULL },
	{ "Start5", 128, NULL },
	{ "Start Variance", 66, NULL },
	{ "Stop", 150, NULL },
	{ "Subproject File", 96, NULL },
	{ "Successors", 71, NULL },
	{ "Summary", 120, NULL },
	{ "SV", 87, NULL },
	{ "Text1", 4, NULL },
	{ "Text2", 5, NULL },
	{ "Text3", 6, NULL },
	{ "Text4", 7, NULL },
	{ "Text5", 8, NULL },
	{ "Text6", 9, NULL },
	{ "Text7", 10, NULL },
	{ "Text8", 11, NULL },
	{ "Text9", 12, NULL },
	{ "Text10", 13, NULL },
	{ "Total Slack", 94, NULL },
	{ "Unique ID", 98, NULL },
	{ "Unique ID Predecessors", 74, NULL },
	{ "Unique ID Successors", 75, NULL },
	{ "Update Needed", 136, NULL },
	{ "WBS", 2, NULL },
	{ "Work", 20, NULL },
	{ "Work Variance", 24, NULL },
	{ NULL, 0, NULL }
};


static const MpxFieldDef *
mpx_get_field_def (const MpxFieldDef *fields, int task_id)
{
	const MpxFieldDef *def;

	for (def = fields; def->id; def++) {
		if (def->id == task_id) {
			break;
		}
	}

	if (def->id == 0) {
		return NULL;
	}

	return def;
}

static void
mpx_insert_task (MpxTaskCtx *tctx)
{
#if 0
	MpxTaskDelayed     *delay;
#endif
	CORBA_Environment ev;
	GM_Id             real_id;
	GM_Id             parent_id;

	if (tctx->outline_level > 0 &&
	    (tctx->outline_level - 1 < tctx->mpx->outlines->len)) {
		parent_id = g_array_index (tctx->mpx->outlines,
					   GM_Id,
					   tctx->outline_level - 1);
	} else {
		parent_id = 0;
	}

	g_array_set_size (tctx->mpx->outlines, tctx->outline_level + 1);

	CORBA_exception_init (&ev);

	tctx->task->taskId = -1;

	if (!tctx->task->name) {
		tctx->task->name = CORBA_string_dup ("[undefined]");
	}

	real_id = GNOME_MrProject_TaskManager_insertTask (
		tctx->mpx->task_manager, tctx->task, parent_id, &ev);

	mpx_add_id (tctx->mpx->map_task_ids, tctx->id, real_id);

	g_array_index (tctx->mpx->outlines,
		       GM_Id,
		       tctx->outline_level) = real_id;

	if (BONOBO_EX (&ev)) {
		g_warning ("Bonobo exception '%s'",
			   bonobo_exception_get_text (&ev));
	}

	CORBA_exception_free (&ev);

	/* Build delayed mappings list */

	/* Disable links for now. */
#if 0
	delay = g_new0 (MpxTaskDelayed, 1);
	delay->id = real_id;
	delay->parent_id = tctx->parent_id;
	tctx->mpx->delayed_mappings = 
		g_slist_prepend (tctx->mpx->delayed_mappings, delay);
#endif
}

static void
mpx_process_delayed_mappings (MpxReadContext *mpx)
{
	GSList            *l;
	CORBA_Environment  ev;

	CORBA_exception_init (&ev);

	for (l = mpx->delayed_mappings; l; l = l->next) {
		MpxTaskDelayed *delay = l->data;
		GM_Id           real_parent;

		real_parent = mpx_id_from_mpx (mpx->map_task_ids, 
					       delay->parent_id);

		GNOME_MrProject_TaskManager_linkTasks (
			mpx->task_manager,
			real_parent, delay->id,
			GNOME_MrProject_DEPENDENCY_SS, /* FIXME */
			&ev);

		CORBA_exception_free (&ev);

		g_free (delay);
	}

	g_slist_free (mpx->delayed_mappings);
	mpx->delayed_mappings = NULL;
}

static void
mpx_read_fields (MpxReadContext    *mpx,
		 const MpxFieldDef *fields,
		 GArray            *offsets,
		 gpointer           closure)
{
	char  *tok;
	guint  field = 0;

	while ((tok = mpx_next_token (mpx))) {
		const MpxFieldDef *def;
		guint              idx;

		if (field >= offsets->len) {
			g_warning ("Too many fields at line %d", 
				   mpx->cur_line);
			break;
		}

		idx = g_array_index (offsets, guint, field);

		def = mpx_get_field_def (fields, idx);

		if (def) {
			if (def->fn) {
				def->fn (closure, idx, tok);
			}
		} /* else - already warned: drop silently*/

		g_free (tok);
		field++;
	}
}

static void
mpx_read_task (MpxReadContext *mpx)
{
	MpxTaskCtx  rtctx;
	MpxTaskCtx *tctx = &rtctx;

	memset (tctx, 0, sizeof (rtctx));

	tctx->mpx = mpx;
	tctx->task = GNOME_MrProject_Task__alloc ();

	mpx_read_fields (mpx, mpx_task_defs, mpx->task_offsets, tctx);

#ifdef MPX_DEBUG
	{
		char *startt = 
			isodate_from_time_t ((time_t) tctx->task->start);
		char *endt = isodate_from_time_t ((time_t) tctx->task->end);

		fprintf (stderr, "Task %d (%d) '%s' %s->%s\n",
			 tctx->id, tctx->outline_level,
			 tctx->task->name, startt, endt);

		g_free (startt);
		g_free (endt);
	}
#endif

	mpx_insert_task (tctx);

	/* Cleanup tctx */
	CORBA_free (tctx->task);
}

typedef struct {
	MpxReadContext *mpx;
	GM_Resource    *resource;
	guint           id;
	guint           group_id;
} MpxResourceCtx;

enum {
	MPX_RESOURCE_DEF_NAME          = 1,
	MPX_RESOURCE_DEF_CODE          = 4,
	MPX_RESOURCE_DEF_GROUP         = 3,
	MPX_RESOURCE_DEF_ID            = 40,
	MPX_RESOURCE_DEF_EMAIL         = 11,
	MPX_RESOURCE_DEF_STD_RATE      = 42,
	MPX_RESOURCE_DEF_OVT_RATE      = 43
};

static void
mpx_resource_handle_int (gpointer ctx, int res_id, const char *tok)
{
	MpxResourceCtx *rctx = ctx;

	switch (res_id) {
	case MPX_RESOURCE_DEF_ID:
		rctx->id = atoi (tok);
		break;
	case MPX_RESOURCE_DEF_CODE:
		/* FIXME: what does this do ? assume type */
		if (tok && tok [0] == '\0') {
			/* FIXME */
			rctx->resource->type = 
				GNOME_MrProject_RESOURCE_TYPE_PEOPLE;
		} else {
			g_warning ("Unknown resource code '%s'", tok);
		}
		break;
	case MPX_RESOURCE_DEF_GROUP:
		rctx->group_id = atoi (tok);
		break;
	default:
		g_assert_not_reached ();
		break;
	}
}

static void
mpx_resource_handle_rate (gpointer ctx, int res_id, const char *tok)
{
	MpxResourceCtx *rctx = ctx;
	const char     *p;

/*	fprintf (stderr, "Handle float %d = '%s'\n", res_id, tok); */

	if (tok [0] == '$') { /* FIXME - hardcoded - other currencies ? */
		p = tok + 1;
	} else {
		p = "0";
	}

	switch (res_id) {
	case MPX_RESOURCE_DEF_STD_RATE:
		rctx->resource->stdRate = atof (p);
		break;
	case MPX_RESOURCE_DEF_OVT_RATE:
		rctx->resource->ovtRate = atof (p);
		break;
	default:
		g_assert_not_reached ();
		break;
	}
}

static void
mpx_resource_handle_string (gpointer ctx, int res_id, const char *tok)
{
	MpxResourceCtx *rctx = ctx;

	switch (res_id) {
	case MPX_RESOURCE_DEF_NAME:
		rctx->resource->name = CORBA_string_dup (tok);
		break;
	case MPX_RESOURCE_DEF_EMAIL:
		rctx->resource->email = CORBA_string_dup (tok);
		break;
	default:
		g_assert_not_reached ();
		break;
	}
}

static const MpxFieldDef
mpx_resource_defs [] = {
	{ "Work Complete", 26, NULL },
	{ "Accrue At", 45, NULL },
	{ "Actual Cost", 32, NULL },
	{ "Actual Work", 22, NULL },
	{ "Base Calendar", 48, NULL },
	{ "Baseline Cost", 31, NULL },
	{ "Baseline Work", 21, NULL },
	{ "Code", MPX_RESOURCE_DEF_CODE, mpx_resource_handle_int },
	{ "Cost", 30, NULL },
	{ "Cost Per Use", 44, NULL },
	{ "Cost Variance", 34, NULL },
	{ "Email Address", MPX_RESOURCE_DEF_EMAIL, mpx_resource_handle_string },
	{ "Group", MPX_RESOURCE_DEF_GROUP, mpx_resource_handle_int },
	{ "ID", MPX_RESOURCE_DEF_ID, mpx_resource_handle_int },
	{ "Initials", 2, NULL },
	{ "Linked Fields", 51, NULL },
	{ "Max Units", 41, NULL },
	{ "Name", MPX_RESOURCE_DEF_NAME, mpx_resource_handle_string },
	{ "Notes", 10, NULL },
	{ "Objects", 50, NULL },
	{ "Overallocated", 46, NULL },
	{ "Overtime Rate", MPX_RESOURCE_DEF_OVT_RATE, mpx_resource_handle_rate },
	{ "Overtime Work", 24, NULL },
	{ "Peak", 47, NULL },
	{ "Remaining Cost", 33, NULL },
	{ "Remaining Work", 23, NULL },
	{ "Standard Rate", MPX_RESOURCE_DEF_STD_RATE, mpx_resource_handle_rate },
	{ "Text1", 5, NULL },
	{ "Text2", 6, NULL },
	{ "Text3", 7, NULL },
	{ "Text4", 8, NULL },
	{ "Text5", 9, NULL },
	{ "Unique ID", 49, NULL },
	{ "Work", 20, NULL },
	{ "Work Variance", 25, NULL },
	{ NULL, 0, NULL }
};

static void
mpx_insert_resource (MpxResourceCtx *rctx)
{
#ifdef PORTED
	MpxTaskDelayed    *delay;
#endif
	CORBA_Environment  ev;
	GM_Id              real_id;

	CORBA_exception_init (&ev);

	if (!rctx->resource->name) {
		rctx->resource->name = CORBA_string_dup ("[undefined]");
	}

	if (!rctx->resource->email) {
		rctx->resource->email = CORBA_string_dup ("");
	}

	real_id = GNOME_MrProject_ResourceManager_insertResource (
		rctx->mpx->resource_manager, rctx->resource, &ev);

	mpx_add_id (rctx->mpx->map_resource_ids, rctx->id, real_id);

	if (BONOBO_EX (&ev)) {
		g_warning ("Bonobo exception '%s'",
			   bonobo_exception_get_text (&ev));
	}

	CORBA_exception_free (&ev);

#ifdef PORTED
	/* Build delayed mappings list */
	delay = g_new0 (MpxTaskDelayed, 1);
	delay->id = real_id;
	delay->parent_id = tctx->parent_id;
	tctx->mpx->delayed_mappings = 
		g_slist_prepend (tctx->mpx->delayed_mappings, delay);
#endif
}

static void
mpx_read_resource (MpxReadContext *mpx)
{
	MpxResourceCtx  rrctx;
	MpxResourceCtx *rctx = &rrctx;

	memset (rctx, 0, sizeof (rrctx));

	rctx->mpx = mpx;
	rctx->resource = GNOME_MrProject_Resource__alloc ();

	/* Some kosha defaults */
	rctx->resource->resourceId = -1;
	rctx->resource->groupId = -1;
	rctx->resource->units = -1;
	rctx->resource->type = GNOME_MrProject_RESOURCE_TYPE_PEOPLE;
	rctx->resource->stdRate = 123.0;
	rctx->resource->ovtRate = 456.0;

	mpx_read_fields (mpx, mpx_resource_defs, mpx->resource_offsets, rctx);

#ifdef MPX_DEBUG
	fprintf (stderr, "Resource %d '%s'\n", rctx->id, rctx->resource->name);
#endif

	mpx_insert_resource (rctx);

	/* Cleanup rctx */
	CORBA_free (rctx->resource);
}

static void
mpx_build_field_table (MpxReadContext    *mpx,
		       GArray            *offsets,
		       const MpxFieldDef *fields)
{
	char *tok;

	while ((tok = mpx_next_token (mpx))) {
		const MpxFieldDef *def;
		guint              idx = atoi (tok);

		def = mpx_get_field_def (fields, idx);

		if (!def) {
			g_warning ("Unknown field '%s'", tok);
		}
#ifdef MPX_DEBUG
		else {
			fprintf (stderr, "Field %d = '%s'\n", 
				 offsets->len,
				 def->name);
		}
#endif

		g_array_append_val (offsets, idx);

		g_free (tok);
	}
}

static void
mpx_read_header (MpxReadContext *mpx, Bonobo_PropertyBag pb)
{
	char    *proj_name;
	char    *company, *manager, *calendar;
	char    *start_date, *end_date;
	GM_Time  start, end;

	if (!(proj_name = mpx_next_token (mpx))) {
		bonobo_property_bag_client_set_value_string (
			pb, "Name", proj_name, NULL);
	}
	
	g_free (proj_name);
	
	if (!(company = mpx_next_token (mpx))) {
		bonobo_property_bag_client_set_value_string (
			pb, "Company", company, NULL);
	}

	g_free (company);

	if (!(manager = mpx_next_token (mpx))) {
		bonobo_property_bag_client_set_value_string (
			pb, "Manager", manager, NULL);
	}
	
	g_free (manager);

	if (!(calendar = mpx_next_token (mpx))) {
		return;
	}
	
	g_free (calendar);

	if (!(start_date = mpx_next_token (mpx))) {
		return;
	}
	
	start = (GM_Time) time_from_msdate (start_date);
	g_free (start_date);

	if (!(end_date = mpx_next_token (mpx))) {
		return;
	}
	
	end = (GM_Time) time_from_msdate (end_date);
	g_free (end_date);

	bonobo_property_bag_client_set_value_glong (pb, "Start", start, NULL);

	bonobo_property_bag_client_set_value_glong (pb, "Finish", end, NULL);

	/* Ignore some other fields */

	mpx_next_line (mpx);
}

typedef enum {
	MPX_COMMENT                     = 0,

	MPX_CURRENCY_SETTINGS           = 10,
	MPX_DEFAULT_SETTINGS            = 11,
	MPX_DATE_AND_TIME_SETTINGS      = 12,
	MPX_BASE_CALENDAR_DEFINITION    = 20,
	MPX_BASE_CALENDAR_HOURS         = 25,
	MPX_BASE_CALENDAR_EXCEPTION     = 26,
	MPX_PROJECT_HEADER              = 30,

	MPX_TEXT_RESOURCE_TABLE_DEF     = 40,
	MPX_NUMERIC_RESOURCE_TABLE_DEF  = 41,
	MPX_RESOURCE                    = 50,
	MPX_RESOURCE_NOTES              = 51,
	MPX_RESOURCE_CALENDAR_DEF       = 55,
	MPX_RESOURCE_CALENDAR_HOURS     = 56,
	MPX_RESOURCE_CALENDAR_EXCEPTION = 57,

	MPX_TEXT_TASK_TABLE_DEF         = 60,
	MPX_NUMERIC_TASK_TABLE_DEF      = 61,
	MPX_TASK                        = 70,
	MPX_TASK_NOTES                  = 71,
	MPX_TASK_RECURRING              = 72,
	MPX_TASK_RESOURCE_ASSIGNMENT    = 75,
	MPX_TASK_WORKGROUP_ASSIGNMENT   = 76,

	MPX_PROJECT_NAMES               = 80,
	MPX_DDE_OLE_CLIENT_LINKS        = 81
} MpxMainId;

static void
mpx_handle_token (MpxReadContext *mpx, MpxMainId id)
{
	switch (id) {
	case MPX_COMMENT:
		mpx_next_line (mpx);

	case MPX_TEXT_TASK_TABLE_DEF:
		break; /* ignore */

	case MPX_NUMERIC_TASK_TABLE_DEF:
		mpx_build_field_table (mpx, mpx->task_offsets, mpx_task_defs);
		break;

	case MPX_TASK:
		mpx_read_task (mpx);
		break;

	case MPX_TEXT_RESOURCE_TABLE_DEF:
		break; /* ignore */

	case MPX_NUMERIC_RESOURCE_TABLE_DEF:
		mpx_build_field_table (mpx, mpx->resource_offsets,
				       mpx_resource_defs);
		break;

	case MPX_RESOURCE:
		mpx_read_resource (mpx);
		break;

	case MPX_PROJECT_HEADER: {
		Bonobo_PropertyBag pb;
		CORBA_Environment  ev;

		CORBA_exception_init (&ev);

		pb = Bonobo_Unknown_queryInterface (
			mpx->project, "IDL:Bonobo/PropertyBag:1.0", &ev);

		if (BONOBO_EX (&ev) || pb == CORBA_OBJECT_NIL) {
			g_warning ("Error '%s'", 
				   bonobo_exception_get_text (&ev));
			CORBA_exception_free (&ev);
			return;
		}

		mpx_read_header (mpx, pb);

		bonobo_object_release_unref (pb, NULL);
		CORBA_exception_free (&ev);
		break;
	}

	/* Not yet implemented */
	case MPX_CURRENCY_SETTINGS:
	case MPX_DEFAULT_SETTINGS:
	case MPX_DATE_AND_TIME_SETTINGS:
	case MPX_BASE_CALENDAR_DEFINITION:
	case MPX_BASE_CALENDAR_HOURS:
	case MPX_BASE_CALENDAR_EXCEPTION:

	case MPX_RESOURCE_NOTES:
	case MPX_RESOURCE_CALENDAR_DEF:
	case MPX_RESOURCE_CALENDAR_HOURS:
	case MPX_RESOURCE_CALENDAR_EXCEPTION:

	case MPX_TASK_NOTES:
	case MPX_TASK_RECURRING:
	case MPX_TASK_RESOURCE_ASSIGNMENT:
	case MPX_TASK_WORKGROUP_ASSIGNMENT:

	case MPX_PROJECT_NAMES:
	case MPX_DDE_OLE_CLIENT_LINKS:
		mpx_next_line (mpx);
		break;
	}

}

static gboolean
mpx_read_project (const char *location,
		  GM_Project  project)
{
	MpxReadContext    *mpx = mpx_new (location);
	CORBA_Environment  ev;
	char              *tok;

	if (!mpx)
		return FALSE;

	mpx->project = project;

	CORBA_exception_init (&ev);

	mpx->task_manager = Bonobo_Unknown_queryInterface (
		project, "IDL:GNOME/MrProject/TaskManager:1.0",
		&ev);

	if (BONOBO_EX (&ev)) {
		g_warning ("Could not get TaskManager interface.");
		CORBA_exception_free (&ev);
	}
	
	mpx->resource_manager = Bonobo_Unknown_queryInterface (
		project, "IDL:GNOME/MrProject/ResourceManager:1.0", &ev);

	if (BONOBO_EX (&ev)) {
		g_warning ("Could not get ResourceManager interface.");
		CORBA_exception_free (&ev);
	}
	
	mpx->allocation_manager = Bonobo_Unknown_queryInterface (
		project, "IDL:GNOME/MrProject/AllocationManager:1.0", &ev);

	if (BONOBO_EX (&ev)) {
		g_warning ("Could not get AllocationManager interface.");
		CORBA_exception_free (&ev);
	}

	if (!(tok = mpx_next_token (mpx))) {
		g_warning ("No mpx header");
		return FALSE;
	} else if (strcmp (tok, "MPX")) {
		g_warning ("Not an mpx '%s'", tok);
		return FALSE;
	} else {
		char *ver_str = mpx_next_token (mpx);
		char *ver = mpx_next_token (mpx);
		char *std = mpx_next_token (mpx);

		fprintf (stderr, "Reading '%s', '%g', '%s'\n",
			 ver_str, atof (ver), std);

		mpx_next_line (mpx);
	}

	do {
		char *str;
		int   id;

		str = mpx_next_token (mpx);
		if (!str) {/* rather a short line */
			continue;
		}
		
		id = atoi (str);
		g_free (str);

		mpx_handle_token (mpx, id);

	} while (!mpx_eof (mpx));

	mpx_process_delayed_mappings (mpx);

	if (mpx->task_manager) {
		GNOME_MrProject_TaskManager__set_state (
			mpx->task_manager, 
			GNOME_MrProject_PROJECT_STATE_SAVED, &ev);

		if (BONOBO_EX (&ev)) {
			CORBA_exception_free (&ev);
		}

		bonobo_object_release_unref (mpx->task_manager, NULL);
	}

	if (mpx->resource_manager) {
		GNOME_MrProject_ResourceManager__set_state (
			mpx->resource_manager, 
			GNOME_MrProject_PROJECT_STATE_SAVED, &ev);

		if (BONOBO_EX (&ev)) {
			CORBA_exception_free (&ev);
		}

		bonobo_object_release_unref (mpx->resource_manager, NULL);
	}

	if (mpx->allocation_manager) {
		GNOME_MrProject_AllocationManager__set_state (
			mpx->allocation_manager,
			GNOME_MrProject_PROJECT_STATE_SAVED, &ev);

		if (BONOBO_EX (&ev)) {
			CORBA_exception_free (&ev);
		}

		bonobo_object_release_unref (mpx->allocation_manager, NULL);
	}

	mpx_free (mpx);
	
	fprintf (stderr, "Done\n");

	return TRUE;
}

static gboolean
mff_probe (FileFilter        *filter,
	   const gchar       *uri,
	   CORBA_Environment *ev)
{
	return TRUE;
}

static void
mff_load (FileFilter        *filter,
	  const gchar       *uri,
	  const GM_Project   project,
	  CORBA_Environment *ev)
{
	gchar *location;

	location = gnome_vfs_get_local_path_from_uri (uri);

	if (!location) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_GNOME_MrProject_FileFilter_Failure,
				     NULL);
		return;
	}
	
	if (!mpx_read_project (location, project)) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_GNOME_MrProject_FileFilter_Failure,
				     NULL);
	}

	g_free (location);
}

static void
mff_load_summary (FileFilter         *filter,
		  const gchar        *uri,
		  GM_FileSummary    **summary,
		  CORBA_Environment  *ev)
{
	CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
			     ex_GNOME_MrProject_FileFilter_Failure,
			     NULL);
}

static void
mff_save (FileFilter        *filter,
	  const gchar       *uri,
	  const GM_Project   project,
	  CORBA_Environment *ev)
{
	CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
			     ex_GNOME_MrProject_FileFilter_Unimplemented,
			     NULL);
}

static void
mpx_file_filter_class_init (MpxFileFilterClass *klass)
{
	FileFilterClass *file_filter_class = (FileFilterClass *) klass;

	file_filter_class->probe        = mff_probe;
	file_filter_class->load         = mff_load;
	file_filter_class->load_summary = mff_load_summary;
	file_filter_class->save         = mff_save;
}

static void
mpx_file_filter_init (MpxFileFilter *mpx_file_filter)
{
}

static BonoboObject *
factory (BonoboGenericFactory *this,
	 const char           *object_id,
	 void                 *data)
{
	g_return_val_if_fail (object_id != NULL, NULL);

	if (!strcmp (object_id, "OAFIID:GNOME_MrProject_MpxFileFilter")) {
		FileFilter *filter;

		filter = gtk_type_new (TYPE_MPX_FILE_FILTER);
		return BONOBO_OBJECT (filter);
	} else {
		g_warning ("Failing to manufacture a '%s'", object_id);
	}
	
	return NULL;
}

BONOBO_OAF_SHLIB_FACTORY_MULTI ("OAFIID:GNOME_MrProject_MpxFileFilterFactory",
				"Mr Project MPX file filter factory",
				factory,
				NULL);

