/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Mikael Hallendal <micke@codefactory.se>
 * Copyright (C) 2001 Richard Hult <rhult@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Mikael Hallendal <micke@codefactory.se>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <bonobo.h>
#include <gtk/gtkobject.h>
#include "util/corba-utils.h"
#include "util/id-map.h"
#include "util/type-utils.h"
#include "file-resource-manager.h"

/* GtkObject. */
static void file_resource_manager_init       (FileResourceManager       *frm);
static void file_resource_manager_class_init (FileResourceManagerClass  *frmc);

static void frm_destroy            (GtkObject                       *object);

static GNOME_MrProject_Id frm_insert_resource (ResourceManager           *rm,
					       const GNOME_MrProject_Resource *res);

static GNOME_MrProject_Resource * 
frm_get_resource                   (ResourceManager                 *rm,
				    GNOME_MrProject_Id               res_id);

static GSList * frm_get_all_resources (ResourceManager                 *rm);

static gboolean
frm_update_resource                (ResourceManager                 *rm,
				    const GNOME_MrProject_Resource  *res);

static void     
frm_remove_resources                (ResourceManager                 *rm,
				     GSList                          *res_ids);

static GNOME_MrProject_Id
frm_insert_group                 (ResourceManager                      *rm,
				  const GNOME_MrProject_ResourceGroup  *group);

static GNOME_MrProject_ResourceGroup * 
frm_get_group                      (ResourceManager                 *rm,
				    GNOME_MrProject_Id               group_id);

static GSList * frm_get_all_groups (ResourceManager                 *rm);
static gboolean 
frm_update_group             (ResourceManager                       *rm,
			      const GNOME_MrProject_ResourceGroup   *group);
static void     frm_remove_group   (ResourceManager                 *rm,
				    GNOME_MrProject_Id               group_id);
 
static GNOME_MrProject_Id
frm_get_default_group              (ResourceManager                 *rm);

static void     
frm_set_default_group              (ResourceManager                 *rm,
				    GNOME_MrProject_Id               group_id);


struct _FileResourceManagerPriv {
	/* Resource Management */
        GSList   *resources;
	IdMap    *resource_map;
	
	/* Group Management */
	IdMap    *group_map;
	gint      default_group;
};

GNOME_CLASS_BOILERPLATE (FileResourceManager, file_resource_manager,
			 ResourceManager, resource_manager);

/* Resource Management */

static void 
frm_destroy (GtkObject *object)
{
        FileResourceManager        *frm;
        FileResourceManagerPriv    *priv;
	GSList                     *list;
	GSList                     *group_list;
	
        g_return_if_fail (object != NULL);
        g_return_if_fail (IS_FILE_RESOURCE_MANAGER (object));

        frm  = FILE_RESOURCE_MANAGER (object);
        priv = frm->priv;
        
	for (list = priv->resources; list; list = list->next) {
		CORBA_free (list->data);
	}

	g_slist_free (priv->resources);
	gtk_object_destroy (GTK_OBJECT (priv->resource_map));
	
	group_list = id_map_get_objects (priv->group_map);
	
	for (list = group_list; list; list = list->next) {
		CORBA_free (list->data);
	}

	g_slist_free (group_list);
	gtk_object_destroy (GTK_OBJECT (priv->group_map));

        GNOME_CALL_PARENT_HANDLER (GTK_OBJECT_CLASS, destroy, (object));
}

static GNOME_MrProject_Id
frm_insert_resource (ResourceManager                  *rm, 
		     const GNOME_MrProject_Resource   *resource)
{
	FileResourceManager        *frm;
	FileResourceManagerPriv    *priv;
	GNOME_MrProject_Resource   *copy;
	gint                       id;
	
	g_return_val_if_fail (rm != NULL, -1);
	g_return_val_if_fail (IS_FILE_RESOURCE_MANAGER (rm), -1);
	
	frm  = FILE_RESOURCE_MANAGER (rm);
	priv = frm->priv;

	copy             = corba_util_resource_duplicate (resource);
	id               = id_map_insert (priv->resource_map, copy);
	copy->resourceId = id;
	priv->resources  = g_slist_prepend (priv->resources, copy);

	return id;
}

static GNOME_MrProject_Resource *
frm_get_resource (ResourceManager *rm, GNOME_MrProject_Id resource_id)
{
	FileResourceManager        *frm;
	FileResourceManagerPriv    *priv;
	GNOME_MrProject_Resource   *resource;
	
	g_return_val_if_fail (rm != NULL, NULL);
	g_return_val_if_fail (IS_FILE_RESOURCE_MANAGER (rm), NULL);
	
	frm  = FILE_RESOURCE_MANAGER (rm);
	priv = frm->priv;

	resource = id_map_lookup (priv->resource_map, resource_id);

	return resource;
}

static GSList *
frm_get_all_resources (ResourceManager *rm)
{
	FileResourceManager       *frm;
	FileResourceManagerPriv   *priv;
	GSList                    *list;
	
	g_return_val_if_fail (rm != NULL, NULL);
	g_return_val_if_fail (IS_FILE_RESOURCE_MANAGER (rm), NULL);
	
	frm  = FILE_RESOURCE_MANAGER (rm);
	priv = frm->priv;
	
 	list = g_slist_copy (priv->resources); 
 	list = g_slist_reverse (list); 

	return list;
}

static gboolean
frm_update_resource (ResourceManager                  *rm, 
		     const GNOME_MrProject_Resource   *resource)
{
	FileResourceManager        *frm;
	FileResourceManagerPriv    *priv;
	GNOME_MrProject_Resource   *original;
	
	g_return_val_if_fail (rm != NULL, FALSE);
	g_return_val_if_fail (IS_FILE_RESOURCE_MANAGER (rm), FALSE);
	
	frm  = FILE_RESOURCE_MANAGER (rm);
	priv = frm->priv;

	original = id_map_lookup (priv->resource_map, resource->resourceId);
	
	if (!original) {
		return FALSE;
	}

	return corba_util_resource_update (original, resource);
}

static void
frm_remove_resources (ResourceManager *rm, GSList *res_ids)
{
	FileResourceManager     *frm;
	FileResourceManagerPriv *priv;
	GM_Resource             *resource;
	GSList                  *node;
	GM_Id                    res_id;
	
	g_return_if_fail (rm != NULL);
	g_return_if_fail (IS_FILE_RESOURCE_MANAGER (rm));
	
	frm  = FILE_RESOURCE_MANAGER (rm);
	priv = frm->priv;

	for (node = res_ids; node; node = node->next) {
		res_id = GPOINTER_TO_INT (node->data);
		resource = id_map_lookup (priv->resource_map, res_id);
		
		if (!resource) {
			continue;
		}

		priv->resources = g_slist_remove (priv->resources, resource);
		id_map_remove (priv->resource_map, res_id);
		
		CORBA_free (resource);
	}
}

static void
frm_foreach_resource (ResourceManager              *rm,
		      ResourceManagerForeachFunc    func,
		      gpointer                      user_data)
{
	GNOME_MrProject_Resource   *resource;
	FileResourceManager        *frm;
	FileResourceManagerPriv    *priv;
	GSList                     *copy, *node;
	
	frm   = FILE_RESOURCE_MANAGER (rm);
	priv  = frm->priv;
	copy  = g_slist_copy (priv->resources);
	node  = g_slist_reverse (copy);

	for (; node; node = node->next) {
		resource = (GNOME_MrProject_Resource *) node->data;

		if (resource == NULL) {
			g_print ("RESOURCE IS NULL\n");
		}
		
		func (rm, resource, user_data);
	}

	g_slist_free (copy);
}

/* Group Management */

static GNOME_MrProject_Id
frm_insert_group (ResourceManager                       *rm, 
		  const GNOME_MrProject_ResourceGroup   *group)
{
	FileResourceManager             *frm;
	FileResourceManagerPriv         *priv;
	GNOME_MrProject_ResourceGroup   *copy;
	
	g_return_val_if_fail (rm != NULL, -1);
	g_return_val_if_fail (IS_FILE_RESOURCE_MANAGER (rm), -1);
	
	frm  = FILE_RESOURCE_MANAGER (rm);
	priv = frm->priv;
	
	copy = corba_util_resource_group_duplicate (group);

	if (copy->groupId > 0) {
		id_map_insert_id (priv->group_map, copy->groupId, copy);
	} else {
		copy->groupId = id_map_insert (priv->group_map, copy);
	}

	return copy->groupId;
}

static GNOME_MrProject_ResourceGroup *
frm_get_group (ResourceManager *rm, const GNOME_MrProject_Id   group_id)
{
	FileResourceManager             *frm;
	FileResourceManagerPriv         *priv;
	GNOME_MrProject_ResourceGroup   *group;
	
	g_return_val_if_fail (rm != NULL, NULL);
	g_return_val_if_fail (IS_FILE_RESOURCE_MANAGER (rm), NULL);
	
	frm  = FILE_RESOURCE_MANAGER (rm);
	priv = frm->priv;

	group = id_map_lookup (priv->group_map, group_id);
	
	return group;
}

static GSList *
frm_get_all_groups (ResourceManager *rm)
{
	FileResourceManager       *frm;
	FileResourceManagerPriv   *priv;
	GSList                    *groups;
	
	g_return_val_if_fail (rm != NULL, NULL);
	g_return_val_if_fail (IS_FILE_RESOURCE_MANAGER (rm), NULL);
	
	frm  = FILE_RESOURCE_MANAGER (rm);
	priv = frm->priv;
	
	groups = id_map_get_objects (priv->group_map);
	
	return groups;
}

static gboolean
frm_update_group (ResourceManager                       *rm, 
		  const GNOME_MrProject_ResourceGroup   *group)
{
	FileResourceManager             *frm;
	FileResourceManagerPriv         *priv;
	GNOME_MrProject_ResourceGroup   *original;
	
	g_return_val_if_fail (rm != NULL, FALSE);
	g_return_val_if_fail (IS_FILE_RESOURCE_MANAGER (rm), FALSE);
	
	frm  = FILE_RESOURCE_MANAGER (rm);
	priv = frm->priv;

	original = id_map_lookup (priv->group_map, group->groupId);
	
	if (!original) {
		return FALSE;
	}

	return corba_util_resource_group_update (original, group);
}

static void
frm_remove_group (ResourceManager *rm, GNOME_MrProject_Id group_id)
{
	FileResourceManager             *frm;
	FileResourceManagerPriv         *priv;
	GNOME_MrProject_ResourceGroup   *group;
	
	g_return_if_fail (rm != NULL);
	g_return_if_fail (IS_FILE_RESOURCE_MANAGER (rm));
	
	frm  = FILE_RESOURCE_MANAGER (rm);
	priv = frm->priv;

	group = id_map_lookup (priv->group_map, group_id);
	
	if (!group) {
		return;
	}

	id_map_remove (priv->group_map, group_id);
	
	CORBA_free (group);
}

static GNOME_MrProject_Id
frm_get_default_group (ResourceManager *rm)
{
	FileResourceManager       *frm;
	FileResourceManagerPriv   *priv;
	
	g_return_val_if_fail (rm != NULL, -1);
	g_return_val_if_fail (IS_FILE_RESOURCE_MANAGER (rm), -1);
	
	frm  = FILE_RESOURCE_MANAGER (rm);
	priv = frm->priv;

	return priv->default_group;
}

static void
frm_set_default_group (ResourceManager *rm, GNOME_MrProject_Id group_id)
{
	FileResourceManager       *frm;
	FileResourceManagerPriv   *priv;
	
	g_return_if_fail (rm != NULL);
	g_return_if_fail (IS_FILE_RESOURCE_MANAGER (rm));
	
	frm  = FILE_RESOURCE_MANAGER (rm);
	priv = frm->priv;

	/* Check to see if group exists */
	if (id_map_lookup (priv->group_map, group_id)) {
		priv->default_group = group_id;
	}
}

static void
frm_foreach_group (ResourceManager *rm,
		   GHFunc           func,
		   gpointer         user_data)
{
	FileResourceManager        *frm;
	FileResourceManagerPriv    *priv;
	
	frm   = FILE_RESOURCE_MANAGER (rm);
	priv  = frm->priv;

	id_map_foreach (priv->group_map, func, user_data);
}

static void
file_resource_manager_init (FileResourceManager   *manager)
{
        FileResourceManagerPriv   *priv;

        priv                 = g_new0 (FileResourceManagerPriv, 1);
        manager->priv        = priv;
        
	/* Resource Management */
	priv->resources      = NULL;
        priv->resource_map   = id_map_new (1);
	
	/* Group Management */
        priv->group_map      = id_map_new (1);
	priv->default_group  = -1;
}

static void
file_resource_manager_class_init (FileResourceManagerClass *klass)
{
        GtkObjectClass         *object_class;
        ResourceManagerClass   *rmc;
	
        object_class = (GtkObjectClass *) klass;
        rmc          = (ResourceManagerClass *) klass;
	
	/* GtkObject functions */
        object_class->destroy           = frm_destroy;
	
	/*
	 * Resource Management
	 */

	/* CORBA methods. */
	rmc->insert_resource   = frm_insert_resource;
	rmc->get_resource      = frm_get_resource;
	rmc->get_all_resources = frm_get_all_resources;
	rmc->update_resource   = frm_update_resource;
	rmc->remove_resources  = frm_remove_resources;

	/* C methods. */
	rmc->foreach_resource  = frm_foreach_resource;

	/*
	 * Group Management
	 */

	/* CORBA methods. */
	rmc->insert_group      = frm_insert_group;
	rmc->get_group         = frm_get_group;
	rmc->get_all_groups    = frm_get_all_groups;
	rmc->update_group      = frm_update_group;
	rmc->remove_group      = frm_remove_group;
	rmc->get_default_group = frm_get_default_group;
	rmc->set_default_group = frm_set_default_group;

	/* C methods. */
	rmc->foreach_group     = frm_foreach_group;
}

ResourceManager *
file_resource_manager_new (BonoboEventSource *event_source)
{
        FileResourceManager *frm;
        
        frm = gtk_type_new (TYPE_FILE_RESOURCE_MANAGER);

	resource_manager_construct (RESOURCE_MANAGER (frm),
				    event_source);
        
        return RESOURCE_MANAGER (frm);
}

