/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Richard Hult <rhult@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtk.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-preferences.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-stock.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gdk-pixbuf/gdk-pixbuf-loader.h>
#include <gal/widgets/gtk-combo-box.h>

#include "util/time-utils.h"
#include "goto-popup.h"

static GtkWidget *image_from_file (const gchar *filename);

enum {
	DIALOG_SHOWN,
	DIALOG_HIDDEN,
	DATE_CLICKED,
	TODAY_CLICKED,
	SELECTED_CLICKED,
	START_CLICKED,
	LAST_SIGNAL
};

typedef struct {
	GotoPopup   *popup;
	GtkCalendar *calendar;
} GotoData;

static gint signals [LAST_SIGNAL] = { 0, };

static void
goto_popup_destroy (GtkObject *object)
{
	GotoPopup *popup;
	GtkObjectClass *parent_class;

	popup = GOTO_POPUP (object);
	
	g_free (popup->title);

	parent_class = gtk_type_class (gtk_combo_box_get_type ());
	parent_class->destroy (GTK_OBJECT (popup));
}

static void
goto_popup_init (GotoPopup *popup)
{
}

static void
goto_popup_class_init (GotoPopupClass *klass)
{
        GtkObjectClass *object_class;

        object_class = (GtkObjectClass *) klass;

	object_class->destroy = goto_popup_destroy;
	
	signals [DIALOG_SHOWN] =
		gtk_signal_new ("dialog-shown",
				GTK_RUN_LAST,
				object_class->type,
				0,
				gtk_marshal_BOOL__NONE,
				GTK_TYPE_BOOL, 0);

	signals [DIALOG_HIDDEN] =
		gtk_signal_new ("dialog-hidden",
				GTK_RUN_LAST,
				object_class->type,
				0,
				gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE, 0);

	signals [DATE_CLICKED] =
		gtk_signal_new ("date-clicked",
				GTK_RUN_LAST,
				object_class->type,
				0,
				gtk_marshal_NONE__INT,
				GTK_TYPE_NONE, 1, GTK_TYPE_LONG);

	signals [TODAY_CLICKED] =
		gtk_signal_new ("today-clicked",
				GTK_RUN_LAST,
				object_class->type,
				0,
				gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE, 0);

	signals [SELECTED_CLICKED] =
		gtk_signal_new ("selected-clicked",
				GTK_RUN_LAST,
				object_class->type,
				0,
				gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE, 0);

	signals [START_CLICKED] =
		gtk_signal_new ("start-clicked",
				GTK_RUN_LAST,
				object_class->type,
				0,
				gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE, 0);
	
	gtk_object_class_add_signals (object_class, signals, LAST_SIGNAL);
}

GtkType
goto_popup_get_type (void)
{
        static GtkType goto_type = 0;

        if (!goto_type)
        {
                const GtkTypeInfo goto_info =
                {
                        "GotoPopup",
                        sizeof (GotoPopup),
                        sizeof (GotoPopupClass),
                        (GtkClassInitFunc)  goto_popup_class_init,
                        (GtkObjectInitFunc) goto_popup_init,
                        NULL, /* Reserved 1 */
                        NULL, /* Reserved 2 */
                        (GtkClassInitFunc) NULL,
                };

                goto_type = gtk_type_unique (gtk_combo_box_get_type (), &goto_info);
        }

        return goto_type;
}

static void
date_selected (GtkWidget *widget, GotoData *data)
{
	time_t t;
	guint  year, month, day;

	g_return_if_fail (data != NULL);
	g_return_if_fail (data->calendar != NULL);
	g_return_if_fail (data->popup != NULL);
	
	gtk_calendar_get_date (data->calendar,
			       &year, &month, &day);

	t = time_from_day (year, month, day);
	
	gtk_combo_box_popup_hide (GTK_COMBO_BOX (data->popup));
	gtk_signal_emit (GTK_OBJECT (data->popup), signals[DATE_CLICKED], t);
}

static void
today_clicked (GtkWidget *widget, GotoData *data)
{
	g_return_if_fail (data != NULL);
	g_return_if_fail (data->calendar != NULL);
	g_return_if_fail (data->popup != NULL);
	
	gtk_combo_box_popup_hide (GTK_COMBO_BOX (data->popup));
	gtk_signal_emit (GTK_OBJECT (data->popup), signals[TODAY_CLICKED], NULL);
}

static void
start_clicked (GtkWidget *widget, GotoData *data)
{
	g_return_if_fail (data != NULL);
	g_return_if_fail (data->calendar != NULL);
	g_return_if_fail (data->popup != NULL);

	gtk_combo_box_popup_hide (GTK_COMBO_BOX (data->popup));
	gtk_signal_emit (GTK_OBJECT (data->popup), signals[START_CLICKED], NULL);
}

static void
selected_clicked (GtkWidget *widget, GotoData *data)
{
	g_return_if_fail (data != NULL);
	g_return_if_fail (data->calendar != NULL);
	g_return_if_fail (data->popup != NULL);

	gtk_combo_box_popup_hide (GTK_COMBO_BOX (data->popup));
	gtk_signal_emit (GTK_OBJECT (data->popup), signals[SELECTED_CLICKED], NULL);
}

static GtkWidget *
create_goto_frame (GotoData *data, GotoButtonFlags flags)
{
	GtkWidget *frame, *calendar;
	GtkWidget *vbox, *button_box, *button;
	time_t     cur_time;
	gint       year, month, day;
	GtkWidget *label, *alignment;

	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_NONE);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (frame), vbox);
	gtk_widget_show (vbox);
	
	calendar = gtk_calendar_new ();
	data->calendar = GTK_CALENDAR (calendar);
	gtk_box_pack_start (GTK_BOX (vbox), calendar, FALSE, FALSE, 0);
	gtk_widget_show (calendar);

	label = gtk_label_new (_("Double click on a date to go there"));
	alignment = gtk_alignment_new (0.95, 0, 0, 0);
	gtk_container_add (GTK_CONTAINER (alignment), label);
	gtk_widget_show_all (alignment);
	gtk_box_pack_start (GTK_BOX (vbox), alignment, FALSE, FALSE, 4);

	button_box = gtk_hbutton_box_new ();
	gtk_container_set_border_width (GTK_CONTAINER (button_box), 4);
	gtk_button_box_set_spacing (GTK_BUTTON_BOX (button_box), 2);
	gtk_button_box_set_child_ipadding (GTK_BUTTON_BOX (button_box), 2, 0);
	gtk_button_box_set_child_size (GTK_BUTTON_BOX (button_box), 0, 0);
	gtk_widget_show (button_box);

	gtk_box_pack_start (GTK_BOX (vbox), button_box, FALSE, FALSE, 0);

	if (flags & GOTO_TODAY) {
		button = gtk_button_new_with_label (_("Today"));
		gtk_widget_show (button);
		gtk_box_pack_start (GTK_BOX (button_box), button, FALSE, FALSE, 0);
		gtk_signal_connect (GTK_OBJECT (button),
				    "clicked",
				    GTK_SIGNAL_FUNC (today_clicked),
				    data);
	}
	
	if (flags & GOTO_START) {
		button = gtk_button_new_with_label (_("Project start"));
		gtk_widget_show (button);
		gtk_box_pack_start (GTK_BOX (button_box), button, FALSE, FALSE, 0);
		gtk_signal_connect (GTK_OBJECT (button),
				    "clicked",
				    GTK_SIGNAL_FUNC (start_clicked),
				    data);
	}
	
	if (flags & GOTO_SELECTED) {
		button = gtk_button_new_with_label (_("Selected task"));
		gtk_widget_show (button);
		gtk_box_pack_start (GTK_BOX (button_box), button, FALSE, FALSE, 0);
		gtk_signal_connect (GTK_OBJECT (button),
				    "clicked",
				    GTK_SIGNAL_FUNC (selected_clicked),
				    data);
	}

	gtk_signal_connect (GTK_OBJECT (calendar),
			    "day-selected-double-click",
			    date_selected,
			    data);
	
	cur_time = time (NULL);
	time_split (cur_time, &year, &month, &day);
	
	gtk_calendar_select_month (GTK_CALENDAR (calendar), month, year);
	gtk_calendar_select_day (GTK_CALENDAR (calendar), day);

	return frame;
}

static gboolean
goto_dialog_close (GtkWidget *widget, GotoData *data)
{
	gtk_signal_emit (GTK_OBJECT (data->popup),
			 signals[DIALOG_HIDDEN],
			 NULL);

	data->popup = NULL;
	data->calendar = NULL;
	g_free (data);
	
	return FALSE; /* Yes, close the window. */
}

static void
goto_dialog_button_clicked (GtkWidget *widget,
			    gint       btn,
			    GotoData  *data)
{
	gnome_dialog_close (GNOME_DIALOG (widget));
}

static void
goto_button_clicked (GtkWidget *widget, GotoPopup *popup)
{
	GotoData  *data;
	GtkWidget *dialog, *frame;
	gboolean   ok_to_show;

	/* First, see if it's ok to show the dialog. E.g. if there
	 * already is one shown, we shouldn't show one more.
	 */
	gtk_signal_emit (GTK_OBJECT (popup),
			 signals[DIALOG_SHOWN],
			 &ok_to_show);
	
	if (!ok_to_show) {
		return;
	}

	data = g_new (GotoData, 1);
	data->popup = popup;
	
	dialog = gnome_dialog_new (popup->title,
				   GNOME_STOCK_BUTTON_CLOSE,
				   NULL);

	frame = create_goto_frame (data, popup->flags);
	gtk_widget_show (frame);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox),
			    frame, FALSE, FALSE, 0);
	
	gtk_signal_connect (GTK_OBJECT (dialog),
			    "close",
			    GTK_SIGNAL_FUNC (goto_dialog_close),
			    data);

	gtk_signal_connect (GTK_OBJECT (dialog),
			    "clicked",
			    goto_dialog_button_clicked,
			    data);

	gtk_widget_show (dialog);
}

GtkWidget *
goto_popup_new (const gchar     *title,
		GotoButtonFlags  flags)
{
	GtkWidget *popup;
	GtkWidget *vbox, *image, *frame;
	GtkWidget *button;
	GotoData  *data;

        popup = gtk_type_new (goto_popup_get_type ());

	GOTO_POPUP (popup)->title = g_strdup (title);
	GOTO_POPUP (popup)->flags = flags;
	
	image = image_from_file (MRPROJECT_IMAGEDIR "24_calendar.png");
	gtk_widget_show (image);

	button = gtk_button_new ();
	if (!gnome_preferences_get_toolbar_relief_btn ()) {
		gtk_button_set_relief (GTK_BUTTON (button),
				       GTK_RELIEF_NONE);
	}
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);

	vbox = gtk_vbox_new (FALSE, 2);

	gtk_box_pack_start (GTK_BOX (vbox), image, TRUE, TRUE, 0);
	/*gtk_box_pack_end (GTK_BOX (vbox), gtk_label_new (_("Go to")), FALSE, TRUE, 0);*/
	
	gtk_container_add (GTK_CONTAINER (button), vbox);
	gtk_widget_show (button);

	gtk_signal_connect (GTK_OBJECT (button),
			    "clicked",
			    goto_button_clicked,
			    popup);

	data = g_new (GotoData, 1);
	data->popup = GOTO_POPUP (popup);

	frame = create_goto_frame (data, flags);

	gtk_combo_box_construct (GTK_COMBO_BOX (popup),
				 button,
				 frame);

	if (!gnome_preferences_get_toolbar_relief_btn ()) {
		gtk_combo_box_set_arrow_relief (GTK_COMBO_BOX (popup),
						GTK_RELIEF_NONE);
	}

	/* We don't want it to be tearable since we have our own
	 * toplevel dialog for this.
	 */
	gtk_combo_box_set_tearable (GTK_COMBO_BOX (popup), FALSE);

	gtk_combo_box_set_title (GTK_COMBO_BOX (popup), _("Go to"));
	gtk_container_set_border_width (GTK_CONTAINER (popup), 0);
	
        gtk_widget_show_all (popup);

	return popup;
}

static GtkWidget *
image_from_file (const gchar *filename)
{
	GtkWidget *image;
	GdkPixmap *pixmap, *bitmap;
	GdkPixbuf *pixbuf;

	pixbuf = gdk_pixbuf_new_from_file (filename);
	gdk_pixbuf_render_pixmap_and_mask (pixbuf, &pixmap, &bitmap, 127);
	image = gtk_pixmap_new (pixmap, bitmap);

	gdk_pixmap_unref (pixmap);
	gdk_pixmap_unref (bitmap);
	gdk_pixbuf_unref (pixbuf);
	
	return image;
}

