/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 Ximian Inc. 
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Richard Hult <rhult@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Based on ECellCombo and ECellDateEdit, written by Damon Chaplin <damon@ximian.com>
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/time.h>
#include <unistd.h>
#include <locale.h>
#include <glib.h>
#include <gdk/gdkkeysyms.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <gal/util/e-util.h>
#include <gal/e-table/e-cell.h>
#include <gal/e-table/e-cell-text.h>
#include <gal/e-table/e-table-item.h>
#include "util/type-utils.h"
#include "util/time-utils.h"
#include "m-cell-date.h"

static void m_cell_date_class_init        (GtkObjectClass *object_class);
static void m_cell_date_init              (GtkObject      *object);

static gint m_cell_date_do_popup	  (ECellPopup     *ecp,
					   GdkEvent       *event,
					   int             row,
					   int             view_col);
static gint m_cell_date_key_press	  (GtkWidget	  *popup_window,
					   GdkEventKey	  *event,
					   MCellDate	  *mcd);
static void m_cell_date_set_calendar_date (MCellDate      *mcd);
static void m_cell_date_update_cell	  (MCellDate	  *mcd);
static void m_cell_date_double_click      (GtkCalendar    *calendar, MCellDate *mcd);
static void m_cell_date_ok_clicked        (GtkWidget      *button, MCellDate *mcd);
static void m_cell_date_cancel_clicked    (GtkWidget      *button, MCellDate *mcd);
static void m_cell_date_today_clicked     (GtkWidget      *button, MCellDate *mcd);


GNOME_CLASS_BOILERPLATE (MCellDate, m_cell_date, ECellPopup, e_cell_popup);


static void
m_cell_date_class_init (GtkObjectClass *object_class)
{
	ECellPopupClass *ecpc = (ECellPopupClass *) object_class;

	ecpc->popup = m_cell_date_do_popup;
}

static void
m_cell_date_init (GtkObject *object)
{
	MCellDate *mcd = M_CELL_DATE (object);
	GtkWidget *frame, *vbox, *bbox, *button;

	mcd->popup_window = gtk_window_new (GTK_WINDOW_POPUP);
	gtk_window_set_policy (GTK_WINDOW (mcd->popup_window),
			       TRUE, TRUE, FALSE);

	frame = gtk_frame_new (NULL);
	gtk_container_add (GTK_CONTAINER (mcd->popup_window), frame);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_OUT);
	gtk_widget_show (frame);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (frame), vbox);
        gtk_widget_show (vbox);
	
	mcd->calendar = gtk_calendar_new ();
	gtk_box_pack_start (GTK_BOX (vbox), mcd->calendar, TRUE, TRUE, 0);
	gtk_widget_show (mcd->calendar);

	bbox = gtk_hbutton_box_new ();
	gtk_container_set_border_width (GTK_CONTAINER (bbox), 4);
	gtk_button_box_set_spacing (GTK_BUTTON_BOX (bbox), 2);
	gtk_button_box_set_child_ipadding (GTK_BUTTON_BOX (bbox), 2, 0);
	gtk_button_box_set_child_size (GTK_BUTTON_BOX (bbox), 0, 0);
	gtk_box_pack_start (GTK_BOX (vbox), bbox, FALSE, FALSE, 0);
        gtk_widget_show (bbox);

	button = gtk_button_new_with_label (_("Today"));
	gtk_container_add (GTK_CONTAINER (bbox), button);
	gtk_widget_set_sensitive (button, FALSE);
        gtk_widget_show (button);
	gtk_signal_connect (GTK_OBJECT (button), "clicked",
			    GTK_SIGNAL_FUNC (m_cell_date_today_clicked),
			    mcd);
	
	button = gtk_button_new_with_label (_("OK"));
	gtk_container_add (GTK_CONTAINER (bbox), button);
        gtk_widget_show (button);
	gtk_signal_connect (GTK_OBJECT (button), "clicked",
			    GTK_SIGNAL_FUNC (m_cell_date_ok_clicked),
			    mcd);

	button = gtk_button_new_with_label (_("Cancel"));
	gtk_container_add (GTK_CONTAINER (bbox), button);
        gtk_widget_show (button);
	gtk_signal_connect (GTK_OBJECT (button), "clicked",
			    GTK_SIGNAL_FUNC (m_cell_date_cancel_clicked),
			    mcd);

	gtk_signal_connect (GTK_OBJECT (mcd->calendar), "day-selected-double-click", 
			    m_cell_date_double_click, 
			    mcd);

	gtk_signal_connect (GTK_OBJECT (mcd->popup_window),
			    "key_press_event",
			    GTK_SIGNAL_FUNC (m_cell_date_key_press), mcd);
}

ECell *
m_cell_date_new (const char *fontname, GtkJustification justify)
{
	MCellDate *mcd = gtk_type_new (m_cell_date_get_type ());
      
	return (ECell *) mcd;
}

static void
m_cell_date_get_popup_pos (MCellDate	*mcd,
			   int           row,
			   int           view_col,
			   gint		*x,
			   gint		*y)
{
	ECellPopup     *ecp = E_CELL_POPUP (mcd);
	ETableItem     *eti = E_TABLE_ITEM (ecp->popup_cell_view->cell_view.e_table_item_view);
	GtkWidget      *canvas = GTK_WIDGET (GNOME_CANVAS_ITEM (eti)->canvas);
	GtkBin         *popwin;
	GtkRequisition  req;
	gint            screen_height, screen_width;
	gint            row_height;
	double          x1, y1;
	gdouble         cx, cy;

	popwin = GTK_BIN (mcd->popup_window);
  
	gdk_window_get_origin (canvas->window, x, y);

	x1 = e_table_header_col_diff (eti->header, 0, view_col + 1);
	y1 = e_table_item_row_diff (eti, 0, row + 1);
	row_height = e_table_item_row_diff (eti, row,
					    row + 1);

	gnome_canvas_item_i2w (GNOME_CANVAS_ITEM (eti), &x1, &y1);

	gnome_canvas_world_to_window (GNOME_CANVAS (GNOME_CANVAS_ITEM (eti)->canvas),
				      x1,
				      y1,
				      &cx,
				      &cy);

	*x += cx;
	/* The ETable positions don't include the grid lines, I think, so we
	   add 1. */
	*y += cy + 1;

	screen_height = gdk_screen_height () - *y;
	screen_width = gdk_screen_width ();
  
	gtk_widget_size_request (mcd->calendar, &req);
  
	/* Check if it fits in the available height. */
	if (req.height > screen_height) {
		/* It doesn't fit, so we see if we have the minimum space needed. */
		if (req.height > screen_height
		    && *y - row_height > screen_height) {
			/* We don't, so we show the popup above the cell
			   instead of below it. */
			screen_height = *y - row_height;
			*y -= (req.height + row_height);
			if (*y < 0)
				*y = 0;
		}
	}

	/* We try to line it up with the right edge of the column, but we don't
	   want it to go off the edges of the screen. */
	if (*x > screen_width)
		*x = screen_width;
	*x -= req.width;
	if (*x < 0)
		*x = 0;
}

static void
m_cell_date_hide_popup (MCellDate *mcd)
{
	gtk_grab_remove (mcd->popup_window);
	gtk_widget_hide (mcd->popup_window);
	E_CELL_POPUP (mcd)->popup_shown = FALSE;
}

static void
m_cell_date_show_popup (MCellDate *mcd, int row, int view_col)
{
	gint x, y;

	m_cell_date_get_popup_pos (mcd, row, view_col, &x, &y);

	gtk_widget_set_uposition (mcd->popup_window, x, y);
	gtk_widget_realize (mcd->popup_window);
	gtk_widget_show (mcd->popup_window);

	E_CELL_POPUP (mcd)->popup_shown = TRUE;
}

static gint 
m_cell_date_do_popup (ECellPopup *ecp,
		      GdkEvent   *event,
		      int         row,
		      int         view_col)
{
	MCellDate *mcd = M_CELL_DATE (ecp);

	m_cell_date_show_popup (mcd, row, view_col);
	m_cell_date_set_calendar_date (mcd);

	gtk_grab_add (mcd->popup_window);

	return TRUE;
}

/* This handles key press events in the popup window. If the Escape key is
   pressed we hide the popup, and do not change the cell contents. */
static int
m_cell_date_key_press			(GtkWidget	*popup_window,
					 GdkEventKey	*event,
					 MCellDate	*mcd)
{
	/* If the Escape key is pressed we hide the popup. */
	if (event->keyval != GDK_Escape
	    && event->keyval != GDK_Return
	    && event->keyval != GDK_KP_Enter
	    && event->keyval != GDK_ISO_Enter
	    && event->keyval != GDK_3270_Enter)
		return FALSE;

	m_cell_date_hide_popup (mcd);

	if (event->keyval != GDK_Escape)
		m_cell_date_update_cell (mcd);

	/*m_cell_date_restart_edit (mcd);*/

	return TRUE;
}

static void
m_cell_date_set_calendar_date (MCellDate *mcd)
{
	ECellPopup  *ecp = E_CELL_POPUP (mcd);
	ECellView   *ecv = (ECellView*) ecp->popup_cell_view;
	ETableItem  *eti = E_TABLE_ITEM (ecv->e_table_item_view);
	ETableCol   *ecol;
	time_t       time;
	guint        year, month, day;
	
	ecol = e_table_header_get_column (eti->header, ecp->popup_view_col);

	time = (time_t) e_table_model_value_at (ecv->e_table_model,
						ecol->col_idx,
						ecp->popup_row);
	
	time_split (time, &year, &month, &day);
	
	gtk_calendar_select_month (GTK_CALENDAR (mcd->calendar), month, year);
	gtk_calendar_select_day (GTK_CALENDAR (mcd->calendar), day);
}

static void
m_cell_date_update_cell (MCellDate *mcd)
{
	ECellPopup *ecp = E_CELL_POPUP (mcd);
	ECellView  *ecv = (ECellView*) ecp->popup_cell_view;
	ETableItem *eti = E_TABLE_ITEM (ecv->e_table_item_view);
	ETableCol  *ecol;
	guint       year, month, day;
	time_t      time;
	
	ecol = e_table_header_get_column (eti->header, ecp->popup_view_col);
	
	gtk_calendar_get_date (GTK_CALENDAR (mcd->calendar),
			       &year,
			       &month,
			       &day);

	/* FIXME: We should update finish too, so that the task is moved,
	 * and not just the start time being changed.
	 */
	time = time_from_day (year, month, day);
	e_table_model_set_value_at (ecv->e_table_model,
				    ecol->col_idx, ecp->popup_row,
				    (void *) time);
}

static void
m_cell_date_double_click (GtkCalendar *calendar, MCellDate *mcd)
{
	m_cell_date_update_cell (mcd);
	m_cell_date_hide_popup (mcd);
}

static void
m_cell_date_ok_clicked (GtkWidget *button, MCellDate *mcd)
{
	m_cell_date_update_cell (mcd);
	m_cell_date_hide_popup (mcd);
}

static void
m_cell_date_cancel_clicked (GtkWidget *button, MCellDate *mcd)
{
	m_cell_date_hide_popup (mcd);
}

static void
m_cell_date_today_clicked (GtkWidget *button, MCellDate *mcd)
{
	/* FIXME: Go to today. */
	
	m_cell_date_update_cell (mcd);
	m_cell_date_hide_popup (mcd);
}


 
