/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Mikael Hallendal <micke@codefactory.se>
 * Copyright (C) 2001 Richard Hult     <rhult@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Mikael Hallendal
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <bonobo/bonobo-listener.h>
#include <bonobo/bonobo-shlib-factory.h>
#include <liboaf/liboaf.h>
#include "util/id-map.h"
#include "util/type-utils.h"
#include "util/corba-utils.h"
#include "libmrproject/GNOME_MrProject.h"
#include "client/manager-clients/resource-manager-client.h"
#include "client/manager-clients/task-manager-client.h"
#include "client/manager-clients/allocation-manager-client.h"
#include "task-dialog.h"
#include "task-dialog-gui.h"

#include "util/debug.h"

static void      task_dialog_init           (TaskDialog         *dialog);
static void      task_dialog_class_init     (TaskDialogClass    *klass);

static void      td_set_initial_resources   (TaskDialog         *dialog);
static void      td_set_initial_allocations (TaskDialog         *dialog);

static void      td_gui_destroyed_cb        (GtkWidget          *widget,
					     BonoboControl      *control);

/* Called when something changes in the GUI */
static void      td_gui_task_updated_cb     (TaskDialogGui      *gui,
					     GM_Task            *task,
					     gpointer            user_data);

static void td_gui_resource_allocated_cb    (TaskDialogGui      *gui,
					     GM_Id               task_id,
					     GM_Id               res_id,
					     gpointer            user_data);
static void td_gui_resource_deallocated_cb  (TaskDialogGui      *gui,
					     GM_Id               task_id,
					     GM_Id               res_id,
					     gpointer            user_data);
static void      td_gui_note_changed_cb     (TaskDialogGui      *gui,
					     GM_Id               task_id,
					     gchar              *text,
					     gpointer            user_data);
/* Called when tasks get updated in the server */
static void      td_task_updated_cb         (TaskDialog         *dialog, 
					     GM_Task            *task,
					     gpointer            user_data);

static void      td_tasks_removed_cb        (TaskDialog         *dialog, 
					     GSList             *tasks,
					     gpointer            user_data);

static void      td_note_changed_cb         (TaskDialog         *dialog,
					     GM_Id               task_id,
					     gchar              *note);

/* Called when resources get updated in the server */

static void      td_resource_inserted_cb    (TaskDialog         *dialog,
					     GM_Resource        *resource,
					     gpointer            user_data);

static void      td_resource_updated_cb     (TaskDialog         *dialog,
					     GM_Resource        *resource,
					     gpointer            user_data);

static void      td_resources_removed_cb    (TaskDialog         *dialog,
					     GSList             *resource,
					     gpointer            user_data);

static void      td_allocation_added_cb     (TaskDialog         *dialog,
					     GM_Allocation      *allocation);

static void      td_allocations_removed_cb  (TaskDialog         *dialog, 
					     GSList             *allocations);
static void      td_allocations_clear_cb    (TaskDialog         *dialog);


#define PARENT_TYPE BONOBO_X_OBJECT_TYPE
static GtkObjectClass *parent_class;

struct _TaskDialogPriv {
	GtkWidget               *gui;
	GM_Id                    id;

	ResourceManagerClient   *rm_client;
	TaskManagerClient       *tm_client;
	AllocationManagerClient *am_client;
	
	IdMap                   *allocations;
};

static void
impl_GNOME_MrProject_TaskDialog_show (PortableServer_Servant  servant, 
				      CORBA_Environment      *ev)
{
	TaskDialog *dialog;

	d(puts(__FUNCTION__));

	dialog = TASK_DIALOG (bonobo_object_from_servant (servant));
}

static void
impl_GNOME_MrProject_TaskDialog_setTaskId (PortableServer_Servant  servant,
					   const GM_Id             id,
					   CORBA_Environment      *ev)
{
	TaskDialog     *dialog;
	TaskDialogPriv *priv;
	GM_Task        *task;
	gchar          *note;
	
	d(puts(__FUNCTION__));

	dialog = TASK_DIALOG (bonobo_object_from_servant (servant));
	priv   = dialog->priv;

	priv->id = id;
	task = task_mc_get_task (priv->tm_client, id, ev);
	note = task_mc_get_note (priv->tm_client, id, ev);

	if (!BONOBO_EX (ev) && task) {
		task_dialog_gui_update_task (TASK_DIALOG_GUI (priv->gui), 
					     task);
		task_dialog_gui_update_note (TASK_DIALOG_GUI (priv->gui),
					     note);
	}
	
   	td_set_initial_resources (dialog); 
   	td_set_initial_allocations (dialog);   
}

static void
impl_GNOME_MrProject_TaskDialog_showPage (PortableServer_Servant  servant,
					  GM_TaskDialogPage       page,
					  CORBA_Environment      *ev)
{
	TaskDialog     *dialog;
	TaskDialogPriv *priv;
	
	d(puts(__FUNCTION__));

	dialog = TASK_DIALOG (bonobo_object_from_servant (servant));
	priv   = dialog->priv;

	task_dialog_gui_show_page (TASK_DIALOG_GUI (priv->gui), page);
}

static void
impl_GNOME_MrProject_TaskDialog_setShell (PortableServer_Servant  servant,
					  const GM_Shell          shell,
					  CORBA_Environment      *ev)
{
	TaskDialog     *dialog;
	TaskDialogPriv *priv;

	d(puts (__FUNCTION__));

	dialog = TASK_DIALOG (bonobo_object_from_servant (servant));
	priv   = dialog->priv;

	priv->rm_client = RESOURCE_MANAGER_CLIENT (resource_mc_new (shell, 
								    TRUE, 
								    ev));
	priv->tm_client = TASK_MANAGER_CLIENT (task_mc_new (shell, TRUE, ev));
	priv->am_client = ALLOCATION_MANAGER_CLIENT (allocation_mc_new (shell,
									TRUE, 
									ev));

	gtk_signal_connect_object (GTK_OBJECT (priv->rm_client),
				   "resource_inserted",
				   GTK_SIGNAL_FUNC (td_resource_inserted_cb),
				   GTK_OBJECT (dialog));

	gtk_signal_connect_object (GTK_OBJECT (priv->rm_client),
				   "resource_updated",
				   GTK_SIGNAL_FUNC (td_resource_updated_cb),
				   GTK_OBJECT (dialog));

	gtk_signal_connect_object (GTK_OBJECT (priv->rm_client),
				   "resources_removed",
				   GTK_SIGNAL_FUNC (td_resources_removed_cb),
				   GTK_OBJECT (dialog));

	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "task_updated",
				   GTK_SIGNAL_FUNC (td_task_updated_cb),
				   GTK_OBJECT (dialog));
	
	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "tasks_removed",
				   GTK_SIGNAL_FUNC (td_tasks_removed_cb),
				   GTK_OBJECT (dialog));

	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "note_changed",
				   GTK_SIGNAL_FUNC (td_note_changed_cb),
				   GTK_OBJECT (dialog));

	gtk_signal_connect_object (GTK_OBJECT (priv->am_client),
				   "allocation_added",
				   GTK_SIGNAL_FUNC (td_allocation_added_cb),
				   GTK_OBJECT (dialog));

	gtk_signal_connect_object (GTK_OBJECT (priv->am_client),
				   "allocations_removed",
				   GTK_SIGNAL_FUNC (td_allocations_removed_cb),
				   GTK_OBJECT (dialog));

	gtk_signal_connect_object  (GTK_OBJECT (priv->am_client),
				   "items_clear",
				   GTK_SIGNAL_FUNC (td_allocations_clear_cb),
				   GTK_OBJECT (dialog));
}

static void
task_dialog_destroy (GtkObject *object)
{
	TaskDialog        *dialog;
	TaskDialogPriv    *priv;
	CORBA_Environment  ev;

	dialog  = TASK_DIALOG (object);
	priv    = dialog->priv;

	d(puts(__FUNCTION__));

	CORBA_exception_init (&ev);
	
	gtk_object_destroy (GTK_OBJECT (priv->rm_client));
	gtk_object_destroy (GTK_OBJECT (priv->tm_client));
	gtk_object_destroy (GTK_OBJECT (priv->am_client));

	(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
task_dialog_class_init (TaskDialogClass *klass)
{
	POA_GNOME_MrProject_TaskDialog__epv *epv;
	GtkObjectClass                      *object_class;

	d(puts(__FUNCTION__));

	epv                   = &klass->epv;
	object_class          = (GtkObjectClass *) klass;
	
	parent_class          = gtk_type_class (PARENT_TYPE);

	object_class->destroy = task_dialog_destroy;

	epv->show             = impl_GNOME_MrProject_TaskDialog_show;
	epv->setTaskId        = impl_GNOME_MrProject_TaskDialog_setTaskId;
	epv->showPage         = impl_GNOME_MrProject_TaskDialog_showPage;
	epv->setShell         = impl_GNOME_MrProject_TaskDialog_setShell;
}

static void 
task_dialog_init (TaskDialog *dialog)
{
	TaskDialogPriv *priv;
	
	d(puts(__FUNCTION__));

	priv              = g_new0 (TaskDialogPriv, 1);
	dialog->priv      = priv;
	priv->id          = -1;
	priv->allocations = id_map_new (1);
}

BONOBO_X_TYPE_FUNC_FULL (TaskDialog, 
			 GNOME_MrProject_TaskDialog,
			 PARENT_TYPE,
			 task_dialog);

static void
td_task_updated_cb (TaskDialog *dialog, GM_Task *task, gpointer user_data)
{
	g_return_if_fail (dialog != NULL);
	g_return_if_fail (IS_TASK_DIALOG (dialog));
	g_return_if_fail (task != NULL);

	d(puts(__FUNCTION__));

	if (task->taskId != dialog->priv->id) {
		return;
	}
	
	task_dialog_gui_update_task (TASK_DIALOG_GUI (dialog->priv->gui),
				     task);
}

static void
td_tasks_removed_cb (TaskDialog *dialog, GSList *tasks, gpointer user_data)
{
	TaskDialogPriv *priv;
	GSList         *node;
	GM_Id           id;
	
	g_return_if_fail (dialog != NULL);
	g_return_if_fail (IS_TASK_DIALOG (dialog));
	g_return_if_fail (tasks != NULL);
	
	d(puts(__FUNCTION__));

	priv = dialog->priv;

	for (node = tasks; node; node = node->next) {
		id = GPOINTER_TO_INT (node->data);
		if (id == priv->id) {
			gtk_widget_destroy (GTK_WIDGET (priv->gui));
		}
	}
}

static void
td_note_changed_cb (TaskDialog *dialog, GM_Id task_id, gchar *note)
{
	TaskDialogPriv *priv;

	g_return_if_fail (dialog != NULL);
	g_return_if_fail (IS_TASK_DIALOG (dialog));
	g_return_if_fail (note != NULL);
	
	priv = dialog->priv;

	if (task_id != priv->id) {
		return;
	}

	task_dialog_gui_update_note (TASK_DIALOG_GUI (priv->gui), note);
}

static void
td_resource_inserted_cb (TaskDialog  *dialog,
			 GM_Resource *resource,
			 gpointer     user_data)
{
	g_return_if_fail (dialog != NULL);
	g_return_if_fail (IS_TASK_DIALOG (dialog));

	d(puts(__FUNCTION__));

	task_dialog_gui_add_resource (TASK_DIALOG_GUI (dialog->priv->gui), 
				      resource);
}


static void      
td_resource_updated_cb (TaskDialog  *dialog, 
			GM_Resource *resource, 
			gpointer     user_data)
{
	g_return_if_fail (dialog != NULL);
	g_return_if_fail (IS_TASK_DIALOG (dialog));

	d(puts(__FUNCTION__));

	task_dialog_gui_update_resource (TASK_DIALOG_GUI (dialog->priv->gui), 
					 resource);
}


static void      
td_resources_removed_cb (TaskDialog *dialog,
			 GSList     *resources,
			 gpointer    user_data)
{
	TaskDialogPriv *priv;
	GSList         *node;
	GM_Id           id;
	
	g_return_if_fail (dialog != NULL);
	g_return_if_fail (IS_TASK_DIALOG (dialog));

	d(puts(__FUNCTION__));

	priv = dialog->priv;
	
	for (node = resources; node; node = node->next) {
		id = GPOINTER_TO_INT (node->data);
		id_map_remove (priv->allocations, id);
		task_dialog_gui_remove_resource (TASK_DIALOG_GUI (dialog->priv->gui),
						 id);
	}
}

static void
td_allocation_added_cb (TaskDialog *dialog, GM_Allocation *allocation)
{
	TaskDialogPriv *priv;

	g_return_if_fail (dialog != NULL);
	g_return_if_fail (IS_TASK_DIALOG (dialog));

	priv = dialog->priv;

	d(puts (__FUNCTION__));
	
	if (allocation->taskId == priv->id) {

		id_map_insert_id (priv->allocations,
				  allocation->resourceId,
				  corba_util_allocation_duplicate(allocation));
		
		task_dialog_gui_set_allocation (TASK_DIALOG_GUI (priv->gui),
						allocation->resourceId,
						TRUE);
	}
}

static void
td_allocations_clear_cb (TaskDialog *dialog)
{
	g_return_if_fail (dialog != NULL);
	g_return_if_fail (IS_TASK_DIALOG (dialog));

	d(g_print("%s: destroying myself\n", __FUNCTION__));

	gtk_widget_destroy (GTK_WIDGET (dialog->priv->gui));
}

static void
td_allocations_removed_cb (TaskDialog *dialog, GSList *allocations)
{
	TaskDialogPriv *priv;
	GSList         *node;
	GM_Allocation  *allocation;
	GM_Allocation  *tmp_allocation;
	
	g_return_if_fail (dialog != NULL);
	g_return_if_fail (IS_TASK_DIALOG (dialog));

	d(puts(__FUNCTION__));

	priv = dialog->priv;

	if (allocations) {
		for (node = allocations; node; node = node->next) {
			allocation = (GM_Allocation *) node->data;

			d(g_print ("GOT A REMOVE_REQUEST [%d <-> %d]\n",
				   allocation->taskId, 
				   allocation->resourceId));

			if (allocation->taskId == priv->id) {
				tmp_allocation = 
					id_map_lookup (priv->allocations,
						       allocation->resourceId);

				if (tmp_allocation) {

					id_map_remove (priv->allocations, 
						       allocation->resourceId);

					task_dialog_gui_set_allocation (TASK_DIALOG_GUI (priv->gui),
									allocation->resourceId,
									FALSE);
					CORBA_free (tmp_allocation);
				}
			}
		}
	}
}


static void
td_gui_task_updated_cb (TaskDialogGui *gui, GM_Task *task, gpointer user_data)
{
	TaskDialog        *dialog;
	TaskDialogPriv    *priv;
	CORBA_Environment  ev;

	g_return_if_fail (gui != NULL);
	g_return_if_fail (IS_TASK_DIALOG_GUI (gui));
	g_return_if_fail (task != NULL);
	g_return_if_fail (user_data != NULL);
	g_return_if_fail (IS_TASK_DIALOG (user_data));
	
	dialog = TASK_DIALOG (user_data);
	priv   = dialog->priv;

	d(puts(__FUNCTION__));

	CORBA_exception_init (&ev);

	/* FIXME: A workaround for the lack of logic in the server.
	 * Remove after M2.
	 */
	if (task->end <= task->start) {
		task->end = task->start + 8 * 60 * 60;
	}
	
 	task_mc_update_task (priv->tm_client, task->taskId, task, &ev);

	if (BONOBO_EX (&ev)) {
		g_warning ("Could not update task.");
	}

	CORBA_exception_free (&ev);
}

static void
td_set_initial_resources (TaskDialog *dialog)
{
	TaskDialogPriv    *priv;
	CORBA_Environment  ev;
	GSList            *resources;
	GSList            *node;
	
	priv = dialog->priv;

	d(puts(__FUNCTION__));

	CORBA_exception_init (&ev);

	resources = resource_mc_get_all_resources (priv->rm_client, &ev);

	for (node = resources; node; node = node->next) {
		task_dialog_gui_add_resource (TASK_DIALOG_GUI (priv->gui), 
					      (GM_Resource *) node->data);
		CORBA_free (node->data);
	}

	g_slist_free (resources);

	CORBA_exception_free (&ev);
}

static void
td_set_initial_allocations (TaskDialog *dialog)
{
	TaskDialogPriv    *priv;
	CORBA_Environment  ev;
	GSList            *allocations, *node;
	
	priv = dialog->priv;

	d(puts(__FUNCTION__));

	CORBA_exception_init (&ev);
	
	allocations = allocation_mc_get_allocations_by_task (priv->am_client,
							     priv->id,
							     &ev);

	if (BONOBO_EX (&ev)) {
		CORBA_exception_free (&ev);
		return;
	}
	
	for (node = allocations; node; node = node->next) {
		td_allocation_added_cb (dialog, (GM_Allocation *) node->data);
	}

	g_slist_free (allocations);

	CORBA_exception_free (&ev);
}

static void
td_gui_resource_allocated_cb (TaskDialogGui *gui,
			      GM_Id          task_id,
			      GM_Id          res_id,
			      gpointer       user_data)
{
	TaskDialog        *dialog;
	TaskDialogPriv    *priv;
 	CORBA_Environment  ev;

	g_return_if_fail (user_data != NULL);
	g_return_if_fail (IS_TASK_DIALOG (user_data));
	
	dialog = TASK_DIALOG (user_data);
	priv   = dialog->priv;
	
	d(g_print ("ALLOCATION: [%d <-> %d]\n", task_id, res_id));

	CORBA_exception_init (&ev);
	
	allocation_mc_allocate (priv->am_client, 
				task_id, res_id, 
				DEFAULT_UNITS,  /* defined in task-dialog.h */
				&ev);

	CORBA_exception_free (&ev);
}


static void 
td_gui_resource_deallocated_cb (TaskDialogGui *gui,
				GM_Id          task_id,
				GM_Id          res_id,
				gpointer       user_data)
{
	TaskDialog        *dialog;
	TaskDialogPriv    *priv;
	CORBA_Environment  ev;
	GM_Allocation     *allocation;

	g_return_if_fail (user_data != NULL);
	g_return_if_fail (IS_TASK_DIALOG (user_data));
	
	dialog = TASK_DIALOG (user_data);
	priv   = dialog->priv;
	
	d(g_print ("DEALLOCATION: [%d <-> %d]\n", task_id, res_id));

	allocation = (GM_Allocation *) id_map_lookup (priv->allocations,
						      res_id);
	if (!allocation) {
		return;
	}

	CORBA_exception_init (&ev);
		
	allocation_mc_deallocate (priv->am_client, allocation, &ev);

	CORBA_exception_free (&ev);
}

static void
td_gui_note_changed_cb (TaskDialogGui *gui,
			GM_Id          task_id,
			gchar         *text,
			gpointer       user_data)
{
	TaskDialog        *dialog;
	TaskDialogPriv    *priv;
	CORBA_Environment  ev;

	g_return_if_fail (user_data != NULL);
	g_return_if_fail (IS_TASK_DIALOG (user_data));
	
	dialog = TASK_DIALOG (user_data);
	priv   = dialog->priv;
	
	CORBA_exception_init (&ev);

	task_mc_set_note (priv->tm_client, task_id, text, &ev);

	CORBA_exception_free (&ev);
}

static void
td_gui_destroyed_cb (GtkWidget *widget, BonoboControl *control)
{
	d(puts (__FUNCTION__));
	
	bonobo_object_unref (BONOBO_OBJECT (control));
}

static BonoboControl *
task_dialog_control_new (void)
{
	TaskDialog     *task_dialog;
	TaskDialogPriv *priv;
	BonoboControl  *control;

	task_dialog = TASK_DIALOG (gtk_type_new (TASK_DIALOG_TYPE));
	priv        = task_dialog->priv;
		
	d(puts(__FUNCTION__));

	/* Create the dialog. */
	priv->gui = task_dialog_gui_new ();
	gtk_widget_show (priv->gui);
	
	control = bonobo_control_new (priv->gui);
	
	gtk_signal_connect (GTK_OBJECT (priv->gui),
			    "task_updated",
			    GTK_SIGNAL_FUNC (td_gui_task_updated_cb),
			    GTK_OBJECT (task_dialog));

	gtk_signal_connect (GTK_OBJECT (priv->gui),
			    "resource_allocated",
			    GTK_SIGNAL_FUNC (td_gui_resource_allocated_cb),
			    GTK_OBJECT (task_dialog));

	gtk_signal_connect (GTK_OBJECT (priv->gui),
			    "resource_deallocated",
			    GTK_SIGNAL_FUNC (td_gui_resource_deallocated_cb),
			    GTK_OBJECT (task_dialog));
	gtk_signal_connect (GTK_OBJECT (priv->gui),
			    "note_changed",
			    GTK_SIGNAL_FUNC (td_gui_note_changed_cb),
			    GTK_OBJECT (task_dialog));

	bonobo_object_add_interface (BONOBO_OBJECT (control),
				     BONOBO_OBJECT (task_dialog));
	
	gtk_signal_connect (GTK_OBJECT (priv->gui),
			    "destroy",
			    GTK_SIGNAL_FUNC (td_gui_destroyed_cb),
			    GTK_OBJECT (control));

	return control;
}

static BonoboObject *
task_dialog_control_factory (BonoboGenericFactory *this,
			     const char           *object_id,
			     void                 *data)
{
	g_return_val_if_fail (object_id != NULL, NULL);

	d(puts(__FUNCTION__));

	if (!strcmp (object_id, "OAFIID:GNOME_MrProject_TaskDialog")) {
		return BONOBO_OBJECT (task_dialog_control_new ());
	} else {
		g_warning ("Failing to manufacture a '%s'", object_id);
	}
	
	return NULL;
}

BONOBO_OAF_SHLIB_FACTORY_MULTI ("OAFIID:GNOME_MrProject_TaskDialogFactory",
				"Mr Project task dialog factory",
				task_dialog_control_factory,
				NULL);
