/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Mikael Hallendal <micke@codefactory.se>
 * Copyright (C) 2001 Richard Hult <rhult@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Mikael Hallendal <micke@codefactory.se>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <time.h> /* must be included before gnome-dateedit.h> */
#include <gal/widgets/e-unicode.h>
#include <gal/e-table/e-table.h>
#include <gal/e-table/e-table-extras.h>
#include <gal/e-table/e-table-scrolled.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtkentry.h>
#include <gtk/gtklabel.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtksignal.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-dateedit.h>
#include <glade/glade.h>
#include <stdio.h> /* Remove when puts goes away */
#include "alloc-table-model.h"
#include "util/type-utils.h"
#include "libmrproject/GNOME_MrProject.h"
#include "task-dialog-gui.h"

#include "util/debug.h"

static void      task_dialog_gui_init          (TaskDialogGui    *gui);
static void      task_dialog_gui_class_init    (GtkObjectClass   *klass);

static void      tdg_destroy                   (GtkObject        *object);

static gboolean  tdg_name_changed_cb           (GtkWidget        *object,
						GdkEventFocus    *focus_event,
						gpointer          user_data);
static void      tdg_date_time_changed_cb      (GnomeDateEdit    *date_edit,
						gpointer          user_data);
static void      tdg_complete_changed_cb       (GtkWidget        *sb,
						gpointer          user_data);
static void      tdg_priority_changed_cb       (GtkWidget        *sb,
						gpointer          user_data);

static void tdg_atm_resource_allocated_cb      (AllocTableModel  *atm,
						GM_Id             res_id,
						gpointer          user_data);
static void tdg_atm_resource_deallocated_cb    (AllocTableModel  *atm,
					        GM_Id             res_id,
						gpointer          user_data);
static void tdg_note_apply_cb                  (GtkWidget        *text,
						gpointer          user_data);

static void tdg_gtk_editable_set_text_utf8     (GtkEditable      *editable,
						const gchar      *text);
static void      tdg_update_labels             (TaskDialogGui    *gui);
static void      tdg_create_gui                (TaskDialogGui    *gui);
static void      tdg_connect_signals           (TaskDialogGui    *gui);


struct _TaskDialogGuiPriv {
	GM_Task     *task;
	
	gboolean     activate_complete;

	GtkWidget   *notebook;

	/* Page one, properties */
        GtkWidget   *label_id_general;
        GtkWidget   *entry_name;
        GtkWidget   *date_start;
        GtkWidget   *date_end;
        GtkWidget   *sb_complete;
        GtkWidget   *sb_priority;
	GtkWidget   *button_close;

	/* Page two, allocations */
        GtkWidget   *label_id_resources;
	GtkWidget   *alloc_table;

	ETableModel *atm;

	/* Page three, notes */
        GtkWidget   *label_id_notes;
        GtkWidget   *text_note;
        GtkWidget   *button_stamp;
	GtkWidget   *button_apply;
};

enum {
        TASK_UPDATED,
        DEPENDENCY_ADDED,
        DEPENDENCY_REMOVED,
        RESOURCE_ALLOCATED,
        RESOURCE_DEALLOCATED,
	NOTE_CHANGED,
        LAST_SIGNAL
};

static gint signals[LAST_SIGNAL] = { 0 };

GNOME_CLASS_BOILERPLATE (TaskDialogGui, task_dialog_gui,
                         GtkWindow, gtk_window);


static void
task_dialog_gui_init (TaskDialogGui *gui)
{
        TaskDialogGuiPriv *priv;
        
        priv      = g_new0 (TaskDialogGuiPriv, 1);
        gui->priv = priv;

	gtk_window_set_wmclass (GTK_WINDOW (gui),
				"Task Properties",
				"MrProject");
	gtk_window_set_title (GTK_WINDOW (gui),
			      _("Task Properties - MrProject"));
	
        tdg_create_gui (gui);
        tdg_connect_signals (gui);
}

static void
task_dialog_gui_class_init (GtkObjectClass *klass)
{
        g_return_if_fail (klass != NULL);
        g_return_if_fail (IS_TASK_DIALOG_GUI_CLASS (klass));
        
        /* GtkObject */
        klass->destroy = tdg_destroy;

        /* Signals */
        signals[TASK_UPDATED] = 
                gtk_signal_new ("task_updated",
                                GTK_RUN_LAST,
                                klass->type,
                                GTK_SIGNAL_OFFSET (TaskDialogGuiClass,
                                                   task_updated),
                                gtk_marshal_NONE__POINTER,
                                GTK_TYPE_NONE,
                                1, GTK_TYPE_POINTER);

        signals[DEPENDENCY_ADDED] =
                gtk_signal_new ("dependency_added",
                                GTK_RUN_LAST,
                                klass->type,
                                GTK_SIGNAL_OFFSET (TaskDialogGuiClass,
                                                   dependency_added),
                                gtk_marshal_NONE__INT_INT,
                                GTK_TYPE_NONE,
                                2, GTK_TYPE_INT, GTK_TYPE_INT);

	signals[DEPENDENCY_REMOVED] =
		gtk_signal_new ("dependency_removed",
				GTK_RUN_LAST,
				klass->type,
				GTK_SIGNAL_OFFSET (TaskDialogGuiClass,
						   dependency_removed),
				gtk_marshal_NONE__INT,
				GTK_TYPE_NONE,
				1, GTK_TYPE_INT);

	signals[RESOURCE_ALLOCATED] =
		gtk_signal_new ("resource_allocated",
				GTK_RUN_LAST,
				klass->type,
				GTK_SIGNAL_OFFSET (TaskDialogGuiClass,
						   resource_allocated),
				gtk_marshal_NONE__INT_INT,
				GTK_TYPE_NONE,
				2, GTK_TYPE_INT, GTK_TYPE_INT);

	signals[RESOURCE_DEALLOCATED] = 
		gtk_signal_new ("resource_deallocated",
				GTK_RUN_LAST,
				klass->type,
				GTK_SIGNAL_OFFSET (TaskDialogGuiClass,
						   resource_deallocated),
				gtk_marshal_NONE__INT_INT,
				GTK_TYPE_NONE,
				2, GTK_TYPE_INT, GTK_TYPE_INT);

	signals[NOTE_CHANGED] = 
		gtk_signal_new ("note_changed",
				GTK_RUN_LAST,
				klass->type,
				GTK_SIGNAL_OFFSET (TaskDialogGuiClass,
						   note_changed),
				gtk_marshal_NONE__INT_POINTER,
				GTK_TYPE_NONE,
				2, GTK_TYPE_INT, GTK_TYPE_POINTER);
	
	gtk_object_class_add_signals (klass, signals, LAST_SIGNAL);
}

static void
tdg_destroy (GtkObject *object)
{
	TaskDialogGui *gui;
	
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_TASK_DIALOG_GUI (object));

	gui = TASK_DIALOG_GUI (object);
}

static gboolean
tdg_name_changed_cb (GtkWidget     *widget,
		     GdkEventFocus *focus_event,
		     gpointer       user_data)
{
	TaskDialogGui *gui;
	GM_Task       *task;
	gchar         *text;
	
	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (GTK_IS_ENTRY (widget), FALSE);
	g_return_val_if_fail (user_data != NULL, FALSE);
	g_return_val_if_fail (IS_TASK_DIALOG_GUI (user_data), FALSE);
	
	gui  = TASK_DIALOG_GUI (user_data);
	task = gui->priv->task;

	text = e_utf8_gtk_entry_get_text (GTK_ENTRY (widget));
	
	if (strcmp (task->name, text)) {
		CORBA_free (task->name);
		task->name = CORBA_string_dup (text);
		tdg_update_labels (gui);
		gtk_signal_emit (GTK_OBJECT (gui), 
				 signals[TASK_UPDATED], 
				 task);
	}
	
	g_free (text);

	return FALSE;
}

static void
tdg_date_time_changed_cb (GnomeDateEdit *date_edit, gpointer user_data)
{
	TaskDialogGui     *gui;
	TaskDialogGuiPriv *priv;
	GM_Task           *task;
	time_t             time;
	gboolean           changed = FALSE;
	
	g_return_if_fail (date_edit != NULL);
	g_return_if_fail (GNOME_IS_DATE_EDIT (date_edit));
	g_return_if_fail (user_data != NULL);
	g_return_if_fail (IS_TASK_DIALOG_GUI (user_data));
	
	gui  = TASK_DIALOG_GUI (user_data);
	priv = gui->priv;
	task = priv->task;

	time = gnome_date_edit_get_date (date_edit);

	if (date_edit == GNOME_DATE_EDIT (priv->date_start)) {
		if (task->start != time) {
			changed = TRUE;
			task->start = time;
		}
	}
	else if (date_edit == GNOME_DATE_EDIT (gui->priv->date_end)) {
		if (task->end != time) {
			changed = TRUE;
			task->end = time;
		}
	}

	if (changed) {
		gtk_signal_emit (GTK_OBJECT (gui), 
				 signals[TASK_UPDATED], 
				 task);
	}
}

static void
tdg_complete_changed_cb (GtkWidget *sb, gpointer user_data)
{
	TaskDialogGui     *gui;
	TaskDialogGuiPriv *priv;

	g_return_if_fail (sb != NULL);
	g_return_if_fail (GTK_IS_SPIN_BUTTON (sb));
	g_return_if_fail (user_data != NULL);
	g_return_if_fail (IS_TASK_DIALOG_GUI (user_data));

	d(puts (__FUNCTION__));
	
	gui  = TASK_DIALOG_GUI (user_data);
	priv = gui->priv;

	priv->task->percentComplete = 
		gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (sb));

	gtk_signal_emit (GTK_OBJECT (gui),
			 signals[TASK_UPDATED],
			 priv->task);
}

static void
tdg_priority_changed_cb (GtkWidget *sb, gpointer data)
{
	/* FIX: Implement */
}

static gint
tdg_key_event (GtkWidget   *widget,
	       GdkEventKey *event,
	       GtkWidget   *close_button)
{
	if (event->keyval == GDK_Escape) {
		gtk_button_clicked (GTK_BUTTON (close_button));
		return TRUE;
	} else {
		return FALSE;
	}
}

static void
tdg_atm_resource_allocated_cb (AllocTableModel *atm,
			       GM_Id            res_id,
			       gpointer         user_data)
{
	TaskDialogGui *gui;
	
	g_return_if_fail (user_data != NULL);
	g_return_if_fail (IS_TASK_DIALOG_GUI (user_data));
	
	gui = TASK_DIALOG_GUI (user_data);

	gtk_signal_emit (GTK_OBJECT (gui), 
			 signals[RESOURCE_ALLOCATED],
			 gui->priv->task->taskId,
			 res_id);
}

static void
tdg_atm_resource_deallocated_cb (AllocTableModel *atm,
				 GM_Id            res_id,
				 gpointer         user_data)
{
	TaskDialogGui *gui;
	
	g_return_if_fail (user_data != NULL);
	g_return_if_fail (IS_TASK_DIALOG_GUI (user_data));
	
	gui = TASK_DIALOG_GUI (user_data);

	gtk_signal_emit (GTK_OBJECT (gui), 
			 signals[RESOURCE_DEALLOCATED],
			 gui->priv->task->taskId,
			 res_id);
}

static void
tdg_note_apply_cb (GtkWidget *button, gpointer user_data)
{
	TaskDialogGui     *gui;
	TaskDialogGuiPriv *priv;
	GM_Task           *task;
	gchar             *note;
	
	g_return_if_fail (user_data != NULL);
	g_return_if_fail (IS_TASK_DIALOG_GUI (user_data));
	
	gui  = TASK_DIALOG_GUI (user_data);
	priv = gui->priv;
	task = priv->task;

	note = e_utf8_gtk_editable_get_text (GTK_EDITABLE (priv->text_note));

	gtk_signal_emit (GTK_OBJECT (gui), 
			 signals[NOTE_CHANGED],
			 (gint) task->taskId,
			 note);

	g_free (note);
}

static void
tdg_note_stamp_clicked_cb (TaskDialogGui *gui, GtkWidget *button)
{
	TaskDialogGuiPriv *priv;
	time_t             t;
	struct tm         *tm;
	gchar              stamp[128];
	gint               position;
	
	g_return_if_fail (gui != NULL);
	g_return_if_fail (IS_TASK_DIALOG_GUI (gui));
	
	priv = gui->priv;

	t = time (NULL);
	tm = localtime (&t);
	
	strftime (stamp, sizeof (stamp), _("\n%a %d %b %Y, %I:%M %p\n"), tm);
		  
	position = gtk_text_get_length (GTK_TEXT (priv->text_note));
	
	gtk_editable_insert_text (GTK_EDITABLE (priv->text_note), stamp, 
				  strlen (stamp), &position);

	gtk_widget_grab_focus (priv->text_note);
}

static void
tdg_create_gui (TaskDialogGui *gui)
{
        GladeXML          *glade;
        TaskDialogGuiPriv *priv;
        GtkWidget         *window;
	GtkWidget         *contents;
	GtkWidget         *resource_box;
	
        priv = gui->priv;

        glade = glade_xml_new (MRPROJECT_DATADIR "task-dialog-gui.glade",
                               "task_dialog");
	
	window = glade_xml_get_widget (glade, "task_dialog");
	
	contents = glade_xml_get_widget (glade, "dialog_contents");

	gtk_widget_reparent (contents, GTK_WIDGET (gui));
	gtk_widget_destroy  (window);

	priv->notebook           = glade_xml_get_widget (glade, "notebook");
        priv->label_id_general   = glade_xml_get_widget (glade, 
							 "label_id_general");
        priv->label_id_resources = glade_xml_get_widget (glade, 
							 "label_id_resources");
        priv->label_id_notes     = glade_xml_get_widget (glade, 
							 "label_id_notes");
        priv->entry_name         = glade_xml_get_widget (glade, "entry_name");
	priv->date_start         = glade_xml_get_widget (glade, "date_start");
        priv->date_end           = glade_xml_get_widget (glade, "date_end");
        priv->sb_complete        = glade_xml_get_widget (glade, "sb_complete");
        priv->sb_priority        = glade_xml_get_widget (glade, "sb_priority");
	priv->button_close       = glade_xml_get_widget (glade, "button_close");
	priv->text_note          = glade_xml_get_widget (glade, "text_note");
	priv->button_stamp       = glade_xml_get_widget (glade, "button_stamp");
	priv->button_apply       = glade_xml_get_widget (glade, "button_apply");
	resource_box             = glade_xml_get_widget (glade, "resource_box");

	gtk_widget_set_usize (priv->entry_name, 150, -1);
	
	gtk_signal_connect (GTK_OBJECT (gui), "key_press_event",
			    GTK_SIGNAL_FUNC (tdg_key_event), 
			    priv->button_close);
	
	priv->atm = alloc_table_model_new ();
	
	priv->alloc_table = e_table_scrolled_new_from_spec_file (
		E_TABLE_MODEL (priv->atm),
		e_table_extras_new (),
		MRPROJECT_DATADIR "task-dialog-alloc-table.etspec",
		MRPROJECT_DATADIR "task-dialog-alloc-table.etstate");

	gtk_widget_show (priv->alloc_table);

	gtk_box_pack_start (GTK_BOX (resource_box), priv->alloc_table, TRUE, TRUE, 0);
	
	gtk_object_unref (GTK_OBJECT (glade));
}

static void
tdg_connect_signals (TaskDialogGui *gui)
{
        TaskDialogGuiPriv *priv;
        
        priv = gui->priv;

	gtk_signal_connect (GTK_OBJECT (priv->entry_name),
			    "focus_out_event",
			    GTK_SIGNAL_FUNC (tdg_name_changed_cb),
			    GTK_OBJECT (gui));

	gtk_signal_connect (GTK_OBJECT (priv->date_start),
			    "date_changed",
			    GTK_SIGNAL_FUNC (tdg_date_time_changed_cb),
			    GTK_OBJECT (gui));
	
	gtk_signal_connect (GTK_OBJECT (priv->date_start),
			    "time_changed",
			    GTK_SIGNAL_FUNC (tdg_date_time_changed_cb),
			    GTK_OBJECT (gui));

	gtk_signal_connect (GTK_OBJECT (priv->date_end),
			    "date_changed",
			    GTK_SIGNAL_FUNC (tdg_date_time_changed_cb),
			    GTK_OBJECT (gui));
	
	gtk_signal_connect (GTK_OBJECT (priv->date_end),
			    "time_changed",
			    GTK_SIGNAL_FUNC (tdg_date_time_changed_cb),
			    GTK_OBJECT (gui));
			    
	gtk_signal_connect (GTK_OBJECT (priv->sb_complete),
			    "changed",
			    GTK_SIGNAL_FUNC (tdg_complete_changed_cb),
			    GTK_OBJECT (gui));
	
	gtk_signal_connect (GTK_OBJECT (priv->sb_priority),
			    "changed",
			    GTK_SIGNAL_FUNC (tdg_priority_changed_cb),
			    GTK_OBJECT (gui));

	gtk_signal_connect (GTK_OBJECT (priv->atm),
			    "resource_allocated",
			    GTK_SIGNAL_FUNC (tdg_atm_resource_allocated_cb),
			    GTK_OBJECT (gui));

	gtk_signal_connect (GTK_OBJECT (priv->atm),
			    "resource_deallocated",
			    GTK_SIGNAL_FUNC (tdg_atm_resource_deallocated_cb),
			    GTK_OBJECT (gui));
	
	gtk_signal_connect_object (GTK_OBJECT (priv->button_close),
				   "clicked",
				   GTK_SIGNAL_FUNC (gtk_object_destroy),
				   GTK_OBJECT (gui));

	gtk_signal_connect (GTK_OBJECT (priv->button_apply),
			    "clicked",
			    GTK_SIGNAL_FUNC (tdg_note_apply_cb),
			    GTK_OBJECT (gui));

	gtk_signal_connect_object (GTK_OBJECT (priv->button_stamp),
				   "clicked",
				   GTK_SIGNAL_FUNC (tdg_note_stamp_clicked_cb),
				   GTK_OBJECT (gui));
}

static void
tdg_update_labels (TaskDialogGui *gui)
{
	TaskDialogGuiPriv *priv;
	gchar             *name;
	
	g_return_if_fail (IS_TASK_DIALOG_GUI (gui));
	
	priv = gui->priv;
	
	name = e_utf8_to_locale_string (priv->task->name);
	
	gtk_label_set_text (GTK_LABEL (priv->label_id_general), name);
	gtk_label_set_text (GTK_LABEL (priv->label_id_resources), name);
	gtk_label_set_text (GTK_LABEL (priv->label_id_notes), name);

	g_free (name);
}

static void
tdg_update_gui (TaskDialogGui *gui)
{
	TaskDialogGuiPriv *priv;
	GM_Task           *task;
	
	priv = gui->priv;
	task = priv->task;

	if (!task) {
		g_warning ("Trying to update GUI without a task");
		return;
	}

	tdg_update_labels (gui);
	
	e_utf8_gtk_entry_set_text (GTK_ENTRY (priv->entry_name), task->name);

	gnome_date_edit_set_time (GNOME_DATE_EDIT (priv->date_start), 
				  task->start);
	gnome_date_edit_set_time (GNOME_DATE_EDIT (priv->date_end), 
				  task->end);

	if (task->type == GNOME_MrProject_TASK_NORMAL) {
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (priv->sb_complete),
					   (gfloat) task->percentComplete);
		gtk_widget_set_sensitive (priv->sb_complete, TRUE);
		gtk_widget_set_sensitive (priv->date_start, TRUE);
		gtk_widget_set_sensitive (priv->date_end, TRUE);
	} else if (task->type == GNOME_MrProject_TASK_SUMMARY) {
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (priv->sb_complete),
					   (gfloat) 0);
		gtk_widget_set_sensitive (priv->sb_complete, FALSE);
		gtk_widget_set_sensitive (priv->date_start, FALSE);
		gtk_widget_set_sensitive (priv->date_end, FALSE);
	}
}

GtkWidget  *
task_dialog_gui_new (void)
{
	TaskDialogGui *gui;
	
	gui = gtk_type_new (TASK_DIALOG_GUI_TYPE);
	
	return GTK_WIDGET (gui);
}

void  
task_dialog_gui_update_task (TaskDialogGui *gui, GM_Task *task)
{
	TaskDialogGuiPriv *priv;
	
	g_return_if_fail (gui != NULL);
	g_return_if_fail (IS_TASK_DIALOG_GUI (gui));
	g_return_if_fail (task != NULL);
	
	priv = gui->priv;

	if (!priv->task) {
		priv->task = corba_util_task_duplicate (task);

		tdg_update_gui (gui);
		return;
	}

	if (priv->task->taskId != task->taskId) {
		return;
	}
	
	if (!corba_util_task_update (gui->priv->task, task, TASK_CHANGE_ALL)) {
		return;
	}

	tdg_update_gui (gui);
}

void  
task_dialog_gui_add_resource (TaskDialogGui *gui, GM_Resource *res)
{
	g_return_if_fail (gui != NULL);
	g_return_if_fail (IS_TASK_DIALOG_GUI (gui));
	g_return_if_fail (res != NULL);

	alloc_table_model_add_resource (ALLOC_TABLE_MODEL (gui->priv->atm),
					res);
}

void 
task_dialog_gui_update_resource (TaskDialogGui *gui, GM_Resource *res)
{
	g_return_if_fail (gui != NULL);
	g_return_if_fail (IS_TASK_DIALOG_GUI (gui));
	g_return_if_fail (res != NULL);

	alloc_table_model_update_resource (ALLOC_TABLE_MODEL (gui->priv->atm), 
					   res);
}

void  
task_dialog_gui_remove_resource (TaskDialogGui *gui, GM_Id id)
{
	g_return_if_fail (gui != NULL);
	g_return_if_fail (IS_TASK_DIALOG_GUI (gui));

	alloc_table_model_remove_resource (ALLOC_TABLE_MODEL (gui->priv->atm),
					   id);
}

void
task_dialog_gui_set_allocation (TaskDialogGui *gui, GM_Id id, gboolean alloc)
{
	g_return_if_fail (gui != NULL);
	g_return_if_fail (IS_TASK_DIALOG_GUI (gui));

	alloc_table_model_set_allocation (ALLOC_TABLE_MODEL (gui->priv->atm),
					  id, alloc);
}

void
task_dialog_gui_add_group (TaskDialogGui *gui, GM_ResourceGroup *group)
{
	g_return_if_fail (gui != NULL);
	g_return_if_fail (IS_TASK_DIALOG_GUI (gui));
	g_return_if_fail (group != NULL);
}

void  
task_dialog_gui_remove_group (TaskDialogGui *gui, GM_Id gid)
{
	g_return_if_fail (gui != NULL);
	g_return_if_fail (IS_TASK_DIALOG_GUI (gui));
}

void
task_dialog_gui_update_note (TaskDialogGui *gui, gchar *note)
{
	TaskDialogGuiPriv *priv;
	gint               position;
	
	g_return_if_fail (gui != NULL);
	g_return_if_fail (IS_TASK_DIALOG_GUI (gui));
	g_return_if_fail (note != NULL);

	priv = gui->priv;

        /* Prevent a signal loop. */
	gtk_signal_handler_block_by_func (GTK_OBJECT (priv->button_apply), 
					  tdg_note_apply_cb, gui);

	position = gtk_editable_get_position (GTK_EDITABLE (priv->text_note));
	tdg_gtk_editable_set_text_utf8 (GTK_EDITABLE (priv->text_note), note);
	gtk_editable_set_position (GTK_EDITABLE (priv->text_note), position);

	gtk_signal_handler_unblock_by_func (GTK_OBJECT (priv->button_apply), 
					    tdg_note_apply_cb, gui);
}

void
task_dialog_gui_show_page (TaskDialogGui     *gui,
			   GM_TaskDialogPage  page)
{
	switch (page) {
	case GNOME_MrProject_PAGE_GENERAL:
		gtk_notebook_set_page (GTK_NOTEBOOK (gui->priv->notebook),
				       0);
		break;

	case GNOME_MrProject_PAGE_RESOURCES:
		gtk_notebook_set_page (GTK_NOTEBOOK (gui->priv->notebook),
				       1);
		break;

	case GNOME_MrProject_PAGE_NOTES:
		gtk_notebook_set_page (GTK_NOTEBOOK (gui->priv->notebook),
				       2);
		break;

	default:
		g_assert_not_reached ();
	}
}

static void
tdg_gtk_editable_set_text_utf8 (GtkEditable *editable, const gchar *text)
{
	gint   position = 0;
	gchar *tmp;

	gtk_editable_delete_text(editable, 0, -1);
	if (text) {
		tmp = e_utf8_to_gtk_string (GTK_WIDGET (editable), text);
		gtk_editable_insert_text (editable, tmp, strlen (tmp), &position);
	}
}






