/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 Richard Hult <rhult@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <bonobo/bonobo-shlib-factory.h>

#include "util/corba-utils.h"
#include "libmrproject/GNOME_MrProject.h"
#include "client/manager-clients/task-manager-client.h"
#include "network-canvas.h"
#include "network-component.h"

#define DEBUG 0
#include "util/debug.h"

#define OAFIID "OAFIID:GNOME_MrProject_NetworkComponent"

static void network_component_init       (NetworkComponent       *component);
static void network_component_class_init (NetworkComponentClass  *klass);
static void nc_task_inserted_cb          (NetworkComponent       *component,
					  GM_Task                *task,
					  GM_Id                   sibling_id,
					  GM_TaskOrderType        type,
					  TaskManagerClient      *mc);
static void nc_tasks_removed_cb          (NetworkComponent       *component,
					  GSList                 *tasks,
					  TaskManagerClient      *mc);
static void nc_task_updated_cb           (NetworkComponent       *component,
					  GM_Task                *task,
					  TaskManagerClient      *mc);
static void nc_task_reparented_cb        (NetworkComponent       *component,
					  GM_Id                   task_id,
					  GM_Id                   new_parent_id,
					  TaskManagerClient      *mc);
static void nc_tasks_linked_cb           (NetworkComponent       *component,
					  GM_Dependency          *dependency,
					  TaskManagerClient      *mc);
static void nc_tasks_unlinked_cb         (NetworkComponent       *component,
					  GM_Dependency          *dependency,
					  TaskManagerClient      *mc);
static void nc_items_clear_cb            (NetworkComponent       *component,
					  TaskManagerClient      *mc);
static void nc_load_begin_cb             (NetworkComponent       *component,
					  TaskManagerClient      *mc);
static void nc_load_end_cb               (NetworkComponent       *component,
					  TaskManagerClient      *mc);
static BonoboControl *nc_control_new     (NetworkComponent       *component);
static void nc_sort                      (NetworkComponent       *component);

#define PARENT_TYPE SHELL_COMPONENT_TYPE
static GtkObjectClass *parent_class;

/* Private members. */
struct _NetworkComponentPriv {
	BonoboControl     *control;
	GtkWidget         *network_canvas;
	TaskManagerClient *tm_client;
	guint              sort_idle_id;
	gboolean           loading;
};

static gboolean
network_component_add_to_shell (ShellComponent *component, 
				GM_Shell        shell)
{
	CORBA_Environment     ev;
	NetworkComponentPriv *priv;
	
	g_return_val_if_fail (component != NULL, FALSE);
	g_return_val_if_fail (IS_NETWORK_COMPONENT (component), FALSE);

	d(puts(__FUNCTION__));

	priv = NETWORK_COMPONENT (component)->priv;

	CORBA_exception_init (&ev);

	/* Create a control. */
	priv->control = nc_control_new (NETWORK_COMPONENT (component));

	GNOME_MrProject_Shell_addControl (component->shell,
					  BONOBO_OBJREF (component),
					  BONOBO_OBJREF (priv->control),
					  OAFIID,
					  _("Network"),
					  MRPROJECT_IMAGEDIR "network.png",
					  NULL);
	
	priv->tm_client = TASK_MANAGER_CLIENT (task_mc_new (shell, FALSE, &ev));
	
	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "task_inserted",
				   GTK_SIGNAL_FUNC (nc_task_inserted_cb),
				   GTK_OBJECT (component));
	
	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "tasks_removed",
				   GTK_SIGNAL_FUNC (nc_tasks_removed_cb),
				   GTK_OBJECT (component));

 	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "task_updated",
				   GTK_SIGNAL_FUNC (nc_task_updated_cb),
				   GTK_OBJECT (component));
	
	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "task_reparented",
				   GTK_SIGNAL_FUNC (nc_task_reparented_cb),
				   GTK_OBJECT (component));
	
	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "tasks_linked",
				   GTK_SIGNAL_FUNC (nc_tasks_linked_cb),
				   GTK_OBJECT (component));
	
	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "tasks_unlinked",
				   GTK_SIGNAL_FUNC (nc_tasks_unlinked_cb),
				   GTK_OBJECT (component));

	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "items_clear",
				   GTK_SIGNAL_FUNC (nc_items_clear_cb),
				   GTK_OBJECT (component));

	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "load_begin",
				   GTK_SIGNAL_FUNC (nc_load_begin_cb),
				   GTK_OBJECT (component));

	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "load_end",
				   GTK_SIGNAL_FUNC (nc_load_end_cb),
				   GTK_OBJECT (component));
	
	CORBA_exception_free (&ev);

	return TRUE;
}

static void
network_component_remove_from_shell (ShellComponent *component)
{
	NetworkComponentPriv *priv;
	
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_NETWORK_COMPONENT (component));

	d(puts(__FUNCTION__));
	
	priv = (NETWORK_COMPONENT (component))->priv;

	if (priv->tm_client) {
		gtk_object_destroy (GTK_OBJECT (priv->tm_client));
		priv->tm_client = NULL;
	}
}

static void
network_component_set_debug_flags (ShellComponent *component, 
				   glong           flags)
{
	NetworkComponent     *network_component;
	NetworkComponentPriv *priv;

	d(puts(__FUNCTION__));

	network_component = NETWORK_COMPONENT (component);
	priv              = network_component->priv;
}

static void
nc_canvas_realize (GtkWidget *widget, NetworkComponent *component)
{
	/* We need to sort on realize because there are no actual
	 * canvas items created before realized so we don't know their
	 * dimensions.
	 */
	nc_sort (component);
}
	
static BonoboControl *
nc_control_new (NetworkComponent *component)
{
	NetworkComponentPriv *priv;
	BonoboControl        *control;
	GtkWidget            *sw;
	
	priv = component->priv;

	priv->network_canvas = network_canvas_new ();

	gtk_signal_connect (GTK_OBJECT (priv->network_canvas),
			    "realize",
			    nc_canvas_realize,
			    component);
	
	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);


	gtk_container_add (GTK_CONTAINER (sw), priv->network_canvas);
	
	control = bonobo_control_new (sw);
	
	gtk_widget_show_all (bonobo_control_get_widget (control));

	bonobo_control_set_automerge (control, FALSE);

	return control;
}

static void
network_component_destroy (GtkObject *object)
{
	NetworkComponent     *component;
	NetworkComponentPriv *priv;

	d(puts(__FUNCTION__));

	component = NETWORK_COMPONENT (object);
	priv      = component->priv;

	if (priv->sort_idle_id) {
		g_source_remove (priv->sort_idle_id);
		priv->sort_idle_id = 0;
	}

	if (priv->tm_client) {
		gtk_object_destroy (GTK_OBJECT (priv->tm_client));
		priv->tm_client = NULL;
	}
		
	g_free (priv);
	component->priv = NULL;

	(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
network_component_class_init (NetworkComponentClass *klass)
{
	GtkObjectClass      *object_class;
	ShellComponentClass *shell_component_class;

	object_class                             = (GtkObjectClass *) klass;
	shell_component_class                    = SHELL_COMPONENT_CLASS (klass);
	parent_class                             = gtk_type_class (PARENT_TYPE);

	object_class->destroy                    = network_component_destroy;

	shell_component_class->add_to_shell      = network_component_add_to_shell;
	shell_component_class->remove_from_shell = network_component_remove_from_shell;
	shell_component_class->set_debug_flags   = network_component_set_debug_flags;
}

static void
network_component_init (NetworkComponent *component)
{
	NetworkComponentPriv   *priv;
	
	priv = g_new0 (NetworkComponentPriv, 1);
	
	component->priv = priv;
}

static BonoboObject *
network_component_factory (BonoboGenericFactory *this,
			   const char           *object_id,
			   void                 *data)
{
	g_return_val_if_fail (object_id != NULL, NULL);

	if (!strcmp (object_id, OAFIID)) {
		return BONOBO_OBJECT (gtk_type_new (NETWORK_COMPONENT_TYPE));
	} else {
		g_warning ("Failing to manufacture a '%s'", object_id);
	}
	
	return NULL;
}

static void
nc_task_inserted_cb (NetworkComponent  *component,
		     GM_Task           *task,
		     GM_Id              sibling_id,
		     GM_TaskOrderType   type,
		     TaskManagerClient *mc)
{
	NetworkCanvas *network_canvas;

	network_canvas = NETWORK_CANVAS (component->priv->network_canvas);

	network_canvas_insert_task (network_canvas, task);

	nc_sort (component);
}

static void
nc_tasks_removed_cb (NetworkComponent    *component,
		     GSList              *tasks,
		     TaskManagerClient   *mc)
{
	NetworkCanvas *network_canvas;

	network_canvas = NETWORK_CANVAS (component->priv->network_canvas);

	d(puts (__FUNCTION__));
	
	network_canvas_remove_tasks (network_canvas, tasks);

	nc_sort (component);
}

static void
nc_task_updated_cb (NetworkComponent  *component,
		    GM_Task           *task,
		    TaskManagerClient *mc)
{
	NetworkCanvas *network_canvas;

	/*d(puts (__FUNCTION__));*/
	
	network_canvas = NETWORK_CANVAS (component->priv->network_canvas);

	network_canvas_update_task (network_canvas, task);
}

static void
nc_task_reparented_cb (NetworkComponent  *component,
		       GM_Id              task_id,
		       GM_Id              new_parent_id,
		       TaskManagerClient *mc)
{
	NetworkCanvas *network_canvas;

	network_canvas = NETWORK_CANVAS (component->priv->network_canvas);

	network_canvas_reparent_task (network_canvas, task_id, new_parent_id);

	nc_sort (component);
}

static void
nc_tasks_linked_cb (NetworkComponent  *component,
		    GM_Dependency     *dependency,
		    TaskManagerClient *mc)
{
	NetworkCanvas *network_canvas;

	network_canvas = NETWORK_CANVAS (component->priv->network_canvas);

	network_canvas_link (network_canvas, dependency);

	nc_sort (component);
}

static void
nc_tasks_unlinked_cb (NetworkComponent  *component,
		      GM_Dependency     *dependency,
		      TaskManagerClient *mc)
{
	NetworkCanvas *network_canvas;

	network_canvas = NETWORK_CANVAS (component->priv->network_canvas);

	network_canvas_unlink (network_canvas, dependency);

	nc_sort (component);
}

static void
nc_items_clear_cb (NetworkComponent  *component,
		   TaskManagerClient *mc)
{
	NetworkCanvas *canvas;

	d(puts (__FUNCTION__));

	if (component->priv->sort_idle_id) {
		g_source_remove (component->priv->sort_idle_id);
		component->priv->sort_idle_id = 0;
	}
	
	canvas = NETWORK_CANVAS (component->priv->network_canvas);
	network_canvas_clear (canvas);
}

static void
nc_load_begin_cb (NetworkComponent  *component,
		  TaskManagerClient *mc)
{
	component->priv->loading = TRUE;

	d(g_print ("**************** begin\n"));
	
	if (component->priv->sort_idle_id) {
		g_source_remove (component->priv->sort_idle_id);
		component->priv->sort_idle_id = 0;
	}
}

static void
nc_load_end_cb (NetworkComponent  *component,
		TaskManagerClient *mc)
{
	component->priv->loading = FALSE;

	d(g_print ("**************** end\n"));

	nc_sort (component);
}

static gboolean
sort_idle_cb (gpointer user_data)
{
	NetworkComponent  *component;
	CORBA_Object       mgr;
	CORBA_Environment  ev;
	GM_IdSeq          *sorted_ids;

	d(puts (__FUNCTION__));

	component = NETWORK_COMPONENT (user_data);

	if (component->priv->loading) {
		component->priv->sort_idle_id = 0;
		return FALSE;
	}
	
	mgr = task_mc_get_manager (component->priv->tm_client);
	
	CORBA_exception_init (&ev);
	sorted_ids = GNOME_MrProject_TaskManager_getAllTaskIdsSorted (mgr, &ev);
	if (BONOBO_EX (&ev)) {
		CORBA_exception_free (&ev);

		component->priv->sort_idle_id = 0;
		return FALSE;
	}
	CORBA_exception_free (&ev);
		
	network_canvas_sort (NETWORK_CANVAS (component->priv->network_canvas), sorted_ids);
	CORBA_free (sorted_ids);
	
	component->priv->sort_idle_id = 0;

	return FALSE;
}

static void
nc_sort (NetworkComponent *component)
{
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_NETWORK_COMPONENT (component));

	/* Note: This is kind of a hack. We don't schedule a sort if loading is TRUE.
	 * First, we don't need to sort while loading. Second, we can't make CORBA calls
	 * from the client to the engine during the load, or the tasks will be queued up
	 * and delivered in the reversed order (and the parents will be inserted after the
	 * children so nothing will work. I suspect that this is because the shlib
	 * components make oneway calls in event source not work.
	 */
	if (component->priv->loading) {
		return;
	}

	if (!component->priv->sort_idle_id) {
		component->priv->sort_idle_id = g_idle_add (sort_idle_cb, component);
	}
}

BONOBO_OAF_SHLIB_FACTORY_MULTI ("OAFIID:GNOME_MrProject_NetworkComponentFactory",
				"Mr Project network component factory",
				network_component_factory,
				NULL);

BONOBO_X_TYPE_FUNC (NetworkComponent, 
		    PARENT_TYPE,
		    network_component);
