/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Richard Hult <rhult@codefactory.se>
 * Copyright (C) 2001 Mikael Hallendal <micke@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <math.h>
#include <string.h>
#include <gtk/gtksignal.h>
#include <libgnomeui/gnome-canvas.h>
#include <libgnomeui/gnome-canvas-util.h>
#include <gal/widgets/e-font.h>

#include "util/time-utils.h"
#include "util/type-utils.h"
#include "util/corba-utils.h"
#include "util/marshallers.h"

#include "gantt-model.h"
#include "gantt-scale.h"
#include "gantt-row-item.h"

/* FIXME: Move to a common file. */
#define TEXT_PAD 4

static const char *left_bracket_xpm[] = {
"4 6 2 1",
" 	c None",
".	c #000000",
"....",
"....",
"....",
"... ",
"..  ",
".   "};

static const char *right_bracket_xpm[] = {
"4 6 2 1",
" 	c None",
".	c #000000",
"....",
"....",
"....",
" ...",
"  ..",
"   ."};


/* Arguments. */
enum {
	ARG_0,
	ARG_TASK,
	ARG_LEAF,
	ARG_X,
	ARG_Y,
	ARG_WIDTH,
	ARG_HEIGHT,
	ARG_SELECTED
};

typedef enum {
	DRAG_NONE		= 0,
	DRAG_PRE_LEFT   	= 1 << 0,
	DRAG_PRE_RIGHT		= 1 << 1,
	DRAG_PRE_MOVE	 	= 1 << 2,
	DRAG_PRE_MASK		= 0x07,
	DRAG_LEFT		= 1 << 3,
	DRAG_RIGHT	        = 1 << 4,
	DRAG_MOVE		= 1 << 5,
} DragState;

/* Private members. */
struct _GanttRowItemPriv {
	guint	      leaf : 1;
	guint	      selected : 1;
	gint	      idle_update;
	GdkGC	     *fill_gc;
	GdkGC        *fill_complete_gc;
	GM_Task      *task;
	gdouble	      x, y;
	gdouble	      width;
	gdouble	      height;
	gdouble       label_width;

	DragState     state;
	
	GSList       *resources;
	gchar	     *resources_text;
};

typedef struct {
	gint   resource_id;
	gchar *name;
} AssignedResource;

enum {
	CHANGED,
	CLICKED,
	LAST_SIGNAL
};

/* GtkObject. */
static void gantt_row_item_class_init (GanttRowItemClass *klass);
static void gantt_row_item_init       (GanttRowItem      *item);
static void draw_bracket              (GdkDrawable       *drawable,
				       gint               y1,
				       gint               x1,
				       gint               x2,
				       gboolean           left,
				       gboolean           right);

static GdkGC     *frame_gc, *text_gc;
static GdkPixbuf *left_bracket_pixbuf, *right_bracket_pixbuf;
static EFont	 *font;

GNOME_CLASS_BOILERPLATE (GanttRowItem,
			 gantt_row_item,
			 GnomeCanvasItem,
			 gnome_canvas_item);

static guint signals[LAST_SIGNAL] = { 0 };


static void
gantt_row_item_destroy (GtkObject *object)
{
	GanttRowItem *gantt;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GANTT_ROW_ITEM (object));

	gantt = GANTT_ROW_ITEM (object);

	if (gantt->priv->idle_update) {
		g_source_remove (gantt->priv->idle_update);
		gantt->priv->idle_update = 0;
	}

	g_free (gantt->priv->resources_text);
	
	g_free (gantt->priv);
	gantt->priv = NULL;

	GNOME_CALL_PARENT_HANDLER (GTK_OBJECT_CLASS, destroy, (object));
}

static gboolean
real_idle_update (gpointer data)
{
	GnomeCanvasItem *item = data;
	GanttRowItem    *row = data;

	g_return_val_if_fail (row != NULL, FALSE);

	gtk_signal_emit (GTK_OBJECT (row), 
			 signals[CHANGED], 
			 row->priv->x,
			 row->priv->x + row->priv->width,
			 row->priv->y,
			 row->priv->y + row->priv->height);
	
	gnome_canvas_item_request_update (item);
	row->priv->idle_update = 0;
	return FALSE;
}

static void
idle_update (GanttRowItem *item)
{
	if (item->priv->idle_update != 0) {
		return;
	}

	item->priv->idle_update = gtk_idle_add_priority (GTK_PRIORITY_DEFAULT,
							 real_idle_update,
							 item);
}

static void
gantt_row_item_set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	GanttRowItem     *row;
	GanttRowItemPriv *priv;
	
	row = GANTT_ROW_ITEM (object);
	priv = row->priv;

	switch (arg_id) {
	case ARG_LEAF:
		priv->leaf = GTK_VALUE_BOOL (*arg);
		break;

	case ARG_TASK:
		priv->task = GTK_VALUE_POINTER (*arg);
		break;

	case ARG_X:
		priv->x = GTK_VALUE_DOUBLE (*arg);
		break;

	case ARG_Y:
		priv->y = GTK_VALUE_DOUBLE (*arg);
		break;

	case ARG_WIDTH:
		priv->width = GTK_VALUE_DOUBLE (*arg);
		break;

	case ARG_HEIGHT:
		priv->height = GTK_VALUE_DOUBLE (*arg);
		break;
		
	case ARG_SELECTED:
		priv->selected = GTK_VALUE_BOOL (*arg);
		break;

	default:
		return;
	}

	idle_update (row);
}

static void
gantt_row_item_get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	GanttRowItem     *row;
	GanttRowItemPriv *priv;
	
	row = GANTT_ROW_ITEM (object);
	priv = row->priv;

	switch (arg_id) {
	case ARG_LEAF:
		GTK_VALUE_BOOL (*arg) = priv->leaf;
		break;

	case ARG_TASK:
		GTK_VALUE_POINTER (*arg) = priv->task;
		break;

	case ARG_X:
		GTK_VALUE_DOUBLE (*arg) = priv->x;
		break;

	case ARG_Y:
		GTK_VALUE_DOUBLE (*arg) = priv->y;
		break;

	case ARG_WIDTH:
		GTK_VALUE_DOUBLE (*arg) = priv->width;
		break;

	case ARG_HEIGHT:
		GTK_VALUE_DOUBLE (*arg) = priv->height;
		break;

	default:
		arg->type = GTK_TYPE_INVALID;
		break;
	}
}

static GdkBitmap *stipple = NULL;
static gchar      stipple_pattern[] = { 0x02, 0x01 };

static void
gantt_row_item_realize (GnomeCanvasItem *item)
{
	GanttRowItem *row = GANTT_ROW_ITEM (item);
	GtkStyle     *style;
	GdkColor      color, black_color;

	GNOME_CALL_PARENT_HANDLER (GNOME_CANVAS_ITEM_CLASS, realize, (item));

	style = GTK_WIDGET (item->canvas)->style;

	if (!stipple) {
		stipple = gdk_bitmap_create_from_data (NULL, stipple_pattern, 2, 2);
	}
	
	if (!text_gc) {
		text_gc = gdk_gc_new (item->canvas->layout.bin_window);
		gdk_gc_set_foreground (text_gc, &style->fg [GTK_STATE_NORMAL]);
	}

	if (!frame_gc) {
		frame_gc = gdk_gc_new (item->canvas->layout.bin_window);
		gdk_gc_set_foreground (frame_gc, &style->fg [GTK_STATE_NORMAL]);
	}

	if (left_bracket_pixbuf == NULL) {
		left_bracket_pixbuf = gdk_pixbuf_new_from_xpm_data (left_bracket_xpm);
	}
		
	if (right_bracket_pixbuf == NULL) {
		right_bracket_pixbuf = gdk_pixbuf_new_from_xpm_data (right_bracket_xpm);
	}
	
	gnome_canvas_get_color (item->canvas, "light green", &color);
	row->priv->fill_gc = gdk_gc_new (item->canvas->layout.bin_window);
	gdk_gc_set_foreground (row->priv->fill_gc, &color);

	gnome_canvas_get_color (item->canvas, "black", &black_color);
	row->priv->fill_complete_gc = gdk_gc_new (item->canvas->layout.bin_window);
	gdk_gc_set_foreground (row->priv->fill_complete_gc, &black_color);
	gdk_gc_set_background (row->priv->fill_complete_gc, &color);
	gdk_gc_set_stipple (row->priv->fill_complete_gc, stipple);
	gdk_gc_set_fill (row->priv->fill_complete_gc, GDK_OPAQUE_STIPPLED);

	if (font == NULL) {
		font = e_font_from_gdk_font (style->font);
	}
}

static void
gantt_row_item_unrealize (GnomeCanvasItem *item)
{
	GanttRowItem *row = GANTT_ROW_ITEM (item);

	GNOME_CALL_PARENT_HANDLER (GNOME_CANVAS_ITEM_CLASS, unrealize, (item));

	gdk_gc_unref (row->priv->fill_gc);
	gdk_gc_unref (row->priv->fill_complete_gc);
	
	/* We leak the gcs and the font here, but they are "static" data
	 * so it's only _one_ each per application. No big deal, and
	 * faster and simpler code.
	 */
}

static int
gantt_row_item_event (GnomeCanvasItem *item,
		      GdkEvent        *e)
{
	GanttRowItem     *row_item;
	GanttRowItemPriv *priv;
	GanttModel       *gantt_model;
	GanttScale       *gantt_scale;
	GdkCursor        *cursor;
	static gint       drag_dx;
	GM_Task          *task;
	time_t            t1, t2;

	row_item = GANTT_ROW_ITEM (item);
	priv = row_item->priv;
	task = priv->task;
	
	gantt_model = gtk_object_get_data (GTK_OBJECT (item->canvas),
					   "gantt_model");
	gantt_scale = gtk_object_get_data (GTK_OBJECT (item->canvas),
					   "gantt_scale");
	
	switch (e->type) {
	case GDK_BUTTON_PRESS: {
		switch (e->button.button) {
		case 1:
			if (!gantt_model_task_is_leaf (gantt_model, task->taskId)) {
				return TRUE;
			}
			else if (abs (e->button.x - priv->x) < 5) {
				priv->state = DRAG_LEFT;
				drag_dx = gantt_scale_t2w (gantt_scale, task->start) - e->button.x;
			}
			else if (abs (e->button.x - (priv->x + priv->width)) < 5) {
				priv->state = DRAG_RIGHT;
				drag_dx = gantt_scale_t2w (gantt_scale, task->end) - e->button.x;
			} else {
				priv->state = DRAG_MOVE;
				drag_dx = gantt_scale_t2w (gantt_scale, task->start) - e->button.x;
			}
			
			gnome_canvas_item_grab (item,
						GDK_POINTER_MOTION_MASK |
						GDK_POINTER_MOTION_HINT_MASK |
						GDK_BUTTON_RELEASE_MASK,
						NULL,
						e->button.time);

			gtk_signal_emit (GTK_OBJECT (item),
					 signals[CLICKED],
					 e);

			return TRUE;
			break;

		case 2:
		case 3:
		case 4:
		case 5:
		default:
			return FALSE;
		}
		break;
	}
	break;

	case GDK_2BUTTON_PRESS: {
		switch (e->button.button) {
		case 1:
			if (priv->state == DRAG_NONE) {
				gtk_signal_emit (GTK_OBJECT (item),
						 signals[CLICKED],
						 e);
				return TRUE;
			}
			break;

		case 2:
		case 3:
		case 4:
		case 5:
			break;
		}
	}
	break;
	
	case GDK_BUTTON_RELEASE: {
		switch (e->button.button) {
		case 1:
			gnome_canvas_item_ungrab (item, e->button.time);
			gdk_window_set_cursor (((GtkWidget *) item->canvas)->window, NULL);
			priv->state = DRAG_NONE;

			return TRUE;
			break;
			
		case 2:
		case 3:
		case 4:
		case 5:
			return FALSE;
		}
	}
	break;

	case GDK_LEAVE_NOTIFY: {
		if (priv->state == DRAG_NONE) {
			/* For some reason, we get a leave notify when pressing 
			 * button 1 over the item. We don't want to reset the 
			 * cursor when that happens.
			 */
			if (!(e->crossing.state & GDK_BUTTON1_MASK)) {
				GtkWidget *widget;

				widget = (GtkWidget *) item->canvas;
				gdk_window_set_cursor (widget->window, NULL);
			}
		}
		return TRUE;
	}
	break;

	case GDK_MOTION_NOTIFY: {
		if (e->motion.is_hint) {
			gint x, y, offx, offy;
			gdk_window_get_pointer (e->motion.window, &x, &y, NULL);
			gnome_canvas_get_scroll_offsets (item->canvas, &offx, &offy);
			x += offx;
			y += offy;

			gnome_canvas_c2w (item->canvas, x, y, &e->motion.x, &e->motion.y);
		}

		if ((priv->state & (DRAG_MOVE | DRAG_LEFT | DRAG_RIGHT)) == 0) {
			/* No dragging, just change the cursor. */
			if (!gantt_model_task_is_leaf (gantt_model, task->taskId)) {
				cursor = NULL;
			}
			else if (abs (e->motion.x - priv->x) < 5) {
				cursor = gdk_cursor_new (GDK_LEFT_SIDE);
			}
			else if (abs (e->motion.x - (priv->x + priv->width)) < 5) {
				cursor = gdk_cursor_new (GDK_RIGHT_SIDE);
			}
			else {
				cursor = gdk_cursor_new (GDK_SB_H_DOUBLE_ARROW);
			}
			
			gdk_window_set_cursor (((GtkWidget *) item->canvas)->window, cursor);
			if (cursor) {
				gdk_cursor_destroy (cursor);
			}

			return TRUE;
		} 
		else if (priv->state == DRAG_MOVE) {
			if (e->motion.x + drag_dx < 0) {
				return TRUE;
			}

			t1 = gantt_scale_w2t (gantt_scale, e->motion.x + drag_dx);
			t1 = time_hour_begin (t1);

			if (t1 != task->start) {
				gantt_model_task_moved (gantt_model, task->taskId, t1, GNOME_MrProject_TASK_START);
			}
			
			return TRUE;
		}
		else if (priv->state == DRAG_LEFT) {
			time_t t1;

			if (e->motion.x + drag_dx < 0) {
				return TRUE;
			}

			t1 = gantt_scale_w2t (gantt_scale, e->motion.x + drag_dx);
			t1 = time_hour_begin (t1);

			if (t1 != task->start && t1 < task->end) {
				gantt_model_task_duration_changed (gantt_model,
								   task->taskId,
								   task->end - t1,
								   GNOME_MrProject_TASK_END);
			}
			return TRUE;
		}
		else if (priv->state == DRAG_RIGHT) {
			if (e->motion.x + drag_dx < 0) {
				return TRUE;
			}

			t2 = gantt_scale_w2t (gantt_scale, e->motion.x + drag_dx);
			t2 = time_hour_begin (t2);

			if (t2 != task->end && t2 > task->start) {
				gantt_model_task_duration_changed (gantt_model,
								   task->taskId,
								   t2 - task->start,
								   GNOME_MrProject_TASK_START);
			}
			return TRUE;
		}
	}
	break;

	default:
		break;
	}

	return FALSE;
}

static double
gantt_row_item_point (GnomeCanvasItem   *item,
		      double             x,
		      double             y,
		      int                cx,
		      int                cy,
		      GnomeCanvasItem  **actual_item)
{
	GanttRowItem     *row_item;
	GanttRowItemPriv *priv;

	row_item = GANTT_ROW_ITEM (item);
	priv = row_item->priv;

	if (x >= priv->x && x <= priv->x + priv->width &&
	    y >= priv->y && y <= priv->y + priv->height) {
		*actual_item = item;
		return 0.0;
	} else {
		return 1.0;
	}
}

static void
gantt_row_item_bounds_item_coordinates (GnomeCanvasItem *item,
					double          *x1,
					double          *y1,
					double          *x2,
					double          *y2)
{
	GanttRowItem     *row;
	GanttRowItemPriv *priv;
	gint              label_width;

	row = GANTT_ROW_ITEM (item);
	priv = row->priv;

	g_return_if_fail (priv->task);

	label_width = 0;
	if (priv->resources && priv->resources_text) {
		label_width = e_font_utf8_text_width (
			font,
			E_FONT_PLAIN,
			priv->resources_text,
			strlen (priv->resources_text)) + 10;
	}

	*x1 = priv->x - 1;
	*y1 = priv->y - 1;
	*x2 = priv->x + priv->width + 2 + label_width;
	*y2 = priv->y + priv->height + 1;
}

static void
gantt_row_item_bounds_canvas_coordinates (GnomeCanvasItem *item,
					  double          *x1,
					  double          *y1,
					  double          *x2,
					  double          *y2)
{
	double            i2c[6];
	ArtPoint          c1, c2, i1, i2;
	GanttRowItem     *row;
	GanttRowItemPriv *priv;

	row = GANTT_ROW_ITEM (item);
	priv = row->priv;

	gantt_row_item_bounds_item_coordinates (item,
						&i1.x,
						&i1.y,
						&i2.x,
						&i2.y);

	gnome_canvas_item_i2c_affine (item, i2c);
	art_affine_point (&c1, &i1, i2c);
	art_affine_point (&c2, &i2, i2c);
	*x1 = c1.x;
	*y1 = c1.y;
	*x2 = c2.x;
	*y2 = c2.y;
}

static void
gantt_row_item_update (GnomeCanvasItem *item,
		       double          *affine,
		       ArtSVP          *clip_path,
		       int              flags)
{
	GanttRowItem     *row;
	GanttRowItemPriv *priv;
	gdouble           x1, y1, x2, y2;

	row = GANTT_ROW_ITEM (item);
	priv = row->priv;

	GNOME_CALL_PARENT_HANDLER (GNOME_CANVAS_ITEM_CLASS,
				   update, (item, affine, clip_path, flags));

	gantt_row_item_bounds_canvas_coordinates (item, &x1, &y1, &x2, &y2);
	gnome_canvas_update_bbox (item, x1, y1, x2, y2);
	if (priv->resources && priv->resources_text) {
		priv->label_width = e_font_utf8_text_width (
			font,
			E_FONT_PLAIN,
			priv->resources_text,
			strlen (priv->resources_text)) + 10;
	}
	else {
		priv->label_width = 0;
	}
}

static void 
gantt_row_item_draw (GnomeCanvasItem *item,
		     GdkDrawable     *drawable,
		     int              x,
		     int              y,
		     int              width,
		     int              height)
{
	GanttRowItem     *row;
	GanttRowItemPriv *priv;
	gint              cx1, cy1, cx2, cy2, cx2_label;
	gint              tx, twidth;
	gint              complete_width, comp_x;
	ArtPoint          pc, pi;
	double            i2c[6];
	gboolean          left, right;
	GdkRectangle      rect;
	gboolean          draw_bar;
	
	row = GANTT_ROW_ITEM (item);
	priv = row->priv;

	if (priv->width <= 0 || priv->height <= 0) {
		return;
	}

	gnome_canvas_item_i2c_affine (item, i2c);

	pi.x = priv->x;
	pi.y = priv->y;
	art_affine_point (&pc, &pi, i2c);
	cx1 = floor (pc.x + 0.5) - x;
	cy1 = floor (pc.y + 0.5) - y;
	
	pi.x = priv->x + priv->width;
	pi.y = priv->y + priv->height;
	art_affine_point (&pc, &pi, i2c);
	cx2 = floor (pc.x + 0.5) - x;
	cy2 = floor (pc.y + 0.5) - y;

	cx2_label = floor (pc.x + priv->label_width + 0.5) - x;

	/* GtkLayout emulates a coord. system > 2^16, but it only
	 * works if each individual item is smaller than 2^16. So
	 * we only draw the part actually shown on screen (probably
	 * a good idea anyway...).
	 *
	 * This is done by splitting the bar into three parts,
	 * left, middle, right. Find out which parts actually
	 * need to be drawn and do that.
	 */

/* 	g_print ("cx1: %d, cx2: %d, cx2_label: %d, width: %d\n", cx1, cx2, cx2_label, width); */

	complete_width = (cx2 - cx1) * ((gfloat)priv->task->percentComplete / 100.0);
	comp_x = cx1 + complete_width;
	
	left = right = FALSE;
	draw_bar = TRUE;
	tx = twidth = 0;
	
	if (cx1 >= 0 && cx1 <= width) {
		left = TRUE;
	}

	if (cx2 >= 0 && cx2 <= width) {
		right = TRUE;
	}

	if (left && right) {
		/* Draw the whole thing. */

		tx = cx1;
		twidth = cx2 - cx1;
	}
	else if (left) {
		rect.x = cx1 - 1;
		rect.y = cy1;
		rect.width = width - cx1 + 1;
		rect.height = cy2 - cy1;

		gdk_gc_set_clip_rectangle (frame_gc, &rect);

		tx = cx1;
		twidth = width - cx1 + 10;
	}
	else if (right) {
		rect.x = 0;
		rect.y = cy1;
		rect.width = cx2 + 10;
		rect.height = cy2 - cy1;

		gdk_gc_set_clip_rectangle (frame_gc, &rect);

		tx = -10;
		twidth = cx2 + 10;
	}
	else if (cx2 > 0) {
		/* middle */

		rect.x = cx1;
		rect.y = cy1;
		rect.width = cx2 - cx1;
		rect.height = cy2 - cy1;

		gdk_gc_set_clip_rectangle (frame_gc, &rect);

		tx = cx1 - 1;
		twidth = cx2 - cx1 + 2;
 	}
	else {
		draw_bar = FALSE;
	}

	if (draw_bar && priv->leaf) {
		gdk_gc_set_ts_origin (priv->fill_complete_gc, cx1, cy1);

		gdk_draw_rectangle (drawable,
				    priv->fill_gc,
				    TRUE,
				    tx, cy1 + 4,
				    twidth, cy2 - cy1 - 8);

		if (comp_x > tx) {
			gdk_draw_rectangle (drawable,
					    priv->fill_complete_gc,
					    TRUE,
					    tx, cy1 + 4 + 4,
					    comp_x - tx, cy2 - cy1 - 8 - 7);
		}

		gdk_draw_rectangle (drawable,
				    frame_gc,
				    FALSE,
				    tx, cy1 + 4,
				    twidth, cy2 - cy1 - 8);
		
		if (priv->selected) {
			gdk_draw_rectangle (drawable,
					    frame_gc,
					    FALSE,
					    tx + 1, cy1 + 5,
					    twidth - 2, cy2 - cy1 - 8 - 2);
		}

		gdk_gc_set_clip_rectangle (frame_gc, NULL);
	}
	else if (draw_bar) {
		draw_bracket (drawable, cy1, tx, tx + twidth, left, right);
		gdk_gc_set_clip_rectangle (frame_gc, NULL);
	}

	if (priv->resources && priv->resources_text) {
		e_font_draw_utf8_text (drawable,
				       font,
				       E_FONT_PLAIN,
				       text_gc,
				       cx2 + 10,
				       cy1 + priv->height / 4 + e_font_ascent (font),
				       priv->resources_text,
				       strlen (priv->resources_text));
	}
}

static void
draw_bracket (GdkDrawable *drawable,
	      gint         y1,
	      gint         x1,
	      gint         x2,
	      gboolean     left,
	      gboolean     right)
{
	gint top, mid;
	gint left_width, left_height, right_width, right_height;

	top = y1 + TEXT_PAD / 2 + 2;
	mid = top + TEXT_PAD / 2 - 1;

	gdk_draw_rectangle (drawable,
			    frame_gc,
			    TRUE,
			    x1 + 1, mid,
			    x2 - x1 - 2, 3);

	if (left) {
		left_width = gdk_pixbuf_get_width (left_bracket_pixbuf);
		left_height = gdk_pixbuf_get_height (left_bracket_pixbuf);
		gdk_pixbuf_render_to_drawable_alpha (
			left_bracket_pixbuf,
			drawable,
			0, 0,
			x1,
			top + 1,
			left_width,
			left_height,
			GDK_PIXBUF_ALPHA_BILEVEL,
			127,
			GDK_RGB_DITHER_NORMAL,
			left_width,
			left_height);
	}

	if (right) {
		right_width = gdk_pixbuf_get_width (right_bracket_pixbuf);
		right_height = gdk_pixbuf_get_height (right_bracket_pixbuf);
		gdk_pixbuf_render_to_drawable_alpha (
			right_bracket_pixbuf,
			drawable,
			0, 0,
			x2 - right_width,
			top + 1,
			right_width,
			right_height,
			GDK_PIXBUF_ALPHA_BILEVEL,
			127,
			GDK_RGB_DITHER_NORMAL,
			right_width,
			right_height);
	}
}

static void
rebuild_resources_text (GanttRowItem *row)
{
	gchar            *text, *tmp;
	AssignedResource *resource;
	GSList           *list;
	
	g_free (row->priv->resources_text);
	row->priv->resources_text = NULL;

	if (!row->priv->resources) {
		return;
	}

	text = NULL;
	for (list = row->priv->resources; list; list = list->next) {
		resource = list->data;

		if (text == NULL) { /* First resource. */
			text = g_strdup (resource->name);
			if (!list->next) {
				break;
			}
		} else {
			tmp = g_strconcat (text, ", ", resource->name, NULL);
			g_free (text);
			text = tmp;
		}
	}

	row->priv->resources_text = text;
}

void
gantt_row_item_add_resource (GanttRowItem *row,
			     gint          resource_id,
			     const gchar  *name)
{
	AssignedResource *resource;
	
	g_return_if_fail (row != NULL);
	g_return_if_fail (IS_GANTT_ROW_ITEM (row));

	resource = g_new0 (AssignedResource, 1);
	resource->name = g_strdup (name);
	resource->resource_id = resource_id;
	
	row->priv->resources = g_slist_append (row->priv->resources, resource);

	rebuild_resources_text (row);
	idle_update (row);
}

void
gantt_row_item_remove_resource (GanttRowItem *row,
				gint          resource_id)
{
	GSList *list;
	
	g_return_if_fail (row != NULL);
	g_return_if_fail (IS_GANTT_ROW_ITEM (row));

	for (list = row->priv->resources; list; list = list->next) {
		AssignedResource *resource;

		resource = list->data;
		if (resource->resource_id == resource_id) {
			row->priv->resources = g_slist_remove_link (row->priv->resources, list);
			g_free (resource->name);
			g_free (resource);
			break;
		}
	}
	
	rebuild_resources_text (row);
	idle_update (row);
}

void 
gantt_row_item_update_resource (GanttRowItem *row,
				GM_Id         resource_id,
				const gchar  *new_name)
{
	GSList *list;
	
	g_return_if_fail (row != NULL);
	g_return_if_fail (IS_GANTT_ROW_ITEM (row));

	for (list = row->priv->resources; list; list = list->next) {
		AssignedResource *resource;

		resource = list->data;
		if (resource->resource_id == resource_id) {
			g_free (resource->name);
			
			resource->name = g_strdup (new_name);
			break;
		}
	}
	
	rebuild_resources_text (row);
	idle_update (row);
}

static void
gantt_row_item_class_init (GanttRowItemClass *klass)
{
	GtkObjectClass       *object_class;
	GnomeCanvasItemClass *item_class;
	
	object_class = (GtkObjectClass*) klass;
	item_class = (GnomeCanvasItemClass*) klass;

	/* Gtk Object methods. */
	object_class->destroy = gantt_row_item_destroy;
	object_class->set_arg = gantt_row_item_set_arg;
	object_class->get_arg = gantt_row_item_get_arg;

	/* GnomeCanvasItem methods. */
	item_class->update    = gantt_row_item_update;
	item_class->draw      = gantt_row_item_draw;
	item_class->point     = gantt_row_item_point;
	item_class->bounds    = gantt_row_item_bounds_item_coordinates;
	item_class->event     = gantt_row_item_event;
	item_class->realize   = gantt_row_item_realize;
	item_class->unrealize = gantt_row_item_unrealize;

	signals[CHANGED] = 
		gtk_signal_new ("changed",
				GTK_RUN_FIRST,
				object_class->type,
				0,
				mrproject_marshal_NONE__DOUBLE_DOUBLE_DOUBLE_DOUBLE,
				GTK_TYPE_NONE, 
				4, 
				GTK_TYPE_DOUBLE, 
				GTK_TYPE_DOUBLE,
				GTK_TYPE_DOUBLE, 
				GTK_TYPE_DOUBLE);

	signals[CLICKED] = 
		gtk_signal_new ("clicked",
				GTK_RUN_FIRST,
				object_class->type,
				0,
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

	gtk_object_class_add_signals (object_class, signals, LAST_SIGNAL);

	/* Arguments. */
	gtk_object_add_arg_type ("GanttRowItem::task", GTK_TYPE_POINTER, GTK_ARG_READWRITE, ARG_TASK);
	gtk_object_add_arg_type ("GanttRowItem::x", GTK_TYPE_DOUBLE, GTK_ARG_READWRITE, ARG_X);
	gtk_object_add_arg_type ("GanttRowItem::y", GTK_TYPE_DOUBLE, GTK_ARG_READWRITE, ARG_Y);
	gtk_object_add_arg_type ("GanttRowItem::width", GTK_TYPE_DOUBLE, GTK_ARG_READWRITE, ARG_WIDTH);
	gtk_object_add_arg_type ("GanttRowItem::height", GTK_TYPE_DOUBLE, GTK_ARG_READWRITE, ARG_HEIGHT);
	gtk_object_add_arg_type ("GanttRowItem::leaf", GTK_TYPE_BOOL, GTK_ARG_READWRITE, ARG_LEAF);
	gtk_object_add_arg_type ("GanttRowItem::selected", GTK_TYPE_BOOL, GTK_ARG_WRITABLE, ARG_SELECTED);
}

static void
gantt_row_item_init (GanttRowItem *item)
{
	item->priv         = g_new0 (GanttRowItemPriv, 1);
	item->priv->x      = 0;
	item->priv->y      = 0;
	item->priv->height = 0;
	item->priv->width  = 0;
	item->priv->leaf   = TRUE;
}

void
gantt_row_item_get_geometry (GanttRowItem *item,
			     gdouble      *x1,
			     gdouble      *y1,
			     gdouble      *x2,
			     gdouble      *y2)
{
	GanttRowItemPriv *priv;
	
	g_return_if_fail (item != NULL);
	g_return_if_fail (IS_GANTT_ROW_ITEM (item));

	priv = item->priv;
	
	if (x1)
		*x1 = priv->x;
	if (y1)
		*y1 = priv->y;
	if (x2)
		*x2 = priv->x + priv->width;
	if (y2)
		*y2 = priv->y + priv->height;
}
			     
