/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001  Richard Hult 
 * Copyright (C) 2001  CodeFactory AB 
 * 
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the 
 * License, or (at your option) any later version. 
 * 
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <math.h>
#include <time.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-paper.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnomeui/gnome-uidefs.h>
#include <libgnomeui/gnome-paper-selector.h>
#include <libgnomeprint/gnome-printer.h>
#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-printer-dialog.h>
#include <libgnomeprint/gnome-print-master.h>
#include <libgnomeprint/gnome-print-master-preview.h>
#include <gal/widgets/e-unicode.h>

#include "util/time-utils.h"
#include "gantt-print.h"
#include "gantt-model.h"

void
gantt_print_rectangle (GanttPrintInfo *print_info,
		       gdouble         x1,
		       gdouble         y1,
		       gdouble         x2,
		       gdouble         y2) 
{
	gnome_print_newpath (print_info->context);
	gnome_print_setlinewidth (print_info->context, 0);
	gnome_print_moveto (print_info->context, x1, y1);
	gnome_print_lineto (print_info->context, x1, y2);
	gnome_print_lineto (print_info->context, x2, y2);
        gnome_print_lineto (print_info->context, x2, y1);
        gnome_print_lineto (print_info->context, x1, y1);
        gnome_print_closepath (print_info->context);
        gnome_print_stroke (print_info->context);
}

void
gantt_print_line (GanttPrintInfo *print_info,
		  gdouble         x1,
		  gdouble         y1,
		  gdouble         x2,
		  gdouble         y2) 
{
	gnome_print_newpath (print_info->context);
	gnome_print_setlinewidth (print_info->context, 0);
	gnome_print_moveto (print_info->context, x1, y1);
        gnome_print_lineto (print_info->context, x2, y2);
        gnome_print_stroke (print_info->context);
}

void
gantt_print_arrow (GanttPrintInfo *print_info,
		   gdouble         x1,
		   gdouble         y1,
		   gdouble         x2,
		   gdouble         y2) 
{
	gnome_print_newpath (print_info->context);
	gnome_print_setlinewidth (print_info->context, 0);
	gnome_print_moveto (print_info->context, x1, y1);
        gnome_print_lineto (print_info->context, x2, y2);
        gnome_print_stroke (print_info->context);
}

void
gantt_print_text (GanttPrintInfo *print_info,
		  gdouble         x,
		  gdouble         y,
		  const gchar    *text)
{
	gdouble  text_base;
	gchar   *str;

	text_base = y - gnome_font_get_size (print_info->font) +
		gnome_font_get_ascender (print_info->font);
	
	str = e_utf8_from_locale_string (text);
	gnome_print_moveto (print_info->context, x, text_base);
	gnome_print_gsave (print_info->context);
	gnome_print_scale (print_info->context, 1, -1);
	gnome_print_setfont (print_info->context, print_info->font);
	gnome_print_show (print_info->context, str);
	gnome_print_grestore (print_info->context);
}

#define SWAP(a,b) { double tmp = a; a = b; b = tmp; } 

static void
setup_rotation (GanttPrintInfo *print_info)
{
	double  affine [6];

	/* Landscape, with (0,0) in the upper left corner,
	 * growing down/right. */
	art_affine_rotate (affine, 90.0);
	gnome_print_concat (print_info->context, affine);
	art_affine_scale (affine, 1, -1);
	gnome_print_concat (print_info->context, affine);
}

static void
setup_paper (GanttPrintInfo   *print_info,
	     const GnomePaper *paper)
{
	gdouble width, height;

	width = gnome_paper_pswidth (paper);
	height = gnome_paper_psheight (paper);

	print_info->lmargin = gnome_paper_lmargin (paper);
	print_info->tmargin = gnome_paper_tmargin (paper);
	print_info->rmargin = gnome_paper_rmargin (paper);
	print_info->bmargin = gnome_paper_bmargin (paper);
	print_info->paper = paper;

	print_info->width = width - (print_info->lmargin + print_info->rmargin);
	print_info->height = height - (print_info->tmargin + print_info->bmargin);

	/* Since we rotate the paper, switch around height/width. */
	SWAP (print_info->width, print_info->height);
	SWAP (print_info->lmargin, print_info->tmargin);
	SWAP (print_info->rmargin, print_info->bmargin);
}

static GanttPrintInfo *
gantt_print_info_new (GnomePrintContext *context, 
		      const GnomePaper  *paper)
{
	GanttPrintInfo *print_info;

	print_info = g_new0 (GanttPrintInfo, 1);
	print_info->context = context;

	print_info->font = gnome_font_new_closest ("helvetica", 
						   GNOME_FONT_BOOK, 
						   FALSE, 
						   8);

	setup_paper (print_info, paper);
	
	/* FIXME: where to do this? (and don't hardcode it :) */
	print_info->scale = 1.0;
	
	return print_info;
}

static void
gantt_print_info_free (GanttPrintInfo *print_info)
{
	gnome_font_unref (print_info->font);
	g_free (print_info);
}

static void
print_chart (GanttPrintInfo    *print_info,
	     GanttModel        *gantt_model,
	     GanttItem         *gantt,
	     GanttHeaderItem   *header)
{
	GanttPrintable *header_printable, *gantt_printable;
	gdouble         gantt_height;
	gdouble         header_width, header_height;
	gdouble         row_height;
	gdouble         tree_width;
	gdouble         width_per_page;
	gdouble         hscale, vscale;
	time_t          first_time, last_time, t;
	gint            time_per_page;
	gint            r, rows_per_page;
	gint            vertical_pages, vp;
	gint            horizontal_pages, hp;

	/* FIXME: We print everything from the first task to the last one for now. */
	first_time = gantt_model_get_first_time (gantt_model);
	last_time = gantt_model_get_last_time (gantt_model);

	if (first_time == -1 || last_time == -1) {
		return;
	}

	first_time = time_day_begin (first_time);
	last_time = time_day_end (last_time);

	first_time = time_add_day (first_time, -1);
	last_time = time_add_day (last_time, 1);
	
	if (last_time <= (first_time + 60*60*24*7)) {
		last_time = time_add_day (first_time, 7);
	}
	
	header_printable = gantt_header_item_get_printable (header);
	gantt_printable = gantt_item_get_printable (gantt);

	/* In points. */
	header_width = gantt_printable_get_width (header_printable, print_info, first_time, last_time);
	header_height = gantt_printable_get_height (header_printable, print_info);
	gantt_height = gantt_printable_get_height (gantt_printable, print_info);

	/* FIXME */
	tree_width = 0;

	/* FIXME */
	row_height = gantt_printable_get_row_height (gantt_printable, print_info);
	rows_per_page = (print_info->height - header_height) / row_height;

	vertical_pages = ceil (gantt_height / (print_info->height - header_height));
	horizontal_pages = ceil ((tree_width + header_width) / print_info->width);

	/* Shouldn't be neccessary, but it happens once in a while, why? */
	if (vertical_pages == 0)
		vertical_pages = 1;
	if (horizontal_pages == 0)	
		horizontal_pages = 1;
	
	time_per_page = (last_time - first_time) / horizontal_pages;
	width_per_page = gantt_printable_get_width (header_printable,
						    print_info,
						    first_time,
						    first_time + time_per_page);
	hscale = print_info->width / width_per_page;
	vscale = 1.0;

	r = 0;
	for (vp = 0; vp < vertical_pages; vp++) {
		t = first_time;
		for (hp = 0; hp < horizontal_pages; hp++) {
			ArtDRect target;
			
			gnome_print_beginpage (print_info->context, "");
			gnome_print_gsave (print_info->context);
			setup_rotation (print_info);

			target.x0 = print_info->lmargin;
			target.y0 = print_info->tmargin;
			target.x1 = print_info->lmargin + print_info->width;
			target.y1 = print_info->tmargin + print_info->height;
			if (hp == 0) {
				gantt_print_rectangle (print_info,
						       target.x0,
						       target.y0,
						       target.x0 + tree_width,
						       target.y1);
				target.x0 += tree_width;
			}
			
			gantt_printable_print_page (header_printable,
						    print_info,
						    &target,
						    hscale,
						    vscale,
						    t,
						    t + time_per_page,
						    0,
						    0);

			target.x0 = print_info->lmargin;
			target.y0 = print_info->tmargin + header_height;
			target.x1 = print_info->lmargin + print_info->width;
			target.y1 = print_info->tmargin + print_info->height;
			if (hp == 0) {
				target.x0 += tree_width;
			}
			
			gantt_printable_print_page (gantt_printable,
						    print_info,
						    &target,
						    hscale,
						    vscale,
						    t,
						    t + time_per_page,
						    r,
						    r + rows_per_page - 1);

			gnome_print_grestore (print_info->context);
			gnome_print_showpage (print_info->context);

			t += time_per_page;
		}
		r += rows_per_page;
	}
	
}

void
print_gantt (GanttModel      *gantt_model,
	     GanttItem       *gantt,
	     GanttHeaderItem *header,
	     gboolean         preview_only)
{
	GtkWidget         *dialog;
	gchar             *paper_name;
	gdouble            scale;
	GnomePrintMaster  *print_master;
	GnomePrintContext *ctx;
	const GnomePaper  *paper;
	gboolean           preview;
       	int                btn;
	GanttPrintInfo    *print_info;

	
	scale      = 1.0;
	dialog     = NULL;
	paper_name = NULL;

	preview_only = FALSE;

	preview = FALSE;
	if (!preview_only) {

		dialog= gnome_print_dialog_new (_("Print Gantt Chart"), 0);
		gtk_window_set_wmclass (GTK_WINDOW (dialog),
					"Print Gantt Chart",
					"MrProject");

		btn = gnome_dialog_run (GNOME_DIALOG (dialog));
		switch (btn) {
		case -1:
			return;

		case GNOME_PRINT_CANCEL:
			gtk_widget_destroy (dialog);
			return;
			
		case GNOME_PRINT_PREVIEW:
			preview = TRUE;
			break;
		default:
			break;
		};
	} else {
		preview = TRUE;
	}

	print_master = gnome_print_master_new_from_dialog (
		GNOME_PRINT_DIALOG (dialog));

	/* Get the paper metrics. */
	if (paper_name) {
		paper = gnome_paper_with_name (paper_name);
	} else {
		paper = gnome_paper_with_name (gnome_paper_name_default ());
	}

	gnome_print_master_set_paper (print_master, paper);

	ctx = gnome_print_master_get_context (print_master);

	print_info = gantt_print_info_new (ctx, paper);
	print_chart (print_info, 
		     gantt_model, 
		     gantt, 
		     header);
	gantt_print_info_free (print_info);

	gnome_print_master_close (print_master);

	if (preview) {
		gboolean landscape = TRUE;
		GnomePrintMasterPreview *preview;
		
		preview = gnome_print_master_preview_new_with_orientation (
			print_master, _("Print Preview"), landscape);
		gtk_window_set_wmclass (GTK_WINDOW (preview),
					"PrintPreviewGantt",
					"MrProject");
		gtk_widget_show (GTK_WIDGET (preview));
	} else {
		int result;

		result = gnome_print_master_print (print_master);

		if (result == -1) {
			g_warning (_("Printing failed."));
		}
	}

	/* Done. */
	gtk_object_unref (GTK_OBJECT (print_master));
	if (!preview_only) {
		gtk_widget_destroy (dialog);
	}
}


