/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2000-2001 CodeFactory AB
 * Copyright (C) 2000-2001 Richard Hult <rhult@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gal/widgets/e-unicode.h>
#include <libgnomeui/gnome-canvas.h>
#include <libgnomeui/gnome-canvas-util.h>
#include <math.h>
#include <string.h>
#include "util/type-utils.h"
#include "util/time-utils.h"
#include "gantt-util.h"
#include "gantt-model.h"
#include "gantt-scale.h"
#include "gantt-header-item.h"
#include "gantt-print.h"


/* Arguments. */
enum {
	ARG_0,
	ARG_GANTT_MODEL,
	ARG_GANTT_SCALE,
	ARG_HEIGHT
};

/* Private members. */
struct _GanttHeaderItemPriv {
	GdkGC		*text_gc;
	GdkGC		*line_gc;
	GdkFont		*font;
	gdouble		 x1, y1;
	gdouble		 x2, y2;

	GanttScale      *gantt_scale;
	GanttModel      *gantt_model;
};

/* GtkObject. */
static void gantt_header_item_class_init (GanttHeaderItemClass *klass);
static void gantt_header_item_init       (GanttHeaderItem      *item);


GNOME_CLASS_BOILERPLATE (GanttHeaderItem, gantt_header_item,
			 GnomeCanvasItem, gnome_canvas_item);

static void
gantt_header_item_destroy (GtkObject *object)
{
	GanttHeaderItem *gantt;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GANTT_HEADER_ITEM (object));

	gantt = GANTT_HEADER_ITEM (object);

	g_free (gantt->priv);
	gantt->priv = NULL;

	GNOME_CALL_PARENT_HANDLER (GTK_OBJECT_CLASS, destroy, (object));
}

static void
units_changed (GanttScale *gantt_scale, GanttHeaderItem *header)
{
	gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (header));
}

static void
scale_changed (GanttScale *gantt_scale, GanttHeaderItem *header)
{
	GnomeCanvas *canvas;

	canvas = GNOME_CANVAS_ITEM (header)->canvas;

	header->priv->x1 = gantt_scale->x1;
	header->priv->x2 = gantt_scale->x2;

	gnome_canvas_set_scroll_region (
		canvas,
		header->priv->x1,
		header->priv->y1,
		header->priv->x2 - 1,
		header->priv->y2 - 1);
	
	gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (header));
}

static void
viewport_changed (GanttScale *gantt_scale, GanttHeaderItem *header)
{
	GnomeCanvas *canvas;

	canvas = GNOME_CANVAS_ITEM (header)->canvas;

	header->priv->x1 = gantt_scale->x1;
	header->priv->x2 = gantt_scale->x2;

	gnome_canvas_set_scroll_region (
		canvas,
		header->priv->x1,
		header->priv->y1,
		header->priv->x2 - 1,
		header->priv->y2 - 1);
	
	gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (header));
}

static void
gantt_header_item_set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	GanttHeaderItem     *header;
	GanttHeaderItemPriv *priv;
	
	header = GANTT_HEADER_ITEM (object);
	priv = header->priv;

	switch (arg_id) {
	case ARG_GANTT_MODEL:
		/* We don't handle changing models after setting it (yet?). */
		if (priv->gantt_model != NULL) {
			g_warning ("Changing gantt model is not allowed.");
			return;
		}
		priv->gantt_model = GANTT_MODEL (GTK_VALUE_OBJECT (*arg));
		break;

	case ARG_GANTT_SCALE:
		/* We don't handle changing scales after setting it (yet?). */
		if (priv->gantt_scale != NULL) {
			g_warning ("Changing gantt scale is not allowed.");
			return;
		}
		priv->gantt_scale = GANTT_SCALE (GTK_VALUE_OBJECT (*arg));

		priv->x1 = priv->gantt_scale->x1;
		priv->x2 = priv->gantt_scale->x2;

		scale_changed (priv->gantt_scale, header);

		gtk_signal_connect (GTK_OBJECT (priv->gantt_scale),
				    "scale_changed",
				    scale_changed,
				    header);
		
		gtk_signal_connect (GTK_OBJECT (priv->gantt_scale),
				    "viewport_changed",
				    viewport_changed,
				    header);
		
		gtk_signal_connect (GTK_OBJECT (priv->gantt_scale),
				    "units_changed",
				    units_changed,
				    header);
		break;

	default:
		return;
	}

	gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (object));
}

static void
gantt_header_item_get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	GanttHeaderItem     *header;
	GanttHeaderItemPriv *priv;
	
	header = GANTT_HEADER_ITEM (object);
	priv = header->priv;

	switch (arg_id) {
	case ARG_HEIGHT:
		GTK_VALUE_DOUBLE (*arg) = priv->y2 - priv->y1;
		break;

	default:
		arg->type = GTK_TYPE_INVALID;
		break;
	}
}

static void
gantt_header_item_realize (GnomeCanvasItem *item)
{
	GtkStyle *style;
	gdouble   y2;
	
	GanttHeaderItem *header = GANTT_HEADER_ITEM (item);

	GNOME_CALL_PARENT_HANDLER (GNOME_CANVAS_ITEM_CLASS, realize, (item));

	style = GTK_WIDGET (item->canvas)->style;
	header->priv->text_gc = gdk_gc_new (GTK_WIDGET (item->canvas)->window);
	gdk_gc_set_foreground (header->priv->text_gc,
			       &style->fg [GTK_STATE_NORMAL]);
	header->priv->line_gc = gdk_gc_new (GTK_WIDGET (item->canvas)->window);
	gdk_gc_set_foreground (header->priv->line_gc,
			       &style->fg [GTK_STATE_INSENSITIVE]);

	header->priv->font = style->font;
	gdk_font_ref (header->priv->font);

	y2 = header->priv->font->ascent + header->priv->font->descent;
	y2 += 2 * (1 + style->klass->ythickness);
	y2 *= 2;
	header->priv->y2 = y2;

	scale_changed (header->priv->gantt_scale, header);
}

static void
gantt_header_item_unrealize (GnomeCanvasItem *item)
{
	GanttHeaderItem *header = GANTT_HEADER_ITEM (item);

	GNOME_CALL_PARENT_HANDLER (GNOME_CANVAS_ITEM_CLASS,
				   unrealize,
				   (item));

	gdk_gc_unref (header->priv->text_gc);
	gdk_font_unref (header->priv->font);
}

static double 
gantt_header_item_point (GnomeCanvasItem  *item,
			 double            x,
			 double            y,
			 int               cx,
			 int               cy,
			 GnomeCanvasItem **actual_item)
{
	*actual_item = NULL;
	return 1.0;
}

static void
gantt_header_item_bounds_item_coordinates (GnomeCanvasItem *item,
					   double          *x1,
					   double          *y1,
					   double          *x2,
					   double          *y2)
{
	GanttHeaderItem     *header;
	GanttHeaderItemPriv *priv;

	header = GANTT_HEADER_ITEM (item);
	priv = header->priv;
	
	*x1 = priv->x1 - 1;
	*y1 = priv->y1 - 1;
	*x2 = priv->x2 + 1;
	*y2 = priv->y2 + 1;
}

static void
gantt_header_item_bounds_canvas_coordinates (GnomeCanvasItem *item,
					     double          *x1,
					     double          *y1,
					     double          *x2,
					     double          *y2)
{
	double           i2c[6];
	ArtPoint         c1, c2, i1, i2;
	GanttHeaderItem *header;

	header = GANTT_HEADER_ITEM (item);

	gantt_header_item_bounds_item_coordinates (item,
						   &i1.x,
						   &i1.y,
						   &i2.x,
						   &i2.y);
	
	gnome_canvas_item_i2c_affine (item, i2c);
	art_affine_point (&c1, &i1, i2c);
	art_affine_point (&c2, &i2, i2c);
	
	*x1 = c1.x;
	*y1 = c1.y;
	*x2 = c2.x;
	*y2 = c2.y;
}

static void
gantt_header_item_update (GnomeCanvasItem *item,
			  double          *affine,
			  ArtSVP          *clip_path,
			  int              flags)
{
	gdouble x1, y1, x2, y2;

	GNOME_CALL_PARENT_HANDLER (GNOME_CANVAS_ITEM_CLASS,
				   update,
				   (item, affine, clip_path, flags));
	gantt_header_item_bounds_canvas_coordinates (item, &x1, &y1, &x2, &y2);
	gnome_canvas_update_bbox (item, x1, y1, x2, y2);
}

static void 
gantt_header_item_draw (GnomeCanvasItem *item,
			GdkDrawable     *drawable,
			int              x,
			int              y,
			int              width,
			int              height)
{
	GanttHeaderItem     *header;
	GanttHeaderItemPriv *priv;
	GtkStyle            *style;
	GdkRectangle         rect;
	gint                 cx1, cy1, cx2, cy2, cy_mid;
	gint                 xd, next_xd;
	gint                 font_height;
	time_t               first_time, last_time, time, next_time, tick_time;
	gint                 ypos1, ypos2, inner_y, inner_height, tick_width;
	double               i2c[6], c2i[6];
	ArtPoint             i1, i2, c1, c2;
	gchar               *text;
	gboolean             draw_minor_tick_marks;

	header = GANTT_HEADER_ITEM (item);
	priv = header->priv;

	font_height = priv->font->ascent + priv->font->descent;

	gnome_canvas_item_i2c_affine (item, i2c);

	c1.x = x;
	c1.y = y;
	c2.x = x + width;
	c2.y = y + height;
	art_affine_invert (c2i, i2c);
	art_affine_point (&i1, &c1, c2i);
	art_affine_point (&i2, &c2, c2i);

	/* i is where we should start drawing, in item coordinates
	 * ( == world coordinates since we never scale the canvas).
	 */
	first_time = gantt_scale_w2t (priv->gantt_scale, i1.x);
	last_time = gantt_scale_w2t (priv->gantt_scale, i2.x);
	
	i1.x = priv->x1;
	i1.y = priv->y1;
	i2.x = priv->x2;
	i2.y = priv->y2;
	art_affine_point (&c1, &i1, i2c); 
	art_affine_point (&c2, &i2, i2c); 

	cx1 = c1.x - x;
	cy1 = c1.y - y;
	cx2 = c2.x - x;
	cy2 = c2.y - y;

	cy_mid = (c2.y - c1.y) / 2;
	style = GTK_WIDGET (item->canvas)->style;
	inner_y = style->klass->ythickness + 1;
	inner_height = cy_mid - 2 * (style->klass->ythickness + 1);
	ypos1 = inner_y + (inner_height - font_height) / 2 + priv->font->ascent;
	ypos2 = ypos1 * 2 + (inner_height - font_height) + 6;

	cy_mid -= y;
	inner_y -= y;
	ypos1 -= y;
	ypos2 -= y;

	/* Draw the separator between the two scales. */
	{
		gint a, b;
		
		i1.x = gantt_scale_t2w (priv->gantt_scale, first_time);
		i1.y = 0;
		art_affine_point (&c1, &i1, i2c);
		a = floor (c1.x + 0.5) - x;
		i1.x = gantt_scale_t2w (priv->gantt_scale, last_time);
		i1.y = 0;
		art_affine_point (&c1, &i1, i2c);
		b = floor (c1.x + 0.5) - x;

		/* Draw a button but only make it look "buttony" at the
		 * bottom, since it looks less cluttered (hence the
		 * -10/+2*10 and GTK_POS_TOP).
		 */
		gtk_draw_extension (style,
				    drawable,
				    GTK_STATE_NORMAL,
				    GTK_SHADOW_OUT,
				    a - 10,
				    cy1,
				    b - a + 2*10,
				    cy2 - cy1,
				    GTK_POS_TOP);
		
		gdk_draw_line (drawable, priv->line_gc, a, cy_mid, b, cy_mid);
	}

	/* Get the pixel width of a minor tick. Use it to decide if to
	 * draw them.
	 */
	i1.x = gantt_scale_t2w (priv->gantt_scale, priv->gantt_scale->t1);
	i1.y = 0;
	art_affine_point (&c1, &i1, i2c);
	tick_width = floor (c1.x + 0.5) - x;

	tick_time = gantt_scale_increase_one_tick (priv->gantt_scale,
						   GANTT_UNIT_MINOR,
						   priv->gantt_scale->t1);

	i1.x = gantt_scale_t2w (priv->gantt_scale, tick_time);
	i1.y = 0;
	art_affine_point (&c1, &i1, i2c);
	tick_width = (floor (c1.x + 0.5) - x) - tick_width;
	
	if (tick_width < 20) {
		/* Too tight, don't draw'em... */
		draw_minor_tick_marks = FALSE;
	}
	else {
		draw_minor_tick_marks = TRUE;
	}
	
	/*
	 * Draw the major (top half) scale.
	 */
	time = gantt_scale_snap_time (priv->gantt_scale,
				      GANTT_UNIT_MAJOR,
				      first_time);
	/* Time to canvas coord. */
	i1.x = gantt_scale_t2w (priv->gantt_scale, time);
	i1.y = 0;
	art_affine_point (&c1, &i1, i2c);
	xd = floor (c1.x + 0.5) - x;

	next_time = gantt_scale_increase_one_tick (priv->gantt_scale,
						   GANTT_UNIT_MAJOR,
						   time);
	/* Time to canvas coord. */
	i1.x = gantt_scale_t2w (priv->gantt_scale, next_time);
	i1.y = 0;
	art_affine_point (&c1, &i1, i2c);
	next_xd = floor (c1.x + 0.5) - x;
       
	do {
		text = gantt_scale_format_time (priv->gantt_scale,
						GANTT_UNIT_MAJOR,
						time);

		/* Clip so we don't make a mess. */
		rect.x = xd;
		rect.y = cy1;
		rect.width = next_xd - xd;
		rect.height = cy2 - cy1;
		gdk_gc_set_clip_rectangle (priv->text_gc,
					   &rect);
		gdk_draw_text (drawable, priv->font, priv->text_gc,
			       xd + 5,
			       ypos1,
			       text,
			       strlen (text));
		/* Unclip. */
		gdk_gc_set_clip_rectangle (priv->text_gc, NULL);
		
		g_free (text);
			
		/* Don't draw a grid line at the edges, since it looks bad. */
		if (xd + x > 0) {
			gdk_draw_line (drawable, priv->line_gc,
				       xd, cy1,
				       xd, draw_minor_tick_marks ? cy_mid : (cy2 - 2));
		}
		
		time = next_time;
		xd = next_xd;
		next_time = gantt_scale_increase_one_tick (priv->gantt_scale,
							   GANTT_UNIT_MAJOR,
							   time);
		/* Time to canvas coord. */
		i1.x = gantt_scale_t2w (priv->gantt_scale, next_time);
		i1.y = 0;
		art_affine_point (&c1, &i1, i2c);
		next_xd = floor (c1.x + 0.5) - x;
	} while (time < last_time);

	if (!draw_minor_tick_marks) {
		return;
	}

	/*
	 * Draw the minor (bottom half) scale.
	 */
	time = gantt_scale_snap_time (priv->gantt_scale,
				      GANTT_UNIT_MINOR,
				      first_time);
	/* Time to canvas coord. */
	i1.x = gantt_scale_t2w (priv->gantt_scale, time);
	i1.y = 0;
	art_affine_point (&c1, &i1, i2c);
	xd = floor (c1.x + 0.5) - x;
	
	next_time = gantt_scale_increase_one_tick (priv->gantt_scale,
						   GANTT_UNIT_MINOR,
						   time);
	/* Time to canvas coord. */
	i1.x = gantt_scale_t2w (priv->gantt_scale, next_time);
	i1.y = 0;
	art_affine_point (&c1, &i1, i2c);
	next_xd = floor (c1.x + 0.5) - x;
	
	do {
		text = gantt_scale_format_time (priv->gantt_scale,
						GANTT_UNIT_MINOR,
						time);

		/* Clip so we don't make a mess. */
		rect.x = xd;
		rect.y = cy1;
		rect.width = next_xd - xd;
		rect.height = cy2 - cy1;
		gdk_gc_set_clip_rectangle (priv->text_gc,
					   &rect);
		gdk_draw_text (drawable, priv->font, priv->text_gc,
			       xd + 5,
			       ypos2,
			       text,
			       strlen (text));
		gdk_gc_set_clip_rectangle (priv->text_gc, NULL);
		g_free (text);
		
		/* Don't draw a grid line at the left edge, since it looks bad. */
		if (xd + x > 0) {
			gdk_draw_line (drawable, priv->line_gc,
				       xd, cy_mid,
				       xd, cy2 - 2);
		}
			
		if (time >= last_time) {
			break;
		}

		time = next_time;
		xd = next_xd;
		
		next_time = gantt_scale_increase_one_tick (priv->gantt_scale,
							   GANTT_UNIT_MINOR,
							   time);
		/* Time to canvas coord. */
		i1.x = gantt_scale_t2w (priv->gantt_scale, next_time);
		i1.y = 0;
		art_affine_point (&c1, &i1, i2c);
		next_xd = floor (c1.x + 0.5) - x;		
	} while (time < last_time);
}

static void
gantt_header_item_class_init (GanttHeaderItemClass *klass)
{
	GtkObjectClass       *object_class;
	GnomeCanvasItemClass *item_class;
	
	object_class = (GtkObjectClass*) klass;
	item_class = (GnomeCanvasItemClass*) klass;

	/* Gtk Object methods. */
	object_class->destroy = gantt_header_item_destroy;
	object_class->set_arg = gantt_header_item_set_arg;
	object_class->get_arg = gantt_header_item_get_arg;

	/* GnomeCanvasItem methods. */
	item_class->update = gantt_header_item_update;
	item_class->draw = gantt_header_item_draw;
	item_class->point = gantt_header_item_point;
	item_class->bounds = gantt_header_item_bounds_item_coordinates;
	item_class->realize = gantt_header_item_realize;
	item_class->unrealize = gantt_header_item_unrealize;

	/* Arguments. */
	gtk_object_add_arg_type ("GanttHeaderItem::gantt_model",
				 GTK_TYPE_POINTER,
				 GTK_ARG_WRITABLE,
				 ARG_GANTT_MODEL);
	gtk_object_add_arg_type ("GanttHeaderItem::gantt_scale",
				 GTK_TYPE_POINTER,
				 GTK_ARG_WRITABLE,
				 ARG_GANTT_SCALE);
	gtk_object_add_arg_type ("GanttHeaderItem::height",
				 GTK_TYPE_DOUBLE,
				 GTK_ARG_READABLE,
				 ARG_HEIGHT);
}

static void
gantt_header_item_init (GanttHeaderItem *item)
{
	item->priv = g_new0 (GanttHeaderItemPriv, 1);
	item->priv->x1 = 0;
	item->priv->y1 = 0;
	item->priv->x2 = 0;
	item->priv->y2 = 0;
}

static gdouble
print_get_row_height (GanttPrintInfo *print_info)
{
	return 1.25 * gnome_font_get_size (print_info->font) + 2;
}

/* bbox is the target print area, where t1 is to be printed
 * at (0, 0) relative that area.
 */
static void
gantt_header_item_print_page (GanttPrintable    *printable,
			      GanttPrintInfo    *print_info,
			      ArtDRect          *bbox,
			      gdouble            hscale,
			      gdouble            vscale,
			      time_t             t1,
			      time_t             t2,
			      gint               r1,
			      gint               r2,
			      GanttHeaderItem   *header)
{
	GanttHeaderItemPriv *priv;
	GnomePrintContext   *context;
	double               x1, y1, x2, y2, y_mid, height;
	time_t               time;
	gint                 xd;
	gchar               *text;
	
	priv = header->priv;
	height = priv->y1 - priv->y2;
	context = print_info->context;

	x1 = 0;
	x2 = hscale * (gantt_scale_t2w (priv->gantt_scale, t2) -
		       gantt_scale_t2w (priv->gantt_scale, t1));

	x1 += bbox->x0;
	x2 += bbox->x0;

	y1 = 0;
	y2 = 2 * print_get_row_height (print_info);
	y_mid = y2 / 2;
	y1 += bbox->y0;
	y2 += bbox->y0;
	y_mid += bbox->y0; 
	
	/* Draw the two frames for the scale. */
	gantt_print_rectangle (print_info, x1, y1, x2, y2);
	gantt_print_line (print_info, x1, y_mid, x2, y_mid);

	/*
	 * Draw the major (top half) scale.
	 */
	time = gantt_scale_snap_time (priv->gantt_scale,
				      GANTT_UNIT_MAJOR,
				      t1);
	xd = 0;
	do {
		text = gantt_scale_format_time (priv->gantt_scale,
						GANTT_UNIT_MAJOR,
						time);
		gantt_print_text (print_info,
				  hscale * xd + bbox->x0,
				  y_mid,
				  text);
		g_free (text);
		
		if (xd > 0) {
			gantt_print_line (print_info,
					  hscale * xd + bbox->x0,
					  y1,
					  hscale * xd + bbox->x0,
					  y2);
		}
		
		time = gantt_scale_increase_one_tick (priv->gantt_scale,
						      GANTT_UNIT_MAJOR,
						      time);
		xd = gantt_scale_t2w (priv->gantt_scale, time) -
			gantt_scale_t2w (priv->gantt_scale, t1);
		
	} while (time < t2);

	/*
	 * Draw the minor (bottom half) scale.
	 */
	time = gantt_scale_snap_time (priv->gantt_scale,
				      GANTT_UNIT_MINOR,
				      t1);
	xd = 0;
	do {
		text = gantt_scale_format_time (priv->gantt_scale,
						GANTT_UNIT_MINOR,
						time);
		gantt_print_text (print_info,
				  hscale * xd + bbox->x0,
				  y2,
				  text);
		g_free (text);

		/* Don't draw a grid line at the left edge, since it looks bad. */
		if (xd > 0) {
			gantt_print_line (print_info,
					  hscale * xd + bbox->x0,
					  y_mid,
					  hscale * xd + bbox->x0,
					  y2);
		}
			
		time = gantt_scale_increase_one_tick (priv->gantt_scale,
						      GANTT_UNIT_MINOR,
						      time);
		xd = gantt_scale_t2w (priv->gantt_scale, time) -
			gantt_scale_t2w (priv->gantt_scale, t1);
	} while (time < t2);
}

static gdouble
gantt_header_item_get_width (GanttPrintable  *printable,
			     GanttPrintInfo  *print_info,
			     time_t           t1,
			     time_t           t2,
			     GanttHeaderItem *item)
{
	gdouble x1, x2;

	x1 = gantt_scale_t2w (item->priv->gantt_scale, t1);
	x2 = gantt_scale_t2w (item->priv->gantt_scale, t2);

	return x2 - x1;
}

static gdouble
gantt_header_item_get_height (GanttPrintable  *printable,
			      GanttPrintInfo  *print_info,
			      GanttHeaderItem *item)
{
	gdouble y1, y2;

	y1 = 0;
	y2 = 2 * print_get_row_height (print_info);
	
	return y2 - y1;
}

GanttPrintable *
gantt_header_item_get_printable (GanttHeaderItem *item)
{
	GanttPrintable *printable;
		
	g_return_val_if_fail (item != NULL, NULL);
	g_return_val_if_fail (GANTT_HEADER_ITEM (item), NULL);

	printable = gantt_printable_new ();

	gtk_signal_connect (GTK_OBJECT (printable),
			    "print_page",
			    gantt_header_item_print_page,
			    item);

	gtk_signal_connect (GTK_OBJECT (printable),
			    "get_width",
			    GTK_SIGNAL_FUNC (gantt_header_item_get_width),
			    item);

	gtk_signal_connect (GTK_OBJECT (printable),
			    "get_height",
			    GTK_SIGNAL_FUNC (gantt_header_item_get_height),
			    item);
	
	return printable;
}

