/*
 * Copyright (C) 2001  CodeFactory AB
 * Copyright (C) 2001  Richard Hult
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult <rhult@codefactory.se>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <gtk/gtk.h>
#include <libgnomeui/gnome-canvas.h>
#include <libgnomeui/gnome-canvas-util.h>
#include <libgnomeui/gnome-canvas-line.h>

#include "util/type-utils.h"
#include "gantt-arrow-item.h"

static void gantt_arrow_item_init	    (GanttArrowItem    *gantt_arrow_item);
static void gantt_arrow_item_class_init   (GanttArrowItemClass *klass);

GNOME_CLASS_BOILERPLATE (GanttArrowItem,
			 gantt_arrow_item,
			 GnomeCanvasLine,
			 gnome_canvas_line);


static void
gantt_arrow_item_class_init (GanttArrowItemClass *klass)
{
}

static void
gantt_arrow_item_init (GanttArrowItem *item)
{
}

static void
update_item (GanttArrowItem *arrow)
{
	GnomeCanvasPoints *points;
	gdouble            y;

	g_return_if_fail (arrow != NULL);
	g_return_if_fail (IS_GANTT_ARROW_ITEM (arrow));
	
	gtk_object_get (GTK_OBJECT (arrow), "points", &points, NULL);

	g_return_if_fail (points != NULL);
	g_return_if_fail (points->num_points == 3);

	/*
                x2,y2_bottom
	       |
	  ------
         x1,y1

	 or

         x1,y1
	  ------
	       |
                 x2,y2_top
	 
	*/
	
	points->coords[0] = arrow->x1;
	points->coords[1] = arrow->y1;

	points->coords[2] = arrow->x2;
	points->coords[3] = arrow->y1;

	/* If the predecessor is below the item, make the arrow point
	 * to the bottom of the item, otherwise the top.
	 */
	if (arrow->y1 > arrow->y2_top) {
		y = arrow->y2_bottom;
	} else {
		y = arrow->y2_top;
	}
	
	points->coords[4] = arrow->x2;
	points->coords[5] = y;

	gnome_canvas_item_set (GNOME_CANVAS_ITEM (arrow),
			       "points", points,
			       NULL);
}

static void
item_moved (GanttRowItem   *item,
	    gdouble         x1,
	    gdouble         y1,
	    gdouble         x2,
	    gdouble         y2,
	    GanttArrowItem *arrow)
{
	GnomeCanvasPoints *points;

	g_return_if_fail (item != NULL);
	g_return_if_fail (IS_GANTT_ROW_ITEM (item));
	g_return_if_fail (arrow != NULL);
	g_return_if_fail (IS_GANTT_ARROW_ITEM (arrow));

	gtk_object_get (GTK_OBJECT (arrow), "points", &points, NULL);

	g_return_if_fail (points != NULL);
	g_return_if_fail (points->num_points == 3);

	gantt_row_item_get_geometry (item, &x1, &y1, &x2, &y2);

	arrow->x2 = x1;
	arrow->y2_top = y1 + 4;
	arrow->y2_bottom = y2 - 4;

	update_item (arrow);
}

static void
predecessor_moved (GanttRowItem   *item,
		   gdouble         x1,
		   gdouble         y1,
		   gdouble         x2,
		   gdouble         y2,
		   GanttArrowItem *arrow)
{
	GnomeCanvasPoints *points;

	g_return_if_fail (item != NULL);
	g_return_if_fail (IS_GANTT_ROW_ITEM (item));
	g_return_if_fail (arrow != NULL);
	g_return_if_fail (IS_GANTT_ARROW_ITEM (arrow));
	
	gtk_object_get (GTK_OBJECT (arrow), "points", &points, NULL);

	g_return_if_fail (points != NULL);
	g_return_if_fail (points->num_points == 3);
	
	gantt_row_item_get_geometry (item, &x1, &y1, &x2, &y2);

	arrow->x1 = x2;
	arrow->y1 = y1 + (y2 - y1) / 2.0;

	update_item (arrow);
}

GanttArrowItem *
gantt_arrow_item_new (GanttRowItem *item, GanttRowItem *predecessor)
{
	GanttArrowItem    *arrow;
	GnomeCanvasGroup  *root;
	GnomeCanvasPoints *points;
	gdouble            x1, y1, x2, y2;

	root = gnome_canvas_root (GNOME_CANVAS_ITEM (item)->canvas);
	points = gnome_canvas_points_new (3);

	arrow = GANTT_ARROW_ITEM (gnome_canvas_item_new (
		root,
		TYPE_GANTT_ARROW_ITEM,
		"points", points,
		"last_arrowhead", TRUE,
		"arrow_shape_a", 6.0,
		"arrow_shape_b", 6.0,
		"arrow_shape_c", 3.0,
		"fill_color", "black",
		"width_pixels", (guint) 1,
		"join_style", GDK_JOIN_MITER,
		NULL));
	
	gnome_canvas_points_unref (points);

	arrow->item = item;
	arrow->predecessor = predecessor;
	
	gantt_row_item_get_geometry (predecessor, &x1, &y1, &x2, &y2);
	arrow->x1 = x2;
	arrow->y1 = y1 + (y2 - y1) / 2;

	gantt_row_item_get_geometry (item, &x1, &y1, &x2, &y2);
	arrow->x2 = x1;
	arrow->y2_top = y1;
	arrow->y2_bottom = y2;

	gtk_signal_connect_while_alive (GTK_OBJECT (item),
					"changed",
					item_moved,
					arrow,
					GTK_OBJECT (arrow));

	gtk_signal_connect_while_alive (GTK_OBJECT (predecessor),
					"changed",
					predecessor_moved,
					arrow,
					GTK_OBJECT (arrow));
	
	update_item (arrow);

	return arrow;
}
