/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 Thomas Nyberg <thomas@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Thomas Nyberg
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <gal/widgets/e-unicode.h>
#include <gal/e-table/e-table.h>
#include <gal/e-table/e-table-extras.h>
#include <gal/e-table/e-table-scrolled.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtksignal.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <glade/glade.h>

#include "util/type-utils.h"
#include "libmrproject/GNOME_MrProject.h"
#include "resource-filter-dialog-gui.h"
#include "filter-table-model.h"
#include "util/id-map.h"

#define DEBUG 1
#include "util/debug.h"

static void rfdg_button_apply_cb (GtkWidget *widget, ResourceFilterDialogGui *gui);
static void rfdg_button_clear_cb (GtkWidget *widget, ResourceFilterDialogGui *gui);
static void resource_filter_dialog_gui_init (ResourceFilterDialogGui *gui);
static void resource_filter_dialog_gui_class_init (ResourceFilterDialogGuiClass *class);
static void rfdg_create_gui (ResourceFilterDialogGui *gui);
static void rfdg_connect_signals (ResourceFilterDialogGui *gui);
static void rfdg_set_sensitivity (ResourceFilterDialogGui *gui);
static void rfdg_resource_deallocated_cb (GtkWidget               *widget, 
					  GM_Id                    id,
					  ResourceFilterDialogGui *gui);
static void rfdg_resource_allocated_cb (GtkWidget               *widget, 
					GM_Id                    id,
					ResourceFilterDialogGui *gui);
struct _ResourceInfo {
	GM_Resource *resource;
	gboolean selected;
};

struct _ResourceFilterDialogGuiPriv {
	
	IdMap *resources; /* contains struct _ResourceInfo */

	GtkWidget   *notebook;
	GtkWidget   *button_close;
	GtkWidget   *button_apply;
	GtkWidget   *button_clear;

	GtkWidget   *filter_table;

	ETableModel *ftm;

	gboolean dirty;
};

enum {
	RESOURCE_CHECKED,
	RESOURCE_UNCHECKED,
	APPLY_FILTER,
	CLEAR_FILTER,
	LAST_SIGNAL
};

static gint signals[LAST_SIGNAL] = { 0 };

GNOME_CLASS_BOILERPLATE (ResourceFilterDialogGui, resource_filter_dialog_gui,
			 GtkWindow, gtk_window);

static void
resource_filter_dialog_gui_init (ResourceFilterDialogGui *gui)
{
	ResourceFilterDialogGuiPriv *priv;

	priv = (ResourceFilterDialogGuiPriv *)g_new0 (ResourceFilterDialogGuiPriv, 1);
	gui->priv = priv;
	priv->dirty = FALSE;
	priv->resources = id_map_new (0);

	gtk_window_set_wmclass (GTK_WINDOW (gui),
                                "Resource Filter",
                                "MrProject");
        gtk_window_set_title (GTK_WINDOW (gui),
                              _("Resource Filter - MrProject"));

	rfdg_create_gui (gui);
	rfdg_connect_signals (gui);
}

static void
resource_filter_dialog_gui_class_init (ResourceFilterDialogGuiClass *class)
{
	GtkObjectClass *object_class;

	object_class = GTK_OBJECT_CLASS (class);

        /* Signals */
        signals[RESOURCE_CHECKED] =
                gtk_signal_new ("resource_checked",
                                GTK_RUN_LAST,
                                object_class->type,
                                GTK_SIGNAL_OFFSET (ResourceFilterDialogGuiClass,
                                                   resource_checked),
                                gtk_marshal_NONE__POINTER,
                                GTK_TYPE_NONE,
                                1, GTK_TYPE_POINTER);
        signals[RESOURCE_UNCHECKED] =
                gtk_signal_new ("resource_unchecked",
                                GTK_RUN_LAST,
                                object_class->type,
                                GTK_SIGNAL_OFFSET (ResourceFilterDialogGuiClass,
                                                   resource_unchecked),
                                gtk_marshal_NONE__POINTER,
                                GTK_TYPE_NONE,
                                1, GTK_TYPE_POINTER);
        signals[APPLY_FILTER] =
                gtk_signal_new ("apply_filter",
                                GTK_RUN_LAST,
                                object_class->type,
                                GTK_SIGNAL_OFFSET (ResourceFilterDialogGuiClass,
                                                   apply_filter),
                                gtk_marshal_NONE__POINTER,
                                GTK_TYPE_NONE,
                                1, GTK_TYPE_POINTER);
        signals[CLEAR_FILTER] =
                gtk_signal_new ("clear_filter",
                                GTK_RUN_LAST,
                                object_class->type,
                                GTK_SIGNAL_OFFSET (ResourceFilterDialogGuiClass,
                                                   clear_filter),
                                gtk_marshal_NONE__NONE,
                                GTK_TYPE_NONE,
                                0);

	gtk_object_class_add_signals (object_class, signals, LAST_SIGNAL);
}


static void
rfdg_create_gui (ResourceFilterDialogGui *gui)
{
	GladeXML *glade;
	GtkWidget *window;
	GtkWidget *contents;
	GtkWidget *resource_box;

	g_assert (gui != NULL);
	g_assert (IS_RESOURCE_FILTER_DIALOG_GUI (gui));
	g_assert (gui->priv != NULL);

	d(puts(__FUNCTION__));

	glade = glade_xml_new (MRPROJECT_DATADIR "resource-filter-dialog-gui.glade", NULL);

	window = glade_xml_get_widget (glade, "resource_filter_dialog");
	contents = glade_xml_get_widget (glade, "dialog_contents");

	gtk_widget_reparent (contents, GTK_WIDGET (gui));
	gtk_widget_destroy (window);

	gui->priv->notebook = glade_xml_get_widget (glade, "notebook");
	gui->priv->button_close = glade_xml_get_widget (glade, "button_close");
	gui->priv->button_apply = glade_xml_get_widget (glade, "button_apply");
	gui->priv->button_clear = glade_xml_get_widget (glade, "button_clear");

	resource_box = glade_xml_get_widget (glade, "resource_box");

	gui->priv->ftm = filter_table_model_new ();
	gui->priv->filter_table = 
		e_table_scrolled_new_from_spec_file (E_TABLE_MODEL (gui->priv->ftm), 
						     e_table_extras_new (), 
						     MRPROJECT_DATADIR "resource-filter-table.etspec", 
						     MRPROJECT_DATADIR "resource-filter-table.etstate");
	gtk_box_pack_start (GTK_BOX (resource_box), 
			    gui->priv->filter_table, 
			    TRUE, TRUE, 0);

	
	gtk_object_unref (GTK_OBJECT (glade));
}

static void
rfdg_connect_signals (ResourceFilterDialogGui *gui)
{
	g_assert (gui != NULL);
	g_assert (IS_RESOURCE_FILTER_DIALOG_GUI (gui));

	d(puts(__FUNCTION__));
	
	gtk_signal_connect_object (GTK_OBJECT (gui->priv->button_close),
				   "clicked",
				   GTK_SIGNAL_FUNC (gtk_object_destroy),
				   GTK_OBJECT (gui));
	gtk_signal_connect (GTK_OBJECT (gui->priv->button_apply),
			    "clicked",
			    GTK_SIGNAL_FUNC (rfdg_button_apply_cb),
			    gui);
	gtk_signal_connect (GTK_OBJECT (gui->priv->button_clear),
			    "clicked",
			    GTK_SIGNAL_FUNC (rfdg_button_clear_cb),
			    gui);
	gtk_signal_connect (GTK_OBJECT (gui->priv->ftm),
			    "resource_allocated",
			    GTK_SIGNAL_FUNC (rfdg_resource_allocated_cb),
			    gui);
	gtk_signal_connect (GTK_OBJECT (gui->priv->ftm),
			    "resource_deallocated",
			    GTK_SIGNAL_FUNC (rfdg_resource_deallocated_cb),
			    gui);
}

static void
rfdg_resource_allocated_cb (GtkWidget               *widget, 
			    GM_Id                    id,
			    ResourceFilterDialogGui *gui)
{
	ResourceFilterDialogGuiPriv *priv;
	struct _ResourceInfo *res_info;

	g_return_if_fail (gui != NULL);
	g_return_if_fail (IS_RESOURCE_FILTER_DIALOG_GUI (gui));

	d(puts(__FUNCTION__));

	priv = gui->priv;

	res_info = (struct _ResourceInfo *)id_map_lookup (priv->resources,
							  id);

	g_assert (res_info != NULL); /* just in case */

	res_info->selected = TRUE;
	priv->dirty = TRUE;

	rfdg_set_sensitivity (gui);

	gtk_signal_emit (GTK_OBJECT (gui), signals[RESOURCE_CHECKED], id);
}

static void
rfdg_resource_deallocated_cb (GtkWidget               *widget, 
			      GM_Id                    id,
			      ResourceFilterDialogGui *gui)
{
	ResourceFilterDialogGuiPriv *priv;
	struct _ResourceInfo *res_info;

	g_return_if_fail (gui != NULL);
	g_return_if_fail (IS_RESOURCE_FILTER_DIALOG_GUI (gui));

	d(puts(__FUNCTION__));

	priv = gui->priv;

	res_info = (struct _ResourceInfo *)id_map_lookup (priv->resources,
							  id);

	g_assert (res_info != NULL); /* just in case */

	res_info->selected = FALSE;
	priv->dirty = TRUE;

	rfdg_set_sensitivity (gui);

	gtk_signal_emit (GTK_OBJECT (gui), signals[RESOURCE_UNCHECKED], id);
}

static void
rfdg_button_clear_cb (GtkWidget *widget, ResourceFilterDialogGui *gui)
{
	g_assert (gui != NULL);
	g_assert (IS_RESOURCE_FILTER_DIALOG_GUI (gui));
	
	d(puts(__FUNCTION__));

	gtk_signal_emit (GTK_OBJECT (gui), signals[CLEAR_FILTER], NULL);
}

static void
rfdg_button_apply_cb (GtkWidget *widget, ResourceFilterDialogGui *gui)
{
	GSList *node, *list, *filter;
	struct _ResourceInfo *res_info;

	g_assert (gui != NULL);
	g_assert (IS_RESOURCE_FILTER_DIALOG_GUI (gui));
	
	d(puts(__FUNCTION__));


	list = id_map_get_objects (gui->priv->resources);

	filter = NULL;
	for (node = list; node; node = node->next) {
		g_assert (node->data != NULL);

		res_info = (struct _ResourceInfo *)node->data;

		if (res_info->selected) {
			filter = g_slist_prepend (filter, GINT_TO_POINTER (res_info->resource->resourceId));
		}
	}

	gtk_signal_emit (GTK_OBJECT (gui), signals[APPLY_FILTER], filter);

	g_slist_free (filter);
	g_slist_free (list);

	gui->priv->dirty = FALSE;
	rfdg_set_sensitivity (gui);
}

static void 
rfdg_set_sensitivity (ResourceFilterDialogGui *gui)
{
	g_assert (gui != NULL);
	g_assert (IS_RESOURCE_FILTER_DIALOG_GUI (gui));
	g_assert (gui->priv != NULL);

	gtk_widget_set_sensitive (gui->priv->button_apply, 
				  gui->priv->dirty);
}

GtkWidget *
resource_filter_dialog_gui_new (void)
{
	ResourceFilterDialogGui *gui;

	gui = gtk_type_new (RESOURCE_FILTER_DIALOG_GUI_TYPE);

	rfdg_set_sensitivity (gui); /* update sensitivities */

	return GTK_WIDGET (gui);
}


void
resource_filter_dialog_gui_add_resource (ResourceFilterDialogGui *gui,
					 GM_Resource *resource,
					 gboolean selected)
{
	struct _ResourceInfo *resinfo;

	g_return_if_fail (gui != NULL);
	g_return_if_fail (IS_RESOURCE_FILTER_DIALOG_GUI (gui));
	g_return_if_fail (resource != NULL);

	d(puts(__FUNCTION__));

	resinfo = (struct _ResourceInfo *)g_new0 (struct _ResourceInfo, 1);
	resinfo->resource = resource;
	resinfo->selected = selected;

	id_map_insert_id (gui->priv->resources, resource->resourceId, resinfo);

	filter_table_model_add_resource (FILTER_TABLE_MODEL (gui->priv->ftm),
					 resource);
	filter_table_model_set_allocation (FILTER_TABLE_MODEL (gui->priv->ftm),
					   resource->resourceId, selected);
}

void
resource_filter_dialog_gui_remove_resource (ResourceFilterDialogGui *gui,
					    GM_Id                    id)
{
	struct _ResourceInfo *resinfo;

	g_return_if_fail (gui != NULL);
	g_return_if_fail (IS_RESOURCE_FILTER_DIALOG_GUI (gui));
	
	d(puts(__FUNCTION__));

	resinfo = (struct _ResourceInfo *)id_map_lookup (gui->priv->resources,
							 id);
	g_assert (resinfo != NULL);

	filter_table_model_remove_resource (FILTER_TABLE_MODEL (gui->priv->ftm),
					    id);
	id_map_remove (gui->priv->resources, id);
	/* remember, resinfo->resource is _not_ a copy */
	g_free (resinfo);
}

void
resource_filter_dialog_gui_add_resources (ResourceFilterDialogGui *gui,
					  GSList *resources)
{
	g_return_if_fail (gui != NULL);
	g_return_if_fail (IS_RESOURCE_FILTER_DIALOG_GUI (gui));
	g_return_if_fail (resources != NULL);

	d(puts(__FUNCTION__));
}

void
resource_filter_dialog_gui_update_resource (ResourceFilterDialogGui *gui,
					    GM_Resource *resource)
{
	g_return_if_fail (gui != NULL);
	g_return_if_fail (IS_RESOURCE_FILTER_DIALOG_GUI (gui));
	g_return_if_fail (resource != NULL);

	d(puts(__FUNCTION__));

	filter_table_model_update_resource (FILTER_TABLE_MODEL (gui->priv->ftm), 
					    resource);
}
