/* -*- tab-width: 4 -*- */

/*
 *  mod_bt - Making Things Better For Seeders
 *  Copyright 2004, 2005, 2006 Tyler MacDonald <tyler@yi.org>
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  Small snippets of code in this file (mod_bt.c) were copied from portions
 *  of the Apache 2.0.49 source code tree.
 */

/* apache */
#include "httpd.h"
#include "http_config.h"
#include "http_log.h"
#include "http_request.h"
#include "http_protocol.h"
#include "http_core.h"
#include "util_filter.h"

#include "apr.h"
#include "apr_strings.h"
#include "apr_hash.h"
#include "apr_general.h"
#include "apr_portable.h"
#include "apr_lib.h"
#include "apr_user.h"
#include "mpm.h"
#include "apr_dbm.h"
#include "apr_optional.h"
#define APR_WANT_STRFUNC
#include "apr_want.h"
#include "apr_signal.h"
#include "apr_pools.h"

#include "mod_bt.h"

#include <stdio.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <arpa/inet.h>

#include <libbttracker.h>

module bt_module;

#define VARY_KEY_THIS "mod_bt-Vary"

static server_rec *mys = NULL;
static char mytop = 1;
static apr_sigfunc_t *osig = NULL;

/** MODULE CONFIGURATION & CALLBACKS **/

static const char *modbt_config_Tracker(cmd_parms *cmd, void* dconf, int flag) {
 modbt_config_t* conf = ap_get_module_config(cmd->server->module_config, &bt_module);
 conf->enabled = flag;
 return NULL;
}

static const char* modbt_config_TrackerHome(cmd_parms* cmd, void* dconf, const char* val) {
 modbt_config_t* conf = ap_get_module_config(cmd->server->module_config, &bt_module);
 conf->tcfg.db.dir[sizeof(conf->tcfg.db.dir)-1] = 0;
 strncpy(conf->tcfg.db.dir, val, sizeof(conf->tcfg.db.dir) - 1);
 conf->tcfg.homedir = strdup(val);
 strcpy(conf->tcfg.c->db_dir, conf->tcfg.db.dir);
 return NULL;
}

static const char* modbt_config_TrackerStyleSheet(cmd_parms* cmd, void* dconf, const char* val)
{
 modbt_config_t* conf = ap_get_module_config(cmd->server->module_config, &bt_module);
 conf->tcfg.c->stylesheet[sizeof(conf->tcfg.c->stylesheet)-1] = 0;
 strncpy(conf->tcfg.c->stylesheet, val, sizeof(conf->tcfg.c->stylesheet) - 1);
 return NULL;
}

static const char* modbt_config_TrackerDetailURL(cmd_parms* cmd, void* dconf, const char* val)
{
 modbt_config_t* conf = ap_get_module_config(cmd->server->module_config, &bt_module);
 conf->tcfg.c->detail_url[sizeof(conf->tcfg.c->detail_url)-1] = 0;
 strncpy(conf->tcfg.c->detail_url, val, sizeof(conf->tcfg.c->detail_url) - 1);
 return NULL;
}

static const char* modbt_config_TrackerRootInclude(cmd_parms* cmd, void* dconf, const char* val)
{
 modbt_config_t* conf = ap_get_module_config(cmd->server->module_config, &bt_module);
 conf->tcfg.c->root_include[sizeof(conf->tcfg.c->root_include)-1] = 0;
 strncpy(conf->tcfg.c->root_include, val, sizeof(conf->tcfg.c->root_include) - 1);
 return NULL;
}


static const char* modbt_config_TrackerReturnPeers(cmd_parms* cmd, void* dconf, const char* val)
{
 modbt_config_t* conf = ap_get_module_config(cmd->server->module_config, &bt_module);
 int i = atoi(val);
 if(i)
 {
  conf->tcfg.c->return_peers = i;
  return NULL;
 }
 else
 {
  return "TrackerReturnPeers must be a number.";
 }
}

static const char* modbt_config_TrackerReturnMax(cmd_parms* cmd, void* dconf, const char* val)
{
 modbt_config_t* conf = ap_get_module_config(cmd->server->module_config, &bt_module);
 int i = atoi(val);
 if(i)
 {
  conf->tcfg.c->return_max = i;
  return NULL;
 }
 else
 {
  return "TrackerReturnMax must be a number.";
 }
}

static const char* modbt_config_TrackerReturnInterval(cmd_parms* cmd, void* dconf, const char* val)
{
 modbt_config_t* conf = ap_get_module_config(cmd->server->module_config, &bt_module);
 long i = atol(val);
 if(i)
 {
  conf->tcfg.c->return_interval = i;
  return NULL;
 }
 else
 {
  return "TrackerReturnInterval must be a number.";
 }
}

static const char* modbt_config_TrackerReturnPeerFactor(cmd_parms* cmd, void* dconf, const char* val)
{
 modbt_config_t* conf = ap_get_module_config(cmd->server->module_config, &bt_module);
 int i = atoi(val);
 if(i)
 {
  conf->tcfg.c->return_peer_factor = i;
  return NULL;
 }
 else
 {
  return "TrackerReturnPeerFactor must be a number.";
 }
}

static const char* modbt_config_TrackerHashWatermark(cmd_parms* cmd, void* dconf, const char* val)
{
 modbt_config_t* conf = ap_get_module_config(cmd->server->module_config, &bt_module);
 int i = atoi(val);
 if(i)
 {
  conf->tcfg.c->hash_watermark = i;
  return NULL;
 }
 else
 {
  return "TrackerHashWatermark must be a number.";
 }
}

static const char* modbt_config_TrackerHashMinAge(cmd_parms* cmd, void* dconf, const char* val)
{
 modbt_config_t* conf = ap_get_module_config(cmd->server->module_config, &bt_module);
 long i = atol(val);
 if(i)
 {
  conf->tcfg.c->hash_min_age = i;
  return NULL;
 }
 else
 {
  return "TrackerReturnHashMinAge must be a number.";
 }
}

static const char* modbt_config_TrackerHashMaxAge(cmd_parms* cmd, void* dconf, const char* val)
{
 modbt_config_t* conf = ap_get_module_config(cmd->server->module_config, &bt_module);
 long i = atol(val);
 if(i)
 {
  conf->tcfg.c->hash_max_age = i;
  return NULL;
 }
 else
 {
  return "TrackerHashMaxAge must be a number.";
 }
}

static const char* modbt_config_TrackerFlags(cmd_parms* cmd, void* dconf, const char* val)
{
 modbt_config_t* conf = ap_get_module_config(cmd->server->module_config, &bt_module);
 
 int i = 0;
 int bad = 0;
 const char* opts = NULL;
 char* w = NULL;
 char* badflag = NULL;
 int onoff = 1;
 int flag = 0;
 
 if(val && *val)
  opts = apr_pstrdup(cmd->pool, val);
 
 while(opts && *opts)
 {
  onoff=1;
  w = ap_getword_conf(cmd->pool, &opts);
  if(w && *w)
  {
   if(*w == '+')
   {
    w++;
   }
   else if(*w == '-')
   {
    onoff = 0;
    w++;
   }
   
   if(w && *w)
   {
   	flag = 0;
    for(i=0;(!flag) && (btt_tracker_flags[i].flag);i++)
     if(!strcasecmp(btt_tracker_flags[i].config_name, w))
      flag = btt_tracker_flags[i].flag;

    if(flag)
    {
     if(onoff)
      conf->tcfg.c->flags = conf->tcfg.c->flags | flag;
     else if(conf->tcfg.c->flags & flag)
      conf->tcfg.c->flags -= flag;
    }
    else
    {
     bad++;
     badflag = apr_pstrdup(cmd->pool, w);
    }
   }
  }
 }
 
 if(bad)
 {
  w = "The following flags are valid (+/- = current setting)";
  for(i=0;btt_tracker_flags[i].flag;i++)
  {
   if(new_btt_tracker_config.flags & btt_tracker_flags[i].flag)
    w = apr_pstrcat(cmd->pool, w, "; +", btt_tracker_flags[i].config_name, ": ", btt_tracker_flags[i].description, NULL);
   else
    w = apr_pstrcat(cmd->pool, w, "; -", btt_tracker_flags[i].config_name, ": ", btt_tracker_flags[i].description, NULL);
  }
  return w;
 }
 else
 {
  return NULL;
 }
}

static const command_rec modbt_commands[] =
{
 AP_INIT_FLAG( "Tracker",            modbt_config_Tracker,            NULL,   RSRC_CONF,  "Enable the BitTorrent tracker system"),
 AP_INIT_TAKE1( "TrackerHome",				modbt_config_TrackerHome,				NULL,	RSRC_CONF,	"Directory that mod_bt should use to store its data files"),
 AP_INIT_RAW_ARGS( "TrackerFlags",			modbt_config_TrackerFlags,				NULL,	RSRC_CONF,	"Set boolean tracker options"),
 AP_INIT_TAKE1( "TrackerStyleSheet",		modbt_config_TrackerStyleSheet,			NULL,	RSRC_CONF,	"Stylesheet URL to use for generated HTML pages"),
 AP_INIT_TAKE1( "TrackerDetailURL",			modbt_config_TrackerDetailURL,			NULL,	RSRC_CONF,	"Infohash Detail URL to use for generated HTML pages"),
 AP_INIT_TAKE1( "TrackerRootInclude",		modbt_config_TrackerRootInclude,		NULL,	RSRC_CONF,	"File to SSI-include on root HTML info page"),
 AP_INIT_TAKE1( "TrackerReturnPeers",		modbt_config_TrackerReturnPeers,		NULL,	RSRC_CONF,	"Default number of peers to return on an /announce request"),
 AP_INIT_TAKE1( "TrackerReturnMax",			modbt_config_TrackerReturnMax,			NULL,	RSRC_CONF,	"Maximum number of peers to return on an /announce request"),
 AP_INIT_TAKE1( "TrackerReturnInterval",	modbt_config_TrackerReturnInterval,		NULL,	RSRC_CONF,	"Number of seconds to tell peers to wait before using /announce again"),
 AP_INIT_TAKE1( "TrackerReturnPeerFactor",	modbt_config_TrackerReturnPeerFactor,	NULL,	RSRC_CONF,	"Increase Return Interval for a hash once it has this many peers. (hash interval = ((peers / factor) + 1) * tracker interval)"),
 AP_INIT_TAKE1( "TrackerHashWatermark",		modbt_config_TrackerHashWatermark,		NULL,	RSRC_CONF,	"If there are less than this many hashes, only dead hashes are deleted; if there are more, inactive ones are deleted too"),
 AP_INIT_TAKE1( "TrackerHashMinAge",		modbt_config_TrackerHashMinAge,			NULL,	RSRC_CONF,	"How many seconds of inactivity until a hash is considered inactive"),
 AP_INIT_TAKE1( "TrackerHashMaxAge",		modbt_config_TrackerHashMaxAge,			NULL,	RSRC_CONF,	"How many seconds of inactivity until a hash is considered dead"),
 {NULL}
};


/** HANDLERS **/
static apr_status_t modbt_shutdown_tracker(void *data)
{
 server_rec* s = data;
 modbt_config_t* cfg = ap_get_module_config(s->module_config, &bt_module);

 if((!cfg->tracker) || (!cfg->enabled))
  return APR_SUCCESS;
    
 if(!btt_tracker_disconnect_mem(cfg->tracker))
 {
  ap_log_error(APLOG_MARK, APLOG_CRIT, HTTP_INTERNAL_SERVER_ERROR, s, "modbt_shutdown_tracker(): bt_tracker_disconnect_mem failed! (pid %d)", getpid());
  return HTTP_INTERNAL_SERVER_ERROR;
 }
 
 if(!btt_tracker_free(&cfg->tracker, 1))
 {
  ap_log_error(APLOG_MARK, APLOG_CRIT, HTTP_INTERNAL_SERVER_ERROR, s, "modbt_shutdown_tracker(): bt_tracker_free failed! (pid %d)", getpid());
  return HTTP_INTERNAL_SERVER_ERROR;
 }

 ap_log_error(APLOG_MARK, APLOG_INFO, APR_SUCCESS, s, "modbt_shutdown_tracker(): Master BitTorrent tracker shut down. (pid %d)", getpid());
 return APR_SUCCESS;
}

static apr_status_t modbt_shutdown_child(void *data)
{
 server_rec* s = data;
 modbt_config_t* cfg = ap_get_module_config(s->module_config, &bt_module);

 if((!cfg->tracker) || (!cfg->enabled))
  return APR_SUCCESS;
 
 if(!btt_tracker_disconnect(cfg->tracker))
 {
  ap_log_error(APLOG_MARK, APLOG_CRIT, HTTP_INTERNAL_SERVER_ERROR, s, "modbt_shutdown_child(): bt_tracker_disconnect failed! (pid %d)", getpid());
  return HTTP_INTERNAL_SERVER_ERROR;
 }
 
 if(!btt_tracker_free(&cfg->tracker, 0))
 {
  ap_log_error(APLOG_MARK, APLOG_CRIT, HTTP_INTERNAL_SERVER_ERROR, s, "modbt_shutdown_child(): bt_tracker_free failed! (pid %d)", getpid());
  return HTTP_INTERNAL_SERVER_ERROR;
 }

 ap_log_error(APLOG_MARK, APLOG_INFO, APR_SUCCESS, s, "modbt_shutdown_child(): Child BitTorrent tracker shut down. (pid %d)", getpid());
 return APR_SUCCESS;
}

#if APR_HAS_OTHER_CHILD
static void modbt_child_maint(int reason, void *data, apr_wait_t status)
{
 server_rec *s = data;

 if(reason == APR_OC_REASON_RESTART) {
  modbt_shutdown_child((void*)s);
 }
}
#endif

static void* modbt_server_config(apr_pool_t* p, server_rec* s)
{
 modbt_config_t* cfg;
 cfg = (modbt_config_t*) apr_pcalloc(p, sizeof(modbt_config_t));
 cfg->tcfg = new_btt_tracker;
 cfg->tcfg.c = apr_pcalloc(p, sizeof(new_btt_tracker_config));
 *(cfg->tcfg.c) = new_btt_tracker_config;
 cfg->tracker = NULL;
 cfg->started = 0;
 cfg->enabled = 0;
 cfg->tcfg.c->parent_server = apr_pstrdup(p, ap_get_server_version());

 return (void*)cfg;
}

static void modbt_sig_term (int signum)
{
 fprintf(stderr, "Caught SIGTERM, cleanly shutting down mod_bt.\n");
 fflush(stderr);
 if(mys) {
  if(mytop)
   modbt_shutdown_tracker((void*)mys);
  else
   modbt_shutdown_child((void*)mys);
  
  mys = NULL;
 }
 apr_signal(signum, osig);
 osig = NULL;
 kill(getpid(), SIGHUP);
}

static apr_status_t modbt_init(apr_pool_t *p, apr_pool_t *plog, apr_pool_t *ptemp, server_rec *s)
{
 modbt_config_t *cfg = ap_get_module_config(s->module_config, &bt_module);
 
 if(!cfg->enabled)
  return APR_SUCCESS;
    
 apr_sigfunc_t *oosig;
 const char* userdata_key = "modbt_init";
 int *started = NULL;

 mys = s;
 mytop = 1;

 apr_pool_userdata_get((void**)&started, userdata_key, s->process->pool);

 if(!started || !*started)
 {
  started = apr_pcalloc(s->process->pool, sizeof(*started));
  *started = 1;
  apr_pool_userdata_set((const void *)started, userdata_key, apr_pool_cleanup_null, s->process->pool);
  return APR_SUCCESS;
 }

if(cfg->enabled) {
#if !defined(OS2) && !defined(WIN32) && !defined(BEOS) && !defined(NETWARE)
 unsigned int iuid = 0;
 if(geteuid() == 0)
 {
  iuid++;
  seteuid(unixd_config.user_id);
  setegid(unixd_config.group_id);
 }
#endif
 
  if(!bt_random_seed())
  {
   ap_log_error(APLOG_MARK, APLOG_CRIT, HTTP_INTERNAL_SERVER_ERROR, s, "modbt_init(): bt_random_seed() failed!");
   return HTTP_INTERNAL_SERVER_ERROR;
  }

  if(!(cfg->tracker = btt_tracker_alloc(s->process->pool, cfg->tcfg.homedir, 1)))
  {
   ap_log_error(APLOG_MARK, APLOG_CRIT, HTTP_INTERNAL_SERVER_ERROR, s, "modbt_init(): bt_tracker_alloc() failed!");
   return HTTP_INTERNAL_SERVER_ERROR;
  }

  if(!btt_tracker_connect(cfg->tracker, 1))
  {
   ap_log_error(APLOG_MARK, APLOG_CRIT, HTTP_INTERNAL_SERVER_ERROR, s, "modbt_init(): bt_tracker_connect() failed!");
   return HTTP_INTERNAL_SERVER_ERROR;
  }

  *(cfg->tracker->c) = *(cfg->tcfg.c);
  cfg->tracker->s->master_pid = getpid();
  cfg->tracker->s->start_t = time(NULL);
 
  if(!btt_tracker_disconnect_db(cfg->tracker))
  {
   ap_log_error(APLOG_MARK, APLOG_CRIT, HTTP_INTERNAL_SERVER_ERROR, s, "modbt_init(): bt_tracker_disconnect_db() failed!");
   return HTTP_INTERNAL_SERVER_ERROR;
  }
 
 #if !defined(OS2) && !defined(WIN32) && !defined(BEOS) && !defined(NETWARE)
  if(iuid)
  {
   seteuid(0);
   setegid(0);
  }
 #endif
 
  oosig = apr_signal(SIGTERM, modbt_sig_term);
  if(!osig)
   osig = oosig;

  apr_pool_cleanup_register(p, (void*)s, modbt_shutdown_tracker, apr_pool_cleanup_null);
  ap_log_error(APLOG_MARK, APLOG_INFO, APR_SUCCESS, s, "modbt_init(): New master process started (pid %d)", getpid());
 }
 
 return APR_SUCCESS;
}

static void modbt_child_init(apr_pool_t *p, server_rec *s)
{
 modbt_config_t *cfg = ap_get_module_config(s->module_config, &bt_module);
 apr_sigfunc_t *oosig;
 
 if(!cfg->enabled)
  return;
    
 if(!cfg->tracker)
 {
  ap_log_error(APLOG_MARK, APLOG_INFO, APR_SUCCESS, s, "modbt_child_init(): Child process started with no tracker!!!!\n");
  exit(1);
  return;
 }

 if(!(cfg->tracker = btt_tracker_alloc(s->process->pool, cfg->tcfg.homedir, 0)))
 {
  ap_log_error(APLOG_MARK, APLOG_CRIT, HTTP_INTERNAL_SERVER_ERROR, s, "modbt_child_init(): bt_tracker_alloc() failed!");
  exit(1);
  return;
 }

 if(!btt_tracker_connect(cfg->tracker, 0))
 {
  ap_log_error(APLOG_MARK, APLOG_CRIT, HTTP_INTERNAL_SERVER_ERROR, s, "modbt_child_init(): bt_tracker_connect() failed!");
  exit(1);
  return;
 }

 ap_log_error(APLOG_MARK, APLOG_DEBUG, APR_SUCCESS, s, "modbt_child_init(): New child process started (pid %d, master pid %d)", getpid(), cfg->tracker->s->master_pid);
 apr_pool_cleanup_register(p, (void*)s, modbt_shutdown_child, apr_pool_cleanup_null);
 mys = s;
 mytop = 0;
 oosig = apr_signal(SIGTERM, modbt_sig_term);
 if(!osig)
  osig = oosig;
#if APR_HAS_OTHER_CHILD
 apr_proc_other_child_register((apr_proc_t*)s->process, modbt_child_maint, s, NULL, p);
#endif
 return;
}

/* CONTENT HANDLERS */

/*
 * the lookup_header() function shamelessly stolen from
 * mod_rewrite by Ralf S. Engelschall
 * (httpd-2.0.49/modules/mappers/mod_rewrite.c)
 * you'd figure this would be a standard request_rec type thing..
 */

static char *lookup_header(request_rec *r, const char *name)
{
 const apr_array_header_t *hdrs_arr;
 const apr_table_entry_t *hdrs;
 int i;

 hdrs_arr = apr_table_elts(r->headers_in);
 hdrs = (const apr_table_entry_t *)hdrs_arr->elts;

 for (i = 0; i < hdrs_arr->nelts; ++i)
 {
  if (hdrs[i].key == NULL)
   continue;

  if (strcasecmp(hdrs[i].key, name) == 0)
  {
   apr_table_merge(r->notes, VARY_KEY_THIS, name);
   return hdrs[i].val;
  }
 }

 return NULL;
}

static int modbt_announce_handler(request_rec *r, modbt_config_t *cfg)
{
 int len;
 int rv;
 char *content;
 struct sockaddr_in address =
 {
  AF_INET,
  ntohs(r->connection->remote_addr->port),
  { 0 }
 };

 inet_aton(r->connection->remote_ip, &(address.sin_addr));
 rv = btt_cxn_announce(cfg->tracker, r->pool, NULL, r->args, lookup_header(r, "User-Agent"), address, &content, &len);
 
 if(len)
 {
  ap_set_content_type(r, "text/plain");
  ap_rwrite(content, len, r);
 }

 if(rv == HTTP_OK)
  rv = APR_SUCCESS;
 
 return rv;
}

static int modbt_scrape_handler(request_rec *r, modbt_config_t *cfg)
{
 int len;
 int rv;
 char *content;
 
 rv = btt_cxn_scrape(cfg->tracker, r->pool, NULL, r->args, r->connection->remote_addr->sa.sin, &content, &len);
 
 if(len)
 {
  if(r->args && strstr(r->args, "xml=1"))
   ap_set_content_type(r, "text/xml");
  else if(r->args && strstr(r->args, "html=1"))
   ap_set_content_type(r, "text/html");
  else
   ap_set_content_type(r, "text/plain");

  ap_rwrite(content, len, r);
 }

 if(rv == HTTP_OK)
  rv = APR_SUCCESS;
 
 return rv;
}

static int modbt_details_handler(request_rec *r, modbt_config_t* cfg)
{
 int len = 0;
 char* content = NULL;
 int rv;

 rv = btt_cxn_details(cfg->tracker, r->pool, NULL, r->args, r->connection->remote_addr->sa.sin, &content, &len);

 if(len)
 {
  ap_set_content_type(r, "text/html");
  ap_rwrite(content, len, r);
 }
 
 if(rv == HTTP_OK)
  rv = APR_SUCCESS;
  
 return rv;
}

static int modbt_root_handler(request_rec *r, modbt_config_t* cfg)
{
 char *args = r->args;

 if(!r->args)
  args = apr_pstrdup(r->pool, "html=1");
 else
  args = apr_pstrcat(r->pool, r->args, "&html=1", NULL);

 r->args = args;
 
 return modbt_scrape_handler(r, cfg);
}

static int modbt_register_handler(request_rec *r, modbt_config_t *cfg)
{
 int len;
 int rv;
 char *content;
 
 rv = btt_cxn_register(cfg->tracker, r->pool, NULL, r->args, r->connection->remote_addr->sa.sin, &content, &len);
 
 if(len)
 {
  ap_set_content_type(r, "text/html");
  ap_rwrite(content, len, r);
 }

 if(rv == HTTP_OK)
  rv = APR_SUCCESS;
 
 return rv;
}


/* MASTER CONTENT HANDLER */

typedef struct modbt_handler_map {
 char*      name;
 int        (*handler) (request_rec *, modbt_config_t *);
} modbt_handler_map;

static modbt_handler_map handler_map[] = {
 { "modbt-announce",    modbt_announce_handler },
 { "modbt-scrape",      modbt_scrape_handler },
 { "modbt-root",        modbt_root_handler },
 { "modbt-register",    modbt_register_handler },
 { "modbt-details",     modbt_details_handler },
 { NULL, NULL }
};

static int modbt_content_handler(request_rec *r)
{
 modbt_config_t *cfg = ap_get_module_config(r->server->module_config, &bt_module);
 int i;
 int f = -1;

 for(i=0;(f == -1) && (handler_map[i].name);i++) {
  if(!strcmp(r->handler, handler_map[i].name)) f = i;
 }

 if(f != -1) {
  if(cfg->enabled) {
   cfg->tracker->s->server_time = time(NULL);
   return handler_map[f].handler(r, cfg);
  } else {
   ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, 0, r, "%s: handler '%s' called when mod_bt is disabled", r->uri, handler_map[f].name);
   return HTTP_INTERNAL_SERVER_ERROR;
  }
 } else {
  return DECLINED;
 }
}


static void modbt_register_hooks(apr_pool_t *p)
{
 /* content handler */
 ap_hook_handler(modbt_content_handler, NULL, NULL, APR_HOOK_MIDDLE);
 /* etc */
 ap_hook_post_config(modbt_init, NULL, NULL, APR_HOOK_MIDDLE);
 ap_hook_child_init(modbt_child_init, NULL, NULL, APR_HOOK_MIDDLE);
 return;
}

/** MODULE DEFINITION **/

module bt_module =
{
 STANDARD20_MODULE_STUFF,
 NULL,					/* dir config creater */
 NULL,					/* dir merger --- default is to override */
 modbt_server_config,	/* server config */
 NULL,					/* merge server config */
 modbt_commands,		/* command table */
 modbt_register_hooks	/* register hooks */ 
};
