# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Load and parse data from ASCII format.

# load libs
import string
import re
import os


class mASCIIDoc:
    """ Get and format data from ASCII document. """

    # ----
    def __init__(self):
        self.data = {
                    'docType': 'ASCII XY',
                    'peaklist':[],
                    'spectrum':[],
                    }
    # ----


    # ----
    def getDocument(self, path):
        """ Read and parse all data from document. """

        status = self.handleSpectrum(path)
        if status:
            return self.data
        else:
            return False
    # ----


    # ----
    def getElement(self, name, path):
        """ Read and parse selected elements' data from document. """

        if name == 'peaklist':
            status = self.handlePeaklist(path)
        elif name == 'spectrum':
            status = self.handleSpectrum(path)

        if status:
            return self.data
        else:
            return False
    # ----


    # ----
    def handleSpectrum(self, path):
        """ Read and parse spectrum data from document. """

        # load file
        try:
            docFile = file(path)
            rawData = docFile.readlines()
            docFile.close()
        except IOError:
            return False

        # parse spectrum data
        spectrum = self.parseSpectrum(rawData)
        if not spectrum:
            return False
        else:
            spectrum.sort()
            self.data['spectrum'] = spectrum
            return True
    # ----


    # ----
    def handlePeaklist(self, path):
        """ Read and parse peaklist data from document. """

        # load file
        try:
            docFile = file(path)
            rawData = docFile.readlines()
            docFile.close()
        except IOError:
            return False

        # parse spectrum data
        peaklist = self.parsePeaklist(rawData)
        if not peaklist:
            return False
        else:
            peaklist.sort()
            self.data['peaklist'] = peaklist
            return True
    # ----


    # ----
    def parseSpectrum(self, data):
        """ Parse given data and get mass/intensity values. """

        pattern = re.compile('^([-0-9\.eE+]+)[ \t]*(;|,)?[ \t]*([-0-9\.eE+]*)$')

        # check each line
        spectrum = []
        for line in data:
            line = string.strip(line)

            # discard comment lines
            if not line or line[0] == '#':
                continue

            # check pattern
            parts = pattern.match(line)
            if parts:
                try:
                    mass = float(parts.group(1))
                    intensity = float(parts.group(3))
                except ValueError:
                    return False
                spectrum.append([mass, intensity])

            else:
                return False

        return spectrum
    # ----


    # ----
    def parsePeaklist(self, data):
        """ Parse given data and get mass/intensity/comments values. """

        pattern = re.compile('^([-0-9\.eE+]+)[ \t]*(;|,)?[ \t]*([-0-9\.eE+]*)[ \t]*(;|,)?[ \t]*(.*)$')

        # check each line
        peaklist = []
        for line in data:
            line = string.strip(line)

            # discard comment lines
            if not line or line[0] == '#':
                continue

            # check pattern
            parts = pattern.match(line)
            if parts:

                # get mass
                try:
                    mass = float(parts.group(1))
                except ValueError:
                    return False

                # get intensity or se zero
                try:
                    intensity = float(parts.group(3))
                except ValueError:
                    intensity = 0

                # get comments
                comments = ''
                if parts.group(5):
                    comments = parts.group(5)

                # add data to peaklist
                peaklist.append([mass, intensity, comments, 0])

        return peaklist
    # ----
