# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Generate peptide fragments.

# load libs
import wx
import wx.grid
import string

# load modules
from nucleus import commfce
from count import mFragCount
from modules.mlistpanel.dlg_mfragpanel import mFragListPanel

class mFrag(wx.Panel):
    """Generate peptide fragments"""

    # ----
    def __init__(self, parent, document):
        wx.Panel.__init__(self, parent, -1)

        self.parent = parent
        self.config = parent.config
        self.docMonitor = parent.docMonitor
        self.docData = document

        # init fragment counter
        self.mFragCount = mFragCount(self.config)

        # init module variables
        self.ions_check = {}
        self.ctrlData = {}
        self.parsedSequence = None
        self.mainFragments = None
        self.internalFragments = None
        self.listPanel = None

        self.ionsNames = {'im':'i',
                        'a0':'a', 'a1':'a-HN3', 'a2':'a-H2O',
                        'a3':'a(2+)', 'a4':'a-HN3(2+)', 'a5':'a-H2O(2+)',
                        'b0':'b', 'b1':'b-NH3', 'b2':'b-H2O',
                        'b3':'b(2+)', 'b4':'b-NH3(2+)', 'b5':'b-H2O(2+)',
                        'y0':'y', 'y1':'y-NH3', 'y2':'y-H2O',
                        'y3':'y(2+)', 'y4':'y-NH3(2+)', 'y5':'y-H2O(2+)',
                        'c0':'c', 'x0':'x', 'z0':'z',
                        'c1':'c(2+)', 'x1':'x(2+)', 'z1':'z(2+)',
                        'int0':'int.', 'int1':'int.-CO', 'int2':'int.-NH3', 'int3':'int.-H2O'}

        # setup colour
        self.colours = {}
        self.colours['matched'] = self.config.cfg['colours']['matched']
        self.colours['filtered'] = self.config.cfg['colours']['grayed']

        # make gui items
        self.makeMainFragmentsTable()
        self.makeInternalFragmentsTable()
        mainIons = self.makeMainIonsBox()
        doublyIons = self.makeDoublyChargedIonsBox()
        internalIons = self.makeInternalIonsBox()
        instrumentPresets = self.makeInstrumentBox()
        generateButton = self.makeGenerateButt()
        listButton = self.makeListButt()

        # set default ions
        self.setDefaultIons()

        # pack main area
        main = wx.BoxSizer(wx.VERTICAL)
        main.Add(self.mainFragmentsTable, 1, wx.EXPAND, 0)
        main.Add(wx.StaticText(self, -1, "Internal Fragments:"), 0, wx.EXPAND|wx.LEFT|wx.TOP, 5)
        main.Add(self.internalFragmentsTable, 1, wx.EXPAND|wx.TOP, 3)

        # pack control elements
        controls = wx.BoxSizer(wx.VERTICAL)
        controls.Add(mainIons, 0, wx.EXPAND|wx.ALL, 10)
        controls.Add(doublyIons, 0, wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, 10)
        controls.Add(internalIons, 0, wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, 10)
        controls.Add(instrumentPresets, 0, wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, 10)
        controls.Add(generateButton, 0, wx.ALIGN_CENTRE|wx.ALL, 5)
        controls.Add(listButton, 0, wx.ALIGN_CENTRE|wx.ALL, 5)

        # pack main frame
        mainSizer = wx.BoxSizer(wx.HORIZONTAL)
        mainSizer.Add(main, 1, wx.EXPAND)
        mainSizer.Add(controls, 0, wx.EXPAND)
        self.SetSizer(mainSizer)
    # ----


    # ----
    def makeMainFragmentsTable(self):
        """ Make teble of main fragments. """

        # set style
        if wx.Platform == '__WXMAC__':
            style=wx.SIMPLE_BORDER
        else:
            style=wx.SUNKEN_BORDER

        # make item
        self.mainFragmentsTable = wx.grid.Grid(self, -1, style=style)
        self.mainFragmentsTable.CreateGrid(0, 0)
        self.mainFragmentsTable.DisableDragGridSize()
        self.mainFragmentsTable.SetDefaultRowSize(19)
        self.mainFragmentsTable.SetDefaultCellAlignment(wx.ALIGN_CENTER, wx.ALIGN_TOP)
        self.mainFragmentsTable.SetLabelFont(wx.Font(8, wx.DEFAULT, wx.NORMAL, wx.NORMAL, 0))
        self.mainFragmentsTable.SetRowLabelSize(0)
        self.mainFragmentsTable.SetColLabelSize(19)
        self.mainFragmentsTable.SetDefaultColSize(50)
        self.mainFragmentsTable.SetDefaultCellBackgroundColour(wx.WHITE)

        if wx.Platform == '__WXMAC__':
            self.mainFragmentsTable.SetDefaultCellFont(wx.Font(10, wx.DEFAULT, wx.NORMAL, wx.NORMAL, 0))
            self.mainFragmentsTable.SetLabelFont(wx.Font(10, wx.DEFAULT, wx.NORMAL, wx.NORMAL, 0))
    # ----


    # ----
    def makeInternalFragmentsTable(self):
        """ Make teble of internal fragments. """

        # set style
        if wx.Platform == '__WXMAC__':
            style=wx.SIMPLE_BORDER
        else:
            style=wx.SUNKEN_BORDER

        # make item
        self.internalFragmentsTable = wx.grid.Grid(self, -1, style=style)
        self.internalFragmentsTable.CreateGrid(0, 0)
        self.internalFragmentsTable.DisableDragGridSize()
        self.internalFragmentsTable.SetDefaultRowSize(19)
        self.internalFragmentsTable.SetDefaultCellAlignment(wx.ALIGN_CENTER, wx.ALIGN_TOP)
        self.internalFragmentsTable.SetLabelFont(wx.Font(8, wx.DEFAULT, wx.NORMAL, wx.NORMAL, 0))
        self.internalFragmentsTable.SetRowLabelSize(0)
        self.internalFragmentsTable.SetColLabelSize(19)
        self.internalFragmentsTable.SetDefaultCellBackgroundColour(wx.WHITE)

        if wx.Platform == '__WXMAC__':
            self.internalFragmentsTable.SetDefaultCellFont(wx.Font(10, wx.DEFAULT, wx.NORMAL, wx.NORMAL, 0))
            self.internalFragmentsTable.SetLabelFont(wx.Font(10, wx.DEFAULT, wx.NORMAL, wx.NORMAL, 0))
    # ----


    # ----
    def makeMainIonsBox(self):
        """ Make box for main ions selection. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Main Ions"), wx.VERTICAL)
        grid = wx.GridBagSizer(4, 4)

        self.ions_check['a0'] = wx.CheckBox(self, -1, "a")
        self.ions_check['a1'] = wx.CheckBox(self, -1, "a*")
        self.ions_check['a2'] = wx.CheckBox(self, -1, "a~")
        self.ions_check['a1'].SetToolTip(wx.ToolTip("a-NH3"))
        self.ions_check['a2'].SetToolTip(wx.ToolTip("a-H2O"))

        self.ions_check['b0'] = wx.CheckBox(self, -1, "b")
        self.ions_check['b1'] = wx.CheckBox(self, -1, "b*")
        self.ions_check['b2'] = wx.CheckBox(self, -1, "b~")
        self.ions_check['b1'].SetToolTip(wx.ToolTip("b-NH3"))
        self.ions_check['b2'].SetToolTip(wx.ToolTip("b-H2O (same as N-term ladder)"))

        self.ions_check['y0'] = wx.CheckBox(self, -1, "y")
        self.ions_check['y1'] = wx.CheckBox(self, -1, "y*")
        self.ions_check['y2'] = wx.CheckBox(self, -1, "y~")
        self.ions_check['y0'].SetToolTip(wx.ToolTip("same as C-term ladder"))
        self.ions_check['y1'].SetToolTip(wx.ToolTip("y-NH3"))
        self.ions_check['y2'].SetToolTip(wx.ToolTip("y-H2O"))

        self.ions_check['c0'] = wx.CheckBox(self, -1, "c")
        self.ions_check['x0'] = wx.CheckBox(self, -1, "x")
        self.ions_check['z0'] = wx.CheckBox(self, -1, "z")

        self.ions_check['im'] = wx.CheckBox(self, -1, "imm.")

        # pack items
        grid.Add(self.ions_check['a0'], (0, 0))
        grid.Add(self.ions_check['a1'], (1, 0))
        grid.Add(self.ions_check['a2'], (2, 0))

        grid.Add(self.ions_check['b0'], (0, 1))
        grid.Add(self.ions_check['b1'], (1, 1))
        grid.Add(self.ions_check['b2'], (2, 1))

        grid.Add(self.ions_check['y0'], (0, 2))
        grid.Add(self.ions_check['y1'], (1, 2))
        grid.Add(self.ions_check['y2'], (2, 2))

        grid.Add(self.ions_check['c0'], (0, 3))
        grid.Add(self.ions_check['x0'], (1, 3))
        grid.Add(self.ions_check['z0'], (2, 3))

        grid.Add(self.ions_check['im'], (3, 0), (1, 2))

        if wx.Platform == '__WXMAC__':
            mainBox.Add(grid, 0, wx.ALL, 0)
        else:
            mainBox.Add(grid, 0, wx.ALL, 5)

        return mainBox
    # ----


    # ----
    def makeDoublyChargedIonsBox(self):
        """ Make box for doubly charged ions selection. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Doubly Charged Ions (2+)"), wx.VERTICAL)
        grid = wx.GridBagSizer(4, 4)

        self.ions_check['a3'] = wx.CheckBox(self, -1, "a")
        self.ions_check['a4'] = wx.CheckBox(self, -1, "a*")
        self.ions_check['a5'] = wx.CheckBox(self, -1, "a~")
        self.ions_check['a4'].SetToolTip(wx.ToolTip("a-NH3"))
        self.ions_check['a5'].SetToolTip(wx.ToolTip("a-H2O"))

        self.ions_check['b3'] = wx.CheckBox(self, -1, "b")
        self.ions_check['b4'] = wx.CheckBox(self, -1, "b*")
        self.ions_check['b5'] = wx.CheckBox(self, -1, "b~")
        self.ions_check['b4'].SetToolTip(wx.ToolTip("b-NH3"))
        self.ions_check['b5'].SetToolTip(wx.ToolTip("b-H2O"))

        self.ions_check['y3'] = wx.CheckBox(self, -1, "y")
        self.ions_check['y4'] = wx.CheckBox(self, -1, "y*")
        self.ions_check['y5'] = wx.CheckBox(self, -1, "y~")
        self.ions_check['y4'].SetToolTip(wx.ToolTip("y-NH3"))
        self.ions_check['y5'].SetToolTip(wx.ToolTip("y-H2O"))

        self.ions_check['c1'] = wx.CheckBox(self, -1, "c")
        self.ions_check['x1'] = wx.CheckBox(self, -1, "x")
        self.ions_check['z1'] = wx.CheckBox(self, -1, "z")

        # pack items
        grid.Add(self.ions_check['a3'], (0, 0))
        grid.Add(self.ions_check['a4'], (1, 0))
        grid.Add(self.ions_check['a5'], (2, 0))

        grid.Add(self.ions_check['b3'], (0, 1))
        grid.Add(self.ions_check['b4'], (1, 1))
        grid.Add(self.ions_check['b5'], (2, 1))

        grid.Add(self.ions_check['y3'], (0, 2))
        grid.Add(self.ions_check['y4'], (1, 2))
        grid.Add(self.ions_check['y5'], (2, 2))

        grid.Add(self.ions_check['c1'], (0, 3))
        grid.Add(self.ions_check['x1'], (1, 3))
        grid.Add(self.ions_check['z1'], (2, 3))

        if wx.Platform == '__WXMAC__':
            mainBox.Add(grid, 0, wx.ALL, 0)
        else:
            mainBox.Add(grid, 0, wx.ALL, 5)

        return mainBox
    # ----


    # ----
    def makeInternalIonsBox(self):
        """ Make box for internal ions selection. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Internal Ions"), wx.VERTICAL)
        grid = wx.GridBagSizer(4, 10)

        self.ions_check['int0'] = wx.CheckBox(self, -1, "main")
        self.ions_check['int1'] = wx.CheckBox(self, -1, "-CO")
        self.ions_check['int2'] = wx.CheckBox(self, -1, "-NH3")
        self.ions_check['int3'] = wx.CheckBox(self, -1, "-H2O")

        # pack items
        grid.Add(self.ions_check['int0'], (0, 0))
        grid.Add(self.ions_check['int1'], (1, 0))
        grid.Add(self.ions_check['int2'], (0, 1))
        grid.Add(self.ions_check['int3'], (1, 1))

        if wx.Platform == '__WXMAC__':
            mainBox.Add(grid, 0, wx.ALL, 0)
        else:
            mainBox.Add(grid, 0, wx.ALL, 5)

        return mainBox
    # ----


    # ----
    def makeInstrumentBox(self):
        """ Make box for ion presets. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Instrument Presets"), wx.VERTICAL)

        # get/make choices
        instrument_choices = ['Default',
            'ESI-TRAP',
            'ESI-QUAD',
            'ESI-QUAD-TOF',
            'ESI-4SECTOR',
            'ESI-FTICR',
            'FTMS-ECD',
            'MALDI-TOF-PSD',
            'MALDI-TOF-TOF',
            'MALDI-QUAD-TOF',
            'MALDI-QIT-TOF',
            ]
        if self.config.cfg['mfrag']['pres-custom1']:
            instrument_choices.append('Custom1')
        if self.config.cfg['mfrag']['pres-custom2']:
            instrument_choices.append('Custom2')
        if self.config.cfg['mfrag']['pres-custom3']:
            instrument_choices.append('Custom3')
        instrument_choices.append('User specified')

        # make combo
        self.instrument_combo = wx.ComboBox(self, -1, 'Default', size=(90, -1), choices=instrument_choices, style=wx.CB_READONLY)

        if wx.Platform == '__WXMAC__':
            mainBox.Add(self.instrument_combo, 0, wx.ALL|wx.EXPAND, 0)
        else:
            mainBox.Add(self.instrument_combo, 0, wx.ALL|wx.EXPAND, 5)

        # set events
        self.instrument_combo.Bind(wx.EVT_COMBOBOX, self.onInstrument)

        return mainBox
    # ----


    # ----
    def makeGenerateButt(self):
        """ Make 'Fragment' button. """

        generateButt = wx.Button(self, -1, "Fragment")
        generateButt.Bind(wx.EVT_BUTTON, self.onGenerate)
        return generateButt
    # ----


    # ----
    def makeListButt(self):
        """ Make 'List' button. """

        listButt = wx.Button(self, -1, "Show in panel")
        listButt.Bind(wx.EVT_BUTTON, self.onShowInPanel)
        return listButt
    # ----


    # ----
    def setDefaultIons(self):
        """ Check default ions. """

        # check ions
        for ion in self.config.cfg['mfrag']['pres-default'].split(';'):
            try:
                self.ions_check[ion].SetValue(True)
            except KeyError:
                pass

        # set event
        for item in self.ions_check:
            self.ions_check[item].Bind(wx.EVT_CHECKBOX, self.onIonSelected)
    # ----


    # ----
    def updateMainFragmentsTable(self, fragments):
        """ Update table of main fragments. """

        # get selected (counted) ions
        ions = self.ctrlData['mainions']

        # delete old data
        if self.mainFragmentsTable.GetNumberCols() != 0:
            self.mainFragmentsTable.DeleteCols(0, self.mainFragmentsTable.GetNumberCols())
            self.mainFragmentsTable.DeleteRows(0, self.mainFragmentsTable.GetNumberRows())

        # if no ions selected
        if not ions:
            return

        # add cols
        self.mainFragmentsTable.AppendCols(len(ions) + 4)

        # set columns atributes
        for x in range(len(ions)+4):
            cellAttr = wx.grid.GridCellAttr()
            cellAttr.SetReadOnly(True)
            if 1 < x < len(ions)+3:
                cellAttr.SetAlignment(wx.ALIGN_RIGHT, wx.ALIGN_TOP)
            self.mainFragmentsTable.SetColAttr(x, cellAttr)

        # create labels
        self.mainFragmentsTable.SetColLabelValue(0, "N#")
        self.mainFragmentsTable.SetColLabelValue(1, "Seq.")
        col = 2
        for ion in ions:
            self.mainFragmentsTable.SetColLabelValue(col, self.ionsNames[ion])
            self.mainFragmentsTable.SetColFormatFloat(col, 0, self.config.cfg['common']['digits'])
            col += 1
        self.mainFragmentsTable.SetColLabelValue(col, "Seq.")
        self.mainFragmentsTable.SetColLabelValue(col+1, "C#")

        # paste data
        digitsFormat = '%0.' + `self.config.cfg['common']['digits']` + 'f'
        for row in range(len(fragments)):
            self.mainFragmentsTable.AppendRows(1)

            # set start labels
            self.mainFragmentsTable.SetCellValue(row, 0, str(row + 1))
            self.mainFragmentsTable.SetCellValue(row, 1, fragments[row]['amino'])

            # print ions
            col = 1
            for ion in ions:
                col += 1

                # discard some ions
                if (row == 0 and ion in 'y0;y1;y2;x0;z0;y3;y4;y5;x1;z1;') \
                    or (row == len(fragments)-1 and not ion in 'im;y0;y1;y2;x0;z0;y3;y4;y5;x1;z1;'):
                    continue

                # get and paste mass
                mass = digitsFormat % fragments[row][ion][0]
                self.mainFragmentsTable.SetCellValue(row, col, str(mass))

                # mark filtered
                if ion in fragments[row]['filter']:
                    self.mainFragmentsTable.SetCellTextColour(row, col, self.colours['filtered'])

                # mark matched ions
                if fragments[row][ion][1] != '':
                    self.mainFragmentsTable.SetCellBackgroundColour(row, col, self.colours['matched'])

            # set end labels
            self.mainFragmentsTable.SetCellValue(row, col+1, str(fragments[row]['amino']))
            self.mainFragmentsTable.SetCellValue(row, col+2, str(len(fragments)-row))
            self.mainFragmentsTable.SetCellAlignment(row, col+1, wx.ALIGN_CENTRE, wx.ALIGN_TOP)


        # scrollbar hack
        h,w = self.mainFragmentsTable.GetSize()
        self.mainFragmentsTable.SetSize((h+1, w))
        self.mainFragmentsTable.SetSize((h, w))
        self.mainFragmentsTable.ForceRefresh()
        self.mainFragmentsTable.AutoSizeColumns()
    # ----


    # ----
    def updateInternalFragmentsTable(self, fragments):
        """ Update table of main fragments. """

        # get selected (counted) ions
        ions = self.ctrlData['internalions']
        cols = len(ions) + 3

        # delete old data
        if self.internalFragmentsTable.GetNumberCols() != 0:
            self.internalFragmentsTable.DeleteCols(0, self.internalFragmentsTable.GetNumberCols())
            self.internalFragmentsTable.DeleteRows(0, self.internalFragmentsTable.GetNumberRows())

        # if no ions selected
        if not ions:
            return

        # add cols
        self.internalFragmentsTable.AppendCols(cols)

        # set columns atributes
        for x in range(cols):
            cellAttr = wx.grid.GridCellAttr()
            cellAttr.SetReadOnly(True)
            if x == 0 or x == 1:
                cellAttr.SetAlignment(wx.ALIGN_CENTER, wx.ALIGN_TOP)
            elif x == cols - 1:
                cellAttr.SetAlignment(wx.ALIGN_LEFT, wx.ALIGN_TOP)
            else:
                cellAttr.SetAlignment(wx.ALIGN_RIGHT, wx.ALIGN_TOP)
            self.internalFragmentsTable.SetColAttr(x, cellAttr)

        # create labels
        self.internalFragmentsTable.SetColLabelValue(0, "#")
        self.internalFragmentsTable.SetColLabelValue(1, "Range")
        col = 2
        for ion in ions:
            self.internalFragmentsTable.SetColLabelValue(col, self.ionsNames[ion])
            self.internalFragmentsTable.SetColFormatFloat(col, 0, self.config.cfg['common']['digits'])
            col += 1
        self.internalFragmentsTable.SetColLabelValue(col, "Sequence")

        # paste data
        digitsFormat = '%0.' + `self.config.cfg['common']['digits']` + 'f'
        for row in range(len(fragments)):
            self.internalFragmentsTable.AppendRows(1)
            self.internalFragmentsTable.SetCellValue(row, 0, str(row+1))
            self.internalFragmentsTable.SetCellValue(row, 1, fragments[row]['pos'])

            col = 2
            for ion in ions:
                mass = digitsFormat % fragments[row][ion][0]
                self.internalFragmentsTable.SetCellValue(row, col, str(mass))

                # mark filtered ions
                if ion in fragments[row]['filter']:
                    self.internalFragmentsTable.SetCellTextColour(row, col, self.colours['filtered'])

                # mark matched ions
                if fragments[row][ion][1] != '':
                    self.internalFragmentsTable.SetCellBackgroundColour(row, col, self.colours['matched'])

                col += 1
            self.internalFragmentsTable.SetCellValue(row, col, fragments[row]['seq'])

        # scrollbar hack
        h,w = self.internalFragmentsTable.GetSize()
        self.internalFragmentsTable.SetSize((h+1, w))
        self.internalFragmentsTable.SetSize((h, w))
        self.internalFragmentsTable.ForceRefresh()
        self.internalFragmentsTable.AutoSizeColumns()
    # ----


    # ----
    def updateListPanel(self):
        """ Update list panel. """

        fragments = self.getFragmentsList()
        if self.listPanel:
            self.listPanel.updatePanel(fragments)
    # ----


    # ----
    def onShow(self):
        """ Show panel and set focus to main item. """

        self.Show(True)
        self.mainFragmentsTable.SetFocus()
    # ----


    # ----
    def onGenerate(self, evt=None):
        """ Get all controls and generate digest peptides. """

        # get and parse controls
        if not self.getControls():
            return

        # set application working
        self.docMonitor('setAppStatus', "Generating fragments...")

        # send cotrols to counting module
        self.mFragCount.ctrlData = self.ctrlData

        # get main fragments
        self.mainFragments = []
        if self.ctrlData['mainions'] != '':
            self.mainFragments = self.mFragCount.getMainFragments(self.parsedSequence)

        # internal fragments
        self.internalFragments = []
        if self.ctrlData['internalions'] != '':
            self.internalFragments = self.mFragCount.getInternalFragments(self.parsedSequence)

        # update table of main fragments
        self.updateMainFragmentsTable(self.mainFragments)

        # update table of internal fragments
        self.updateInternalFragmentsTable(self.internalFragments)

        # update list panel
        self.updateListPanel()

        # update document status
        status = bool(self.mainFragments) + bool(self.internalFragments)
        self.docData.setDataStatus(status, 'mFrag')

        # update application
        self.docMonitor('onModuleReset', 'mFrag')
        self.docMonitor('setAppStatus', 0)
    # ----


    # ----
    def onInstrument(self, evt):
        """ Set iont for selected instrument. """

        # get instrument presets
        instrument = self.instrument_combo.GetValue()
        if instrument in ('Custom', 'User specified'):
            return
        elif instrument == 'Default':
            presets = self.config.cfg['mfrag']['pres-default']
        else:
            presets = self.config.cfg['mfrag']['pres-' + string.lower(instrument)]

        # uncheck all ions
        for item in self.ions_check:
            self.ions_check[item].SetValue(False)

        # check ions
        for ion in presets.split(';'):
            try:
                self.ions_check[ion].SetValue(True)
            except KeyError:
                pass
    # ----


    # ----
    def onIonSelected(self, evt):
        """ Set 'User' if ion selected manually. """
        self.instrument_combo.SetValue('User specified')
    # ----


    # ----
    def onShowInPanel(self, evt):
        """ Show list of all peptides in separate panel. """

        # get and check data
        fragments = self.getFragmentsList()
        if not fragments:
            return

        # raise panel
        if not self.listPanel:
            self.listPanel = mFragListPanel(self, self.config, fragments)
            self.listPanel.Show()
    # ----


    # ----
    def onSpectrumHighlightPoint(self, point, zoom):
        """ Highlight selected point in the spectrum. """
        self.parent.onSpectrumHighlightPoint(point, zoom)
    # ----


    # ----
    def getControls(self):
        """ Get and check controls. """

        errorMessage = ''

        # get sequence from document
        self.parsedSequence = self.docData.getParsedSequence()

        # get main ions
        self.ctrlData['mainions'] = []
        for ion in ('im','a0','a1','a2','a3','a4','a5','b0','b1','b2','b3','b4','b5','c0','c1','y0','y1','y2','y3','y4','y5','x0','x1','z0','z1'):
            if self.ions_check[ion].IsChecked():
                self.ctrlData['mainions'].append(ion)

        # get internal ions
        self.ctrlData['internalions'] = []
        for ion in ('int0', 'int1', 'int2', 'int3'):
            if self.ions_check[ion].IsChecked():
                self.ctrlData['internalions'].append(ion)

        # get common params
        self.ctrlData['digits'] = self.config.cfg['common']['digits']
        self.ctrlData['masstype'] = self.docData.getMassParam('masstype')

        # validate controls
        if not self.parsedSequence:
            errorMessage = "Peptide sequence is empty!\nCheck sequence editor."
        elif len(self.parsedSequence) > self.config.cfg['mfrag']['maxsequence']:
            errorMessage = "Peptide sequence is too long!\nCheck sequence editor."

        # if controls not OK
        if errorMessage != '':
            errorDlg = wx.MessageDialog(self, errorMessage, "Value Error", wx.OK|wx.ICON_ERROR)
            errorDlg.ShowModal()
            errorDlg.Destroy()
            return False
        else:
            return True
    # ----


    # ----
    def matchDataToPeaklist(self):
        """ Compare generated list of ions masses with the document peaklist. """

        # get peaklist from document
        mainPeaklist = self.docData.getPeaks()

        # get common params
        self.ctrlData['tolerance'] = self.docData.getMassParam('tolerance')
        self.ctrlData['errortype'] = self.docData.getMassParam('errortype')
        self.mFragCount.ctrlData = self.ctrlData

        # match main fragments
        matchMainStatus = False
        if self.ctrlData['mainions'] != '':
            self.mainFragments, matchMainStatus = self.mFragCount.matchDataToPeaklist(mainPeaklist, self.mainFragments, self.ctrlData['mainions'])
            self.updateMainFragmentsTable(self.mainFragments)

        # match internal fragments
        matchIntStatus = False
        if self.ctrlData['internalions'] != '':
            self.internalFragments, matchIntStatus = self.mFragCount.matchDataToPeaklist(mainPeaklist, self.internalFragments, self.ctrlData['internalions'])
            self.updateInternalFragmentsTable(self.internalFragments)

        # update list panel
        self.updateListPanel()

        # update document status
        self.docData.setMatchStatus(matchMainStatus + matchIntStatus, 'mFrag')

        return matchMainStatus + matchIntStatus
    # ----


    # ----
    def getMatchInfo(self):
        """ Get information about current match. """

        # get basic params
        errorType = self.docData.getMassParam('errortype')
        digits = self.config.cfg['common']['digits']

        # get match info
        mainPeaklist = self.docData.getPeaks()
        data = self.mFragCount.getMatchInfo(mainPeaklist, self.mainFragments, self.internalFragments, self.parsedSequence, errorType, digits)

        return data
    # ----


    # ----
    def annotatePeaklist(self):
        """ Annotate matched peaks in the document peaklist. """

        # get peaklist from document
        mainPeaklist = self.docData.getPeaks()

        # copy peaklist structure
        annotations = []
        for x in range(len(mainPeaklist)):
            annotations.append('')

        # annotate main fragments
        if self.ctrlData['mainions'] != '':
            annotations = self.getMainFragmentsAnnots(mainPeaklist, annotations)

        # annotate internal fragments
        if self.ctrlData['internalions'] != '':
            annotations = self.getInternalFragmentsAnnots(mainPeaklist, annotations)

        # update document
        self.docData.annotatePeaklist(annotations)
    # ----


    # ----
    def getMainFragmentsAnnots(self, peaklist, annotations):
        """ Get annotations for main fragments. """

        errorType = self.docData.getMassParam('errortype')
        digits = self.config.cfg['common']['digits']

        # get peaks number for each row and iontype
        for row in range(len(self.mainFragments)):
            for ion in self.mainFragments[row]:
                if ion in self.ctrlData['mainions'] \
                    and self.mainFragments[row][ion][1] != '' \
                    and (ion not in self.mainFragments[row]['filter'] \
                        or self.config.cfg['mfrag']['annotfiltered']):

                    # collect info for each peak
                    peaks = self.mainFragments[row][ion][1].split(';')
                    for peakIndex in peaks:
                        if peakIndex != '':
                            newAnnotation = ''

                            # get info
                            peakIndex = int(peakIndex) # make index
                            ionName = self.ionsNames[ion] # get ion name from ionDic
                            ionMass = self.mainFragments[row][ion][0] # get fragmen mass
                            peakMass = peaklist[peakIndex][0] # get peak mass
                            oldAnnotations = annotations[peakIndex] # get peak comments

                            if oldAnnotations != '':
                                oldAnnotations += '; '

                            # count error
                            error = commfce.calcMassError(peakMass, ionMass, errorType)
                            if errorType == 'Da':
                                error = round(error, digits)

                            # get sequence index for ion
                            if ion in ('y0', 'y1', 'y2', 'x0', 'z0', 'y3', 'y4', 'y5', 'x1', 'z1'):
                                ionIndex = len(self.mainFragments) - row
                            else:
                                ionIndex = row + 1

                            # make ion name
                            ionIndexName = ionName[0] + str(ionIndex) + ionName[1:]

                            # make annotation
                            newAnnotation = '%s (%s %s)' % (ionIndexName, error, errorType)
                            annotations[peakIndex] = oldAnnotations + newAnnotation

        return annotations
    # ----


    # ----
    def getInternalFragmentsAnnots(self, peaklist, annotations):
        """ Get annotations for internal fragments. """

        errorType = self.docData.getMassParam('errortype')
        digits = self.config.cfg['common']['digits']

        for peptide in self.internalFragments:
            for ion in peptide:
                if ion in self.ctrlData['internalions'] \
                    and not ion in ('seq', 'pos', 'filter') \
                    and peptide[ion][1] != '' \
                    and (ion not in peptide['filter'] \
                        or self.config.cfg['mfrag']['annotfiltered']):

                    peaks = peptide[ion][1].split(';')

                    # collect info for each peak
                    for peakIndex in peaks:
                        if peakIndex != '':
                            newAnnotation = ''

                            # get info
                            peakIndex = int(peakIndex)
                            peptidePos = peptide['pos']
                            peptideSeq = peptide['seq']
                            ionName = self.ionsNames[ion]
                            ionMass = peptide[ion][0]
                            peakMass = peaklist[peakIndex][0]
                            oldAnnotations = annotations[peakIndex]

                            if oldAnnotations != '':
                                oldAnnotations += '; '

                            # count error
                            error = commfce.calcMassError(peakMass, ionMass, errorType)
                            if errorType == 'Da':
                                error = round(error, digits)

                            # make annotation
                            newAnnotation = '%s (%s %s) %s [%s]' % (ionName, error, errorType, peptidePos, peptideSeq)
                            annotations[peakIndex] = oldAnnotations + newAnnotation

        return annotations
    # ----


    # ----
    def getFragmentsList(self):
        """ Get list of fragments to export. """

        fragmentsList = []

        # get list of main fragments
        if self.mainFragments:
            for row in range(len(self.mainFragments)):
                for ion in self.mainFragments[row]:
                    if ion in self.ctrlData['mainions']:

                        # discard some ions
                        if (row == 0 and ion in 'y0;y1;y2;x0;z0;y3;y4;y5;x1;z1;') \
                            or (row == len(self.mainFragments)-1 and not ion in 'im;y0;y1;y2;x0;z0;y3;y4;y5;x1;z1;'):
                            continue
                    
                        # get sequence index for ion
                        if ion in ('y0', 'y1', 'y2', 'x0', 'z0', 'y3', 'y4', 'y5', 'x1', 'z1'):
                            ionIndex = len(self.mainFragments) - row
                        else:
                            ionIndex = row + 1

                        # make ion description
                        ionDesription = self.ionsNames[ion][0] + str(ionIndex) + self.ionsNames[ion][1:]

                        # check matched
                        matched = False
                        if self.mainFragments[row][ion][1] != '':
                            matched = True

                        # check filtered
                        filtered = False
                        if ion in self.mainFragments[row]['filter']:
                            filtered = True

                        # add fragment to list
                        fragmentsList.append((self.mainFragments[row][ion][0], ionDesription, matched, filtered))

        # get internal fragments list
        if self.internalFragments:
            for peptide in self.internalFragments:
                for ion in peptide:
                    if ion in self.ctrlData['internalions'] and not ion in ('seq', 'pos', 'filter'):

                        # make ion description
                        ionDesription = '%s %s [%s]' % (self.ionsNames[ion], peptide['pos'], peptide['seq'])

                        # check matched
                        matched = False
                        if peptide[ion][1] != '':
                            matched = True

                        # check filtered
                        filtered = False
                        if ion in peptide['filter']:
                            filtered = True

                        # add fragment to list
                        fragmentsList.append((peptide[ion][0], ionDesription, matched, filtered))

        # return fragments list
        fragmentsList.sort()
        return fragmentsList
    # ----


    # ----
    def destroyListPanel(self):
        """ Destroy list panel. """

        self.listPanel.Destroy()
        self.listPanel = None
    # ----
