# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Basic peaklist definition.

# load libs
import copy

class peaklist:
    """ Basic peaklist definition. """

    # ----
    def __init__(self):

        self.peaks = []
        self.procBuff = []
        self.undoBuff = []
        self.redoBuff = []
        self.lastPeak = None
    # ----


    # ----
    def setPeaklist(self, peaks, undo=True, store=False):
        """ Set new peaks to peaklist. """

        # add data to undo buffer
        self.redoBuff = []
        if undo:
            self.addUndo()
        else:
            self.undoBuff = []

        # set data
        self.peaks = peaks

        # store data
        if store:
            self.store()
    # ----


    # ----
    def getPeaks(self, indexes=None):
        """ Get all or selected peaks. """

        # get all peaks
        if indexes == None:
            return self.peaks

        # get one peak
        elif type(indexes) == int:
            return self.peaks[indexes]

        # get selected peaks
        elif type(indexes) == list:
            peaks = []
            for x in indexes:
                if type(x) == int:
                    peaks.append(self.peaks[x])
            return peaks
    # ----


    # ----
    def getLength(self):
        """ Get number of peaks in peaklist. """
        return len(self.peaks)
    # ----


    # ----
    def getUndoStatus(self):
        """ Get status of undo/redo. """
        return (len(self.undoBuff), len(self.redoBuff))
    # ----


    # ----
    def addPeak(self, peak, store=False):
        """ Add new peak to peaklist. """

        # add data to undo buffer
        self.addUndo()
        self.redoBuff = []

        # check intensity
        if len(peak) == 1:
            peak.append(0)

        # check annotation
        if len(peak) == 2:
            peak.append('')

        # check picking params
        if len(peak) == 3:
            peak.append(0) # label method 0 = point

        # add and sort
        self.peaks.append(peak)
        self.peaks.sort()

        # store peaklist for undo
        if store:
            self.store()
    # ----


    # ----
    def editPeak(self, index, data, store=False):
        """ Edit selected peak. """

        # add data to undo buffer
        self.addUndo()
        self.redoBuff = []

        # update peak
        self.peaks[index] = data

        # store peaklist
        if store:
            self.store()
    # ----


    # ----
    def deletePeaks(self, indexes=None, store=False):
        """ Delete all or selected peaks. """

        # add data to undo buffer
        self.addUndo()
        self.redoBuff = []

        # delete all
        if indexes == None:
            self.peaks = []

        # delete selected
        else:

          # sort indexes to delete from end of the list
          indexes.sort()
          indexes.reverse()

          # delete peaks
          for index in indexes:
              del self.peaks[index]

        # store peaklist
        if store:
            self.store()
    # ----


    # ----
    def annotate(self, annotations, store=False):
        """ Annotate peaklist. """

        # add data to undo buffer
        self.addUndo()
        self.redoBuff = []

        # check length
        if not annotations:
            return
        elif len(annotations) != len(self.peaks):
            raise ValueError, "Peaklist length and annotations lenght must be the same."

        # update peaklist
        for peak in range(len(annotations)):
            if self.peaks[peak][2] == '':
                self.peaks[peak][2] = annotations[peak]
            elif annotations[peak] != '':
                self.peaks[peak][2] += '; ' + annotations[peak]

        # store peaklist
        if store:
            self.store()
    # ----


    # ----
    def deleteAnnotations(self, indexes=None, store=False):
        """ Delete annotations in all or selected peaks. """

        # add data to undo buffer
        self.addUndo()
        self.redoBuff = []

        # delete selected
        if indexes != None:
            for index in indexes:
                self.peaks[index][2] = ''

        # delete all
        else:
            for row in range(len(self.peaks)):
                self.peaks[row][2] = ''

        # store peaklist
        if store:
            self.store()
    # ----


    # ----
    def revert(self):
        """ Revert peaklist data to stored. """

        self.peaks = copy.deepcopy(self.procBuff)
        self.undoBuff = []
        self.redoBuff = []
    # ----


    # ----
    def store(self):
        """ Store current peaklist data as main for processing undo. """
        self.procBuff = copy.deepcopy(self.peaks)
    # ----


    # ----
    def undo(self, store=False):
        """ Revert peaklist data from undo buffer. """

        if self.undoBuff:
            self.addRedo()
            self.peaks = self.undoBuff[-1][:]
            del self.undoBuff[-1]
            if store:
                self.store()
    # ----


    # ----
    def redo(self, store=False):
        """ Revert peaklist data from redo buffer. """

        if self.redoBuff:
            self.addUndo()
            self.peaks = self.redoBuff[-1][:]
            del self.redoBuff[-1]
            if store:
                self.store()
    # ----


    # ----
    def addUndo(self):
        """ Add current peaklist data to undo buffer. """

        self.undoBuff.append(copy.deepcopy(self.peaks))
        if len(self.undoBuff) > 10:
            del self.undoBuff[0]
    # ----


    # ----
    def addRedo(self):
        """ Add current peaklist data to redo buffer. """

        self.redoBuff.append(copy.deepcopy(self.peaks))
        if len(self.redoBuff) > 10:
            del self.redoBuff[0]
    # ----
