# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Make protein digest.

# load libs
import wx

# load modules
from nucleus import mwx
from nucleus import commfce
from count import mCutCount
from modules.mlistpanel.dlg_mcutpanel import mCutListPanel

class mCut(wx.Panel):
    """Make protein digest from document sequence. """

    # ----
    def __init__(self, parent, document):
        wx.Panel.__init__(self, parent, -1)

        self.parent = parent
        self.config = parent.config
        self.docMonitor = parent.docMonitor
        self.docData = document

        # init peptide counter
        self.mCutCount = mCutCount(self.config)

        # init module variables
        self.parsedSequence = None
        self.peptides = None
        self.viewMatchedOnly = None
        self.listPanel = None

        # set default params
        self.ctrlData = {}
        self.ctrlData['enzyme'] = self.config.cfg['mcut']['enzyme']
        self.ctrlData['partials'] = self.config.cfg['mcut']['partials']
        self.ctrlData['maxpartials'] = self.config.cfg['mcut']['maxpartials']
        self.ctrlData['charge'] = self.config.cfg['mcut']['charge']
        self.ctrlData['optmodif'] = self.config.cfg['mcut']['optmodif']
        self.ctrlData['notcleavemodif'] = self.config.cfg['mcut']['notcleavemodif']
        self.ctrlData['minmass'] = self.config.cfg['mcut']['minmass']
        self.ctrlData['maxmass'] = self.config.cfg['mcut']['maxmass']
        self.ctrlData['uselimits'] = self.config.cfg['mcut']['uselimits']
        self.ctrlData['sortby'] = self.config.cfg['mcut']['sortby']
        self.ctrlData['sortdesc'] = self.config.cfg['mcut']['sortdesc']

        # setup colours
        self.colours = {}
        self.colours['matched'] = self.config.cfg['colours']['matched']
        self.colours['modified'] = self.config.cfg['colours']['modified']

        # make gui items
        self.makePeptidesList()
        digestBox = self.makeDigestBox()
        massBox = self.makeMassBox()
        generateButt = self.makeGenerateButt()
        listButton = self.makeListButt()

        # pack control elements
        controls = wx.BoxSizer(wx.VERTICAL)
        controls.Add(digestBox, 0, wx.EXPAND|wx.ALL, 10)
        controls.Add(massBox, 0, wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, 10)
        controls.Add(generateButt, 0, wx.ALIGN_CENTRE|wx.ALL, 5)
        controls.Add(listButton, 0, wx.ALIGN_CENTRE|wx.ALL, 5)

        # pack main frame
        mainSizer = wx.BoxSizer(wx.HORIZONTAL)
        mainSizer.Add(self.peptidesList, 1, wx.EXPAND)
        mainSizer.Add(controls, 0, wx.EXPAND)
        self.SetSizer(mainSizer)
    # ----


    # ----
    def makeDigestBox(self):
        """ Make main digest controls. """

        # get enzymes
        digestEnzyme_choices = []
        for enz in self.config.enz:
            digestEnzyme_choices.append(enz)
        digestEnzyme_choices.sort()

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Digest Options"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        digestEnzyme_label = wx.StaticText(self, -1, "Enzyme: ")
        self.digestEnzyme_combo = wx.ComboBox(self, -1, size=(85, -1), choices=digestEnzyme_choices, style=wx.CB_READONLY)
        self.digestEnzyme_combo.SetToolTip(wx.ToolTip("Select enzyme"))

        digestPartials_label = wx.StaticText(self, -1, "Partials: ")
        self.digestPartials_value = wx.SpinCtrl(self, -1, '', size=(84, -1), min=0, max=self.ctrlData['maxpartials'], initial=0, style=wx.SP_WRAP|wx.SP_ARROW_KEYS )
        self.digestPartials_value.SetToolTip(wx.ToolTip("Number of missed digest sites"))

        self.digestOptModif_check = wx.CheckBox(self, -1, "Optional modifications")
        self.digestNotCleaveModif_check = wx.CheckBox(self, -1, "Don't cleave if modified")

        # pack items
        grid.Add(digestEnzyme_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.digestEnzyme_combo, (0, 1))

        grid.Add(digestPartials_label, (1, 0), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.digestPartials_value, (1, 1))

        if wx.Platform == '__WXMAC__':
            mainBox.Add(grid, 0, 0)
            mainBox.Add(self.digestOptModif_check, 0, wx.TOP, 5)
            mainBox.Add(self.digestNotCleaveModif_check, 0, wx.TOP, 5)
        else:
            mainBox.Add(grid, 0, wx.ALL, 5)
            mainBox.Add(self.digestOptModif_check, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)
            mainBox.Add(self.digestNotCleaveModif_check, 0, wx.ALL, 5)

        # set events
        self.digestOptModif_check.Bind(wx.EVT_CHECKBOX, self.onOptModif)

        # set defaults
        try:
            self.digestEnzyme_combo.Select(digestEnzyme_choices.index(self.ctrlData['enzyme']))
        except ValueError:
            self.digestEnzyme_combo.Select(0)
        self.digestPartials_value.SetValue(self.ctrlData['partials'])
        self.digestOptModif_check.SetValue(self.ctrlData['optmodif'])
        self.digestNotCleaveModif_check.SetValue(self.ctrlData['notcleavemodif'])
        if self.ctrlData['optmodif']:
            self.digestNotCleaveModif_check.Enable(False)

        return mainBox
    # ----


    # ----
    def makeMassBox(self):
        """ Make mass type and limits controls. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Mass Type and Limits"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        massCharge_choices = ['M', '1+', '2+', '1-', '2-']
        massCharge_label = wx.StaticText(self, -1, "Charge: ")
        self.massCharge_combo = wx.ComboBox(self, -1, size=(70, -1), choices=massCharge_choices, style=wx.CB_READONLY)
        self.massCharge_combo.SetToolTip(wx.ToolTip("Digest peptides' charge"))

        massMinMass_label = wx.StaticText(self, -1, "Low mass: ")
        self.massMinMass_value = wx.TextCtrl(self, -1, str(self.ctrlData['minmass']), size=(70, -1), validator=mwx.txtValidator('digits'))
        self.massMinMass_value.SetToolTip(wx.ToolTip("Lowest mass visible"))

        massMaxMass_label = wx.StaticText(self, -1, "High mass: ")
        self.massMaxMass_value = wx.TextCtrl(self, -1, str(self.ctrlData['maxmass']), size=(70, -1), validator=mwx.txtValidator('digits'))
        self.massMaxMass_value.SetToolTip(wx.ToolTip("Highest mass visible"))

        self.massUseLimits_check = wx.CheckBox(self, -1, "Use mass range limits")

        # pack items
        grid.Add(massCharge_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.massCharge_combo, (0, 1))

        grid.Add(massMinMass_label, (1, 0), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.massMinMass_value, (1, 1))

        grid.Add(massMaxMass_label, (2, 0), flag=wx.ALIGN_CENTER_VERTICAL)
        grid.Add(self.massMaxMass_value, (2, 1))

        if wx.Platform == '__WXMAC__':
            mainBox.Add(grid, 0, 0)
            mainBox.Add(self.massUseLimits_check, 0, wx.TOP, 5)
        else:
            mainBox.Add(grid, 0, wx.TOP|wx.LEFT, 5)
            mainBox.Add(self.massUseLimits_check, 0, wx.ALL, 5)

        # set events
        self.massUseLimits_check.Bind(wx.EVT_CHECKBOX, self.onUseLimitsSelected)

        # set defaults
        self.massCharge_combo.Select(massCharge_choices.index(self.ctrlData['charge']))
        self.massUseLimits_check.SetValue(self.ctrlData['uselimits'])

        return mainBox
    # ----


    # ----
    def makeGenerateButt(self):
        """ Make 'Digest' button. """

        generateButt = wx.Button(self, -1, "Digest")
        generateButt.Bind(wx.EVT_BUTTON, self.onGenerate)
        return generateButt
    # ----


    # ----
    def makeListButt(self):
        """ Make 'List' button. """

        listButt = wx.Button(self, -1, "Show in panel")
        listButt.Bind(wx.EVT_BUTTON, self.onShowInPanel)
        return listButt
    # ----


    # ----
    def makePeptidesList(self):
        """ Make main peptides-list. """

        self.peptidesList = mwx.ListCtrl(self, -1)

        # row height hack for MSW
        if wx.Platform == '__WXMSW__':
            il = wx.ImageList(1, 16)
            self.peptidesList.SetImageList(il, wx.IMAGE_LIST_SMALL)

        # make header
        self.peptidesList.InsertColumn(0, "#")
        self.peptidesList.InsertColumn(1, "Range", wx.LIST_FORMAT_CENTER)
        self.peptidesList.InsertColumn(2, "Part.", wx.LIST_FORMAT_CENTER)
        self.peptidesList.InsertColumn(3, "Mass/z", wx.LIST_FORMAT_RIGHT)
        self.peptidesList.InsertColumn(4, "Sequence", wx.LIST_FORMAT_LEFT)

        # set columns width
        self.peptidesList.SetColumnWidth(0, 25)
        self.peptidesList.SetColumnWidth(1, 60)
        self.peptidesList.SetColumnWidth(2, 60)
        self.peptidesList.SetColumnWidth(3, 60)

        # set event
        self.peptidesList.Bind(wx.EVT_LIST_COL_CLICK, self.onSortList)
        self.peptidesList.Bind(wx.EVT_LIST_ITEM_RIGHT_CLICK, self.onRMB)
        self.peptidesList.Bind(wx.EVT_RIGHT_UP, self.onRMB)
    # ----


    # ----
    def updatePeptidesList(self):
        """ Show generated peptides. """

        # clear list
        self.peptidesList.DeleteAllItems()

        # check data
        if not self.peptides:
            for col in range(4):
                self.peptidesList.SetColumnWidth(col, wx.LIST_AUTOSIZE_USEHEADER)
            return

        # sort peptides by specified column
        self.peptides = commfce.sortMultiList(self.peptides, self.ctrlData['sortby'], self.ctrlData['sortdesc'])

        # paste results
        digitsFormat = '%0.' + `self.config.cfg['common']['digits']` + 'f'
        row = -1
        for x in range(len(self.peptides)):

            # show matched peptides only
            if self.viewMatchedOnly == 1 and self.peptides[x][6] == '':
                continue

            # show unmatched peptides only
            elif self.viewMatchedOnly == -1 and self.peptides[x][6] != '':
                continue

            # paste data
            row += 1
            mass = digitsFormat % self.peptides[x][3]
            self.peptidesList.InsertStringItem(row, str(self.peptides[x][0]))
            self.peptidesList.SetStringItem(row, 1, self.peptides[x][1])
            self.peptidesList.SetStringItem(row, 2, str(self.peptides[x][2]))
            self.peptidesList.SetStringItem(row, 3, str(mass))
            self.peptidesList.SetStringItem(row, 4, self.peptides[x][4])

            # mark matched peptides
            if self.peptides[x][6] != '' and self.viewMatchedOnly == None:
                self.peptidesList.SetItemBackgroundColour(row, self.colours['matched'])

            # mark modified peptides
            if self.config.cfg['mcut']['highlightmods'] and self.viewMatchedOnly != None and self.peptides[x][5]:
                self.peptidesList.SetItemBackgroundColour(row, self.colours['modified'])

        # set columns width
        if self.peptidesList.GetItemCount():
            autosize = wx.LIST_AUTOSIZE
        else:
            autosize = wx.LIST_AUTOSIZE_USEHEADER
        for col in range(4):
            self.peptidesList.SetColumnWidth(col, autosize)
        self.peptidesList.SetColumnWidth(2, wx.LIST_AUTOSIZE_USEHEADER)
        self.peptidesList.updateLastCol()
    # ----


    # ----
    def updateListPanel(self):
        """ Update list panel. """

        if self.listPanel:
            self.listPanel.updatePanel(self.peptides)
    # ----


    # ----
    def onShow(self):
        """ Show panel and set focus to main item. """

        self.Show(True)
        self.peptidesList.SetFocus()
    # ----


    # ----
    def onRMB(self, evt):
        """ Raise popup menu on right-click. """

        # make menu items
        menuViewMatchedID = wx.NewId()
        menuViewUnmatchedID = wx.NewId()
        menuViewAllID = wx.NewId()

        # add items to menu
        menu = wx.Menu()
        menu.Append(menuViewMatchedID, "View Matched Only", " View matched peptides only", wx.ITEM_RADIO)
        menu.Append(menuViewUnmatchedID, "View Missed Only", " View unmatched peptides only", wx.ITEM_RADIO)
        menu.Append(menuViewAllID, "View All Peptides", " View all possible peptides", wx.ITEM_RADIO)

        # bind events to menu
        self.Bind(wx.EVT_MENU, self.onViewMatchedOnly, id=menuViewMatchedID)
        self.Bind(wx.EVT_MENU, self.onViewUnmatchedOnly, id=menuViewUnmatchedID)
        self.Bind(wx.EVT_MENU, self.onViewAll, id=menuViewAllID)

        # set defaults
        if self.viewMatchedOnly == 1:
            menu.Check(menuViewMatchedID, True)
        elif self.viewMatchedOnly == -1:
            menu.Check(menuViewUnmatchedID, True)
        else:
            menu.Check(menuViewAllID, True)

        # disable if no data
        if self.peptides == []:
            menu.Enable(menuViewMatchedID, False)
            menu.Enable(menuViewUnmatchedID, False)
            menu.Enable(menuViewAllID, False)

        self.PopupMenu(menu)
        menu.Destroy()
    # ----


    # ----
    def onOptModif(self, evt):
        """ Disable 'Don't cleave if modified' button if optional modifications. """

        # check if checked
        if evt.IsChecked():
            self.digestNotCleaveModif_check.Enable(False)
        else:
            self.digestNotCleaveModif_check.Enable(True)
    # ----


    # ----
    def onUseLimitsSelected(self, evt):
        """ Enable/disable mass limits textboxes. """

        # check if 'Use limits' checked
        if evt.IsChecked():
            self.massMinMass_value.Enable(True)
            self.massMaxMass_value.Enable(True)
        else:
            self.massMinMass_value.Enable(False)
            self.massMaxMass_value.Enable(False)
    # ----


    # ----
    def onSortList(self, evt):
        """ Sort peptides-list by specified column. """

        # get selected column
        selectedCol = evt.m_col
        if selectedCol == 1:
            selectedCol = 0

        # get order
        if self.ctrlData['sortby'] == selectedCol and not self.ctrlData['sortdesc']:
            self.ctrlData['sortdesc'] = 1
        else:
            self.ctrlData['sortdesc'] = 0
            self.ctrlData['sortby'] = selectedCol

        # update peptide list
        self.updatePeptidesList()
    # ----


    # ----
    def onViewMatchedOnly(self, evt=None):
        """ Filter peptides-list to view matched peptides only. """

        self.viewMatchedOnly = 1
        self.updatePeptidesList()
    # ----


    # ----
    def onViewUnmatchedOnly(self, evt=None):
        """ Filter peptides-list to view unmatched peptides only. """

        self.viewMatchedOnly = -1
        self.updatePeptidesList()
    # ----


    # ----
    def onViewAll(self, evt=None):
        """ Remove any filter and show all peptides. """

        self.viewMatchedOnly = None
        self.updatePeptidesList()
    # ----


    # ----
    def onGenerate(self, evt=None):
        """ Get all controls and generate digest peptides. """

        # get and parse controls
        if not self.getControls():
            return

        # set application working
        self.docMonitor('setAppStatus', "Generating peptides...")

        # digest protein
        self.mCutCount.ctrlData = self.ctrlData
        self.peptides = self.mCutCount.digest(self.parsedSequence)

        # update table of peptides
        self.viewMatchedOnly = None
        self.updatePeptidesList()

        # update list panel
        self.updateListPanel()

        # update document status
        self.docData.setDataStatus(bool(self.peptides), 'mCut')

        # update application
        self.docMonitor('onModuleReset', 'mCut')
        self.docMonitor('setAppStatus', 0)
    # ----


    # ----
    def onShowInPanel(self, evt):
        """ Show list of all peptides in separate panel. """

        # check data
        if not self.peptides:
            return

        # raise panel
        if not self.listPanel:
            self.listPanel = mCutListPanel(self, self.config, self.peptides)
            self.listPanel.Show()
    # ----


    # ----
    def onSpectrumHighlightPoint(self, point, zoom):
        """ Highlight selected point in the spectrum. """
        self.parent.onSpectrumHighlightPoint(point, zoom)
    # ----


    # ----
    def getControls(self):
        """ Get and check controls. """

        errorMessage = ''

        # get sequence from document
        self.parsedSequence = self.docData.getParsedSequence()

        # get digest params
        self.ctrlData['enzyme'] = self.digestEnzyme_combo.GetValue()
        self.ctrlData['partials'] = self.digestPartials_value.GetValue()
        self.ctrlData['optmodif'] = self.digestOptModif_check.IsChecked()
        self.ctrlData['notcleavemodif'] = self.digestNotCleaveModif_check.IsChecked()
        if self.ctrlData['optmodif']:
            self.ctrlData['notcleavemodif'] = False

        # get mass params
        charge = self.massCharge_combo.GetValue()
        minMass = self.massMinMass_value.GetValue()
        maxMass = self.massMaxMass_value.GetValue()
        self.ctrlData['uselimits'] = self.massUseLimits_check.IsChecked()

        # set charge
        if charge == 'M':
            self.ctrlData['charge'] = 0
        elif charge == '1+':
            self.ctrlData['charge'] = 1
        elif charge == '2+':
            self.ctrlData['charge'] = 2
        elif charge == '1-':
            self.ctrlData['charge'] = -1
        elif charge == '2-':
            self.ctrlData['charge'] = -2

        # get common params
        self.ctrlData['masstype'] = self.config.cfg['common']['masstype']

        # validate data
        if not self.parsedSequence:
            errorMessage = "Protein sequence is empty!\nCheck the sequence editor."
        elif self.ctrlData['uselimits']:
            try:
                self.ctrlData['minmass'] = int(minMass)
                self.ctrlData['maxmass'] = int(maxMass)
            except ValueError:
                errorMessage = "Low mass and high mass must be integers!"

        # if controls not OK
        if errorMessage != '':
            errorDlg = wx.MessageDialog(self, errorMessage, "Error", wx.OK|wx.ICON_ERROR)
            errorDlg.ShowModal()
            errorDlg.Destroy()
            self.parsedSequence = None
            return False
        else:
            return True
    # ----


    # ----
    def matchDataToPeaklist(self):
        """ Compare generated list of peptides masses with the document peaklist. """

        # get common params
        self.ctrlData['tolerance'] = self.docData.getMassParam('tolerance')
        self.ctrlData['errortype'] = self.docData.getMassParam('errortype')

        # match peptides
        mainPeaklist = self.docData.getPeaks()
        self.mCutCount.ctrlData = self.ctrlData
        self.peptides, matchStatus = self.mCutCount.matchDataToPeaklist(mainPeaklist, self.peptides)

        # update peptide list
        self.viewMatchedOnly = None
        self.updatePeptidesList()

        # update list panel
        self.updateListPanel()

        # update document status
        self.docData.setMatchStatus(matchStatus, 'mCut')

        return matchStatus
    # ----


    # ----
    def getMatchInfo(self):
        """ Get information about current match. """

        # get basic params
        errorType = self.docData.getMassParam('errortype')
        digits = self.config.cfg['common']['digits']

        # get match info
        mainPeaklist = self.docData.getPeaks()
        data = self.mCutCount.getMatchInfo(mainPeaklist, self.peptides, self.parsedSequence, errorType, digits)

        return data
    # ----


    # ----
    def annotatePeaklist(self):
        """ Annotate matched peaks in the document peaklist. """

        # get peaklist from document
        mainPeaklist = self.docData.getPeaks()

        # copy peaklist structure
        annotations = []
        for x in range(len(mainPeaklist)):
            annotations.append('')

        # get basic params
        errorType = self.docData.getMassParam('errortype')
        digits = self.config.cfg['common']['digits']

        # check each peptide
        for peptide in self.peptides:
            # collect info for each matched peak
            if peptide[6] != '':
                # split matched peptides
                peaks = peptide[6].split(';')
                for peakIndex in peaks:
                    # anotate matched peaks
                    if peakIndex != '':
                        newAnnotation = ''

                        # get info
                        peakIndex = int(peakIndex)
                        peptideRange = peptide[1][1:-1]
                        peptideParts = peptide[2]
                        peptideMass = peptide[3]
                        peptideSeq = peptide[4]
                        peakMass = mainPeaklist[peakIndex][0]
                        oldAnnotations = annotations[peakIndex]

                        if oldAnnotations != '':
                            oldAnnotations += '; '

                        # count error
                        error = commfce.calcMassError(peakMass, peptideMass, errorType)
                        if errorType == 'Da':
                            error = round(error, digits)

                        # make annotation
                        newAnnotation = '(%s %s) [%s/%d] [%s]' % (error, errorType, peptideRange, peptideParts, peptideSeq)
                        annotations[peakIndex] = oldAnnotations + newAnnotation

        # update document
        self.docData.annotatePeaklist(annotations)
    # ----


    # ----
    def destroyListPanel(self):
        """ Destroy list panel. """

        self.listPanel.Destroy()
        self.listPanel = None
    # ----
