#-----------------------------------------------------------------------------
#
#  Copyright (c) 2006 by Enthought, Inc.
#  All rights reserved.
#
#  Author: Martin Chilvers <martin@enthought.com>
#  Author: Dave Peterson <dpeterson@enthought.com>
#
#-----------------------------------------------------------------------------

""" Builds menus from action, group, and menu extensions.
"""

# Standard library imports.
import logging

# Enthought library imports.
from enthought.pyface.api import ImageResource
from enthought.pyface.action.api import Group, MenuManager
from enthought.traits.api import Instance

# Local imports.
from action_factory import ActionFactory
from menu_builder import MenuBuilder


# Setup a logger for this module.
logger = logging.getLogger(__name__)


class DefaultMenuBuilder(MenuBuilder):
    """ Builds menus from action, group, and menu extensions.
    """

    ##########################################################################
    # Traits
    ##########################################################################

    #### public 'DefaultMenuBuilder' interface ###############################

    # The factory to use when building menu actions.
    action_factory = Instance(ActionFactory)

    # The application this object builds menus for.
    application = Instance('enthought.envisage.Application')


    ##########################################################################
    # 'MenuBuilder' interface.
    ##########################################################################

    #### protected interface #################################################

    def _create_action(self, extension):
        """ Creates an action implementation from an action extension.
        """
        return self.action_factory.create_action(extension)


    def _create_group(self, extension):
        """ Creates a group implementation from a group extension.
        """
        if len(extension.class_name) > 0:
            klass = self._import_class(extension)
        else:
            klass = Group

        return klass(id=extension.id, separator=extension.separator)


    def _create_menu_manager(self, extension):
        """ Creates a menu manager implementation from a menu extension.
        """
        if len(extension.class_name) > 0:
            klass = self._import_class(extension)
        else:
            klass = MenuManager
        menu_manager = klass(id=extension.id, name=extension.name)
        for group in extension.groups:
            menu_manager.append(self._create_group(group))

        return menu_manager


    ##########################################################################
    # 'DefaultMenuBuilder' interface
    ##########################################################################

    #### protected interface #################################################

    def _action_factory_default(self):
        """ Creates the default action factory for this instance.
        """
        return ActionFactory(application=self.application)


    def _import_class(self, extension):
        """ Imports an extension implementation class.
        """
        # The plugin definition that the extension was defined in.
        plugin_definition = extension._action_set_._definition_

        # We are about to actually import the menu manager class, so make
        # sure that the plugin that contributed it has been started.
        self.application.start_plugin(plugin_definition.id)

        # Import the implementation class
        return self.application.import_symbol(extension.class_name)


#### EOF #####################################################################
