/****************************************************************************

  module      : gbd190.h

  -------------------------------------------------------------------------

  author      : TorstenS
  responsible : TorstenS

  special area: FileDirectory
  description : Cache for syskey generation


  last changed: 2005-06-20  14:00
  see also    : 

  -------------------------------------------------------------------------

  copyright:    (c) 1999-2004 SAP AG



    ========== licence begin  GPL
    Copyright (c) 1999-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*****************************************************************************/

#ifndef GBD190_H
#define GBD190_H

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/


#include "gsp00.h"   // PASCAL: SP_basic_constants_and_types
#include "ggg04.h"   // PASCAL: GG_kernel_constants_and_types_1
#include "ggg00.h"   // PASCAL: GG_kernel_constants_and_types
#include "hsp30.h"   // PASCAL: s30surrogate_incr

#include "Container/Container_Hash.hpp"
#include "RunTime/Synchronisation/RTESync_RWRegion.hpp"
#include "RunTime/MemoryManagement/RTEMem_AllocatorWrapper.hpp"
#include "SAPDBCommon/SAPDB_MemCopyMove.hpp" // Kernel_move_and_fill


class SAPDBMem_IRawAllocator;

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

class bd190SystemKeyCache
{
public:

    static bd190SystemKeyCache& CreateInstance();

    static bd190SystemKeyCache& Instance()
    {
        SAPDBERR_ASSERT_STATE( 0 != m_Instance );
        return *m_Instance;
    }

    void Initialize(){
        m_SysKeyCache.Delete();
    }

    bool GetNewSysKeyForFile(
        const tsp00_PageNo  fileRoot,
        tgg00_Surrogate&    sysKey )
    {
        LockedScope lock( *m_pRwLock );

        Iterator iter = m_SysKeyCache.Find( fileRoot );
        if( ! iter.IsValid())
            return false;
        sysKey = iter->value.GetNextSysKey();
        return true;
    }

    bool AddAndGetNewSysKeyForFile(
        const tsp00_PageNo  fileRoot,
        tgg00_Surrogate&    sysKey )
    {
        LockedScope lock( *m_pRwLock );

        Iterator iter = m_SysKeyCache.Find( fileRoot );
        if( ! iter.IsValid())
        {
            iter = m_SysKeyCache.Insert( fileRoot, CacheItem( sysKey ));
            if( ! iter.IsValid())
                return false;
        }
        sysKey = iter->value.GetNextSysKey();
        return true;
    }

    void DeleteSysKeyFile( const tsp00_PageNo  fileRoot )
    {
        LockedScope lock( *m_pRwLock );

        Iterator iter = m_SysKeyCache.Find( fileRoot );
        m_SysKeyCache.Delete( iter ); // iter.IsValid() is executed inside
    }

private:

    // Copy constructor is not supported
    bd190SystemKeyCache( const bd190SystemKeyCache& );

    // Assignment operator is not supported
    bd190SystemKeyCache& operator= ( const bd190SystemKeyCache& );

    bd190SystemKeyCache( SAPDBMem_IRawAllocator  &allocator );


private: // class

    class CacheItem
    {
    public:
        CacheItem(){
            SAPDB_MemCopyNoCheck( m_SystemKey, NIL_SYSKEY_GG00, sizeof( m_SystemKey ));
        }

        CacheItem( const tgg00_Surrogate&    sysKey )
        : m_SystemKey( sysKey )
        {}

        tgg00_Surrogate GetNextSysKey()
        {
            s30surrogate_incr( m_SystemKey );
            return m_SystemKey;
        }

    private:
        tgg00_Surrogate m_SystemKey;
    };

    class LockedScope
    {
    public:
        LockedScope( RTESync_IRWRegion& rwRegion ): m_RWRegion( rwRegion )
        { m_RWRegion.enter( true );}

        ~LockedScope(){ m_RWRegion.leave( true );}

    private :
        RTESync_IRWRegion& m_RWRegion;
    };

private:

    typedef Container_Hash<tsp00_PageNo, CacheItem> SysKeyCache;
    typedef SysKeyCache::Iterator       Iterator;
    typedef SysKeyCache::ConstIterator  ConstIterator;


    static bd190SystemKeyCache* m_Instance;

    RTEMem_AllocatorWrapper     m_Allocator;
    RTESync_SpinlockPool        m_SpinLock;
    RTESync_IRWRegion*          m_pRwLock;

    SysKeyCache                 m_SysKeyCache;
};

/*===========================================================================*
 *  END CLASS                                                                *
 *===========================================================================*/

#endif // GBD190_H
