/*!---------------------------------------------------------------------
  @file           RTESys_MicroTime.c
  @author         JoergM
  @ingroup        RunTime
  @brief          Microsecond timer

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2001-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



\endif
---------------------------------------------------------------------*/




/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "RunTime/System/RTESys_MicroTime.h"

#if defined(_WIN32)
#include "RunTime/System/RTESys_NTHighResTimer.h" /* nocheck */
#endif

#include <time.h>

#ifdef AIX
#   include     <sys/time.h>
    /*  AIX 4.1 and newer - use routines 'read_real_time' and 'time_base_to_time' */
#define HAS_READ_REAL_TIME
#endif

#ifdef SUN
#   include     <sys/time.h>
#define HAS_GETHRTIME
#endif

#ifdef HPUX
/* defined in /usr/include/time.h */
#define HAS_GETHRTIME
#endif

#ifdef OSF1
#   include     <sys/time.h>
#endif

#ifdef LINUX
#   include     <sys/time.h>
/*
  In a single thread application librt must not be referenced. But clock_gettime()
  is defined in librt, so we excluded this explicitly for single threaded (!_REENTRANT)....
 */
#  if !defined(_REENTRANT) && defined(CLOCK_MONOTONIC)
#    define HAS_CLOCK_MONOTONIC
#  endif
#endif

/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/

#define RTESYS_SECONDS_TO_MICROSECONDS(s_) (((SAPDB_UInt8)s_)*1000000U)
#define RTESYS_NANOSECONDS_TO_MICROSECONDS(s_) (((SAPDB_UInt8)s_)/1000U)

/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/

/*===========================================================================*
 *  STRUCTURES, TYPES, UNIONS ...                                            *
 *===========================================================================*/

/*
  Value needed to return number of microseconds since start
  These make the call RTESys_InitMicroSecTimer() not multithread save, 
  which is acceptable to prevent the otherwise needed spinlock coding...
 */
static SAPDB_UInt8 databaseStartTimeInMicroseconds = 0;

/*===========================================================================*
 *  STATIC/INLINE FUNCTION PROTOTYPES                                        *
 *===========================================================================*/

/*
  Function: CurrentMicroSeconds
  Description: Return the current number of microseconds as returned from high res timer

  The base of the number has nothing to do with database start, but the number is expected to have no overflow...

  Return value: Number of microseconds as returned from high res timer
 */
static SAPDB_UInt8 CurrentMicroSeconds();

/*===========================================================================*
 *  GLOBAL FUNCTIONS                                                         *
 *===========================================================================*/

/*!--------------------------------------------------------------------
   @description    Initialize timer for microseconds since start of database

 --------------------------------------------------------------------*/

void RTESys_InitMicroSecTimer()
{
    databaseStartTimeInMicroseconds = CurrentMicroSeconds();
}

/*
  Function: RTESys_MicroSecTimer
  Description: Return number of microseconds since start of database

  Return value: Number of microseconds since database start
 */
SAPDB_UInt8 RTESys_MicroSecTimer()
{
    return CurrentMicroSeconds() - databaseStartTimeInMicroseconds;
}

/*===========================================================================*
 *  STATUC FUNCTIONS                                                         *
 *===========================================================================*/

static SAPDB_UInt8 CurrentMicroSeconds()
{
#if defined(_WIN32)

    return RTESys_NTHRTimerMicroSeconds();

#elif defined(HAS_READ_REAL_TIME)

    timebasestruct_t time_now ;

    (void) read_real_time    ( & time_now , TIMEBASE_SZ );
    /* This yields processor-specific units - not usable directly */

    (void) time_base_to_time ( & time_now , TIMEBASE_SZ );
    /* Now 'time_now' contains second / nanosecond */

    return ( RTESYS_SECONDS_TO_MICROSECONDS(time_now.tb_high)
           + RTESYS_NANOSECONDS_TO_MICROSECONDS(time_now.tb_low) );

#elif defined(HAS_GETHRTIME)

    hrtime_t currentTimeInNanoSeconds;
    SAPDB_UInt8 resultTime;
    
    currentTimeInNanoSeconds = gethrtime();
    resultTime = RTESYS_NANOSECONDS_TO_MICROSECONDS(currentTimeInNanoSeconds);

    return resultTime;

#elif defined(HAS_CLOCK_MONOTONIC)

    struct timespec monotonTime;
    if ( 0 == clock_gettime(CLOCK_REALTIME, &monotonTime) )
    {
        return ( RTESYS_SECONDS_TO_MICROSECONDS(monotonTime.tv_sec)
               + RTESYS_NANOSECONDS_TO_MICROSECONDS(monotonTime.tv_nsec) );
    }
    else
    {
        struct timeval	tv ;
        (void) gettimeofday ( &tv , (void *)0 );
        return ( RTESYS_SECONDS_TO_MICROSECONDS(tv.tv_sec) + tv.tv_usec );
    }

#else

    /* Fallback of all other platforms */
    struct timeval	tv ;

    (void) gettimeofday ( &tv , (void *)0 );

    return ( RTESYS_SECONDS_TO_MICROSECONDS(tv.tv_sec) + tv.tv_usec );

#endif
}

/*!--------------------------------------------------------------------
   @brief    Returns microsecond since since 1.1.1970 0:00 UTC

   Used as input for RTESys_BuildSQLTimeStamp to generate SQL time stamps. This
   call is not used to get timestamps for intenal timing measurements, so it is
   not called very frequently, but it should be as precise as possible and always
   monotonic ascendend. 

   @return   Microseconds since 1.1.1970 0:00 UTC
 */
externC SAPDB_UInt8 RTESys_MicroSecondTime()
{
#if defined(WIN32)
    static FILETIME the_epoch = { 3577643008UL, 27111902UL };

    SYSTEMTIME currentTime;
    union {
        FILETIME asFILETIME;
        SAPDB_UInt8 asUInt8;
    } now;

    GetSystemTime(&currentTime);
    SystemTimeToFileTime(&currentTime, &now.asFILETIME);
    return ( (now.asUInt8 - (*(SAPDB_UInt8 *)&the_epoch)) / 10 );
#else
    struct timeval	tv ;

    (void) gettimeofday ( &tv , (void *)0 );

    return ( RTESYS_SECONDS_TO_MICROSECONDS(tv.tv_sec) + tv.tv_usec );
#endif
}

/*===========================================================================*
 *  END OF CODE                                                              *
 *===========================================================================*/