#/*!
#\@file           genSQLDBCdoc.pl
#\@author         DL SAP DB INTERFACES
#\@ingroup        SQLDBC
#\@brief          SAP Database Connectivity
#\@see            
#
#       This script patches the generated doxygen documentation 
#       for the c++ interface SQLDBC
#
#\\if EMIT_LICENCE
#
#    ========== licence begin  GPL
#    Copyright (c) 2001-2005 SAP AG
#
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of the GNU General Public License
#    as published by the Free Software Foundation; either version 2
#    of the License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#    ========== licence end

#\\endif
#*/

#use strict;
use Env;
use File::Path;
use Getopt::Long;
use Getopt::Std;
use File::Copy;

my @doxygendirs = ("$ENV{OWN}/tool/bin/", "$ENV{TOOL}/pgm/", "");

my $dotexe=".exe" if($^O=~/.*win32.*/i);
my $doxygen;
# regular expression for the version on doxygen needed 
# may overwritten by option --doxyver
my @doxyversions= ("1.4.*", "1.3.*");
my $doxyverfound="unknown";
my $package="SQLDBC-MAN.TGZ";
my $doxyfile="$ENV{WRK}/fast/obj/SAPDB/Interfaces/SQLDBC/";
my @addfiles=();

my $cmd;
GetOptions("dir=s", "with-floss", "addfiles=s", "help") || usage();

if (!getopts ('h') || $opt_help || !$opt_dir) {
    usage();
}

if (-d "$opt_dir") {
    chdir("$opt_dir");
    unlink <*>;
}

if ($opt_addfiles) {
   @addfiles = split(',', $opt_addfiles);
}

my $found=0;
my $versionok=0;

# find doxygen 
$dir="$ENV{OWN}/tool/bin/";
$doxygen="${dir}doxygen$dotexe";
if (-x "$doxygen") {
    checkDoxygenVersion($doxygen, $ver);
    $found=1;
    $versionok=1;
}
else {
    foreach my $ver (@doxyversions) {
        foreach my $dir (@doxygendirs) {
            $doxygen="${dir}doxygen$dotexe";
            if ($dir eq "" || -x "$doxygen") {
                if (checkDoxygenVersion($doxygen, $ver)) {
                    $found=1;
                    $versionok=1;
                    last;
                }
            }
        }
        if ($found) {
            last;
        }
    }
}

if (!$found) {
    print "ERROR: Could not found $doxygen\n";
    exit 1;    
}
else {
    if (!$versionok) {
        print "ERROR: Wrong doxygen version '$doxyverfound' nedded '@doxyversions'\n";
        exit 1;    
    }
}

if ($opt_with_floss) {
    $doxyfile.="SQLDBC-FLOSS.doxygen"
}
else {
    $doxyfile.="SQLDBC.doxygen"
}
print "Use $doxygen version $doxyverfound\n";
$cmd = "$doxygen $doxyfile";
print "$cmd\n";
if(system($cmd)) {
    print "ERROR: Can't generate documentation\n";
    exit 1;
}

if ($doxyverfound =~ /1\.3/ ) {
    patchDoxygen1_3();
}
else {
    patchDoxygen1_4();
}
    
chdir("$ENV{OWN}/usr");
    
my $from="HelloWorld.tgz";
my $to="sdk/sqldbc/doc/html/";
File::Copy::copy($from, $to) || die "Can't copy $from to $to : $!";
 
my $cmd = "tar -cvozf $package sdk/sqldbc/doc/* -T @addfiles";
print "$cmd\n";
if (system($cmd)) {
    print "ERROR: '$cmd' failed. Can't pack documentation.\n";
    exit 1;    
}

if (! -d "$ENV{WRK}/fast/install" ) {
    mkdir ("$ENV{WRK}/fast/install", 0777);
}

File::Copy::copy("$package", "$ENV{WRK}/fast/install/$package") || die "Can't copy $package to $ENV{WRK}/fast/install/$package : $!";
print "Package $package copied to $ENV{WRK}/fast/install\n";

exit 0;

sub checkDoxygenVersion
{
    my $doxygen=shift;
    my $doxyver=shift;
    $doxyverfound="";
    open(P1, "$doxygen -h |" ) || die "Can't open pipe '$doxygen -h' : $!\n";
    while(<P1>) {
        if (/Doxygen version (.*)$/) {
            $doxyverfound = $1;
        }
    }
    close(P1);
    if (! ($doxyverfound =~ /$doxyver/) ) {
        return 0;
    }
    if ($doxyverfound =~ /([0-9]\.[0-9])*/ ) {
        $doxyverfound = $1;
    }
    return 1;
}

sub patchDoxygen1_3 
{
    my $infile = "$opt_dir/tree.js";
    my $outfile = "$opt_dir/tree.js.tmp";
    open(FILE_IN, "<$infile") || die "Can't open '$infile' (input): $!";
    open(FILE_OUT, ">$outfile")  || die "Can't open '$outfile' (output): $!";
    my $firstline=1;
    my $sqldbc_c_ref;
    while(<FILE_IN>) {
#
#   Skip if already modified
#
        if ($_=~/\/\/ tree.js modified by genSQLDBCdoc/) {
            exit 0;
        }
        if ($firstline) {
            print  FILE_OUT "// tree.js modified by genSQLDBCdoc\n";
            $firstline=0;
        }
#
#   Insert link for Native C Reference
#
        if ($_=~/\s+insDoc\(.*\"SQLDBC_C.h\".*\"(SQLDBC__C.*)\".*/){
            $sqldbc_c_ref = $1;
        }
        if ($_=~/\s+insDoc\(.*\"Compound Members\".*/){
            print  FILE_OUT "insDoc(foldersTree, gLnk(\"Native C Reference\", \"\", \"$sqldbc_c_ref\"))\n";
        }	      
#
#   Insert Installation chapter
#
        if ($_=~/\s+insDoc\(.*\"SQL Database Connectivity.*\".*/){
            print FILE_OUT $_;
            print FILE_OUT "insDoc(foldersTree, gLnk(\"Installation\", \"\", \"install.html\"))\n";
            next;
        }	          
        print FILE_OUT $_;
    }
    
#
#   Append Examples chapter
#
    printExamplesChapter(\*FILE_OUT);
    
#
#   Append License chapter
#
    printLicenseChapter(\*FILE_OUT);
    
    close(FILE_OUT); close(FILE_IN);
    
    unlink "$infile";
    rename "$outfile", "$infile";

}

sub patchDoxygen1_4 
{
    my $infile = "$opt_dir/tree.html";
    my $outfile = "$opt_dir/tree.html.tmp";
    open(FILE_IN, "<$infile") || die "Can't open '$infile' (input): $!";
    open(FILE_OUT, ">$outfile")  || die "Can't open '$outfile' (output): $!";
    my $firstline=1;
    my $foldernum=0;
    my $sqldbc_c_ref;
    while(<FILE_IN>) {
#
#   Skip if already modified
#
        if ($_=~/\<!-- tree.html modified by genSQLDBCdoc --\>/) {
            exit 0;
        }
        if ($firstline) {
            print  FILE_OUT "\<!-- tree.html modified by genSQLDBCdoc --\>\n";
            $firstline=0;
        }
#
#   Insert link for Native C Reference
#
        if ($_=~/\s.*(SQLDBC__C.*\.html).*SQLDBC_C\.h.*/){
            $sqldbc_c_ref = "$1";
        }	          
        if ($_=~/(\s.*)functions\.html(.*)Class Members(.*)/){
            print  FILE_OUT "$1$sqldbc_c_ref$2Native C Reference$3\n";
        }	      
#
#   Insert Installation chapter
#
        if ($_=~/(\s.*)main.html(.*)SQL Database Connectivity \(SQLDBC\)(.*)/){
            print FILE_OUT $_;
            print FILE_OUT "$1install.html$2Installation$3\n";
            next;
        }	          
        if (s/(\s.*)ftv2plastnode(.*)('folder)(.*)('.*Related Pages.*)/$1ftv2pnode$2$3$4$5/) {
            print FILE_OUT $_;
            $foldernum=$4;
            next;
        }
#
#   Append examples and license chapter
#
        if ($foldernum > 0 && $_=~/\s.*\<\/div\>.*/) {
            print FILE_OUT $_;            
            printExamplesChapter1_4(\*FILE_OUT, $foldernum+1);
            printLicenseChapter1_4(\*FILE_OUT, $foldernum+2);
            $foldernum=0;
        }
        print FILE_OUT $_;
    }
    
    close(FILE_OUT); close(FILE_IN);
    
    unlink "$infile";
    rename "$outfile", "$infile";

}

sub printLicenseChapter {
    my $fh = shift;
    print $fh "aux1 = insFld(foldersTree, gFld(\"License\", \"\", \"license.html\"))\n";
    if ($opt_with_floss) {
        print $fh "insDoc(aux1, gLnk(\"MaxDB FLOSS license exception\", \"\", \"FLOSS.html\"))\n";
    }
}

sub printFolderNode
{
    my $png=shift;
    my $num=shift;
    my $link=shift;
    my $name=shift;
    return "\<p\>\<img src=\"$png\" alt=\"\\\" width=16 height=22 onclick=\"toggleFolder\('folder$num', this)\"\/\>\<img src=\"ftv2folderclosed.png\" alt=\"+\" width=24 height=22 onclick=\"toggleFolder\('folder$num', this\)\"\/\>\<a class=\"el\" href=\"$link\" target=\"basefrm\"\>$name\<\/a\>\<\/p\>\n";
}

sub printFileNode
{
    my $png1=shift;
    my $png2=shift;
    my $link=shift;
    my $name=shift;
    return "\<p\>\<img src=\"$png1\" alt=\"&nbsp;\" width=16 height=22 \/\>\<img src=\"$png2\" alt=\"o\" width=16 height=22 \/\>\<img src=\"ftv2doc.png\" alt=\"*\" width=24 height=22 \/\>\<a class=\"el\" href=\"$link\" target=\"basefrm\"\>$name\<\/a\>\<\/p\>\n";
}

sub printLicenseChapter1_4 {
    my $fh = shift;
    my $fhnum = shift;
    printf $fh printFolderNode("ftv2plastnode.png", "$fhnum", "license.html", "License");
    print $fh "\<div id=\"folder$fhnum\"\>\n";

    print $fh printFileNode("ftv2blank.png", "ftv2lastnode.png", "FLOSS.html", "MaxDB FLOSS license exception");

    print $fh "\</div\>\n";
}

sub printExamplesChapter1_4 {
    my $fh = shift;
    my $fhnum = shift;
    printf $fh printFolderNode("ftv2pnode.png", "$fhnum", "examples.html", "Examples");

    print $fh "\<div id=\"folder$fhnum\"\>\n";

    print $fh printFileNode("ftv2vertline.png", "ftv2node.png", "HelloWorld.cpp.html", "Hello World");

    print $fh "\</div\>\n";
}

sub printExamplesChapter {
    my $fh = shift;
    print $fh "aux1 = insFld(foldersTree, gFld(\"Examples\", \"\", \"examples.html\"))\n";
}

sub usage() {
    print <DATA>;
    exit 1;
}

__DATA__

 USAGE: genSQLDBCdoc [-h] --dir=<path> [options]

           Patches the doxygen generated SQLDBC documnetation.

Options:  --dir=<path>    Path to the html doc tree
          --addfiles=<file> List with additional files to pack
          --help          This help
 
          --with-floss    Includes FLOSS license exception
