#   mta-specific.pl
#
#   MailScanner - SMTP E-Mail Virus Scanner
#   Copyright (C) 2001  Julian Field
#
#   $Id: mta-specific.pl,v 1.24 2002/03/25 12:10:51 jkf Exp $
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#   The author, Julian Field, can be contacted by email at
#      Jules@JulianField.net
#   or by paper mail at
#      Julian Field
#      Dept of Electronics & Computer Science
#      University of Southampton
#      Southampton
#      SO17 1BJ
#      United Kingdom
#

package MTA;

# Can you think of better names than DFile and HFile?
# I would have use D and Q but using Q gets confused
# with a "generic" queue file...
#

# Required vars are:
#
# DFileRegexp:
# A regexp that will verify that a filename is a valid
# "DFile" name and leave the queue id in $1 if it is.
#
# HFileRegexp:
# A regexp that will verify that a filename is a valid
# "HFile" name and leave the queue id in $1 if it is.
#
# TFileRegexp:
# A regexp that will verify that a filename is a valid
# "TFile" name and leave the queue id in $1 if it is.
#
# QueueFileRegexp:
# A regexp that will match any legitimate queue file name
# and leave the queue id in $1.
#
# LockType:
# The way we should usually do spool file locking for
# this MTA ("posix" or "flock")
#
# Required subs are:
#
# DFileName: 
# Take a queue ID and return
# filename for data queue file
#
# HFileName:
# Take a queue ID and return
# filename for envelope queue file
#
# TFileName:
# Take a queue ID and return
# filename for temp queue file
#
# BuildMessageCmd:
# Return the shell command to take a mailscanner header file
# and an MTA message file, and build a plain text message
# (complete with headers)
#
# ReadQf:
# Read an envelope queue file (sendmail qf) and build
# an array of lines which together form all the mail headers.
#
# AddHeader:
# Given a current set of headers (string), and another header
# (key string, value string), return the set of headers with the new one
# added.
#
# DeleteHeader:
# Given a current set of headers (string), and another header
# (string), return the set of headers with the new one removed.
#
# ReplaceHeader:
# Given a current set of headers (string), and another header
# (key string, value string), return the set of headers with the new one
# in place of any existing occurence of the header.
#
# AppendHeader:
# Given a current set of headers (string), another header
# (key string, value string), and a separator string,
# return the set of headers with the new value
# appended to any existing occurrence of the header.
#
# PrependHeader:
# Given a current set of headers (string), another header
# (key string, value string), and a separator string,
# return the set of headers with the new value
# prepended to the front of any existing occurrence of the header.
# Do the header matching in a case-insensitive way.
#
# TextStartsHeader:
# Given a current set of headers (string), another header (string)
# and a search string,
# return true if the search string appears at the start of the
# text of the header.
# Do the matching in a case-insensitive way.
#
# ConstructHeaders:
# Build a set of headers (in a string) ready to go into an MTA
# envelope file.
#
# ReadEnvelope:
# Given filehandle open for reading, read envelope lines into
# string and return it.
# 
# SplitEnvelope:
# Given complete envelope string, separate out header lines and
# return 2 strings, one containing the main part of the envelope,
# the other containing the headers.
#
# MergeEnvelope:
# Given main envelope body (from SplitEnvelope at the moment) and
# string of headers, merge them to form a complete envelope.
#
# MergeEnvelopeParts:
# Given filehandle open for reading, merge envelope data (excepting
# headers) from filehandle with headers from string, and return new
# envelope data in string, ready to be written back to new
# envelope queue file.
#
# KickMessage:
# Given id, tell MTA to make a delivery attempt.
#

my($cat) = "/bin/cat";
my($sed) = "/bin/sed";

# Do conditional once at include time

if ($Config::MTA eq "exim") {

  Log::InfoLog("Configuring mailscanner for Exim mailer...");

  # These need to be improved
  $DFileRegexp = '^([-\\w]*)-D$';
  $HFileRegexp = '^([-\\w]*)-H$';
  $TFileRegexp = '^([-\\w]*)-T$';
  $QueueFileRegexp = '^([-\\w]*)-[A-Z]$';

  $LockType = "posix";

  eval <<'__EOD';

  sub DFileName {
    my($id) = @_;
    return "$id-D";
  }

  sub HFileName {
    my($id) = @_;
    return "$id-H";
  }

  sub TFileName {
    my($id) = @_;
    return "$id-T";
  }

  sub BuildMessageCmd {
    my($hfile, $dfile) = @_;
    return "$sed -e '1d' \"$dfile\" | $cat \"$hfile\" -";
  }
  
  sub ReadQf {
    my($RQf) = @_;
    my($InHeader, $InSubject, $InDel, @results, $msginfo, $from, @to, $subject);
    my($ip);
    my($Line);
    
    $Line = <$RQf>;	# queue file name
    $Line = <$RQf>;	# username, uid, gid that submitted message
    
    $Line = <$RQf>;	# envelope-sender (in <>)
    $from = $Line;
    chomp $from;
    #print "Sender is \"$from\"\n";
    
    $Line = <$RQf>;	# time msg received (seconds since epoch) + number of delay warnings sent
    
    while (<$RQf>) {
      if (/^-(host_address )?(\d+\.\d+\.\d+\.\d+)?/) {
	if ($1 eq "host_address ") {
	  $ip = $2;
	  #print "IP is \"$ip\"\n";
	  #$ip = '127.0.0.2';
	}
	next;
      }
      # lose lines containing excluded/delivered recipients...
      # if they are delivered, we're in trouble.
      # maybe we should handle exclusions!!!
      /^[NXY]{2}/ and next;
      last;
    }
    
    # ignore current line - it contains the number of recipients
    
    while (<$RQf>) {
      $Line = $_;
      chomp $Line;
      # blank line separates recipients from headers
      $Line and push @to, $Line or last;
    }
    
    $InHeader = 0;
    $InSubject = 0;
    $InDel = 0;
    while (<$RQf>) {
      $Line = $_;
      if ($InHeader) {
	$InHeader -= (length($Line));
	if ($InHeader < 0) {
	  Log::InfoLog("Header ($Line) too long (wanted $InHeader) -- using it anyway!!");
	  $InHeader = 0;
	}
	unless ($Line =~ /^[\t ]/) {
	  Log::InfoLog("Header continuation ($Line) doesn't begin with LWSP -- using it anyway!!");
	}
	push @results, $Line unless $InDel;
	# BEWARE: original sendmail version only set $subject to 1st line of subject header
	if ($InSubject) {
	  $subject .= $Line;
	}
	$InDel = ($InDel && $InHeader);
	$InSubject = ($InSubject && $InHeader);
	next;
      }
      if ($Line =~ /^([\d]{3,})([A-Z* ]) (.*)/s) {
	# start of log line
	$InHeader = $1 - (length($3));
	$InDel = ($2 eq '*');
	if ($InHeader < 0) {
	  Log::WarnLog("Header too long! -- using it anyway!!");
	  $InHeader = 0;
	}
	unless ($InDel) {
	  push @results, $3;
	  if ($3 =~ /^Subject:\s+(\S.*)$/i) {
	    $subject = $1;
	    $InSubject = 1;
	    #print "Subject is \"$subject\"\n";
	  }
	}
	$InDel = ($InDel && $InHeader);
	$InSubject = ($InSubject && $InHeader);
	next;
      }
      Log::WarnLog("Apparently invalid line in queue file! - continuing anyway.");
    }
    
    # JKF 25/03/2002 Sanitise the list of headers by removing newline
    # characters from header lines. Particularly a problem with the subject.
    my(@stripped);
    foreach $Line (@results) {
      #print STDERR "Line is \"$Line\"\n";
      $Line =~ s/\015/ /g;
      #print STDERR "Line is now \"$Line\"\n";
      push @stripped, $Line;
    }
    $subject =~ s/\015/ /g;

    $msginfo = $from . "\0" . join(', ', @to) . "\0" . $subject . "\0" . $ip;
    return (\@stripped, $msginfo);
  }

  sub AddHeader {
    my($currentheaders, $newkey, $newvalue) = @_;
    my($newheader);

    $newheader = "$newkey $newvalue";
    return $currentheaders . sprintf("%03d  $newheader\n", length($newheader)+1);
  }
  
  sub DeleteHeader {
    my($currentheaders, $newkey) = @_;
    my($oldlen);

    if ($currentheaders =~ /^(\d+)\ {2}$newkey/mi) {
      $oldlen = $1;
      $currentheaders =~ s/^$oldlen\ {2}(?=$newkey).{$oldlen}//ismx;
    }
    return $currentheaders;
  }

  sub ReplaceHeader {
    my($currentheaders, $newkey, $newvalue) = @_;
    my($oldlen, $newlen, $newheader);

    if ($currentheaders =~ /^(\d+)\ {2}$newkey/mi) {
      $oldlen = $1;
      $newheader = "$newkey $newvalue";
      $newlen = sprintf("%03d", length($newheader)+1);

      $currentheaders =~ s/^$oldlen\ {2}(?=$newkey).{$oldlen}/$newlen  $newheader\n/smix;
      return $currentheaders;
    }
    else {
      return AddHeader($currentheaders, $newkey, $newvalue);
    }
  }

  sub AppendHeader {
    my($currentheaders, $newkey, $newvalue, $sep) = @_;
    my($oldlen, $headerlen, $oldvalue);

    # JKF 7/1/2002 Removed /i from next line to stop perl segfaulting
    if ($currentheaders =~ /^(\d+)\ {2}$newkey/m) {
      $oldlen = $1;
      $headerlen = sprintf("%03d", $oldlen + length($newvalue) + length($sep));
      # If you think about it, this next line is somewhat cunning...
      # Make sure you do think about it before you change it ;)
      # Hint: remember that Exim counts \n in lengths.
      # JKF 7/1/2002 Removed /i from next line to stop perl segfaulting
      $currentheaders =~ s/^$oldlen\ (?=\ $newkey)(.{$oldlen})/$headerlen $1$sep$newvalue/smx;
      return $currentheaders;
    }
    else {
      return AddHeader($currentheaders, $newkey, $newvalue);
    }
  }

  sub PrependHeader {
    my($currentheaders, $newkey, $newvalue, $sep) = @_;
    my($len, $oldlen, $headerlen, $oldkey);

    # Get old length, key, and take account of whether key is followed by a space.
    if ($currentheaders =~ /^(\d+)\ {2}($newkey ?)/mi) {
      $oldlen = $1;
      $oldkey = $2;
      $len = $oldlen - length($oldkey);
      
      $headerlen = sprintf("%03d", $oldlen + length($newvalue) + length($sep)
                           + 1);
      # Since we're *pre*pending, the fact that we match the \n
      # doesn't have any effect.
      $currentheaders =~ s/^$oldlen\ \ $oldkey(.{$len})/$headerlen  $oldkey$newvalue$sep$1/ismx;
      return $currentheaders;
    }
    else {
      return AddHeader($currentheaders, $newkey, $newvalue);
    }
  }

  sub TextStartsHeader {
    my($headers, $name, $search) = @_;

    return 1 if $headers =~ /^(\d+)\ {2}$name\s+\Q$search\E/mi;
    return 0;
  }

  sub ConstructHeaders {
    my($headers) = @_;
    my($InHeader,$Special,$newheaders,$ThisHeaderBody,$Line,@Hlines);
    my %Specials = (
		    "Bcc:"	=> "B",
		    "Cc:"	=> "C",
		    "From:"	=> "F",
		    "Message-id"=> "I",
		    "Received:"	=> "P",
		    "Reply-To:"	=> "R",
		    "Sender:"	=> "S",
		    "To:"	=> "T"
		   );
    
    $InHeader = 0;
    $newheaders = "";

    @Hlines = split "\n", $headers;
    
    while ($Line = shift @Hlines) {
      if ($Line =~ /^[ \t]/) {
	$ThisHeaderBody .= "$Line\n";
	# next;
      } else {
	if ($InHeader > 0) {
	  $Special = " ";
	  foreach (keys %Specials) {
	    if ($ThisHeaderBody =~ /^$_/is) {
	      $Special = $Specials{$_};
	      delete $Specials{$_};
	      last;
	    }
	  }
	  $newheaders .= sprintf("%03d$Special $ThisHeaderBody", length($ThisHeaderBody));
	}
	$ThisHeaderBody = "$Line\n";
	$InHeader = 1;
	# next;
      }
    }
    
    if ($InHeader > 0) {
      $Special = " ";
      foreach (keys %Specials) {
	if ($ThisHeaderBody =~ /^$_/is) {
	  $Special = $Specials{$_};
	  delete $Specials{$_};
	  last;
	}
      }
      $newheaders .= sprintf("%03d$Special $ThisHeaderBody", length($ThisHeaderBody));
    }
    
    return $newheaders;
  }

  sub ReadEnvelope {
    my($fh) = @_;
    my $envelope = "";

    while(<$fh>) {
      $envelope .= $_;
    }
    return $envelope;
  }

  sub SplitEnvelope {
    my($envelope) = @_;

    my($headers,$newenvelope);
    my(@envelope) = split "\n", $envelope;
    my $ref = \$newenvelope;

    while(defined ($_ = shift @envelope)) {
      $$ref .= "$_\n";
      $ref = \$headers if $_ eq "";
    }

    return ($newenvelope,$headers);
  }

  sub MergeEnvelope {
    my ($envelope,$headers) = @_;
    return $envelope . $headers;
  }

  sub MergeEnvelopeParts {
    my($fh, $headers) = @_;

    my $envelope = "";

    while(<$fh>) {
      $envelope .= $_;
      last if $_ eq "\n";
    }

    $envelope .= $headers;
    return $envelope;
  }

  sub KickMessage {
    my(@ids) = @_;
    my($idlist);

    # Need to check this with Nick to discover how to attempt delivery of multiple messages
    $idlist = join(" ", @ids);
    $idlist .= " &" if $Config::DeliverInBackground;
    system("$Config::Sendmail2 -Mc $idlist");
  }

__EOD

}

elsif ($Config::MTA eq "sendmail") {

  Log::InfoLog("Configuring mailscanner for sendmail...");

  # These need to be improved
  $DFileRegexp = '^df([-\\w]*)$';
  $HFileRegexp = '^qf([-\\w]*)$';
  $TFileRegexp = '^tf([-\\w]*)$';
  $QueueFileRegexp = '^.([-\\w]*)$';

  $LockType = "flock";

  eval <<'__EOD';

  sub DFileName {
    my($id) = @_;
    return "df$id";
  }

  sub HFileName {
    my($id) = @_;
    return "qf$id";
  }

  sub TFileName {
    my($id) = @_;
    return "tf$id";
  }

  sub BuildMessageCmd {
    my($hfile, $dfile) = @_;
    return "$cat \"$hfile\" \"$dfile\"";
  }

  sub ReadQf {
    my($RQf) = @_;
    my($InHeader, @results, $msginfo, $from, @to, $subject);
    my($ip);
    my($Line);

    $InHeader = 0;
    while(<$RQf>) {
      last if /^\./; # Bat book section 23.9.19
      $Line = $_;
      if ($Line =~ /^R/) {
        my $Rline = $Line;
        chomp $Rline;
        $Rline =~ s/^R([^:]*:)?//;
        push @to, $Rline;
        #print "Recipient is \"" . join(', ', @to) . "\"\n";
      }
      if ($Line =~ /^S/) {
        $from = $Line;
        chomp $from;
        $from =~ s/^S//;
        #print "Sender is \"$from\"\n";
      }
      if ($Line =~ /^\$_/) {
        $ip = $Line;
        chomp $ip;
        $ip =~ /\[(\d+\.\d+\.\d+.\d+)\]/;
        $ip = $1;
        #print "IP is \"$ip\"\n";
        #$ip = '127.0.0.2';
      }
      $InHeader = 1 if $Line =~ /^H/;
      ($InHeader=0),next unless $Line =~ /^[H\t ]/;
      $Line =~ s/^H//;
      # JKF 18/04/2001 Delete ?flags? for 0 or more flags for sendmail 8.11
      $Line =~ s/^\?[^?]*\?//;
      push @results, $Line;
      if ($Line =~ /^Subject:\s+(\S.*)$/i) {
        $subject = $1;
        #print "Subject is \"$subject\"\n";
      }
    }

    # JKF 25/03/2002 Sanitise the list of headers by removing newline
    # characters from header lines. Particularly a problem with the subject.
    my(@stripped);
    foreach $Line (@results) {
      #print STDERR "Line is \"$Line\"\n";
      $Line =~ s/\015/ /g;
      #print STDERR "Line is now \"$Line\"\n";
      push @stripped, $Line;
    }
    $subject =~ s/\015/ /g;
    
    $msginfo = $from . "\0" . join(', ', @to) . "\0" . $subject . "\0" . $ip;
    return (\@stripped, $msginfo);
  }

  sub AddHeader {
    my($currentheaders, $newkey, $newvalue) = @_;
    return $currentheaders . "H$newkey $newvalue\n";
  }

  sub DeleteHeader {
    my($currentheaders, $newkey) = @_;

    # This version currently only delete single-line headers.
    $currentheaders =~ s/^H(\?[^?]*\?)?$newkey\s+.*[^\n]*\n//mi;
    #if ($currentheaders =~ /^H(\?[^?]*\?)?$newkey\s+.*$/m) {
    #  $currentheaders =~ s/^H(\?[^?]*\?)?$newkey\s+([^\n]*)$((\n^\s+[^\n]*$)*)\n//sm;
    #}
    return $currentheaders;
  }

  sub ReplaceHeader {
    my($currentheaders, $newkey, $newvalue) = @_;
    my($newheader, $headervalue);

    $newheader = "$newkey $newvalue";

    if ($currentheaders =~ /^H(\?[^?]*\?)?$newkey\s+.*$/mi) {
      #$currentheaders =~ s/^H(\?[^?]*\?)?$newkey\s+.*$/H$1$newheader/m;
      $currentheaders =~ s/^H(\?[^?]*\?)?$newkey\s+([^\n]*)$((\n^\s+[^\n]*$)*)/H$1$newheader/smi;
      return $currentheaders;
    } else {
      return AddHeader($currentheaders, $newkey, $newvalue);
    }
  }

  sub AppendHeader {
    my($currentheaders, $newkey, $newvalue, $sep) = @_;
    my($oldvalue);

    # JKF 7/1/2002 Removed /i from next line to stop perl segfaulting
    if ($currentheaders =~ /^H(\?[^?]*\?)?$newkey\s+(.*)$/m) {
      $oldvalue = $2;
      #$currentheaders =~ s/^H(\?[^?]*\?)?$newkey\s+.*$/H$1$newkey $oldvalue, $newvalue/m;
      # JKF 7/1/2002 Removed /i from next line to stop perl segfaulting
      $currentheaders =~ s/^H(\?[^?]*\?)?$newkey\s+([^\n]*)$((\n^\s+[^\n]*$)*)/H$1$newkey $2$3$sep$newvalue/sm;
      return $currentheaders;
    } else {
      return AddHeader($currentheaders, $newkey, $newvalue);
    }
  }

  sub PrependHeader {
    my($currentheaders, $newkey, $newvalue, $sep) = @_;
    my($oldvalue);

    if ($currentheaders =~ /^H(\?[^?]*\?)?$newkey\s+(.*)$/mi) {
      $oldvalue = $2;
      $currentheaders =~ s/^H(\?[^?]*\?)?$newkey\s+/H$1$newkey $newvalue$sep/mi;
      return $currentheaders;
    } else {
      return AddHeader($currentheaders, $newkey, $newvalue);
    }
  }

  sub TextStartsHeader {
    my($headers, $name, $search) = @_;

    return 1 if $headers =~ /^H(\?[^?]*\?)?$name\s+\Q$search\E/mi;
    return 0;
  }

  sub ConstructHeaders {
    my($headers) = @_;
    $headers =~ s/^\S/H$&/mg;
    return $headers;
  }

  sub ReadEnvelope {
    my($fh) = @_;
    my $envelope = "";

    while(<$fh>) {
      last if /^\./; # Bat book section 23.9.19
      $envelope .= $_;
    }
    return $envelope;
  }

  sub SplitEnvelope {
    my($envelope) = @_;

    my ($headers,$newenvelope);
    my(@envelope) = split "\n", $envelope;

    my $InHeader = 0;

    while($_ = shift @envelope) {
      last if /^\./; # Bat book section 23.9.19
      if (/^H/) {
        $InHeader = 1;
        $headers .= "$_\n";
        next;
      }
      if (/^\s/ && $InHeader) {
        $headers .= "$_\n";
        next;
      }
      $InHeader = 0;
      $newenvelope .= "$_\n";
    }

    return ($newenvelope,$headers);
  }

  sub MergeEnvelope {
    my ($envelope,$headers) = @_;
    return "$envelope$headers.\n";
  }

  sub MergeEnvelopeParts {
    my($fh, $headers) = @_;

    my $envelope = "";
    my $InHeader = 0;

    while(<$fh>) {
      last if /^\./; # Bat book section 23.9.19
      ($InHeader = 1),next if /^H/;
      next if /^\s/ && $InHeader;
      $InHeader = 0;
      $envelope .= $_;
    }

    $envelope .= $headers;
    $envelope .= ".\n";
    return $envelope;
  }

  sub KickMessage {
    my(@ids) = @_;
    my($idlist);

    $idlist = join(" -qI", @ids);
    $idlist .= " &" if $Config::DeliverInBackground;
    #print STDERR "About to do \"$Config::Sendmail2 -qI$idlist\"\n";
    system("$Config::Sendmail2 -qI$idlist");
  }

__EOD

}

1;
