/*
Magpie - reference librarian for Debian systems
Copyright (C) 2000  Bear Giles <bgiles@coyotesong.com>

This program is free software; you may redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the license, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char rcsid[] = "$Id$";

/*****
This module provides an annotated list of all packages, grouped by
priority and section.
*****/
#include <assert.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include "magpie.h"

extern int mkdir (const char *, mode_t);

/*+
Comparison function for sorting by priority, section, category, 
name, version
+*/
static int cmp_p_s_c (const void *p, const void *q)
{
	struct package_info *pp = *((struct package_info **) p);
	struct package_info *qq = *((struct package_info **) q);
	int r;
	
	assert (pp->name);
	assert (qq->name);

	r = pp->priority - qq->priority;
	if (r)
		return r;

	r = pp->section - qq->section;
	if (r)
		return r;
	
	r = pp->category - qq->category;
	if (r)
		return r;

	r = strcoll (pp->name, qq->name);
	if (r)
		return r;

	r = pp->section - qq->section;
	if (r)
		return r;

	return -strcoll (pp->version, qq->version);
}


/*+
Print the intermediate index files.
+*/
static int priorities_index (FILE *fp0, int type)
{
	int sidx, pidx, cidx;
	char pathname[256];
	FILE *fp;

	if (type != MAGPIE_ALL_PACKAGES)
		return 0;

	mp_title (fp0, 3, "Grouped by Priority");

	/*
	 *	Write the header of the detailed pages, plus the
	 *	complete intermediate file.
	 */
	mp_list_open (fp0);
	for (pidx = 0; pidx < CNT_PRIORITIES; pidx++) {
		if (matrixcnt_p[pidx] == 0)
			continue;

		mp_item_open (fp0);
		mp_url (fp0, "%1$s/index.html.gz", "%1$s", priorities[pidx]);
		fprintf (fp0, " (%d)", matrixcnt_p[pidx]);
		mp_item_close (fp0);

		sprintf (pathname, "%s/index.html", priorities[pidx]);
		fp = fopen (pathname, "w");
		mp_doc_open (fp, "Priority '%s'", priorities[pidx]);
		mp_abstract (fp, "\
This page groups packages by priority, section and then category.");

		mp_list_open (fp);
		for (sidx = 0; sidx < CNT_SECTIONS; sidx++) {
			if (matrixcnt_sp[sidx][pidx] == 0)
				continue;

			mp_item_open (fp);
			mp_url (fp, "%1$s.html.gz", "%1$s", sections[sidx]);
			fprintf (fp, " (%d)", matrixcnt_sp[sidx][pidx]);
			mp_item_close (fp);

			mp_list_open (fp);
			for (cidx = 0; cidx < CNT_CATEGORIES; cidx++) {
				if (matrixcnt_scp[sidx][cidx][pidx] == 0)
					continue;

				mp_item_open (fp);
				mp_url (fp, "%1$s.html.gz#%2$s", "%2$s", 
					sections[sidx], categories[cidx]);
				fprintf (fp, " (%d)", matrixcnt_scp[sidx][cidx][pidx]);
				mp_item_close (fp);
			}
			mp_list_close (fp);
		}
		mp_list_close (fp);

		mp_doc_close (fp);
		fclose (fp);
		gzip (pathname);
	}
	mp_list_close (fp0);

	return 0;
}


/*+
Print the detailed index files.
+*/
static int priorities_init (void)
{
	int sidx, pidx, cidx, i;
	struct package_info *p;
	char pathname[256];
	FILE *fp[CNT_PRIORITIES][CNT_SECTIONS];
	int ss[CNT_PRIORITIES][CNT_SECTIONS];

	memset (fp, 0, sizeof fp);
	memset (ss, 0, sizeof ss);

	qsort (cache, cachecnt, sizeof (cache[0]), cmp_p_s_c);

	/*
	 *	Open the files ...
	 */
	for (pidx = 0; pidx < CNT_PRIORITIES; pidx++) {
		if (matrixcnt_p[pidx] == 0)
			continue;

		mkdir (priorities[pidx], 0755);

		for (sidx = 0; sidx < CNT_SECTIONS; sidx++) {
			if (matrixcnt_sp[sidx][pidx] == 0)
				continue;

			sprintf (pathname, "%s/%s.html", 
				priorities[pidx], sections[sidx]);

			fp[pidx][sidx] = fopen (pathname, "w");
			ss[pidx][sidx] = -1;
			mp_doc_open (fp[pidx][sidx], "Priority '%s', Section '%s'",
				priorities[pidx], sections[sidx]);
		}
	}

	/*
	 *	... write the content ...
	 */
	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		sidx = p->section;
		cidx = p->category;
		pidx = p->priority;

		assert (fp[pidx][sidx]);

		if (ss[pidx][sidx] != p->category) {
			mp_break (fp[pidx][sidx]);
			mp_name (fp[pidx][sidx], categories[p->category]);
			mp_title (fp[pidx][sidx], 3, "Category '%s' (%d)",
				categories[p->category], matrixcnt_scp[sidx][cidx][pidx]);
			ss[pidx][sidx] = p->category;
		}

		mp_package (fp[pidx][sidx], p,
			matrixcnt_sp[sidx][pidx] > 500 ? 2 : 1, 0);
	}

	/*
	 *	... and close them.
	 */
	for (pidx = 0; pidx < CNT_PRIORITIES; pidx++) {
		if (matrixcnt_p[pidx] == 0)
			continue;
		for (sidx = 0; sidx < CNT_SECTIONS; sidx++) {
			if (matrixcnt_sp[sidx][pidx] == 0)
				continue;
			mp_doc_close (fp[pidx][sidx]);
			fclose (fp[pidx][sidx]);

			sprintf (pathname, "%s/%s.html", priorities[pidx], sections[sidx]);
			gzip (pathname);
		}
	}
	return 0;
}



struct magpie_module mod_priorities = { 
	version           : MAGPIE_VERSION,
	description       : "group packages by priorities",
	init              : priorities_init,
	annotated_index   : priorities_index
};
