/*
Magpie - reference librarian for Debian systems
Copyright (C) 2000  Bear Giles <bgiles@coyotesong.com>

This program is free software; you may redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the license, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char rcsid[] = "$Id$";

/*****
The functions in this file is an implementation of the output
abstraction.  It produces HTML.
*****/

#include <assert.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <math.h>
#include <ctype.h>
#include "magpie.h"

/*+
This file contains the manifest constants used by the parser.
They always have the form T_XXX.
+*/
#include "parser.h"

#define OUTPUT_DIR	"html"

extern int mkdir (const char *, int);

extern char timestamp[30];

/*****************************************************************
These routines don't make any special assumptions about the 
document being produced.
*****************************************************************/

/*+
Print a string, quoting special characters.
+*/
void mp_text (
	FILE *fp, 			/*+ output file +*/
	const char *s)		/*+ string to be quoted. +*/
{
	int ch;

	assert (s);
	if (*s == '\0')
		fputs ("<p>\n", fp);
	else {
		while (*s != '\0') {
			switch ((ch = *s++)) {
				case '<': fputs ("&lt;", fp); break;
				case '>': fputs ("&gt;", fp); break;
				case '&': fputs ("&amp;", fp); break;
				case '"': fputs ("&quot;", fp); break;
				default:  fputc (ch, fp);
			}
		}
	}
}


/*+
Print a standard document header.
+*/
void mp_doc_open (
	FILE *fp, 				/*+ output file +*/
	const char *fmt,...)	/*+ var-args parameter list +*/
{
	va_list ap;

	fprintf (fp, "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2//EN\">\n");
	fprintf (fp, "<html>\n");
	fprintf (fp, "<head>\n");
	fprintf (fp, "  <title>");
	va_start (ap, fmt);
	vfprintf (fp, fmt, ap);
	va_end (ap);
	fprintf (fp, "</title>\n");
	fprintf (fp, "</head>\n");
	fprintf (fp, "<body bgcolor=#FFFFFF TEXT=#000000>\n");
	fprintf (fp, "\n");
	fprintf (fp, "<!-- this document was automatically generated -->\n");
	fprintf (fp, "\n");

	fprintf (fp, "<h1>");

/*	fprintf (fp, "<img src=\"./magpie.png\" align=bottom alt=\"\">"); */

	va_start (ap, fmt);
	vfprintf (fp, fmt, ap);
	va_end (ap);
	fprintf (fp, "</h1>\n\n");
}


/*+
Print a standard document footer.
+*/
void mp_doc_close (FILE *fp)	/*+ output file +*/
{
	mp_break (fp);
	fprintf (fp, "Automatically generated by magpie.<br>\n");
	fprintf (fp, "Created at %s\n<br>\n", timestamp);
	fprintf (fp, "\n");
	fprintf (fp, "Author: Bear Giles <a href=\"bgiles@coyotesong.com\">&lt;bgiles@coyotesong.com&gt;</a>\n");
	fprintf (fp, "</body>\n");
	fprintf (fp, "</html>\n");
}


/*+
Print an "abstract" (or "purpose") paragraph.
+*/
void mp_abstract (
	FILE *fp, 				/*+ output file +*/
	const char *fmt,...)	/*+ var-args parameter list +*/
{
	va_list ap;

	va_start (ap, fmt);
	vfprintf (fp, fmt, ap);
	va_end (ap);
	fprintf (fp, "<p>\n");
}


/*+
Indicate an internal label
+*/
void mp_name (
	FILE *fp,					/*+ output file +*/
	const char *name)			/*+ label +*/
{
	fprintf (fp, "<a name=\"");
	mp_text (fp, name);
	fprintf (fp, "\">\n");
}


/*+
Indicate a URL
+*/
void mp_url (
	FILE *fp,					/*+ output file +*/
	const char *fmt1,			/*+ format string for URL */
	const char *fmt2,...)		/*+ format string for label */
{
	va_list ap;

	fprintf (fp, "<a href=\"");

	va_start (ap, fmt2);
	vfprintf (fp, fmt1, ap);
	va_end (ap);

	fprintf (fp, "\">");

	va_start (ap, fmt2);
	vfprintf (fp, fmt2, ap);
	va_end (ap);

	fprintf (fp, "</a>");
}


/*+
Indicate the start of a list.
+*/
void mp_list_open (FILE *fp)	/*+ output file +*/
{
	fprintf (fp, "<ul>\n");
}


/*+
Indicate the end of a list.
+*/
void mp_list_close (FILE *fp)	/*+ output file +*/
{
	fprintf (fp, "</ul>\n");
}


/*+
Indicate the start of an item in a list
+*/
void mp_item_open (FILE *fp)	/*+ output file +*/
{
	fprintf (fp, "  <li>");
}


/*+
Indicate the end of an item in a list.
+*/
void mp_item_close (FILE *fp)	/*+ output file +*/
{
	fprintf (fp, "</li>\n");
}


/*+
Indicate the start of a literal/preformatted region.
+*/
void mp_literal_open (FILE *fp)	/*+ output file +*/
{
	fprintf (fp, "<pre>");
}


/*+
Indicate the end of a literal/preformatted region.
+*/
void mp_literal_close (FILE *fp)	/*+ output file +*/
{
	fprintf (fp, "</pre>\n");
}


/*+
Indicate the start of section title
+*/
void mp_title_open (
	FILE *fp, 				/*+ output file +*/
	int level,				/*+ section level +*/
	const char *fmt,...)	/*+ var-args parameter list +*/
{
	va_list ap;

	assert (0 < level && level <= 6);

	fprintf (fp, "<h%d>", level);
	va_start (ap, fmt);
	vfprintf (fp, fmt, ap);
	va_end (ap);
}


/*+
Indicate the end of section title
+*/
void mp_title_close (
	FILE *fp, 				/*+ output file +*/
	int level)				/*+ section level +*/
{
	assert (0 < level && level <= 6);
	fprintf (fp, "</h%d>\n", level);
}


/*+
Indicate a complete title.  This is the form normally used for section
titles.  The partial forms can be used to embed images or URLs in the
section title.
+*/
void mp_title (
	FILE *fp, 				/*+ output file +*/
	int level,				/*+ section level +*/
	const char *fmt,...)	/*+ var-args parameter list +*/
{
	va_list ap;

	assert (0 < level && level <= 6);

	fprintf (fp, "<h%d>", level);
	va_start (ap, fmt);
	vfprintf (fp, fmt, ap);
	va_end (ap);
	fprintf (fp, "</h%d>\n", level);
}


/*+
Indicate a small space
+*/
void mp_nbsp (FILE *fp)		/*+ output file +*/
{
	fprintf (fp, "&nbsp;\n");
}


/*+
Indicate a page break
+*/
void mp_break (FILE *fp)		/*+ output file +*/
{
	fprintf (fp, "<hr>\n");
}


/*+
Write the string in a monospaced font.
+*/
void mp_tt (
	FILE *fp,					/*+ output file +*/
	const char *str)			/*+ output string +*/
{
	fprintf (fp, "<code>");
	mp_text (fp, str);
	fprintf (fp, "</code>\n");
}


/*****************************************************************
These documents *do* understand the magpie data structures.
*****************************************************************/


/*+
Print a single package name.
If possible, the name will actually be a link to the 'details' or
'provides' entry.
+*/
void mp_item (
	FILE *fp, 						/*+ output file +*/
	const struct package_list *d)	/*+ package information +*/
{
	struct package_info *p;
	char pathname2[256];
	char *format;
	char *version;
	char *summary;

	summary = "";
	version = "";
	sprintf (pathname2, "./provides/%1$1.1s/%1$s.html", d->name);
	p = mp_lookup (d->name);
	if (p) {
		format = "<a href=\"../../details/%1$1.1s/%1$s.html.gz\">%1$s</a> %2$s\n";
		if (p->installed_version) {
			version = p->installed_version;
			format = \
"<a href=\"../../details/%1$1.1s/%1$s.html.gz\">%1$s</a>&nbsp;%2$s&nbsp;\
<font color=\"00CC00\">[%3$s]</font>\n";
		}
	}
	else if (access (pathname2, R_OK) == 0) {
		format = "<a href=\"../../provides/%1$1.1s/%1$s.html\">%1$s</a> %2$s\n";
	}
	else {
		format = "%s %s\n";
	}

	fprintf (fp, format, d->name, (d->restriction ? d->restriction : ""), 
		version);
	
	if (p && p->summary) {
		fputs (p->summary, fp);
	}
}


/*+
Print a list of packages.  Each package is written to a separate
line, unless more than one package is acceptable.  In that case
all alternatives are listed on a single line.

List of packages include 'depends,' 'predepends,'
'suggests,' 'recommends,' and possibly others.
+*/
static void mp_list (
	FILE *fp,						/*+ output file +*/
	const struct package_list *d)	/*+ information on the package +*/

{
	const struct package_list *q;

	fprintf (fp, "<table border=1>\n");
	for ( ; d != NULL; d = d->next) {
		fprintf (fp, "<tr><td>\n");
		mp_item (fp, d);
		for (q = d->down; q; q = q->down) {
			fprintf (fp, "<br>");
			mp_item (fp, q);
		}
		fprintf (fp, "</td></tr>\n");
	}
	fprintf (fp, "</table>\n");
}


/*+
Print the information for a single package in a consistent format.
This procedure 
+*/
void mp_package (
	FILE *fp, 					/*+ the output file +*/
	struct package_info *p,		/*+ the information about this package +*/
	int mode,					/*+ 0 for full, 1 for summary, 2 for concise +*/
	int type)					/*+ additional field item to print +*/
{
	int i;
	char *s;
	char *q1, *q2;

	switch (mode) {
		/* "concise" is the package name, installation, and summary */	
	case 2:
		mp_item_open (fp);
		mp_nbsp (fp);

		switch (type) {
		case T_MD5SUM:
			mp_tt (fp, p->md5sum ? p->md5sum : "(no checksum)");
			break;
		case T_SIZE:
			fprintf (fp, "%1$ld", p->size);
			break;
		case T_INSTALLED_SIZE:
			fprintf (fp, "%1$ldk", p->installed_size);
			break;
		}
		mp_nbsp (fp);

		mp_url (fp, "details/%1$1.1s/%1$s.html.gz", "%1$s", p->name);

		switch (type) {
		case T_INSTALLED_SIZE:
			/* in this case, we're always 'installed' */
			if (p->essential) {
				mp_nbsp (fp);
				mp_text (fp, "(essential)");
			}
			break;

		default:
			if (p->installed) {
				mp_nbsp (fp);
				mp_text (fp, "(installed)");
			}
			else if (p->unpacked) {
				mp_nbsp (fp);
				mp_text (fp, "(unpacked)");
			}
		}

		switch (type) {
		case T_MD5SUM:
			break;
		default:
			mp_text (fp, " ");
			mp_text (fp, p->summary);
		}

		mp_item_close (fp);
		break;

		/* "summary" is the package name, version, installation,
		 * and description */	
	case 1:
		if (p->installed)
			q1 = " (installed)";
		else if (p->unpacked)
			q1 = " (unpacked)";
		else
			q1 = "";
		q2 = "";
		if (p->essential)
			q2 = " (essential)";

		mp_name (fp, p->name);
		mp_title_open (fp, 2, "");
		mp_url (fp, "../details/%1$1.1s/%1$s.html.gz", "%1$s", p->name);
		fprintf (fp, " (%1$s)%2$s%3$s", p->version, q1, q2);
		mp_title_close (fp, 2);

		/* print the description */
		for (i = 0; i < p->desccnt; i++) {
			mp_text (fp, p->description[i]);
			fputs ("\n", fp);
		}
		break;
	
		/* "full" contains *everything* */
	case 0:
		/* print the description */
		for (i = 0; i < p->desccnt; i++) {
			mp_text (fp, p->description[i]);
			fputs ("\n", fp);
		}
		fprintf (fp, "<p>\n");

		fprintf (fp, "<!-- package details -->\n");
		fprintf (fp, "<table border=1>\n");
		fprintf (fp, "  <tr><th>Name</th><td>%s</td></tr>\n", p->name);
		fprintf (fp, "  <tr><th>Summary</th><td>%s</td></tr>\n", p->summary);
		if (p->status[0]) {
			fprintf (fp, "  <tr><th>Status</th><td>%s %s %s</td></tr>\n",
				p->status[0], p->status[1], p->status[2]);
		}
		if (p->version) {
			fprintf (fp, "  <tr><th>Version</th><td>%s</td></tr>\n",
				p->version);
		}
		fprintf (fp, "  <tr><th>Section</th><td>%s/%s</td></tr>\n",
			sections[p->section], categories[p->category]);
		fprintf (fp, "  <tr><th>Priority</th><td>%s %s</td></tr>\n",
			priorities[p->priority], (p->essential ? "(essential)" : ""));
		if (p->filename) {
			fprintf (fp, "  <tr><th>Filename</th><td>%s</td></tr>\n", 
				p->filename);
		}
		fprintf (fp,
			"  <tr><th>Maintainer</th><td><a href=\"../../maintainers.html#");
		mp_text (fp, p->maintainer);
		fprintf (fp, "\">");
		mp_text (fp, p->maintainer);
		fprintf (fp, "</a></td></tr>\n");
		if (p->source) {
			s = p->source->restriction;
			if (s == NULL)
				s = "";
			fprintf (fp, "  <tr><th>Source</th><td>");
			mp_url (fp, "../../source.html#%1$s", "%1$s", p->source->name);
			fprintf (fp, " %s</td></tr>\n", s);
		}

		if (p->predepends) {
			fprintf (fp, "  <tr><th>Pre-depends</th><td>");
			mp_list (fp, p->predepends);
			fprintf (fp, "</td></tr>\n");
		}

		if (p->depends) {
			fprintf (fp, "  <tr><th>Depends</th><td>");
			mp_list (fp, p->depends);
			fprintf (fp, "</td></tr>\n");
		}

		if (p->r_depends) {
			fprintf (fp, "  <tr><th>Required by</th><td>");
			mp_list (fp, p->r_depends);
			fprintf (fp, "</td></tr>\n");
		}

		if (p->recommends) {
			fprintf (fp, "  <tr><th>Recommends</th><td>");
			mp_list (fp, p->recommends);
			fprintf (fp, "</td></tr>\n");
		}
			
		if (p->r_recommends) {
			fprintf (fp, "  <tr><th>Recommended by</th><td>");
			mp_list (fp, p->r_recommends);
			fprintf (fp, "</td></tr>\n");
		}

		if (p->suggests) {
			fprintf (fp, "  <tr><th>Suggests</th><td>");
			mp_list (fp, p->suggests);
			fprintf (fp, "</td></tr>\n");
		}

		if (p->r_suggests) {
			fprintf (fp, "  <tr><th>Suggested by</th><td>");
			mp_list (fp, p->r_suggests);
			fprintf (fp, "</td></tr>\n");
		}

		if (p->provides) {
			fprintf (fp, "  <tr><th>Provides</th><td>");
			mp_list (fp, p->provides);
			fprintf (fp, "</td></tr>\n");
		}
		if (p->replaces) {
			fprintf (fp, "  <tr><th>Replaces</th><td>");
			mp_list (fp, p->replaces);
			fprintf (fp, "</td></tr>\n");
		}
		
		if (p->conflicts) {
			fprintf (fp, "  <tr><th>Conflicts</th><td>");
			mp_list (fp, p->conflicts);
			fprintf (fp, "</td></tr>\n");
		}

		fprintf (fp, "  <tr><th>Size</th><td>%ld</td></tr>\n", p->size);

		if (p->installed_size) {
			fprintf (fp,
				"  <tr><th>Installed Size</th><td>%ldk</td></tr>\n",
				p->installed_size);
		}

		if (p->md5sum) {
			fprintf (fp, "  <tr><th>MD5sum</th><td>");
			mp_tt (fp, p->md5sum);
			fprintf (fp, "</td></tr>\n");
		}

		/* debian package? */
		if (p->section <= 3) {
			fprintf (fp, "  <tr><th>Bugs</th><td>");
			mp_url (fp, 
				"http://cgi.debian.org/cgi-bin/pkgreport.cgi?archive=no&pkg=%1$s",
				"http://cgi.debian.org/cgi-bin/pkgreport.cgi?pkg=%1$s",
				p->name);
			fprintf (fp, "</td></tr>\n");
		}

		if (p->installed) {
			fprintf (fp, "  <tr><th>Files</th><td>");
			mp_url (fp, "file:/var/lib/dpkg/info/%1$s.list",
				"file:/var/lib/dpkg/info/%1$s.list", p->name);
			fprintf (fp, "</td></tr>\n");

			fprintf (fp, "  <tr><th>Docs</th><td>");
			mp_url (fp, "file:/usr/doc/%1$s/",
				"file:/usr/doc/%1$s/", p->name);
			fprintf (fp, "</td></tr>\n");
		}

		fprintf (fp, "</table>\n");

		/*+ list configuration files for package +*/
		if (p->cfcnt) {
			mp_title_open (fp, 2, "Conffiles");
			mp_title_close (fp, 2);
			mp_list_open (fp);
			for (i = 0; i < p->cfcnt; i++) {
				mp_item_open (fp);
				mp_tt (fp, p->conffiles[i]);
				mp_item_close (fp);
			}
			mp_list_close (fp);
			fprintf (fp, "</ul>\n");
		}
	}
}


/*+
Print a histogram of the data.
+*/
void mp_histogram (
	FILE *fp,				/*+ output file +*/
	const int data[], 		/*+ array of counts +*/
	int nelms)				/*+ size of array +*/
{
	int n, x, x2, y;
	int i, j;
	int mx;
	int scale;

	n = x = x2 = y = mx = 0;
	for (i = j = 0; i < nelms; i++) {
		if (data[i]) {
			n++;
			j = i;
			x += data[i];
			x2 += (data[i] * data[i]);
			y += i * data[i];
			if (mx < data[i])
				mx = data[i];
		}
	}

	mp_literal_open (fp);
	fprintf (fp, "<font size=-2>\n");
	while (j >= 0) {
		scale = 1 + (55 * data[j]) / mx;
		if (data[j] == 0) {
			fprintf (fp, "%3d:\n", j);
		}
		else {
			fprintf (fp, "%3d: [%3d]%*.*s\n", j, data[j], scale, scale,
			"************************************************************");
		}
		j--;
	}
	fprintf (fp, "</font>\n");
	fprintf (fp, "\n");
	fprintf (fp, "Totals  : %d, %d\n", x, y);
	fprintf (fp, "Mean    : %7.2f\n", (float) y / (float) x);
#if 0
	if (y > 0) {
		fprintf (fp, "Std dev : %7.2f\n", 
			sqrt (((float) (y * x2 - x * x)) / ((float) (y * (y - 1)))));
	}
#endif
	mp_literal_close (fp);
}


/**********************************************************************/

/*+
+*/
static int html_dump_package (struct package_info *p, const char *filename)
{
	char pathname[256];
	FILE *fp;

	sprintf (pathname, "%1$s/%2$s/%3$1.1s/%3$s.html", 
		OUTPUT_DIR, filename, p->name);

	fp = fopen (pathname, "w");

	mp_doc_open (fp, "Detailed Information for '%s'", p->name);

	mp_package (fp, p, 0, 0);

	mp_doc_close (fp);
	fclose (fp);

	gzip (pathname);
	return 1;
}


/*+
+*/
static void html_dump (const char *filename, const char *title)
{
	char pathname[256], ch;
	struct package_info *p;
	int i;

	sprintf (pathname, "%s/details", OUTPUT_DIR);
	mkdir (pathname, 0755);

	/* weird syntax just in case we're non-ASCII */
	for (ch = 0; ch < 0x7F; ch++) {
		if (islower (ch) || isdigit (ch)) {
			sprintf (pathname, "%s/%s/%c", OUTPUT_DIR, filename, ch);
			mkdir (pathname, 0755);
		}
	}

	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		if (!p->selected)
			continue;

		p->selected = html_dump_package (p, filename);
	}
}


/*+
+*/
static int html_init (void)
{
	int i;

	mkdir (OUTPUT_DIR, 0755);

	/* select all packages */
	for (i = 0; i < cachecnt; i++)
		cache[i]->selected = 1;
	html_dump ("details", NULL);

	return 0;
}


struct magpie_module mod_html = { 
	version           : MAGPIE_VERSION,
	description       : "HTML module",
	init              : html_init
};
