/*
Magpie - reference librarian for Debian systems
Copyright (C) 2000  Bear Giles <bgiles@coyotesong.com>

This program is free software; you may redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the license, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char rcsid[] = "$Id$";

/*****
This module kicks out the reference stacks in debiandoc format.
*****/
#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/wait.h>
#include "magpie.h"

#define	OUTPUT_DIR	"debiandoc"

extern int mkdir (const char *, mode_t);

extern char timestamp[];

static int cnt_s[CNT_SECTIONS];
static int cnt_sc[CNT_SECTIONS][CNT_CATEGORIES];
static int cnt_scp[CNT_SECTIONS][CNT_CATEGORIES][CNT_PRIORITIES];

/*+
Comparison function for sorting by section, category, priority, 
name, version
+*/
static int cmp_s_c_p (const void *p, const void *q)
{
	struct package_info *pp = *((struct package_info **) p);
	struct package_info *qq = *((struct package_info **) q);
	int r;
	
	assert (pp->name);
	assert (qq->name);

	r = pp->section - qq->section;
	if (r)
		return r;
	
	r = pp->category - qq->category;
	if (r)
		return r;

	r = pp->priority - qq->priority;
	if (r)
		return r;

	r = strcoll (pp->name, qq->name);
	if (r)
		return r;

	return -strcoll (pp->version, qq->version);
}


/*+
Write the specified string, quoting any characters that will 
confuse docbook.
+*/
static void debiandoc_quote (FILE *fp, const char *s)
{
	char ch;
	while ((ch = *s++)) {
		switch (ch) {
		case '<': fputs ("&lt;", fp);  break;
		case '>': fputs ("&gt;", fp);  break;
		case '&': fputs ("&amp;", fp); break;
		default:  fputc (ch, fp);
		}
	}
}

/*+
Write a quick reference.
+*/
static void debiandoc_quickref (FILE *fp, const char *name, const char *value)
{
	fputs ("  <sect1>\n", fp);

	fputs ("    <heading>", fp);
	debiandoc_quote (fp, name);
	fputs ("</heading>\n", fp);

	fputs ("    <p>", fp);
	debiandoc_quote (fp, value);
	fputs ("</p>\n", fp);

	fputs ("  </sect1>\n", fp);
}


/*+
Write a quick reference.
+*/
static void debiandoc_quickref_file (FILE *fp, const char *name, const char *value)
{
	fputs ("  <sect1>\n", fp);

	fputs ("    <heading>", fp);
	debiandoc_quote (fp, name);
	fputs ("</heading>\n", fp);

	fputs ("    <p><file>", fp);
	debiandoc_quote (fp, value);
	fputs ("</file></p>\n", fp);

	fputs ("  </sect1>\n", fp);
}

/*+
Write a quick reference.
+*/
static void debiandoc_quickref_int (FILE *fp, const char *name, long value)
{
	fputs ("  <sect1>\n", fp);

	fputs ("    <heading>", fp);
	debiandoc_quote (fp, name);
	fputs ("</heading>\n", fp);

	fprintf (fp, "    <p>%ld</p>\n", value);

	fputs ("  </sect1>\n", fp);
}

/*+
Write a list of packages
+*/
static void debiandoc_list (FILE *fp, 
	const char *title, const struct package_list *d)
{
	struct package_info *p;
	struct package_list *q;

	fputs ("  <sect1>\n", fp);

	fputs ("    <heading>", fp);
	debiandoc_quote (fp, title);
	fputs ("</heading>\n", fp);

	fputs ("    <p>\n", fp);
	fputs ("      <list compact>\n", fp);
	while (d) {
		q = d->down;
		if (q) {
			fputs ("        <item>\n", fp);
			fputs ("          <p>Alternatives:</p>\n", fp);
			fputs ("          <p>\n", fp);
			fputs ("            <list compact>\n", fp);
			while (q) {
				fputs ("              <item>\n", fp);
				fputs ("                <p>\n", fp);
				p = mp_lookup (q->name);
				if (p && p->tag) {
					fprintf (fp, "              <qref id=\"%s\">", p->tag);
					fputs ("<package>", fp);
					debiandoc_quote (fp, q->name);
					fputs ("</package>", fp);
					fputs ("</qref>", fp);
				}
				else {
					fputs ("             <package>", fp);
					debiandoc_quote (fp, q->name);
					fputs ("</package>", fp);
				}
				if (q->restriction) {
					fputc (' ', fp);
					debiandoc_quote (fp, q->restriction);
				}
				fputs ("\n", fp);

				if (p && p->summary) {
					fputs ("                  ", fp);
					debiandoc_quote (fp, p->summary);
					fputs ("\n", fp);
				}
				fputs ("                </p>\n", fp);
				fputs ("              </item>\n", fp);

				q = q->down;
				}
			fputs ("            </list>\n", fp);
			fputs ("          </p>\n", fp);
			fputs ("        </item>\n", fp);
			}
		else {
			fputs ("        <item>\n", fp);
			fputs ("          <p>\n", fp);
			p = mp_lookup (d->name);
			if (p && p->tag) {
				fprintf (fp, "            <qref id=\"%s\">", p->tag);
				fputs ("<package>", fp);
				debiandoc_quote (fp, d->name);
				fputs ("</package>", fp);
				fputs ("</qref>", fp);
				}
			else {
				fputs ("            <package>", fp);
				debiandoc_quote (fp, d->name);
				fputs ("</package>", fp);
			}
			if (d->restriction) {
				fputc (' ', fp);
				debiandoc_quote (fp, d->restriction);
			}
			fputs ("\n", fp);

			if (p && p->summary) {
				fputs ("            ", fp);
				debiandoc_quote (fp, p->summary);
				fputs ("\n", fp);
			}
			fputs ("          </p>\n", fp);
			fputs ("        </item>\n", fp);
		}
		d = d->next;
	}

	fputs ("      </list>\n", fp);
	fputs ("    </p>\n", fp);
	fputs ("  </sect1>\n", fp);
}


/*+
Write the information about the package in DebianDoc format.
+*/
static int debiandoc_dump_package (
	struct package_info *p,
	const char *filename)
{
	FILE *fp;
	char pathname[256];
	char buffer[300];
	int i;

	sprintf (pathname, "%s/%s/%s.sgml", OUTPUT_DIR, filename, p->tag);
	fp = fopen (pathname, "w");

	if (fp == NULL) {
		sprintf (buffer, "fopen (%s)", pathname);
		perror (buffer);
		return 0;
	}

	fprintf (fp, "<sect id=\"%s\">\n", p->tag);
	fputs ("  <heading>", fp);
	fputs ("<package>", fp);
	debiandoc_quote (fp, p->name);
	fputs ("</package> - ", fp);
	debiandoc_quote (fp, p->summary);
	fputs ("</heading>\n", fp);
	fputs ("\n", fp);

	fputs ("  <sect1>\n", fp);
	fputs ("    <heading>Description</heading>\n", fp);
	fputs ("    <p>\n", fp);
	for (i = 0; i < p->desccnt; i++) {
		if (*p->description[i] == '\0')
			fputs ("    </p>\n\n    <p>\n", fp);
		else {
			fputs ("      ", fp);
			debiandoc_quote (fp, p->description[i]);
			fputs ("\n", fp);
		}
	}
	fputs ("    </p>\n", fp);
	fputs ("  </sect1>\n", fp);

	debiandoc_quickref (fp, "Version", p->version);

	debiandoc_quickref (fp, "Maintainer", p->maintainer);

	if (p->predepends)
		debiandoc_list (fp, "Predepends", p->predepends);

	if (p->depends)
		debiandoc_list (fp, "Depends", p->depends);

	if (p->r_depends)
		debiandoc_list (fp, "Required by", p->r_depends);

	if (p->recommends)
		debiandoc_list (fp, "Recommends", p->recommends);

	if (p->r_recommends)
		debiandoc_list (fp, "Recommended by", p->r_recommends);

	if (p->suggests)
		debiandoc_list (fp, "Suggests", p->suggests);

	if (p->r_suggests)
		debiandoc_list (fp, "Suggested by", p->r_suggests);

	if (p->provides)
		debiandoc_list (fp, "Provides", p->provides);

	debiandoc_quickref (fp, "Architecture", architectures[p->architecture]);

	if (p->filename)
		debiandoc_quickref_file (fp, "Filename", p->filename);

	if (p->size)
		debiandoc_quickref_int (fp, "Size", p->size);

	if (p->installed_size)
		debiandoc_quickref_int (fp, "Installed Size", p->installed_size);

	/* we don't show version info */
	if (p->source)
		debiandoc_quickref (fp, "Source", p->source->name);

	/* show Debian bug tracking system */
	if (p->section < 4) {
		fputs ("\n", fp);
		fputs ("  <sect1>\n", fp);

		fputs ("    </p>\n", fp);
		fputs ("      <heading>Bug Information</heading>\n", fp);

		fprintf (fp, 
			"      <httppath>cgi-bin/pkgreport.cgi?archive=no&pkg=%s</httppath>\n",
			p->name);
		fputs ("      ", fp);
		debiandoc_quote (fp, p->name);
		fputs ("\n", fp);
		fputs ("    </p>\n", fp);
		fputs ("  </sect1>\n", fp);
	}

//	also status, md5sum

	fputs ("</sect>", fp);

	fprintf (fp, "\n\
<!-- Keep this comment at the end of the file\n\
Local variables:\n\
mode: sgml\n\
sgml-omittag:nil\n\
sgml-shorttag:t\n\
sgml-namecase-general:t\n\
sgml-general-insert-case:lower\n\
sgml-minimize-attributes:t\n\
sgml-indent-step:2\n\
sgml-indent-data:t\n\
sgml-parent-document:(\"../%s.sgml\", \"book\" \"appendix\")\n\
sgml-exposed-tags:nil\n\
sgml-local-catalogs:nil\n\
sgml-local-ecat-files:nil\n\
End:\n\
-->\n",
		filename);

	fclose (fp);

	return 1;
}


/*+
+*/
static void debiandoc_group_by_section (const char *filename)
{
	int sidx, pidx, cidx, i;
	FILE *fp;
	char pathname[256];
	struct package_info *p;

	qsort (cache, cachecnt, sizeof cache[0], cmp_s_c_p);

	sprintf (pathname, "%s/%s/%s.sgml", OUTPUT_DIR, filename, 
		"group-by-section");
	fp = fopen (pathname, "w");

	fprintf (fp, "<chapt id=\"group-by-section\">\n");
	fprintf (fp, "  <heading>%s</heading>\n", "Grouped by Section");

	fprintf (fp, "  <p>\n");
	fprintf (fp, "    <taglist>\n");
	for (sidx = 0; sidx < CNT_SECTIONS; sidx++) {
		if (cnt_s[sidx] == 0)
			continue;
	
		fprintf (fp, "      <tag>Section '%s' (%d)</tag>\n", 
			sections[sidx], cnt_s[sidx]);
		fprintf (fp, "      <item>\n");
		fprintf (fp, "        <p>\n");
		fprintf (fp, "          <taglist>\n");

		for (cidx = 0; cidx < CNT_CATEGORIES; cidx++) {
			if (cnt_sc[sidx][cidx] == 0)
				continue;

			fprintf (fp, "            <tag>Category '%s' (%d)\n",
				categories[cidx], cnt_sc[sidx][cidx]);

			fprintf (fp, "            <item>\n");
			fprintf (fp, "              <p>\n");
			fprintf (fp, "                <list>\n");
			for (pidx = 0; pidx < CNT_PRIORITIES; pidx++) {
				if (cnt_scp[sidx][cidx][pidx] == 0)
					continue;

				for (i = 0; i < cachecnt; i++) {
					p = cache[i];
					if (!p->selected)
						continue;

					fprintf (fp, "                  <item>\n");
					fprintf (fp, "                    <p>\n");
					fprintf (fp, "                      <qref id=\"%s\">",
						p->tag);
					fprintf (fp, "\n");
					fprintf (fp, "                        ");
					debiandoc_quote (fp, p->summary);
					fprintf (fp, "\n");

					fprintf (fp, "                      </qref>\n");
					fprintf (fp, "                    </p>\n");
					fprintf (fp, "                  </item>\n");
				}
			}
			fprintf (fp, "                </list>\n");
			fprintf (fp, "              </p>\n");
			fprintf (fp, "            </item>\n");
		}

		fprintf (fp, "          </taglist>\n");
		fprintf (fp, "        </p>\n");
		fprintf (fp, "      </item>\n");
	}
	fprintf (fp, "    </taglist>\n");
	fprintf (fp, "  </p>\n");
	fprintf (fp, "</chapt>\n");
	fprintf (fp, "\n");

	fprintf (fp, "\n\
<!-- Keep this comment at the end of the file\n\
Local variables:\n\
mode: sgml\n\
sgml-omittag:nil\n\
sgml-shorttag:t\n\
sgml-namecase-general:t\n\
sgml-general-insert-case:lower\n\
sgml-minimize-attributes:t\n\
sgml-indent-step:2\n\
sgml-indent-data:t\n\
sgml-parent-document:(\"../%s.sgml\", \"book\" \"chapt\")\n\
sgml-exposed-tags:nil\n\
sgml-local-catalogs:nil\n\
sgml-local-ecat-files:nil\n\
End:\n\
-->\n",
		filename);

	fclose (fp);
}


/*+
+*/
static void debiandoc_titlepage (const char *filename, const char *title)
{
	char pathname[256];
	FILE *fp;

	sprintf (pathname, "%s/%s/title-toc.sgml", OUTPUT_DIR, filename);
	fp = fopen (pathname, "w");

	fputs ("<titlepag>\n", fp);
    fprintf (fp, "  <title>%s</title>\n", title);
	fputs ("\n", fp);
    fputs ("  <author>\n", fp);
    fputs ("    <name>Automatically generated by Magpie</name>\n", fp);
    fputs ("  </author>\n", fp);
	fputs ("\n", fp);
    fputs ("  <version><date></version>\n", fp);
	fputs ("\n", fp);
    fputs ("  <abstract>\n", fp);
    fputs ("  </abstract>\n", fp);
	fputs ("\n", fp);
    fputs ("  <copyright>\n", fp);
    fputs ("    <copyrightsummary>\n", fp);
    fputs ("      Copyright &copy; 2000 Bear Giles\n", fp);
    fputs ("    </copyrightsummary>\n", fp);
    fputs ("    <p>\n", fp);
    fputs ("      what is the legal status?\n", fp);
    fputs ("    </p>\n", fp);
    fputs ("  </copyright>\n", fp);
	fputs ("\n", fp);
    fputs ("</titlepag>\n", fp);
	fputs ("\n", fp);
    fputs ("<toc detail=\"chapt\">\n", fp);

	fprintf (fp, "\n\
<!-- Keep this comment at the end of the file\n\
Local variables:\n\
mode: sgml\n\
sgml-omittag:nil\n\
sgml-shorttag:t\n\
sgml-namecase-general:t\n\
sgml-general-insert-case:lower\n\
sgml-minimize-attributes:t\n\
sgml-indent-step:2\n\
sgml-indent-data:t\n\
sgml-parent-document:(\"../%s.sgml\", \"book\" \"titlepag\")\n\
sgml-exposed-tags:nil\n\
sgml-local-catalogs:nil\n\
sgml-local-ecat-files:nil\n\
End:\n\
-->\n",
		filename);

	fclose (fp);
}


/*+
Write the information about all selected packages in DocBook format.
+*/
static int debiandoc_dump (
	const char *filename, 
	const char *title)
{
	FILE *fp;
	int i, n;
	struct package_info *p;
	char pathname[256];
	char buffer[256];
	char *s, *ss;

	memset (cnt_s, 0, sizeof cnt_s);
	memset (cnt_sc, 0, sizeof cnt_sc);
	memset (cnt_scp, 0, sizeof cnt_scp);

	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		if (!p->selected)
			continue;

		cnt_s[p->section]++;
		cnt_sc[p->section][p->category]++;
		cnt_scp[p->section][p->category][p->priority]++;

		ss = buffer;
		for (s = p->name; *s; s++) {
			switch (*s) {
			case '+':
				if (s[1] == '+') {
					s++;
					*ss  = '\0';
					strcat (ss, "-pp-");
					ss += 4;
				}
				else {
					*ss  = '\0';
					strcat (ss, "-n-");
					ss += 3;
				}
				break;
			default:
				*ss++ = *s;
			}
		}
		*ss = '\0';
		p->tag = strdup (buffer);
	}

	sprintf (pathname, "%s/%s", OUTPUT_DIR, filename);
	mkdir (pathname, 0755);

	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		if (p->selected)
			p->selected = debiandoc_dump_package (p, filename);
	}

	debiandoc_titlepage (filename, title);
	debiandoc_group_by_section (filename);

	sprintf (pathname, "%s/%s.sgml", OUTPUT_DIR, filename);
	fp = fopen (pathname, "w");
	fputs ("<!doctype debiandoc public \"-//DebianDoc//DTD DebianDoc//EN\"\n",
		fp);
	fputs ("  [\n", fp);

	fputs ("    <!-- textual data entities -->\n", fp);
	fprintf (fp,
		"    <!entity %2$s                 system \"%1$s/%2$s.sgml\">\n",
		filename, "title-toc");

	fputs ("\n", fp);
	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		if (p->selected) {
			if ((n = 24 - strlen (p->tag)) < 1)
				n = 1;
			fprintf (fp, 
				"    <!entity pkg-%s %.*ssystem \"%s/%s.sgml\">\n",
				p->tag, n, "                                   ", 
				filename, p->tag);
		}
	}
	fputs ("  ]\n", fp);
	fputs (">\n", fp);
	fputs ("\n", fp);
	fputs ("<debiandoc>\n", fp);

	fprintf (fp, "\n");
	fprintf (fp, "  <book id=\"magpie-%s-sgml\">\n", title);
	fprintf (fp, "\n");
	fprintf (fp, "    &title-toc;\n");
	fprintf (fp, "\n");
	fprintf (fp, "    <!-- &ch-group-by-section; -->\n");
	fprintf (fp, "\n");
	fprintf (fp, "    <chapt id=\"Packages\">\n");
	fprintf (fp, "      <heading>Packages</heading>\n");
	fprintf (fp, "\n");

	for (i = 0; i < cachecnt; i++) {
		if (cache[i]->selected)
			fprintf (fp, "      &pkg-%s;\n", cache[i]->tag);
	}

	fprintf (fp, "    </chapt>\n");
	fprintf (fp, "\n");
	fprintf (fp, "  </book>\n");
	fprintf (fp, "</debiandoc>\n");

	fprintf (fp, "\n\
<!-- Keep this comment at the end of the file\n\
Local variables:\n\
mode: sgml\n\
sgml-omittag:nil\n\
sgml-shorttag:t\n\
sgml-namecase-general:t\n\
sgml-general-insert-case:lower\n\
sgml-minimize-attributes:t\n\
sgml-indent-step:2\n\
sgml-indent-data:t\n\
sgml-parent-document:nil\n\
sgml-exposed-tags:nil\n\
sgml-local-catalogs:nil\n\
sgml-local-ecat-files:nil\n\
End:\n\
-->\n");

	fclose (fp);

	for (i = 0; i < cachecnt; i++) {
		if (cache[i]->tag) {
			free (cache[i]->tag);
			cache[i]->tag = 0;
		}
	}

	return 0;
}


static int debiandoc_init (void)
{
	int i;
	struct package_info *p;

	mkdir (OUTPUT_DIR, 0755);

	/* select all packages */
	for (i = 0; i < cachecnt; i++)
		cache[i]->selected = 1;
	debiandoc_dump ("all", "All Packages");

	/* select installed packages */
	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		p->selected = p->installed || p->unpacked;
	}
	debiandoc_dump ("installed", "Installed Packages");

	return 0;
}

struct magpie_module mod_debiandoc = { 
	version           : MAGPIE_VERSION,
	description       : "DebianDoc module",
	init              : debiandoc_init
};
