/*
Magpie - reference librarian for Debian systems
Copyright (C) 2000  Bear Giles <bgiles@coyotesong.com>

This program is free software; you may redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the license, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char rcsid[] = "$Id$";

/*****
This module must always be called first after loading the database.
*****/
#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/wait.h>
#include "magpie.h"

extern int mkdir (const char *, mode_t);

extern int rmrf (const char *);

extern FILE *md5sum (const char *);


/*+
Set up our environment.
+*/
static void setup (void)
{
	char pathname[256], ch;

	mkdir ("details", 0755);
	mkdir ("provides", 0755);

	/* weird syntax just in case we're non-ASCII */
	for (ch = 0; ch < 0x7F; ch++) {
		if (islower (ch) || isdigit (ch)) {
			sprintf (pathname, "details/%c", ch);
			mkdir (pathname, 0755);
			sprintf (pathname, "provides/%c", ch);
			mkdir (pathname, 0755);
		}
	}
}


/*+
Insert a copy of one item into a second list.  The second list is 
sorted by package name, but ignores restrictions.
+*/
static struct package_list *pass2helper (
	struct package_list *d,		/*+ list we're modifying +*/
	struct package_info *q)		/*+ term we're adding +*/
{
	struct package_list *qq, *dd, *dsave;
	int r1, r2;

	if (!d) {
		d = (struct package_list *) malloc (sizeof (struct package_list));
		d->name = strdup (q->name);
		d->restriction = NULL;
		d->down = NULL;
		d->next = NULL;
		return d;
	}

	dsave = d;
	while ((dd = d->next)) {
		r1 = strcoll (d->name, q->name);
		r2 = strcoll (q->name, dd->name);

		if (r1 == 0 || r2 == 0)
			return dsave;

		if (0 < r1 && r2 < 0) {
			qq = (struct package_list *) malloc (sizeof (struct package_list));
			qq->name = strdup (q->name);
			qq->restriction = NULL;
			qq->down = NULL;
			qq->next = dd;
			d->next = qq;
			return dsave;
		}
		d = d->next;
	}

	qq = (struct package_list *) malloc (sizeof (struct package_list));
	qq->name = strdup (q->name);
	qq->restriction = NULL;
	qq->down = NULL;
	qq->next = NULL;
	d->next = qq;
	return dsave;
}

/*+
Second pass: we generate the information for reverse list lookups.
+*/
static void pass2 (void)
{
	char *s;
	struct package_info *p, *pp;
	struct package_list *q, *qq;
	int i;

	for (i = 0; i < cachecnt; i++) {
		p = cache[i];

		/* cache "version" information */
		if (p->installed) {
			p->installed_version = strdup (p->version);
			/* strip Debian version info */
			s = strchr (p->installed_version, '-');	
			if (s)
				*s = '\0';
		}

		/* cache "reverse" information */
		for (q = p->depends; q; q = q->next) {
			pp = mp_lookup (q->name);
			if (pp)
				pp->r_depends = pass2helper (pp->r_depends, p);
			if ((qq = q->down)) {
				while (qq) {
					pp = mp_lookup (qq->name);
					if (pp)
						pp->r_depends = pass2helper (pp->r_depends, p);
					qq = qq->down;
				}
			}
		}

		for (q = p->recommends; q; q = q->next) {
			pp = mp_lookup (q->name);
			if (pp)
				pp->r_recommends = pass2helper (pp->r_recommends, p);
			if ((qq = q->down)) {
				while (qq) {
					pp = mp_lookup (qq->name);
					if (pp)
						pp->r_recommends = pass2helper (pp->r_recommends, p);
					qq = qq->down;
				}
			}
		}

		for (q = p->suggests; q; q = q->next) {
			pp = mp_lookup (q->name);
			if (pp)
				pp->r_suggests = pass2helper (pp->r_suggests, p);
			if ((qq = q->down)) {
				while (qq) {
					pp = mp_lookup (qq->name);
					if (pp)
						pp->r_suggests = pass2helper (pp->r_suggests, p);
					qq = qq->down;
				}
			}
		}
	}
}


/*+
third pass: generate the real "provides" files
+*/
static void pass3 (void)
{
	FILE *fp, *fp1, *in;
	char pathname[256];
	char tmpname[256];
	char buffer[256];
	struct package_info *p;
	struct package_list *q;
	char name[30], *s, *pkgname;
	int i;

	strcpy (tmpname, "provides.XXXXXX");
	mkstemp (tmpname);
	fp = fopen (tmpname, "w");
	for (i = 0; i < cachecnt; i++) {
		p = cache[i];

		/* cache "provides" information */
		for (q = p->provides; q; q = q->next) {
			fprintf (fp, "%s:%s:", q->name, p->name);
			mp_text (fp, p->summary);
			fprintf (fp, "\n");
		}
	}
	fclose (fp);

	/*
	 *	In this section:
	 *	 fp  = global index
	 *	 fp1 = local index
	 */
	sprintf (pathname, "./provides/index.html");
	fp = fopen (pathname, "w");
	mp_doc_open (fp, "List of 'provides' pseudotargets");

	fp1 = NULL;

	sprintf (pathname, "/usr/bin/sort %s", tmpname);
	in = popen (pathname, "r");
	name[0] = '\0';

	mp_list_open (fp);
	while (fgets (buffer, sizeof (buffer), in) != NULL) {
		s = strchr (buffer, ':');
		assert (s != NULL);
		*s++ = '\0';
		pkgname = s;
		s = strchr (pkgname, ':');
		assert (s != NULL);
		*s++ = '\0';

		if (strcmp (name, buffer) != 0) {
			if (name[0]) {
				mp_list_close (fp);

				mp_list_close (fp1);
				mp_doc_close (fp1);
				fclose (fp1);
				gzip (pathname);
			}
			strncpy (name, buffer, sizeof name);
			sprintf (pathname, "./provides/%1$1.1s/%1$s.html", name);
			fp1 = fopen (pathname, "w");
			mp_doc_open (fp1, "Packages that provide '%s'", name);
			mp_list_open (fp1);

			mp_item_open (fp);
			mp_url (fp, "%1$1.1s/%1$s.html.gz", "%1$s", name);
			mp_item_close (fp);
			mp_list_open (fp);
		}

		/* this is the list of packages per target */
		mp_item_open (fp1);
		mp_url (fp1, "../../details/%1$1.1s/%1$s.html.gz", "%1$s", pkgname);
		mp_nbsp (fp1);
		mp_text (fp1, s);
		mp_item_close (fp1);

		/* this is the global list of packages */
		mp_item_open (fp);
		mp_url (fp, "../details/%1$1.1s/%1$s.html.gz", "%1$s", pkgname);
		mp_nbsp (fp);
		mp_text (fp, s);
		mp_item_close (fp);
	}
	mp_list_close (fp);
	mp_doc_close (fp);
	fclose (fp);

	if (fp1 != NULL) {
		mp_list_close (fp1);
		mp_doc_close (fp1);
		fclose (fp1);
		gzip (pathname);
	}

	fclose (in);
	unlink (tmpname);

	gzip ("provides/index.html");
}

/*+
fifth pass: check the md5sums file
+*/
#if 0
static void pass5 (void)
{
	int i;
	FILE *fp, *fp1, *fp2;
	char *s;
	struct package_info *p;
	char pathname[256];
	char tmpname[256];
	char buffer[256];
	int n;

	fp2 = fopen ("imd5sums.html", "w");
	mp_doc_open (fp2, "Installed Packages, md5sum check");
	mp_abstract (fp2, "\
This page provides a list of all packages which contain a mismatch between\n\
the md5sums file and the actual file.  A file could appear here because you\n\
have edited it, because the package maintainer made an error, or because\n\
your system is under attack.  This information should not be considered a \n\
replacement for tripwire(8).");
	mp_literal_open (fp2);

	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		if (!p->installed && !p->unpacked)
			continue;

		sprintf (tmpname, "/var/lib/dpkg/info/%s.md5sums", p->name);
		if (access (tmpname, R_OK) == -1)
			continue;

		sprintf (pathname, "./details/%1$1.1s/%1$s.html", p->name);
		fp = fopen (pathname, "a");

		mp_break (fp);
		mp_title (fp, 2, "Results of md5sum check");

		mp_url (fp, "../../imd5sums.html.gz", "Global results");

		mp_literal_open (fp);

		fprintf (fp, "# md5sums -c %s\n", pathname);
		fp1 = md5sum (tmpname);
		while (fgets (buffer, sizeof buffer, fp1) != NULL) {
			if (buffer[0] == '\n')
				continue;

			fputs (buffer, fp);
			n = strlen (buffer);
			buffer[--n] = '\0';

			/* on summary page, we skip past the program name to 
			 * save space for package name. */
			s = strchr (buffer, ' ');
			if (s) {
				fprintf (fp2, "%s ", s);
				mp_url (fp2, "details/%1$1.1s/%1$s.html.gz", 
					"(%1$s)", p->name);
			}
		}
		fclose (fp1);

		mp_literal_close (fp);
		fclose (fp);
	}
	mp_literal_close (fp2);
	mp_doc_close (fp2);
	fclose (fp2);
	gzip ("imd5sums.html");
}
#endif


/*+
Print the complete package information in individual files.
This procedure should be called first, after loading the package
information, since the other packages depend on the of the data
it produces.
+*/
static int core_init (void)
{
//	setup ();

	pass2 ();
//	pass3 ();
//	pass5 ();

	return 0;
}


/*+
Print links to a couple files we produce as a secondary result of
the core analysis.
+*/
static int core_misc (FILE *fp)
{
#if 0
	mp_item_open (fp);
	mp_url (fp, "provides/index.html.gz", "All packages: 'provides' targets");
	mp_item_close (fp);

	/* links to results of pass 5 */
	mp_item_open (fp);
	mp_url (fp, "imd5sums.html.gz", 
		"Installed packages: results of md5sum check");
	mp_item_close (fp);
#endif

	return 0;
}


struct magpie_module mod_core = { 
	version           : MAGPIE_VERSION,
	description       : "core module",
	init              : core_init,
	miscellaneous     : core_misc
};
