/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <errno.h>
#include <pthread.h>
#include <sys/sysctl.h>
#include <sys/iostat.h>
#include <string.h>

#include "libsysactivity.h"
#include "utils.h"

pthread_key_t buffer_key;
pthread_key_t buffer_size_key;
int mib[] = { CTL_HW, HW_IOSTATS, sizeof(struct io_sysctl) };

static void create_keys();

int sa_open_data_storage() {
	pthread_once_t key_once = PTHREAD_ONCE_INIT;
	if (pthread_once(&key_once, create_keys))
		return ENOSYS;

	return 0;
}

int sa_count_data_storages(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	size_t size;
	if (sysctl(mib, 3, NULL, &size, NULL, 0) < 0)
		return ENOSYS;

	*number = size / sizeof(struct io_sysctl);
	return 0;
}

int sa_get_data_storage(char* name, struct sa_data_storage* dst) {
	if (name == NULL || dst == NULL)
		return EINVAL;

	size_t size;
	if (sysctl(mib, 3, NULL, &size, NULL, 0) < 0)
		return ENOSYS;

	struct io_sysctl* buffer = pthread_getspecific(buffer_key);

	size_t buffer_size = (size_t) pthread_getspecific(buffer_size_key);
	if (buffer_size < size) {
		buffer = realloc(buffer, size);
		if (pthread_setspecific(buffer_key, buffer))
			return ENOSYS;
		if (buffer == NULL)
			return ENOMEM;
		buffer_size = size;
		if (pthread_setspecific(buffer_size_key, (void *) buffer_size))
			return ENOSYS;
	}

	if (sysctl(mib, 3, buffer, &size, NULL, 0) < 0)
		return ENOSYS;

	int i, total;
	total = size / sizeof(struct io_sysctl);
	for (i = 0; i < total; i++) {
		if (strncmp(buffer[i].name, name, IOSTATNAMELEN) != 0)
			continue;

		strlcpy(dst->name, buffer[i].name, IOSTATNAMELEN);
		dst->reads = buffer[i].rxfer;
		dst->writes = buffer[i].wxfer;
		dst->bytes_read = buffer[i].rbytes;
		dst->bytes_written = buffer[i].wbytes;

		return 0;
	}

	return ENODEV;
}

int sa_get_data_storages(struct sa_data_storage* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	*written = 0;
	struct io_sysctl* buffer = pthread_getspecific(buffer_key);
	size_t size = dst_size * sizeof(struct io_sysctl);

	size_t buffer_size = (size_t) pthread_getspecific(buffer_size_key);
	if (buffer_size < size) {
		buffer = realloc(buffer, size);
		if (pthread_setspecific(buffer_key, buffer))
			return ENOSYS;
		if (buffer == NULL)
			return ENOMEM;
		buffer_size = size;
		if (pthread_setspecific(buffer_size_key, (void *) buffer_size))
			return ENOSYS;
	}

	if (sysctl(mib, 3, buffer, &size, NULL, 0) < 0)
		return ENOSYS;

	int i;
	for (i = 0; i < dst_size; i++) {
		if (i == size / sizeof(struct io_sysctl))
			return ENODEV;

		strlcpy(dst[i].name, buffer[i].name, IOSTATNAMELEN);
		dst[i].reads = buffer[i].rxfer;
		dst[i].writes = buffer[i].wxfer;
		dst[i].bytes_read = buffer[i].rbytes;
		dst[i].bytes_written = buffer[i].wbytes;
		(*written)++;
	}

	return 0;
}

static void create_keys() {
	static short keys_created = 0;
	if (keys_created) {
		return;
	}

	pthread_key_create(&buffer_key, free_keys);
	pthread_key_create(&buffer_size_key, NULL);

	keys_created = 1;
}
