/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include "support.h"
#include <libsyncml/sml_command_internals.h>
#include <libsyncml/sml_elements_internals.h>

static SmlAssembler *start_assembler(SmlSession *session, unsigned int limit)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;

	SmlAssembler *assm = smlAssemblerNew(SML_MIMETYPE_XML, limit, &error);
	fail_unless(assm != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	if (!smlAssemblerStart(assm, session, &error)) {
		fail_unless(error != NULL, NULL);
		smlAssemblerFree(assm);
		return NULL;
	}
	return assm;
}

START_TEST (check_assembler_message_empty)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	smlLocationUnref(loc);

	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_message_only_header)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	smlLocationUnref(loc);
	
	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_message_status)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerReserveStatus(assm, 1, 1, 1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlStatus *status = smlStatusNew(SML_NO_ERROR, 1, 1, loc, loc, SML_COMMAND_TYPE_SYNC, &error);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	smlLocationUnref(loc);
	
	fail_unless(smlAssemblerAddStatus(assm, status, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlStatusUnref(status);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	g_free(data);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_message_cmd)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = smlCommandNewAlert(SML_ALERT_TWO_WAY, loc, loc, "next", "last", NULL, &error);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd->cmdID = 1;
	
	smlLocationUnref(loc);
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd);
	
	fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
		
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	g_free(data);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_message_subcmd)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd1 = smlCommandNewSync(loc, loc, 0, &error);
	fail_unless(cmd1 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd1->cmdID = 1;
	
	smlLocationUnref(loc);
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "uid", "data", 5, SML_ELEMENT_TEXT_VCARD, &error);
	fail_unless(cmd2 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd2->cmdID = 2;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd1, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd2);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd1);
	
	fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	g_free(data);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_message_several_subcmd)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd1 = smlCommandNewSync(loc, loc, 0, &error);
	fail_unless(cmd1 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd1->cmdID = 1;
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "uid", "data", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 2;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd1, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd2);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "uid2", "data2", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 3;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd1, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd2);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd1);
	
	fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	cmd1 = smlCommandNewSync(loc, loc, 0, &error);
	fail_unless(cmd1 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd1->cmdID = 4;
	
	smlLocationUnref(loc);
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "uid", "data", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 5;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd1, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd2);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "uid2", "data2", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 6;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd1, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd2);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd1);
	
	fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	g_free(data);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_message_flush_open)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd1 = smlCommandNewSync(loc, loc, 0, &error);
	fail_unless(cmd1 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd1->cmdID = 1;
	
	smlLocationUnref(loc);
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "uid", "data", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 2;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd1, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd2);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, FALSE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	g_free(data);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "uid2", "data2", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 3;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd1, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd2);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd1);
	
	fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	g_free(data);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

/** - start sync1
 * - add change to sync1
 * - start sync2
 * - add change to sync2
 * - end sync1
 * - add change to sync1
 * - add change to sync2
 * - end root (sync1)
 * - add change to sync2
 * - end root (sync2)
 */
START_TEST (check_assembler_out_of_order)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd1 = smlCommandNewSync(loc, loc, 0, &error);
	fail_unless(cmd1 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd1->cmdID = 1;
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "change1sync1", "data", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 2;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd1, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd2);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd3 = smlCommandNewSync(loc, loc, 0, &error);
	fail_unless(cmd3 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd3->cmdID = 4;
	
	smlLocationUnref(loc);
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd3, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "change1sync2", "data", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 5;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd3, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd2);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd3, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "change2sync1", "data2", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 3;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd1, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd2);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "change2sync2", "data2", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 6;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd3, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd2);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd3, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "change3sync2", "data2", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 7;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd3, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd2);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd3, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd3);
	smlCommandUnref(cmd1);
	
	fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	g_free(data);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_limit1)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	smlLocationUnref(loc);
	
	SmlAssembler *assm = start_assembler(session, 1);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(!smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_limit2)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	smlLocationUnref(loc);
	
	SmlAssembler *assm = start_assembler(session, 100);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(!smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_limit3)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 220);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerReserveStatus(assm, 1, 1, 1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlStatus *status = smlStatusNew(SML_NO_ERROR, 1, 1, loc, loc, SML_COMMAND_TYPE_SYNC, &error);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	smlLocationUnref(loc);
	
	fail_unless(smlAssemblerAddStatus(assm, status, &error) == SML_ASSEMBLER_RESULT_MISMATCH, NULL);
	fail_unless(error == NULL, NULL);
	
	smlStatusUnref(status);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_limit4)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 380);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerReserveStatus(assm, 1, 1, 1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlStatus *status = smlStatusNew(SML_NO_ERROR, 1, 1, loc, loc, SML_COMMAND_TYPE_SYNC, &error);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	smlLocationUnref(loc);
	
	fail_unless(smlAssemblerAddStatus(assm, status, &error) == SML_ASSEMBLER_RESULT_OK, NULL);
	fail_unless(error == NULL, NULL);
	
	smlStatusUnref(status);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	g_free(data);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_limit5)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 500);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerReserveStatus(assm, 1, 1, 1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlStatus *status = smlStatusNew(SML_NO_ERROR, 1, 1, loc, loc, SML_COMMAND_TYPE_SYNC, &error);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerAddStatus(assm, status, &error) == SML_ASSEMBLER_RESULT_OK, NULL);
	fail_unless(error == NULL, NULL);
	
	smlStatusUnref(status);
	
	SmlCommand *cmd = smlCommandNewAlert(SML_ALERT_TWO_WAY, loc, loc, "next", "last", NULL, &error);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd->cmdID = 1;
	
	smlLocationUnref(loc);
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd, &error) == SML_ASSEMBLER_RESULT_MISMATCH, NULL);
	fail_unless(error == NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, FALSE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	g_free(data);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd, &error) == SML_ASSEMBLER_RESULT_OK, NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd);
	
	fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	g_free(data);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_devinf)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerReserveStatus(assm, 0, 1, 1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
		
	SmlStatus *status = smlStatusNew(SML_NO_ERROR, 0, 1, loc, loc, SML_COMMAND_TYPE_HEADER, &error);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	smlLocationUnref(loc);
	
	fail_unless(smlAssemblerAddStatus(assm, status, &error) == SML_ASSEMBLER_RESULT_OK, NULL);
	fail_unless(error == NULL, NULL);
	
	smlStatusUnref(status);
	
	SmlDevInf *devinf = smlDevInfNew("mydevice", SML_DEVINF_DEVTYPE_WORKSTATION, &error);
	fail_unless(devinf != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = smlDevInfNewPut(devinf, SML_DEVINF_VERSION_11, &error);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd->cmdID = 1;
	
	smlDevInfUnref(devinf);
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd, &error) == SML_ASSEMBLER_RESULT_OK, NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd);
	
	fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, FALSE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	g_free(data);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_order)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerReserveStatus(assm, 0, 1, 1, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlAssemblerReserveStatus(assm, 1, 1, 2, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlAssemblerReserveStatus(assm, 2, 1, 3, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlStatus *status = smlStatusNew(SML_NO_ERROR, 2, 1, loc, loc, SML_COMMAND_TYPE_PUT, &error);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerAddStatus(assm, status, &error) == SML_ASSEMBLER_RESULT_OK, NULL);
	fail_unless(error == NULL, NULL);
	
	smlStatusUnref(status);
	
	status = smlStatusNew(SML_NO_ERROR, 1, 1, loc, loc, SML_COMMAND_TYPE_SYNC, &error);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerAddStatus(assm, status, &error) == SML_ASSEMBLER_RESULT_OK, NULL);
	fail_unless(error == NULL, NULL);
	
	smlStatusUnref(status);
	
	status = smlStatusNew(SML_NO_ERROR, 0, 1, loc, loc, SML_COMMAND_TYPE_HEADER, &error);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	smlLocationUnref(loc);
	
	fail_unless(smlAssemblerAddStatus(assm, status, &error) == SML_ASSEMBLER_RESULT_OK, NULL);
	fail_unless(error == NULL, NULL);
	
	smlStatusUnref(status);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, FALSE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
	
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	smlHeaderFree(header);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(status->type == SML_COMMAND_TYPE_HEADER, NULL);
	smlStatusUnref(status);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(status->type == SML_COMMAND_TYPE_SYNC, NULL);
	smlStatusUnref(status);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(status->type == SML_COMMAND_TYPE_PUT, NULL);
	smlStatusUnref(status);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlBool final = TRUE;
	SmlBool end = TRUE;
	fail_unless(smlParserEnd(parser, &final, &end, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(final == FALSE, NULL);
	fail_unless(end == FALSE, NULL);
	
	smlParserFree(parser);
		
	g_free(data);
}
END_TEST

/* - Reserve 3 statuses
 * - Add one status
 * - Should report missing statuses
 * - Run(final) -> error since statuses are missing
 * - Run(not final) & Flush -> data should have one status
 * - Should report missing statuses
 * - Add 2 remaining statuses
 * - Should not report missing statuses
 * - Run & Flush -> data should have the 2 remaining statuses
 */
START_TEST (check_assembler_reuse)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerIsStatusMissing(assm) == FALSE);
	
	fail_unless(smlAssemblerReserveStatus(assm, 0, 1, 1, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlAssemblerReserveStatus(assm, 1, 1, 2, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlAssemblerReserveStatus(assm, 2, 1, 3, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	
	SmlStatus *status = smlStatusNew(SML_NO_ERROR, 0, 1, loc, loc, SML_COMMAND_TYPE_HEADER, &error);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	/* Must fail since nothing has been added at all */
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, FALSE, &error), NULL);
	fail_unless(error != NULL, NULL);
	fail_unless(data == NULL, NULL);
	fail_unless(size == 0, NULL);
	smlErrorDeref(&error);
	
	fail_unless(smlAssemblerAddStatus(assm, status, &error) == SML_ASSEMBLER_RESULT_OK, NULL);
	fail_unless(error == NULL, NULL);
	
	smlStatusUnref(status);
	
	fail_unless(smlAssemblerIsStatusMissing(assm) == TRUE);
	
	/* Must fail since not all statuses have been added */
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	fail_unless(data == NULL, NULL);
	fail_unless(size == 0, NULL);
	smlErrorDeref(&error);
	
	/* Must work */
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, FALSE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	fail_unless(smlAssemblerFlush(assm) == 3, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerIsStatusMissing(assm) == TRUE);
	
	SmlParser *parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	smlHeaderFree(header);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(status->type == SML_COMMAND_TYPE_HEADER, NULL);
	smlStatusUnref(status);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlBool final = TRUE;
	SmlBool end = TRUE;
	fail_unless(smlParserEnd(parser, &final, &end, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(final == FALSE, NULL);
	fail_unless(end == FALSE, NULL);
	
	smlParserFree(parser);
	g_free(data);
	
	status = smlStatusNew(SML_NO_ERROR, 1, 1, loc, loc, SML_COMMAND_TYPE_SYNC, &error);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerAddStatus(assm, status, &error) == SML_ASSEMBLER_RESULT_OK, NULL);
	fail_unless(error == NULL, NULL);
	smlStatusUnref(status);
	
	status = smlStatusNew(SML_NO_ERROR, 2, 1, loc, loc, SML_COMMAND_TYPE_PUT, &error);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	smlLocationUnref(loc);
	
	fail_unless(smlAssemblerAddStatus(assm, status, &error) == SML_ASSEMBLER_RESULT_OK, NULL);
	fail_unless(error == NULL, NULL);
	smlStatusUnref(status);
	
	fail_unless(smlAssemblerIsStatusMissing(assm) == FALSE);
	
	data = NULL;
	size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
	
	parser = start_parser(data);
	fail_unless(parser != NULL, NULL);
	
	header = NULL;
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	smlHeaderFree(header);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(status->type == SML_COMMAND_TYPE_SYNC, NULL);
	smlStatusUnref(status);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(status->type == SML_COMMAND_TYPE_PUT, NULL);
	smlStatusUnref(status);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	final = TRUE;
	end = TRUE;
	fail_unless(smlParserEnd(parser, &final, &end, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(final == TRUE, NULL);
	fail_unless(end == TRUE, NULL);
	
	smlParserFree(parser);
	g_free(data);
	
}
END_TEST

START_TEST (check_assembler_leak)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerReserveStatus(assm, 0, 1, 1, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlAssemblerReserveStatus(assm, 1, 1, 2, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlAssemblerReserveStatus(assm, 2, 1, 3, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	/* Must fail since nothing has been added at all */
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, FALSE, &error), NULL);
	fail_unless(error != NULL, NULL);
	fail_unless(data == NULL, NULL);
	fail_unless(size == 0, NULL);
	smlErrorDeref(&error);
	
	fail_unless(smlAssemblerFlush(assm) == 4, NULL);
	fail_unless(error == NULL, NULL);
	
	smlAssemblerFree(assm);
	
	assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlStatus *status = smlStatusNew(SML_NO_ERROR, 0, 1, loc, loc, SML_COMMAND_TYPE_HEADER, &error);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerAddStatus(assm, status, &error) != SML_ASSEMBLER_RESULT_OK, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlStatusUnref(status);
	
	/* Must fail since not all statuses have been added */
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	fail_unless(data == NULL, NULL);
	fail_unless(size == 0, NULL);
	smlErrorDeref(&error);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	smlAssemblerFree(assm);
	
	assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "change1sync1", "data", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 2;
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd2);
	
	/* Must fail since not all statuses have been added */
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error== NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	g_free(data);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	smlLocationUnref(loc);
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_stress)
{
	int num_cmd = 1000;
	int num_status = 1000;
	
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	int i = 0;
	for (i = 0; i < num_status; i++) {
		fail_unless(smlAssemblerReserveStatus(assm, i + 1, 1, 1, &error), NULL);
		fail_unless(error == NULL, NULL);
		
		SmlStatus *status = smlStatusNew(SML_NO_ERROR, i + 1, 1, loc, loc, SML_COMMAND_TYPE_SYNC, &error);
		fail_unless(status != NULL, NULL);
		fail_unless(error == NULL, NULL);
	
	
		fail_unless(smlAssemblerAddStatus(assm, status, &error), NULL);
		fail_unless(error == NULL, NULL);
		
		smlStatusUnref(status);
	}
	
	for (i = 0; i < num_cmd; i++) {
		SmlCommand *cmd1 = smlCommandNewSync(loc, loc, 0, &error);
		fail_unless(cmd1 != NULL, NULL);
		fail_unless(error == NULL, NULL);
		cmd1->cmdID = i + 1 + num_status;
	
		fail_unless(smlAssemblerStartCommand(assm, NULL, cmd1, &error), NULL);
		fail_unless(error == NULL, NULL);
	
		smlCommandUnref(cmd1);
		
		fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
		fail_unless(error == NULL, NULL);
	}
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	g_free(data);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlLocationUnref(loc);
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_stress_limit)
{
	int num_cmd = 1000;
	int num_status = 1000;
	
	int limit = 50000;
	
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, limit);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	int i = 0;
	for (i = 0; i < num_status; i++) {
		fail_unless(smlAssemblerReserveStatus(assm, i + 1, 1, 1, &error), NULL);
		fail_unless(error == NULL, NULL);
		
		SmlStatus *status = smlStatusNew(SML_NO_ERROR, i + 1, 1, loc, loc, SML_COMMAND_TYPE_SYNC, &error);
		fail_unless(status != NULL, NULL);
		fail_unless(error == NULL, NULL);
	
		SmlAssemblerResult res = smlAssemblerAddStatus(assm, status, &error);
		if (res == SML_ASSEMBLER_RESULT_MISMATCH) {
			char *data = NULL;
			unsigned int size = 0;
			fail_unless(smlAssemblerRun(assm, &data, &size, NULL, FALSE, &error), NULL);
			fail_unless(error == NULL, NULL);
			fail_unless(data != NULL, NULL);
			fail_unless(size != 0, NULL);
			
			g_free(data);
			
			smlAssemblerFlush(assm);
			
			fail_unless(smlAssemblerAddStatus(assm, status, &error) == SML_ASSEMBLER_RESULT_OK);
			fail_unless(error == NULL, NULL);
		} else if (res != SML_ASSEMBLER_RESULT_OK)
			fail();
		
		smlStatusUnref(status);
	}
	
	for (i = 0; i < num_cmd; i++) {
		SmlCommand *cmd1 = smlCommandNewSync(loc, loc, 0, &error);
		fail_unless(cmd1 != NULL, NULL);
		fail_unless(error == NULL, NULL);
		cmd1->cmdID = i + 1 + num_status;
	
		SmlAssemblerResult res = smlAssemblerStartCommand(assm, NULL, cmd1, &error);
		
		if (res == SML_ASSEMBLER_RESULT_MISMATCH) {
			char *data = NULL;
			unsigned int size = 0;
			fail_unless(smlAssemblerRun(assm, &data, &size, NULL, FALSE, &error), NULL);
			fail_unless(error == NULL, NULL);
			fail_unless(data != NULL, NULL);
			fail_unless(size != 0, NULL);
			
			g_free(data);
			
			fail_unless(smlAssemblerFlush(assm) == 1, NULL);
			fail_unless(error == NULL, NULL);
			
			fail_unless(smlAssemblerStartCommand(assm, NULL, cmd1, &error), NULL);
			fail_unless(error == NULL, NULL);
		} else if (res != SML_ASSEMBLER_RESULT_OK)
			fail();
	
		smlCommandUnref(cmd1);
		
		fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
		fail_unless(error == NULL, NULL);
	}
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	g_free(data);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlLocationUnref(loc);
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_stress_large_limit)
{
	int num_cmd = 1000;
	int num_status = 1000;
	
	int limit = 500000000;
	
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, limit);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	int i = 0;
	for (i = 0; i < num_status; i++) {
		fail_unless(smlAssemblerReserveStatus(assm, i + 1, 1, 1, &error), NULL);
		fail_unless(error == NULL, NULL);
		
		SmlStatus *status = smlStatusNew(SML_NO_ERROR, i + 1, 1, loc, loc, SML_COMMAND_TYPE_SYNC, &error);
		fail_unless(status != NULL, NULL);
		fail_unless(error == NULL, NULL);
		
		fail_unless(smlAssemblerAddStatus(assm, status, &error) == SML_ASSEMBLER_RESULT_OK);
		fail_unless(error == NULL, NULL);
		
		smlStatusUnref(status);
	}
	
	for (i = 0; i < num_cmd; i++) {
		SmlCommand *cmd1 = smlCommandNewSync(loc, loc, 0, &error);
		fail_unless(cmd1 != NULL, NULL);
		fail_unless(error == NULL, NULL);
		cmd1->cmdID = i + 1 + num_status;
	
		fail_unless(smlAssemblerStartCommand(assm, NULL, cmd1, &error) == SML_ASSEMBLER_RESULT_OK, NULL);
		fail_unless(error == NULL, NULL);
	
		smlCommandUnref(cmd1);
		
		fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
		fail_unless(error == NULL, NULL);
	}
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	g_free(data);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlLocationUnref(loc);
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_large_obj)
{
	int limit = 500000;
	
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, limit);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	
	SmlCommand *cmd1 = smlCommandNewSync(loc, loc, 0, &error);
	fail_unless(cmd1 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd1->cmdID = 1;
	
	smlAssemblerStartCommand(assm, NULL, cmd1, &error);
	
	SmlCommand *cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "uid", "data", 5, SML_ELEMENT_TEXT_VCARD, &error);
	fail_unless(cmd2 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd2->cmdID = 2;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd1, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd2);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	cmd2 = smlCommandNewPartialChange(SML_CHANGE_ADD, "uid", "data", 200, 5, SML_ELEMENT_TEXT_VCARD, &error);
	fail_unless(cmd2 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd2->cmdID = 2;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd1, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd2);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "uid", "data", 5, SML_ELEMENT_TEXT_VCARD, &error);
	fail_unless(cmd2 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd2->cmdID = 2;
	
	fail_unless(!smlAssemblerStartCommand(assm, cmd1, cmd2, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlCommandUnref(cmd2);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	g_free(data);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlLocationUnref(loc);
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

Suite *assembler_suite(void)
{
	Suite *s = suite_create("Assembler");
	//Suite *s2 = suite_create("Assembler");
	
	create_case(s, "check_assembler_message_empty", check_assembler_message_empty);
	create_case(s, "check_assembler_message_only_header", check_assembler_message_only_header);
	create_case(s, "check_assembler_message_status", check_assembler_message_status);
	create_case(s, "check_assembler_message_cmd", check_assembler_message_cmd);
	create_case(s, "check_assembler_message_subcmd", check_assembler_message_subcmd);
	create_case(s, "check_assembler_message_several_subcmd", check_assembler_message_several_subcmd);
	create_case(s, "check_assembler_message_flush_open", check_assembler_message_flush_open);
	create_case(s, "check_assembler_out_of_order", check_assembler_out_of_order);
	create_case(s, "check_assembler_limit1", check_assembler_limit1);
	create_case(s, "check_assembler_limit2", check_assembler_limit2);
	create_case(s, "check_assembler_limit3", check_assembler_limit3);
	create_case(s, "check_assembler_limit4", check_assembler_limit4);
	create_case(s, "check_assembler_limit5", check_assembler_limit5);
	create_case(s, "check_assembler_devinf", check_assembler_devinf);
	create_case(s, "check_assembler_order", check_assembler_order);
	create_case(s, "check_assembler_reuse", check_assembler_reuse);
	create_case(s, "check_assembler_leak", check_assembler_leak);
	create_case(s, "check_assembler_stress", check_assembler_stress);
	create_case(s, "check_assembler_stress_limit", check_assembler_stress_limit);
	create_case(s, "check_assembler_stress_large_limit", check_assembler_stress_large_limit);
	
	create_case(s, "check_assembler_large_obj", check_assembler_large_obj);
	
	
	return s;
}

int main(void)
{
	int nf;

	Suite *s = assembler_suite();
	
	SRunner *sr;
	sr = srunner_create(s);
	srunner_run_all(sr, CK_NORMAL);
	nf = srunner_ntests_failed(sr);
	srunner_free(sr);
	return (nf == 0) ? 0 : 1;
}
