/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * vim:expandtab:autoindent:tabstop=4:shiftwidth=4:filetype=c:cindent:textwidth=0:
 *
 * Copyright (C) 2005 Dell Inc.
 *  by Michael Brown <Michael_E_Brown@dell.com>
 * Licensed under the Open Software License version 2.1
 *
 * Alternatively, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.

 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 */

// compat header should always be first header if including system headers
#define LIBSMBIOS_SOURCE
#include "smbios/compat.h"

#include <sstream>

#include "smbios/IMemory.h"
#include "SmbiosImpl.h"

// message.h should be included last.
#include "smbios/message.h"

using namespace smbiosLowlevel;
using namespace std;

#if defined(DEBUG_SMBIOS_STRATEGY)
#   define DCOUT(line) do { cout << line; } while(0)
#   define DCERR(line) do { cerr << line; } while(0)
#else
#   define DCOUT(line) do {} while(0)
#   define DCERR(line) do {} while(0)
#endif

namespace smbios
{
    // validate the smbios table entry point
    bool validateTableEntryPoint(
        const smbiosLowlevel::smbios_table_entry_point *tempTEP,
        bool strict,
        ParseExceptionImpl &parseException
        )
    {
        // This code checks for the following:
        //       entry point structure checksum : As per the specs
        //       smbios major version : As per the specs
        //       Intermediate anchor string : As per the specs
        //
        // This code does not check the following:
        //      intermediate checksum: the main checksum covers the
        //      entire area
        //          and should be sufficient, plus there is a
        //          possibility for
        //          BIOS bugs in this area.
        //
        //      minor version: according to the spec, this parser should
        //      work
        //          with any change in minor version. The spec says this
        //          parser
        //          will break if major version changes, so we check
        //          that.
        //

        bool retval = true;

        u8 checksum = 0;
        const u8 *ptr = reinterpret_cast<const u8*>(tempTEP);
        // don't overrun tempTEP if BIOS is buggy... (note sizeof() test here)
        //      added especially to deal with buggy Intel BIOS.
        for( unsigned int i = 0; (i < static_cast<unsigned int>(tempTEP->eps_length)) && (i < sizeof(*tempTEP)); ++i )
        {
            // stupid stuff to avoid MVC++ .NET runtime exception check for cast to different size
            checksum = (checksum + ptr[i]) & 0xFF;
        }

        ostringstream oss;
        oss << _("validation of table entry point failed") << endl;

        if(memcmp(tempTEP->dmi_anchor,"_DMI_",5)!=0) // Checking intermediate anchor string
        {
            oss << _("Intermediate anchor string does not match. anchor string: %(dmi_anchor)s") << endl;
            retval = false;  // validation failed
        }
        if(checksum!=0x00) // Checking entry point structure checksum
        {
            oss << _("Checksum check for table entry point should be zero. checksum: %(checksum)i ") << endl;
            retval = false;  // validation failed
        }
        if(tempTEP->major_ver!=0x02)     // Checking smbios major version
        {
            oss << _("Major version of table entry point should be 2: %(major_version)i") << endl;
            retval = false;  // validation failed
        }
        // Entry Point Length field is at least 0x1f.
        if(tempTEP->eps_length < 0x0f)
        {
            oss << _("Entry Point Length field is at least 0x1f : %(eps_length)i") << endl;
            retval = false;  // validation failed
        }

        parseException.setParameter("dmi_anchor", reinterpret_cast<const char *>(tempTEP->dmi_anchor));
        parseException.setParameter("checksum", static_cast<int>(checksum));
        parseException.setParameter("major_version", static_cast<int>(tempTEP->major_ver));
        parseException.setParameter("eps_length", static_cast<int>(tempTEP->eps_length));
        parseException.setMessageString(oss.str());

        return strict ? retval : 1;
    }





    bool SmbiosMemoryStrategy::getSmbiosTable(const u8 **smbiosBuffer, smbiosLowlevel::smbios_table_entry_point *table_header, bool strict)
    {
        bool ret = false;
        try
        {
            // allocates no mem
            DCERR("trying SmbiosMemoryStrategy" << endl);
            getSmbiosTableHeader(table_header, strict);

            // allocates mem, but frees on exception
            getSmbiosTableBuf(smbiosBuffer, *table_header);
            if(smbiosBuffer)
                    ret = true;
        }
        catch(const exception &e)
        {
            UNREFERENCED_PARAMETER(e); // avoid unused var warning when !DEBUG
            DCERR("got Exception: " << e.what() << endl);
        }

        DCERR("  ret for SmbiosMemoryStrategy is: " << ret << endl);
        return ret;
    }

    void SmbiosMemoryStrategy::getSmbiosTableBuf(const u8 **smbiosBuffer, smbiosLowlevel::smbios_table_entry_point table_header)
    {
        memory::IMemory *mem = memory::MemoryFactory::getFactory()->getSingleton();

        // new throws exception, no need to test.
        u8 *newSmbiosBuffer = new u8[table_header.table_length];
        try
        {
            mem->fillBuffer( newSmbiosBuffer, table_header.table_address, table_header.table_length );

            //delete old one, if necessary
            if( 0 != *smbiosBuffer )
            {
                memset (const_cast<u8 *>(*smbiosBuffer), 0, sizeof (**smbiosBuffer));
                delete [] const_cast<u8 *>(*smbiosBuffer);
                *smbiosBuffer = 0;
            }
        }
        catch(...)
        {
            delete [] newSmbiosBuffer;
            newSmbiosBuffer = 0;
            throw;
        }

        *smbiosBuffer = reinterpret_cast<const u8 *>(newSmbiosBuffer);
    }

    // allocates no memory, constructs no objects.
    // can raise an exception
    void SmbiosMemoryStrategy::getSmbiosTableHeader(smbiosLowlevel::smbios_table_entry_point *table_header, bool strict)
    {
        memory::IMemory *mem = memory::MemoryFactory::getFactory()->getSingleton();

        unsigned long fp = E_BLOCK_START;
        if( offset )
            fp = offset;

        ParseExceptionImpl parseException;
        if( offset )
        {
            parseException.setMessageString(_("SMBIOS Header not found at offset: %(offsetValue)i"));
            parseException.setParameter("offsetValue",offset);
        }
        else
        {
            parseException.setMessageString(_("SMBIOS Header not found in search."));
        }

        smbios_table_entry_point tempTEP;
        memset(&tempTEP, 0, sizeof(tempTEP));
        while ( (fp + sizeof(tempTEP)) < F_BLOCK_END)
        {
            mem->fillBuffer(
                reinterpret_cast<u8 *>(&tempTEP),
                fp,
                sizeof(tempTEP)
            );

            // search for promising looking headers
            if (memcmp (&tempTEP, "_SM_", 4) == 0)
            {
                if(validateTableEntryPoint(&tempTEP, strict, parseException))
                {
                    break;
                }
            }

            // previous if() would have broken out if we have a valid
            // table header. if offset is set, then we are not supposed
            // to be scanning through memory. We didn't find a table,
            // so there is nothing to do but raise an exception.
            if (offset)
                throw parseException; // previously set up.

            fp += 16;
        }

        // bad stuff happened if we got to here and fp > 0xFFFFFL
        if ((fp + sizeof(tempTEP)) >= F_BLOCK_END)
            throw parseException; // previously set up.

        // found it. set offset for future reference (no need to search.)
        offset = fp;
        memcpy( const_cast<smbios_table_entry_point *>(table_header), &tempTEP, sizeof(*table_header) );
    }
}
